Extension { #name : 'RcCounter' }

{ #category : 'Instance Creation' }
RcCounter class >> new [

"Returns a new RcCounter with an initial size of 30.  The counter can handle
 at least 10 user sessions, plus the global components, and default
 system sessions."

^ super new: 30

]

{ #category : 'Instance Creation' }
RcCounter class >> new: initialNumberOfUsers [

"Returns a new RcCounter with a size that supports initialNumberOfUsers.  The
 new RcCounter will handle more users, but will have subcomponents created
 for initialNumberOfUsers."

| newOne |
newOne := super new: initialNumberOfUsers + 1.
^ newOne initialize

]

{ #category : 'Private' }
RcCounter >> _calculateValue [

"Calculates the cumulative total of all session's counter values and place it
 in the RC value cache.  Returns the total."

| total sessionElement |
total := 0.
1 to: self size do: [ :i |
    sessionElement := self at: i.
    sessionElement ~~ nil
        ifTrue: [
            total := total + sessionElement value.
        ]
].
System rcValueCacheAt: #value put: total for: self.
^ total

]

{ #category : 'Support' }
RcCounter >> _getInvalidSessions [

"Returns an Array of Booleans, where the value is true if the session ID at the
 corresponding index is not active."

| invalidSessions rcSize allSess |
rcSize := self size.
invalidSessions := Array new: rcSize.
1 to: rcSize do: [ :i | invalidSessions at: i put: true ].

allSess := System currentSessions .
1 to: allSess size do: [ :n | | each |
  each := allSess at: n .
  (each < rcSize) ifTrue: [ invalidSessions at: each + 1 put: false ]
].
^ invalidSessions

]

{ #category : 'Testing' }
RcCounter >> _isLarge [

"Returns true if the object is implemented as a tree of private smaller objects"

^ true

]

{ #category : 'Private' }
RcCounter >> _privateDecrementBy: aNumber [

"Decrement the current session's counter value by the given amount."

| sessionElement systm |
aNumber _isNumber ifFalse:[ aNumber _validateClass: Number ].
sessionElement := self _rcSessionElementFor: self _thisSessionIndex .
(systm := System) _addRootObjectToRcReadSet: sessionElement.
sessionElement decrementValueBy: aNumber.

systm rcValueCacheIncrement: #value for: self by: 0 - aNumber .

]

{ #category : 'Private' }
RcCounter >> _privateIncrementBy: aNumber [

"Increment the current session's counter value by the given amount."

| sessionElement systm |
aNumber _isNumber ifFalse:[ aNumber _validateClass: Number ].
sessionElement := self _rcSessionElementFor: self _thisSessionIndex .
systm := System .
systm _addRootObjectToRcReadSet: sessionElement.
sessionElement incrementValueBy: aNumber.

systm rcValueCacheIncrement: #value for: self by: aNumber .

]

{ #category : 'Private' }
RcCounter >> _rcSessionElementFor: offset [

"Returns the RcCounterElement for the given offset.  If one does not exist,
 then create it."

| sessionElement |
(sessionElement := self at: offset) ifNil:[
   self _rcAt: offset . "put receiver in rcReadSet" 
   sessionElement := RcCounterElement new.
   sessionElement objectSecurityPolicy: self objectSecurityPolicy .
   self at: offset put: sessionElement.
   System redoLog addConflictObject: self for: self
] ifNotNil:[
  "no need to put receiver in rcReadSet"
].
^ sessionElement

]

{ #category : 'Private' }
RcCounter >> _resolveRcConflictsWith: conflictObjects [

"A logical write-write conflict has occurred on the receiver.  The objects that
 had the actual physical write-write conflicts are in the conflictObjects
 Array.  Returns whether the conflicts could be successfully resolved."

| sz |
(sz := conflictObjects size) == 0 ifTrue: [ ^ true ].  "no conflicts"

(sz = 1 and: [ (conflictObjects at: 1) == self ]) ifTrue: [ | offset mySessionElement |
   offset := self _thisSessionIndex .
   mySessionElement := self at: offset .
   self _selectiveAbort .

   offset := self _thisSessionIndex.  "grow root obj if needed"
   self at: offset put: mySessionElement.  "replay insertion"

   ^ true
].

 ^ false "no other replay possible"

]

{ #category : 'Private' }
RcCounter >> _sessionElementFor: offset [

"Returns the RcCounterElement for the given offset.  
 If one does not exist, then create it. Not Rc"

| sessionElement |
(sessionElement := self at: offset) ifNil:[
   sessionElement := RcCounterElement new.
   sessionElement objectSecurityPolicy: self objectSecurityPolicy .
   self at: offset put: sessionElement.
].
^ sessionElement

]

{ #category : 'Private' }
RcCounter >> _thisSessionIndex [

"Returns the session ID + 1 (add 1 since the first RcCounter slot is reserved
 as the global counter value)."

| sessionIndex |
sessionIndex := System session + 1.
(sessionIndex > self size)
    ifTrue: [ self size: sessionIndex ].
^ sessionIndex

]

{ #category : 'Support' }
RcCounter >> cleanupCounter [

"For sessions that are not logged in, centralize the individual session
 element's values to the global session element (at index 1).  This may cause
 concurrency conflict if another session performs this operation."

| value invalidSessions |
value := 0.
" get Array where unused session IDs are <true> "
invalidSessions := self _getInvalidSessions.

" set all session element's values to zero "
1 to: self size do: [ :i |
    (invalidSessions at: i) ifTrue:[ | sessionElement |
       sessionElement := self at: i .
       sessionElement ifNotNil:[
          value := value + sessionElement value.
          sessionElement value: 0
       ]
    ]
].
" now set the global value at index 1 "
(self _sessionElementFor: 1) incrementValueBy: value

]

{ #category : 'Decrementing' }
RcCounter >> decrement [

"Decrements the current session's counter value."

self _privateDecrementBy: 1

]

{ #category : 'Decrementing' }
RcCounter >> decrementBy: aNumber [

"Decrements the current session's counter value by the given amount."

self _privateDecrementBy: aNumber

]

{ #category : 'Decrementing' }
RcCounter >> decrementBy: aNumber ifLessThan: minNumber thenExecute: aBlock [

"Determine if decrementing the RcCounter by the given amount would cause the
 total value to fall below the minimum number.  If so, returns the result of
 executing the block; if not, performs the decrement and returns the receiver."

(self value - aNumber >= minNumber)
    " according to this transaction state, the decrement is valid "
    ifTrue: [ self _privateDecrementBy: aNumber ]
    ifFalse: [ ^ aBlock value ]

]

{ #category : 'Decrementing' }
RcCounter >> decrementIfNegative: aBlock [

"This is a convenience method to decrement the counter by one only if the
 counter's value does not become negative.  If it would become negative,
 execute the Block."

^ self decrementBy: 1 ifLessThan: 0 thenExecute: aBlock

]

{ #category : 'Incrementing' }
RcCounter >> increment [

"Increments the current session's counter value by 1."

self _privateIncrementBy: 1

]

{ #category : 'Incrementing' }
RcCounter >> incrementBy: aNumber [

"Increment the current session's counter value by the given amount."

self _privateIncrementBy: aNumber

]

{ #category : 'Initialization' }
RcCounter >> initialize [

"Create subcomponents for all available session IDs.  This can avoid
 initial concurrency conflict when many sessions modify the RcCounter
 for the first time."

 1 to: super _basicSize do: [ :i | self _sessionElementFor: i ]

]

{ #category : 'Accessing' }
RcCounter >> maxSessionId [

"Returns the maximum sessionId that can be used with this RcCounter."

^ super _basicSize

]

{ #category : 'Updating' }
RcCounter >> objectSecurityPolicy: anObjectSecurityPolicy [

"Assigns the receiver and subcomponents to the given security policy."

| sessionElement |
super objectSecurityPolicy: anObjectSecurityPolicy .
1 to: self size do: [ :i |
    sessionElement := self at: i.
    sessionElement ifNotNil:[ sessionElement objectSecurityPolicy: anObjectSecurityPolicy  ]
]

]

{ #category : 'Accessing' }
RcCounter >> value [

"Returns the cumulative total of all session's counter value."

| val |
val := System rcValueCacheAt: #value for: self otherwise: nil.
val ifNil:[ val := self _calculateValue ].
^ val

]
