Extension { #name : 'RcBtreePlusRoot' }

{ #category : 'RC Compatibility' }
RcBtreePlusRoot >> _addBtreePlusNodeToRcReadSet: aBtreePlusNode [
  "Use from hard-to-reach methods where objects need to be added to rc read set."

  System _addRootObjectToRcReadSet: aBtreePlusNode.
  System redoLog addConflictObject: aBtreePlusNode for: self.

]

{ #category : 'RC Accessing' }
RcBtreePlusRoot >> _btreeAt: aKey put: aValue root: rootObject logging: aBoolean [
  "Insert the key/value/rootObject tuple into the root B-tree node.  Must check to see if a
 split occurred.  If so, create a new root node and put the old root and the
 new split node into the new root."

  ^ self
    _btreeAt: aKey
    put: aValue
    root: rootObject
    logging: aBoolean
    selectiveAbortSet: nil

]

{ #category : 'RC Accessing' }
RcBtreePlusRoot >> _btreeAt: aKey put: aValue root: rootObject logging: aBoolean selectiveAbortSet: selectiveAbortSetOrNil [
  "Insert the key/value/rootObject tuple into the root B-tree node.  Must check to see if a
 split occurred.  If so, create a new root node and put the old root and the
 new split node into the new root."

  | returnNode node oldBTreeRootNode |
  returnNode := self btreeRootNode
    at: aKey
    put: aValue
    root: rootObject
    selectiveAbortSet: selectiveAbortSetOrNil.
  " see if a split occurred "
  returnNode == btreeRootNode
    ifFalse: [
      " returnNode is the second half of the split "
      self _addBtreePlusNodeToRcReadSet: returnNode.
      " create the new root node "
      node := self _createNewRootNodeForSplit: returnNode selectiveAbortSet: selectiveAbortSetOrNil.
     self _addBtreePlusNodeToRcReadSet: node.
     btreeRootNode _selectiveAbort: btreeRootNode ifNotIn: selectiveAbortSetOrNil.
     oldBTreeRootNode := btreeRootNode.
     btreeRootNode := node.
     "write dropped node to force a commit conflict and force replay, in case
      another session makes mods to the node"
     oldBTreeRootNode numElements: oldBTreeRootNode numElements ].
  System _addRootObjectToRcReadSet: self.
  aBoolean
    ifTrue: [ self _logBtreeAt: aKey put: aValue root: rootObject ].
  ^ true

]

{ #category : 'Accessing' }
RcBtreePlusRoot >> _createNewRootNodeForSplit: returnNode selectiveAbortSet: selectiveAbortSetOrNil [
  "Create a new root node and put the old root and the new split node into the new root."
  " returnNode is the second half of the split "

  |  node |
  " create the new parent node "
  node := btreeRootNode parentNodeClass new
    rootNode: self;
    objectSecurityPolicy: self objectSecurityPolicy;
    lastValue: returnNode lastValue;
    lastRoot: returnNode lastRoot.
  " insert the first half (the original root) "
  node
     _insertKey: (btreeRootNode at: btreeRootNode _lastKeyIndex)
     value: btreeRootNode
     root: (btreeRootNode at: btreeRootNode _lastRootIndex)
     atIndex: 1
     selectiveAbortSet: selectiveAbortSetOrNil.
  " insert the second half "
  node
    _insertKey: (returnNode at: returnNode _lastKeyIndex)
    value: returnNode
    root: (returnNode at: returnNode _lastRootIndex)
    atIndex: node entrySize + 1
    selectiveAbortSet: selectiveAbortSetOrNil.
  ^ node

]

{ #category : 'RC Logging' }
RcBtreePlusRoot >> _logBtreeAt: aKey put: aValue root: rootObject [
  "Create a log entry for adding the given key/value/root."

  | logEntry |
  logEntry := LogEntry new.
  logEntry
    receiver: self
    selector: #'_btreeAt:put:root:logging:selectiveAbortSet:'
    argArray: {aKey.  aValue.  rootObject.  false.  nil}.
  System redoLog addLogEntry: logEntry

]

{ #category : 'RC Logging' }
RcBtreePlusRoot >> _logBtreeRemoveKey: aKey value: aValue root: rootObject [
  "Create a log entry for removing the given key/value/root."

  | logEntry |
  logEntry := LogEntry new.
  logEntry
    receiver: self
    selector: #'_removeKey:value:root:logging:selectiveAbortSet:'
    argArray: {aKey.  aValue.  rootObject.  false.  nil}.
  System redoLog addLogEntry: logEntry

]

{ #category : 'RC Removing' }
RcBtreePlusRoot >> _removeKey: aKey value: aValue root: rootObject logging: aBoolean [
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

  ^self _removeKey: aKey value: aValue root: rootObject logging: aBoolean selectiveAbortSet: nil

]

{ #category : 'RC Removing' }
RcBtreePlusRoot >> _removeKey: aKey value: aValue root: rootObject logging: aBoolean selectiveAbortSet: selectiveAbortSetOrNil [
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

  (btreeRootNode removeKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil)
    ifFalse: [ ^ false ].
  self updateBtreeRootNodeOnRemoval: selectiveAbortSetOrNil.
  aBoolean
    ifTrue: [ self _logBtreeRemoveKey: aKey value: aValue root: rootObject ].
   ^ true

]

{ #category : 'Private' }
RcBtreePlusRoot >> _resolveRcConflictsWith: conflictObjects [

"A logical write-write conflict has occurred on the receiver.  The objects that
 had the actual physical write-write conflicts are in the conflictObjects
 Array.  Selectively abort the receiver and then attempt to replay the
 operations maintained in the System redo log.  Returns true if all the
 operations could be replayed; otherwise returns false."

"inline implementation of _abortAndReplay: to abort old btrees before
 replaying log entries."

| redoLog logEntries selectiveAbortSet |
redoLog := System _redoLog.

" if no log entries to replay, then we're done "
redoLog == nil ifTrue: [ ^ false ].
logEntries := redoLog getLogEntriesFor: self .
logEntries == nil ifTrue:[ ^ false ].

" cannot perform selective abort if receiver has a dependency tag "
self _hasDependencyList ifTrue: [ ^ false ].

" explicitly abort the btrees that made up original changes, since we need to abort entire tree, before replaying, since
  nodes may split differently."
selectiveAbortSet := IdentitySet new.
1 to: conflictObjects size do:[ :j | | co |
  co := conflictObjects at: j.
  co _selectiveAbort.
  selectiveAbortSet add: co.
].

"abort all of the btree nodes involved in the transaction"
redoLog conflictObjects keysAndValuesDo:[ :conflictObject :redoObject |
  redoObject == self
        ifTrue: [
                conflictObject _selectiveAbort.
                selectiveAbortSet add: conflictObject ].
].

" refresh the state of the receiver "
self _selectiveAbort.

" replay all operations on the receiver"
1 to: logEntries size do:[:j | | logEnt argArray |
  logEnt := logEntries at: j .
  argArray := logEnt argArray.
  argArray at:  argArray size put: selectiveAbortSet. "install the selectiveAbortSet into the #selectiveAbortSet arg slot (last slot)"
  logEnt redo ifFalse:[
    ^ false  "redo failed, cannot commit"
  ].
].
^ true

]

{ #category : 'Accessing' }
RcBtreePlusRoot >> btreeAt: aKey put: aValue root: rootObject [
  "Insert the key/value pair into the root B-tree node.  Must check to see if a
 split occurred.  If so, create a new root node and put the old root and the
 new split node into the new root."

  ^ self _btreeAt: aKey put: aValue root: rootObject logging: true

]

{ #category : 'Removing' }
RcBtreePlusRoot >> removeKey: aKey value: aValue root: rootObject [
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

^ self _removeKey: aKey value: aValue root: rootObject logging: true

]

{ #category : 'Removing' }
RcBtreePlusRoot >> updateBtreeRootNodeOnRemoval: selectiveAbortSetOrNil [
  "If the current root node is an interior node of size one, replace the current root node with its
   single child"

  | oldBTreeRootNode |
  (btreeRootNode numElements == 1 and: [ btreeRootNode isLeaf not ])
    ifTrue: [
      btreeRootNode _selectiveAbort: btreeRootNode ifNotIn: selectiveAbortSetOrNil.
     oldBTreeRootNode := btreeRootNode.
      btreeRootNode := btreeRootNode
        at: btreeRootNode _lastKeyIndex - btreeRootNode keyIndexOffset.
     "write dropped node to force a commit conflict and force replay, in case
      another session makes mods to the node"
     oldBTreeRootNode numElements: oldBTreeRootNode numElements ].
  System _addRootObjectToRcReadSet: self

]
