"
ProfMonitor is a tool for taking snapshots of the execution stack
 at designated moments.  When done monitoring, the results are collected
 and formed into a report showing classes, method selectors, and hit rates.
 Execution stack snapshots can be gathered at either regular time intervals
 or programmatically by using System profMonSample.  Snapshots can also
 be taken during object creation by setting traceObjectCreation to true.
 If traceObjectCreation is true, the number of samples may be dominated
 by object creation samples rather interval based samples.
 
 The interval is the number of nanoseconds between sample points, which 
 must be either zero (for programmatic tracking) or >= 1000 nanoseconds. 
 Methods are provided to specify interval in milliseconds 
 (interval: variants) or in nanoseconds (intervalNs: variants) .

 See the classmethod  computeInterval:  for determining the argument 
 to pass to the intervalNs: keyword.

 Intervals less than 10 milliseconds use high resolution sampling
 and are implemented by reading a high resolution clock at each
 point where interrupts would be checked, (entries to non-trivial 
 non-primitive method, backward branches, and each CALL_PRIMITIVE bytecode).
 For the CALL_PRIMITIVE bytecodes , each CALL_PRIMITIVE in a loaded
 method is transformed to a PROFILING_CALL_PRIMITIVE (or to equivalent
 native code) at start of profiling and the transformation reversed 
 at end of profiling.  Methods loaded during profiling will have the
 transformation done at method load time.
 When using high resolution sampling, code executes at about 10%
 of normal execution speed.

 Intervals >= 10 milliseconds use the legacy timer interrupt implementation
 which has much less effect on execution performance.

 When the interval is zero, samples are not gathered based on clock time,
 but whenever System profMonSample is called.  Developers can use this
 to generate profiling information on specific methods of interest by
 adding calls to System profMonSample to these methods.
 
 The sampleDepth is the number of levels from the top of the stack to
 sample. This controls the depth of the stack for both method calls and
 object creation; it should be at least 20.
 
 The results are recorded in an instance of GsFile for the gem session,
 in binary machine dependent form.  The file is created with a name of 
 gem<processId>.log in one of the following locations:
   1.  The directory referenced by the environment variable $GEMSTONE_CHILD_LOG.
   2.  The current directory.
   3.  /tmp

 For 100000 samples, GEM_TEMPOBJ_CACHE_SIZE should be set to at least 300MB 
 to avoid AlmostOutOfMemory errors during the analysis phase.

 Instances of ProfMonitor can be committed to save the result of runBlock:
 for later analysis. An instance of ProfMonitor embeds a GsFile reference
 to the file of profiling data which can be reopened later for analysis.

 An instances of ProfMonitor which has been sent gatherResults 
 can occupy many MB or GB of memory and committing lots of them could
 cause significant repository growth.  Send removeResults to an instance
 before committing and then reexecute gatherResults to avoid that growth.


Instance Variables
------------------

file
  An instance of GsFile used to record sampling information while profiling
  is active.  The file contains binary data in machine dependent form.

interval
  The interval between sample points, in nanoseconds of CPU time.
  Must be either zero for programmatic sample gathering or >= 1000.

traceObjCreation 
  A SmallInteger with the following bits that control the profiling
    16r1  1=trace object creation enabled
    16r2  1=elapsed time sampling , 0= cpu time sampling(the default)
    16r4  for elapsed time sampling, tally object faults
    16r8  for elapsed time sampling, tally page faults
    16r10  for elapsed time sampling, tally eden bytes used
    16r20  for elapsed time sampling, tally temp obj memory gc time
    16r40  for elapsed time sampling, tally user defined stat 1 (not implemented)
    16r80  for elapsed time sampling, tally user defined stat 2 (not implemented)

numSamples 
  The total number of samples taken

results
  Holds collected, processed snapshot information in instances of a
  class that is private to ProfMonitor.

sampleDepth 
  The number of levels from top of stack to sample.

startTime 
  Starting Real or CPU time in milliseconds.

endTime
  Ending Real or CPU time in milliseconds.

"
Class {
	#name : 'ProfMonitor',
	#superclass : 'Object',
	#instVars : [
		'file',
		'interval',
		'results',
		'sampleDepth',
		'startTime',
		'endTime',
		'traceObjCreation',
		'numSamples'
	],
	#gs_reservedoop : '93441',
	#category : 'PerformanceProfiling'
}

{ #category : 'Estimating' }
ProfMonitor class >> computeInterval: unprofiledCpuTimeSeconds [
 "Returns an estimated profiling interval in nanoseconds for an execution
  taking the specified amount of cpu time, to yield
  approximately 100000 samples.  The number of samples might vary
  by +/- 30% from the target.  The profiled execution will take much
  longer to execute than the same non-profiled execution.

  The 10.0/100000 portion of this computation is an approximation.
  A more sophisticated algorithm might use a different value for
  <= 1 second cpu  than it does for 20 seconds of cpu. There might
  also be some variation in the constant needed between very slow
  and very fast cpu hardware.  As the total cpu time gets smaller,
  the overhead of profiling , plus limited resolution of the software
  clocks could require some non-linearity in the algorithm.
  A more sophisticated algorithm might calibrates itself to a specific
  real or virtualized cpu  and store calibration data in SessionTemps current .
"
 | ns |
 ns := (10.0 * unprofiledCpuTimeSeconds / 100000 / 1.0e-9 + 1) asInteger .
 ns < 1000 ifTrue:[ ns := 1000 ].
 ^ ns

]

{ #category : 'Defaults' }
ProfMonitor class >> defaultIntervalNs [

"Returns the number of CPU nanoseconds used for a monitoring interval if no
 interval is given.
 This value produces about 100000 samples for an unprofiled CPU time of 5 seconds."

 ^ 500000 "500 micro seconds"

]

{ #category : 'Reporting' }
ProfMonitor class >> defaultReports [
  ^{#samples . #stackSamples . #senders . #objCreation}

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock [

"Profile the execution of a block and return a formatted report of the results.
 An interval of 1 millisecond is used, and the results are reported down
 methods that use 3% of the total time"

^self monitorBlock: aBlock
       downTo: 0.03
       intervalNs: self defaultIntervalNs
       options: #()

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock downTo: threshold [

"Profile the execution of a block and return a formatted report of the results.
 An interval of 1 millisecond is used. The threshold argument should be
 a Float 0.01-0.99, or a SmallInteger > 1, specifying the percent of time or
 the number of hits, below which to exclude methods from the report."

 ^self monitorBlock: aBlock
       downTo: threshold
       intervalNs: self defaultIntervalNs
       options: #()

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock downTo: threshold interval: millisecondsPerSample [

"Profile the execution of a block and return a formatted report of the results.
 The millisecondsPerSample argument gives the CPU time interval between
 samples in milliseconds.  The threshold argument should be a Float 0.01-0.99,
 or a SmallInteger > 1, specifying the percent of time or the number of hits,
 below which to exclude methods from the report."

^self monitorBlock: aBlock
       downTo: threshold
       intervalNs: (millisecondsPerSample * 1000000)
       options: #()

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock downTo: threshold intervalNs: nanosecondsPerSample [

"Profile the execution of a block and return a formatted report of the results.
 The nanosecondsPerSample argument gives the CPU time interval between
 samples in nanoseconds. The threshold argument should be a Float 0.01-0.99,
 or a SmallInteger > 1, specifying the percent of time or the number of hits,
 below which to exclude methods from the report."

 ^self monitorBlock: aBlock
       downTo: threshold
       intervalNs: nanosecondsPerSample
       options: #()

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock downTo: threshold intervalNs: nanosecondsPerSample options: anArray [
 "Profile the execution of a block and return a formatted report of the results.
  threshold should be a Float in the range 0.01 to 0.99 specifying a percent of
  total time below which to exclude methods from the report, or a SmallInteger > 1
  specifying a sample count below which to exclude methods from the report.
  The nanosecondsPerSample argument gives the CPU time interval between samples.
  anArray is an Array of Strings or Symbols containing at most one of
    #objFaults #pageFaults #edenUsage #gcTime #objCreation
  and optionally
    #cpu or #real .
  #cpu or #real (if included) specifies real or cpu time sampling, otherwise the i
  default depends on the type of profiling.
    If profiling type is not specified, the default is #cpu.
    #gcTime defaults to #cpu, and is in units of milliseconds  .
    #objFaults and #pageFaults default to #real, and are in units of faults .
    #edenUsage defaults to #reali, and is in units of bytes.
    #objCreation defaults to #cpu, and produces additional reports."

  | inst result |
  inst := self basicNew initializeNoFile .
  inst setOptions: anArray .
  threshold _validateClasses: { Float . SmallInteger }.
  threshold < 0.01 ifTrue:[ ArgumentError signal:'threshold must be > 0.01' ].
  inst _createFile: self newProfileFileName;  intervalNs: nanosecondsPerSample .
  inst runBlock: aBlock.
  result := inst reportDownTo: threshold reports: self defaultReports .
  inst removeResults ; removeFile .
  ^ result

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock downTo: threshold intervalNs: nanosecondsPerSample reports: reportArray [
 "Profile the execution of a block and get a report of the result.
  threshold should be a Float in the range 0.01 to 0.99 specifying a percent of total time below
   which to exclude methods from the report, or a SmallInteger > 1 specifying a sample count below
  which to exclude methods from the report.
  The nanosecondsPerSample argument gives the CPU time interval between samples.
  anArray is an Array of Strings or Symbols.
 reportsArray can include any or all of
      #samples #stackSamples #senders #objCreation #tree #objCreationTree
  specifying which reports to include and the desired order.
  specifying an obj creation report enables object creation; specifying a tree
  report uses ProfMonitorTree."

| inst result |
inst := ((reportArray includes: #tree) or: [reportArray includes: #objCreationTree])
	ifTrue: [(Globals at: #ProfMonitorTree) basicNew initializeNoFile .]
	ifFalse: [self basicNew initializeNoFile .].

((reportArray includes:  #objCreation) or: [reportArray includes: #objCreationTree])
	ifTrue: [inst setOptions: {#objCreation} ].

  threshold _validateClasses: { Float . SmallInteger }.
  threshold < 0.01 ifTrue:[ ArgumentError signal:'threshold must be > 0.01' ].
  inst _createFile: self newProfileFileName;  intervalNs: nanosecondsPerSample .
  inst runBlock: aBlock.
  result := inst reportDownTo: threshold reports: reportArray .
  inst removeResults ; removeFile .
  ^ result

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock intervalNs: nanosecondsPerSample [

"Profile the execution of a block and return a formatted report of the results.
 The nanosecondsPerSample argument gives the CPU time interval
 between samples, and the results are reported down to methods that
 use 3% of the total time"

^ self monitorBlock: aBlock
       downTo: 0.03
       intervalNs: nanosecondsPerSample
       options: #()

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock intervalNs: nanosecondsPerSample options: anArray [
 "Profile the execution of a block and return a formatted report of the results.
  The report includes sampled items each of which is more than 3% of the total.
  The nanosecondsPerSample argument gives the CPU time interval between samples.
  anArray is an Array of Strings or Symbols containing at most one of
  #objFaults #pageFaults #edenUsage #gcTime #objCreation   and optionally
  #cpu or #real .
  For details see commnets in monitorBlock:downTo:intervalNs:options:."

^ self monitorBlock: aBlock
       downTo: 0.03
       intervalNs: nanosecondsPerSample
       options: anArray

]

{ #category : 'Quick Profiling' }
ProfMonitor class >> monitorBlock: aBlock reports: reportsArray [

"Profile the execution of a block and return a formatted report of
 the results. An interval of 1 millisecond is used, and the results
 are reported down to 3% of the number of hits per method.
 reportsArray can include any or all of
      #samples #stackSamples #senders #objCreation #tree #objCreationTree
  specifying which reports to include and the desired order.
  specifying an obj creation report enables object creation; specifying a tree
  report uses ProfMonitorTree."

^self monitorBlock: aBlock
       downTo: 0.03
       intervalNs: self defaultIntervalNs
       reports: reportsArray

]

{ #category : 'Instance Creation' }
ProfMonitor class >> new [

"Returns a new profiler with default initialization."

^ self basicNew initialize

]

{ #category : 'Private' }
ProfMonitor class >> newProfileFileName [

"Returns a String to be used as the name of a temporary file which will
 contain profiling data.  The file name will be gem<processId>_<randomInt>.pro,
 in the directory defined by the environment variable $GEMSTONE_CHILD_LOG,
 the current directory or /tmp if neither of the above can be created."

| result pidStr aFile env filename randInt |
pidStr := (System gemVersionAt: #processId ) printString.
filename := (String withAll: 'gem').
randInt := HostRandom new integerBetween:100000 and:999999 .
filename addAll: pidStr; add: $_ ; addAll: randInt asString; addAll: '.pro'.

env := (System gemEnvironmentVariable: 'GEMSTONE_CHILD_LOG').
env == nil
  ifTrue: [  result := String withAll: './'.]
  ifFalse: [ result := String withAll: env. result addAll: '/'].
result addAll: filename.

aFile := GsFile openWriteOnServer: result.
aFile ifNotNil:[
  aFile close .
  ^ result
  ].
result := (String withAll: '/tmp/').
result addAll: filename.
^ result

]

{ #category : 'Instance Creation' }
ProfMonitor class >> newWithFile: fileName [

"Creates a new profiler with the given output file name and default monitoring
 interval."

^self newWithFile: fileName intervalNs: self defaultIntervalNs

]

{ #category : 'Instance Creation' }
ProfMonitor class >> newWithFile: fileName interval: millisecondsPerSample [

"Creates a new profiler with the specified output file name and monitoring
 interval"

| inst |
(inst := self basicNew) initializeNoFile ;
   _createFile: fileName;
   interval: millisecondsPerSample .
^ inst

]

{ #category : 'Instance Creation' }
ProfMonitor class >> newWithFile: fileName intervalNs: nanosecondsPerSample [

"Creates a new profiler with the specified output file name and monitoring
 interval"

| inst |
(inst := self basicNew) initializeNoFile ;
   _createFile: fileName;
   intervalNs: nanosecondsPerSample .
^ inst

]

{ #category : 'Profiling' }
ProfMonitor class >> profileOn [

"Creates a default instance, starts it monitoring, and returns it.
 To turn off profiling, send the message profileOff to the instance."

| inst |

inst := self new .
inst startMonitoring.
^inst.

]

{ #category : 'Profiling' }
ProfMonitor class >> runBlock: aBlock [

"Profile the execution of a block and return an instance of the receiver.
 The default sample interval will result in about 25000 samples per
 second of CPU time.

 Use gatherResults followed by reporting methods to view results."

| inst |
inst := self basicNew initializeNoFile .
inst _createFile: self newProfileFileName;
             intervalNs: self defaultIntervalNs  .
inst runBlock: aBlock .
^ inst

]

{ #category : 'Profiling' }
ProfMonitor class >> runBlock: aBlock intervalNs: nanosecondsPerSample [

"Profile the execution of a block and return an instance of the receiver.
 The nanosecondsPerSample argument gives the CPU time interval between
 samples.

 Use gatherResults followed by reporting methods to view results."

| inst |
inst := self basicNew initializeNoFile .
inst _createFile: self newProfileFileName;  intervalNs: nanosecondsPerSample .
inst runBlock: aBlock .
^ inst

]

{ #category : 'Private' }
ProfMonitor >> _createFile: fileName [

"Creates the sampling file for writing with the specified file name."
| f |
(f := file) ifNotNil:[
  f isOpen ifTrue:[ f close ].
  file := nil .
].
f := GsFile open: fileName mode: 'wb+' onClient: false .
f ifNil:[
  Error signal: 'Error creating profiling data file ', fileName, ', ', GsFile serverErrorString .
].
file := f .

]

{ #category : 'Private' }
ProfMonitor >> _intervalHeaderString [

| res divisor units tmp |
res := 'monitoring interval:  ' copy.
interval < 1000 ifTrue:[ "currently illegal argument"
  res addAll: interval asString ; addAll: ' ns'.  ^ res  ].

interval >= 1000000
  ifTrue: [divisor := 1000000 . units := ' ms']
  ifFalse: [divisor := 1000 . units := ' us' ].

res addAll: (interval // divisor ) asString.
tmp := (interval \\ divisor ).
tmp > 0 ifTrue:
   [res add: $. ; addAll: (interval \\ divisor ) asString].
res addAll: units .
res add: ' ,  ' ; add: self numberOfSamples asString ; add: ' samples' .
^ res

]

{ #category : 'Private' }
ProfMonitor >> _objCreationReport [

  | classes minTally tallySet rpt nonrep nonreptallies meths counts |

  minTally := self dynamicInstVarAt: #reportThreshold.
  rpt := String new .
  rpt add:'================'; lf ;
      add:'OBJECT CREATION REPORT'; lf;
      addAll: ' tally  class of created object'; lf.
  (sampleDepth > 1)
    ifTrue: [ rpt addAll: '           call stack'; lf ]
    ifFalse: [ rpt addAll: '           callers'; lf ].

  nonrep := 0.  "non reported classes (below threshold)"
  nonreptallies := 0.

  counts := IdentityKeyValueDictionary new.
  (results at: 2) keysAndValuesDo: [:k :d | | cls tally |
    cls := k at: 1.
    tally := (counts at: cls ifAbsent: [ counts at: cls put: 0 ]).
    d valuesDo: [:d2 |
      d2 valuesDo: [:e |
        e cclass == nil  ifFalse: [ tally := tally + (e tally) ]]].
    counts at: cls put: tally ].
  classes := Array new.
  counts keysAndValuesDo: [:k :v | classes add: (Array with: k with: v)].
  classes := classes asSortedCollection: [:x :y | (x at: 2) > (y at: 2)].
  classes := classes collect: [:x | x at: 1].

  classes do: [:class |

    (counts at: class) < minTally ifTrue: [
      nonrep := nonrep + 1.
      nonreptallies := nonreptallies + (counts at: class).
      ]
    ifFalse:[
      rpt
        addAll: '------  -----------------------------------------'; lf.
	  rpt lf ;
	    addAll: ((counts at: class) asString width: -6); addAll: '  ' ;
	    addAll: (class name asString width: 12) ; lf.

    tallySet := IdentitySet new.
    (results at: 2) keysAndValuesDo: [:k :d |
      ((k at: 1) = class) ifTrue: [ | sender |
        sender := (k at: 2).
        d valuesDo: [:d2 |
          d2 valuesDo: [:e |
               e cclass == nil  ifFalse: [
                 tallySet add: e]]]]].

    meths := tallySet sortDescending: 'tally'.
    meths do: [:each | | aMeth |
      aMeth := each cmethod .
      (aMeth _class == GsNMethod _or:[ aMeth == #GCI ]) ifTrue:[

        | aClass |
        aClass := aMeth .
        (sampleDepth > 1) ifTrue: [
          rpt addAll: ' - - -  - - - - - - - - - - - - - - - - - - - - -'; lf ].
        rpt add: '         ';
            add: (each tally asString width: -6);
            add: '  ';
            add: (each asStringWidth: 12) ; lf.
        self _objCreationReportScanParents: each level: 0
             to: rpt seen: Set new ]]]].

  nonrep > 0 ifTrue: [
    rpt lf; lf;
        addAll: '------  -----------------------------------------'; lf;
        addAll: (nonreptallies asString width: -6); addAll: '  ' ;
        addAll: 'Instances of ' ;
        addAll: nonrep asString ;
        addAll: ' other classes'; lf
  ].

  ^rpt

]

{ #category : 'Private' }
ProfMonitor >> _objCreationReportScanParents: entry level: level to: rpt seen: visited [

  "Scan the parents of a given entry, printing them out indented
   according to how deep in the scan we are.  Stop when we hit an
   entry we've already visited. "

  | parents tallies lf |

  lf := Character lf.
  (parents := entry parents) ifNotNil: [
    tallies := entry parentTallies.
    parents := (parents sortDescending: 'tally') asArray.
    parents do: [:parent | |  visit |
      visit := Array with: parent with: entry.
      (visited includes: visit)
      ifFalse: [
        rpt add: '             '.
        level timesRepeat: [ rpt add: '  ' ].
        rpt
          add: ((parent childTallies at: entry otherwise: -1)
               asString width: -4);
          add: '  ';
          add: (parent asStringWidth: 12) ; lf.
          visited add: visit.
               self _objCreationReportScanParents: parent
                   level: (level + 1) to: rpt seen: visited ]]].

]

{ #category : 'Private' }
ProfMonitor >> _openFileAppend [

"Reopens the sampling file for writing."

| f |
(f := file ) ifNil:[ Error signal: 'cannot append to nil file'].
f isOpen ifFalse:[ | fileName |
  fileName := f pathName .
  f := GsFile open: fileName mode: 'ab+' onClient: false .
  f ifNil:[
    Error signal: 'Error opening for append profiling data file ', fileName, ', ', GsFile serverErrorString .
  ].
  file := f
].

]

{ #category : 'Private' }
ProfMonitor >> _openFileRead [

"Reopens the sampling file for reading."

| f fileName |
(f := file ) ifNil:[ Error signal: 'Profiling result file is nil'].
"ignore any lost transient state on the file."
([ f isOpen ] on: Error do:[:ex | ex return: false]) ifTrue:[
  f seekFromBeginning: 0 .
] ifFalse:[
  fileName := f pathName .
  f := GsFile open: fileName mode: 'rb' onClient: false .
  f ifNil:[
    Error signal: 'Error opening profiling data file ', fileName, ', ', GsFile serverErrorString .
  ].
  file := f
].

]

{ #category : 'Private' }
ProfMonitor >> _readSampleFile [

"Returns the raw sample data generated by the monitoring code, for use
 by the output routines.

 result is an Array of the form
    { sampleArray . gcTime . objReads . objWrites . pageFaults . edenBytesUsed }

 sampleArray has the following format:
           aSmallInteger - depth of sampling for next section of Array,
           aSmallInteger - stats sample word
 optional: nil
 optional: aClass
           aGsNMethod 1
           aClass     1
           aGsNMethod 2
           aClass     2
           ...
           aGsNMethod N
           aClass     N

 where N = depth as specified by the initial aSmallInteger.

 The (aGsNMethod, aClass) pairs represent stack frames.
 The optional (nil, aClass) pair at the beginning indicate the sample
 was taken during the creation of an instance of the designated class.
"

| arr |
self _openFileRead.
arr := self _zeroArgPrim: 3.
file close .
(arr at: 1) ifNil:[ ^ nil ].
^ arr

]

{ #category : 'Private' }
ProfMonitor >> _reportHeader: nameStr [
  | rpt |
  rpt := String new.
  rpt add:'================'; lf ; add: nameStr; lf ;
       add: self _timeHeaderString; lf;
	 add: self _intervalHeaderString; lf;
	 add: self _reportThresholdHeaderString; lf.
  ^rpt

]

{ #category : 'Private' }
ProfMonitor >> _reportThresholdHeaderString [

  | pct rpt |
  rpt := String new.
  rpt add: 'report limit threshold: '.
  ((self dynamicInstVarAt: #reportThreshold) = 0 or: [(self dynamicInstVarAt: #totalTallies) = 0])
	ifTrue: [rpt add: ' 0 hits'.  ^rpt].
  pct := (((self dynamicInstVarAt: #reportThreshold) / (self dynamicInstVarAt: #totalTallies)) * 100.0) asStringUsingFormat: #(1 1 false).
  rpt
      add: (self dynamicInstVarAt: #reportThreshold) asString;
	add: ' hits / ';
	add: pct;
	add: '%'.
  ^rpt

]

{ #category : 'Private' }
ProfMonitor >> _samplingReport [

  | rpt lf nonrep nonrepTallies total ave pct meths tallySet nameStr
    thresh tlyName |

  thresh := self dynamicInstVarAt: #reportThreshold.
  total := self dynamicInstVarAt: #totalTallies.

  tallySet := results at: 1 .
  rpt := String new .
  nameStr := 'STATISTICAL SAMPLING RESULTS' .
  lf := Character lf.
  meths := tallySet sortDescending: 'tally'.

  rpt add: (self _reportHeader: nameStr).
  rpt add: (results at: 4"statsSummary"); lf .

  rpt lf;
    add:   (tlyName := self dynamicInstVarAt:#tallyName) ;
    addAll:          '    %   class and method name'; lf;

    addAll: '------   -----   --------------------------------------'; lf.

  total = 0 ifTrue:[
    rpt add: 'total '; add: tlyName; add:' is zero '; lf.
    ^ rpt
  ].

  nonrep := 0.  "non reported methods (below tally threshold.)"
  nonrepTallies := 0.
  meths do: [:each |
    each cmethod class == GsNMethod ifTrue:[ | tly |
      (tly := each tally) < thresh ifTrue: [
	nonrep := nonrep + 1.
	nonrepTallies := nonrepTallies + tly ]
      ifFalse: [
	pct := tly asFloat * 100.0 / total.
	rpt add: (tly  asString width: -6); addAll: '  ' ;
	    add: (pct asStringUsingFormat: #(-6 2 false)); addAll: '   ';
	    add: (each asStringWidth: 25) ; lf ]]].

  nonrep > 0 ifTrue: [
    pct := nonrepTallies asFloat * 100.0 / total.
    rpt addAll: (nonrepTallies asString width: -6); addAll: '  ' ;
        addAll: (pct asStringUsingFormat: #(-6 2 false)); addAll: '   ';
        addAll: nonrep asString ; addAll: ' other methods'; lf ].

  ave := total // (meths size max: 1).
  rpt addAll: (total asString width: -6); addAll: '  100.00   ';
    addAll: 'Total'; lf.
  ^rpt

]

{ #category : 'Private' }
ProfMonitor >> _sendersReport [
"Report formatting is:

  %     %                       Parent
 self total   total local  Method
 Time  Time    secs   %         Child

"

  | tallyThreshold tallySet nameStr rpt meths elapsedTime totalSamples methodMs pMap cMap |

  elapsedTime := self dynamicInstVarAt: #elapsedTime.
  tallyThreshold := self dynamicInstVarAt: #reportThreshold.
  tallySet := results at: 1 .
  nameStr := 'STATISTICAL METHOD SENDERS RESULTS' .
  rpt := String new .
  meths := tallySet sortDescending: 'total'.
  rpt add: (self _reportHeader: nameStr).

  totalSamples := results at: 3 .

  rpt lf;
    addAll: '     %       %                     Parent'; lf;
    addAll: '  self  total   total  local  Method'; lf;
    addAll: '  Time   Time      ms    %         Child'; lf;
    addAll: '------ ------  ------  -----  -----------'; lf.
  meths do: [:each | | aMeth |
    aMeth := each cmethod .
    (aMeth class == GsNMethod or:[ aMeth == #GCI ]) ifTrue:[
      each total < tallyThreshold ifFalse:[ | parents children |
        rpt lf .
        methodMs := each total * elapsedTime asFloat / totalSamples.
        pMap := each parentTallies.
        cMap := each childTallies.
        "PARENTS"
        (parents := each parents) ifNotNil: [
          parents do: [:parent | | pt |
            pt := pMap at: parent otherwise: 0.
            rpt
              add: '                 ';
              add: ((pt * elapsedTime asFloat / totalSamples)
                   asStringUsingFormat: #(-6 1 false)); space;
              add: ((pt * elapsedTime *100.0 / totalSamples / methodMs)
                   asStringUsingFormat: #(-6 1 false)); space; space;
              add: '     ';
              add: (parent asStringWidth: 12) ; lf ]].
        "SELF"
        rpt add:'= ' ;
          add: ((each tally * 100.0 / totalSamples)
               asStringUsingFormat: #(-6 1 false)); space;
          add: ((each total * 100.0  / totalSamples)
               asStringUsingFormat: #(-6 1 false)); space; space;
          add: (methodMs asStringUsingFormat: #(-6 1 false)); space;
          add: ((each tally * elapsedTime *100.0 / totalSamples / methodMs)
               asStringUsingFormat: #(-6 1 false)); space; space;
          add: (each asStringWidth: 12 );
          add: '   [oop ' , aMeth asOop asString ; add: $] ; lf .
        "CHILDREN"
        (children := each children) ~~ nil  ifTrue: [
          children do: [:child | | ct |
            ct := cMap at: child otherwise: 0.
            rpt
              add: '                 ';
              add: ((ct * elapsedTime asFloat / totalSamples)
                   asStringUsingFormat: #(-6 1 false)); space;
              add: ((ct * elapsedTime * 100.0 / totalSamples / methodMs)
                   asStringUsingFormat: #(-6 1 false)); space; space;
              add: '     ';
              add: (child asStringWidth: 12) ; lf ]].
        rpt addAll: '-----------------------------------------------------';
            lf ]]].
  ^rpt

]

{ #category : 'Private' }
ProfMonitor >> _stackReport [

  | rpt nonrep nonreptallies ave pct meths tallySet nameStr
    tlyName tallyThreshold total |

  tallyThreshold := self dynamicInstVarAt: #reportThreshold.
  total := self dynamicInstVarAt: #totalTallies.

  tallySet := results at: 1 .
  nameStr := 'STATISTICAL STACK SAMPLING RESULTS' .
  rpt := String new .

  meths := tallySet sortDescending: 'total'.

  rpt add: (self _reportHeader: nameStr).
  rpt add: (results at: 4"statsSummary"); lf .
  ((tlyName := self dynamicInstVarAt:#tallyName ) at: 1 equals:'tally') ifFalse:[
     rpt add: 'tallying:  ' ; add: tlyName ; lf .
  ].
  rpt lf;
    addAll: ' total       %   class and method name'; lf;
    addAll: '------   -----   --------------------------------------'; lf.
  total = 0 ifTrue:[
    rpt add: 'total '; add: tlyName; add:' is zero '; lf.
    ^ rpt
  ].
  nonrep := 0.  "non reported methods (below tally threshold."
  nonreptallies := 0.
  meths do: [:each | | aTotal |
    each cmethod class == GsNMethod ifTrue:[
      (aTotal := each total) < tallyThreshold ifTrue: [
	      nonrep := nonrep + 1.
	      nonreptallies := nonreptallies + each tally.
      ] ifFalse: [
	      pct := aTotal asFloat * 100.0 / total.
	      rpt add: (aTotal asString width: -6); addAll: '  ' ;
				add: (pct asStringUsingFormat: #(-6 2 false)); addAll: '   ';
			  add: (each asStringWidth: 25);  lf .
      ].
    ].
  ].

  nonrep > 0 ifTrue: [
    pct := nonreptallies asFloat * 100.0 / total.
    rpt addAll: (nonreptallies asString width: -6); addAll: '  ' ;
        addAll: (pct asStringUsingFormat: #(-6 2 false)); addAll: '   ';
        addAll: nonrep asString ; addAll: ' other methods'; lf
  ].

  ave := total // (meths size max: 1).
  rpt addAll: (total asString width: -6); addAll: '  100.00   ';
    addAll: 'Total'; lf.
  ^rpt

]

{ #category : 'Private' }
ProfMonitor >> _tallyInfo [
 | iv |
 "Returns an Array of the form { tallyNameString . statsWordExtractionBlock }."
 iv := traceObjCreation.
 "following must agree with VM C code in profileStatsWord() and code in setOptions:
    16r4  for elapsed time sampling, tally object faults
    16r8  for elapsed time sampling, tally page faults
    16r10  for elapsed time sampling, tally eden bytes used
    16r20  for elapsed time sampling, tally temp obj memory gc time
 "
 (iv bitAnd: 16r4) ~~ 0 ifTrue:[
       "10 bits of objWrites and 13 bits of objReads"
   ^ { ' objFaults' .
     [:w | ((w bitShift: -20) bitAnd: 16r3FF) + ((w bitShift: -30) bitAnd: 16r1FFF)]} ].

 (iv bitAnd: 16r8) ~~ 0 ifTrue:[
   ^ { 'pageFaults' . [:w | w bitAnd: 16r3FF ] "10 bits of page faults"}].

 (iv bitAnd: 16r10) ~~ 0 ifTrue:[  "17 bits of eden bytes used"
   ^ { 'edenBytes' . [:w | (w bitShift: -43) bitAnd: 16r1FFFF ] } ].

 (iv bitAnd: 16r20) ~~ 0 ifTrue:[  "10 bits of gc time"
   ^ { 'gcTimeMs ' . [:w | (w bitShift: -10) bitAnd: 16r3FF ]} ].

 ^   { 'tally    ' . nil } "assume just time sampling"

]

{ #category : 'Private' }
ProfMonitor >> _timeHeaderString [

  | rpt |
  rpt := String new.
  rpt
	add: 'elapsed ';
      add: (self sampling == #cpu ifTrue:['CPU'] ifFalse:['REAL']) ;
      add: ' time:    ';
      add: (self dynamicInstVarAt: #elapsedTime) asString; add: ' ms' .
  ^rpt

]

{ #category : 'Private' }
ProfMonitor >> _zeroArgPrim: opcode [

"opcode      function
   1		start profiling - returns receiver
   2		stop profiling - returns number of samples written to sample file
   3    read sample file - returns contents of file in an Array
   4    suspend high res sampling
   5    resume  high res sampling
   6    get and clear overrun info, result is a String or nil
   7    read clock
"
<primitive: 191>
self _primitiveFailed: #_zeroArgPrim: args: { opcode }

]

{ #category : 'Accessing' }
ProfMonitor >> endTime [

	^endTime

]

{ #category : 'Accessing' }
ProfMonitor >> fileName [

"Returns the name of the active file."

file ifNil:[ ^ nil ].
^ file name

]

{ #category : 'Private' }
ProfMonitor >> forChild: childEntry tallyParentMethod: aSender rcvrClass: rcvrClass in: tempDict
increment: anInteger [

"Private."

| children parentEntry d childTallies cTally parentTallies |
(aSender class == GsNMethod) ifFalse:[
  "ignore methods that were garbage collected and reused."
  ^ nil
  ].

"Add parent to child"
(d := tempDict at: aSender otherwise: nil ) ifNil:[
	d := IdentityKeyValueDictionary new.
	tempDict at: aSender put: d ].

(parentEntry := d at: rcvrClass otherwise: nil) ifNil:[
  parentEntry := ProfMonitorEntry newForMethod: aSender
                    receiverClass: rcvrClass.
  parentEntry total: anInteger .
  d at: rcvrClass put: parentEntry
] ifNotNil:[
  parentEntry incrementTotal: anInteger .
].

childTallies := parentEntry childTallies.
cTally := childTallies at: childEntry otherwise: 0.
childTallies at: childEntry put: (cTally + 1).

childEntry parents add: parentEntry.

"Add child to parent"
(children := parentEntry children ) ifNil:[
  children := IdentitySet new .
  parentEntry children: children .
].
children add: childEntry.

parentTallies := childEntry parentTallies.
cTally := parentTallies at: parentEntry otherwise: 0.
parentTallies at: parentEntry put: (cTally + 1).

^parentEntry

]

{ #category : 'Reporting' }
ProfMonitor >> gatherResults [
"If the receiver's file of sampling data has not been read, read it into memory,
 analyze the samples and store the results of analysis in the
 results instance variable of the receiver.
 After gatherResults has run, this instances' sample file is no longer needed
 and can be deleted with  ProfMonitor>>removeFile .

 For 100000 samples, GEM_TEMPOBJ_CACHE_SIZE should
 be set to at least 300MB to avoid AlmostOutOfMemory errors ."

  | resultSet mthDict objDict rawArray j sysRepos rawArraySiz totalSamples elapsedTime
    statBlock tlyInfo fileData statStr |

  "Now process the statistical sample file."
  results ifNotNil:[ ^ self "sample file was already read" ].

  mthDict := IdentityKeyValueDictionary new .
  objDict := KeyValueDictionary new .
  fileData := self _readSampleFile .
  fileData ifNil:[ ^ 'No profiling data available: check profiling parameters' ].

  rawArray := fileData at: 1 .
  rawArraySiz := rawArray size .
  rawArraySiz == 0 ifTrue:[
    results := nil .
    ^ self .
  ].
  sysRepos := SystemRepository .
  j := 1 .
  tlyInfo := self _tallyInfo .
  statBlock := tlyInfo at: 2 .
  self dynamicInstVarAt:#tallyName put: (tlyInfo at: 1).

  [j < rawArraySiz ] whileTrue:[
    | sample recursionSet aMethod rcvrClass depth statsWord sTly |
    sample := rawArray at: j .
    sample == sysRepos
      ifFalse: [ Error signal: 'valid sample file: bad frame header' ]
      ifTrue:[   "sample header"
        j := j + 1 .
        (sample := rawArray at: j) _isSmallInteger
           ifTrue:[ depth := sample.  j := j + 1 ]
           ifFalse:[ Error signal:'invalid sample file: invalid depth' ].
        (sample := rawArray at: j) _isSmallInteger
           ifTrue:[ statsWord := sample. j := j + 1 ]
           ifFalse:[ Error signal:'invalid sample file: invalid statsWord'].
        sTly := statBlock ifNil:[ 1 ] ifNotNil:[ statBlock value: statsWord ]
      ].
    recursionSet := IdentitySet new.
    aMethod := rawArray at: j .
    aMethod == true ifTrue:[ "an object creation sample"
      | newObjClass aSender rcvr key d d2 anEntry child |
      newObjClass := rawArray at: j + 1 .
      j := j + 2.
      aSender := rawArray at: j.
      rcvr := rawArray at: j + 1.
      rcvr := Object. "suppress polymorphism reporting, TODO add enable/disable"
      key := (Array with: newObjClass with: aSender).
      (d := objDict at: key otherwise: nil) ifNil: [
        d := KeyValueDictionary new.
        objDict at: key put: d. ].
      (d2 := d at: aSender otherwise: nil) ifNil: [
        d2 := IdentityKeyValueDictionary new.
        d at: aSender put: d2 ].

      (anEntry := d2 at: rcvr otherwise: nil) ifNil: [
        anEntry := ProfMonitorEntry newForClass: newObjClass
                        method: aSender receiverClass: rcvr.
        d2 at: rcvr put: anEntry ].
       anEntry incrementTally: 1 .
      "Now scan through call stack"
      child := anEntry.
      recursionSet add: child.
      1 to: depth-1 do: [:m |
        child ifNotNil: [
          aSender := rawArray at: j + (m * 2).
          rcvr := rawArray at: j + (m * 2) + 1.
          aSender ifNotNil: [ rcvr ifNotNil:[
            rcvr := Object .  "supress polymorphism reporting,"
                              "TODO add enable/disable "
            child := self forChild: child
                          tallyParentMethod: aSender
                          rcvrClass: rcvr
                          in: d increment: 1 .
            (recursionSet includes: child) ifTrue: [
              child recursed: true].
            recursionSet add: child ]]]]
    ] ifFalse:[ "statistical method execution sample"
     sTly > 0 ifTrue: [
      rcvrClass := rawArray at: j + 1.
      ((aMethod class == GsNMethod or:[ aMethod == #GCI ])
         and:[ rcvrClass isBehavior]) ifTrue:[
        | d anEntry |
        "method still exists or is a reenter marker"
        (d := mthDict at: aMethod otherwise: nil) ifNil: [
          d := IdentityKeyValueDictionary new.
          mthDict at: aMethod put: d. ].
        rcvrClass := Object .  "supress polymorphism reporting,"
                               "TODO add enable/disable "
        (anEntry := d at: rcvrClass otherwise: nil) ifNil:[
          anEntry := ProfMonitorEntry newForMethod: aMethod
                        receiverClass: rcvrClass.
          d at: rcvrClass put: anEntry ].
        anEntry incrementTally: sTly  .
        depth > 1 ifTrue:[ | child rcvr |
          child := anEntry.
          recursionSet add: child.
          0 to: depth-2 do: [:m |
            child ifNotNil: [ | aSender |
              aSender := rawArray at: j + (m * 2) + 2.
              aSender ifNotNil: [
              rcvr := rawArray at: j + (m * 2) + 3.
              rcvr ifNotNil: [
                rcvr := Object .  "supress polymorphism reporting,"
                                  "TODO add enable/disable "
                child := self forChild: child
                              tallyParentMethod: aSender
                              rcvrClass: rcvr
                              in: mthDict increment: sTly  .
                (recursionSet includes: child) ifTrue: [
                  child recursed: true].
                recursionSet add: child. ]]]]]]]].
    j := j + (depth * 2).
  ].

  (endTime >= startTime) ifTrue: [
    elapsedTime := endTime - startTime.
  ] ifFalse: [
    elapsedTime := endTime + (16rffffffff - startTime).
  ].

  "Consolidate results"
  resultSet := IdentityBag new.
  totalSamples := 0.
  mthDict valuesDo: [:d |
    d valuesDo: [ :anEntry |
      resultSet add: anEntry .
      anEntry cmethod class == GsNMethod ifTrue:[
        totalSamples := totalSamples + anEntry tally ]]].

  (statStr := String new) add: (fileData at: 5) asString ; add:' pageFaults  ';
    add: ((fileData at: 3"reads") + (fileData at: 4"wrts")) asString; add: ' objFaults  ';
    add: (fileData at: 2) asString; add:' gcMs  ' ;
    add: (fileData at: 6) asString; add:' edenBytesUsed' .
  results := { resultSet . objDict . totalSamples . statStr } .

]

{ #category : 'Private' }
ProfMonitor >> initialize [

"Private."

self initializeNoFile .
self _createFile: self class newProfileFileName .
file close .

]

{ #category : 'Private' }
ProfMonitor >> initializeNoFile [

"Private."

interval := self class defaultIntervalNs .
sampleDepth := 1000.
traceObjCreation := 0 . "object creation trace disabled, cpu time sampling"

]

{ #category : 'Accessing' }
ProfMonitor >> interval [

	^interval

]

{ #category : 'Updating' }
ProfMonitor >> interval: milliseconds [

"Assign the sampling interval of the receiver."

| ns |
milliseconds _isSmallInteger ifFalse:[ ns _validateClass: SmallInteger ].
ns := milliseconds * 1000000 .
self intervalNs: ns .

]

{ #category : 'Updating' }
ProfMonitor >> intervalNs: nanoseconds [
  "Assign the sampling interval of the receiver.
   See also class method  computeInterval:  .
  "

nanoseconds _isSmallInteger ifFalse:[ nanoseconds _validateClass: SmallInteger ].
((nanoseconds < 1000) and: [nanoseconds ~~ 0]) ifTrue:[
  OutOfRange new name:'' min: 1000 actual: nanoseconds;
   details: 'invalid sampling interval(nanoseconds)'; signal.
].
interval := nanoseconds

]

{ #category : 'Monitoring' }
ProfMonitor >> monitorBlock: aBlock [

"This method starts profiling, executes the block, and terminates profiling,
 and reads the sample file into memory .
 Use reportDownTo: to generate results"

self runBlock: aBlock .
self gatherResults ; removeFile .

]

{ #category : 'Sampling' }
ProfMonitor >> numberOfSamples [
  "Returns a SmallInteger, the number of stack samples actually taken during
   the profiled execution.  The result is zero until stopMonitoring
   has been sent at least once."
  ^ numSamples ifNil:[ 0 ]

]

{ #category : 'Private' }
ProfMonitor >> prepareForReportWithThreshold: thresholdArg [
  "Reporting uses some dynamic instance variables for the thresholds, time, and totals"

  | total  |
  results ifNil:[ self gatherResults; removeFile  ].
  results ifNil:[ ^self].

  total := 0.
  (results at: 1) do: [:each |
    each cmethod class == GsNMethod ifTrue:[
      total := total + each tally ]].
  self dynamicInstVarAt: #totalTallies put: total.
  self dynamicInstVarAt: #reportThreshold put: (thresholdArg < 1.0
	ifTrue:[ (thresholdArg * total) asInteger ]
	ifFalse:[ thresholdArg asInteger ]).

  self dynamicInstVarAt: #elapsedTime put: ((endTime >= startTime)
	ifTrue: [endTime - startTime]
	ifFalse: [endTime + (16rffffffff - startTime)]).

]

{ #category : 'Monitoring' }
ProfMonitor >> profileOff [
 |rpt |
"Stop the given monitor and report."

self stopMonitoring; gatherResults; removeFile.
rpt := self reportDownTo: 0.03 .
^ rpt

]

{ #category : 'Updating' }
ProfMonitor >> removeFile [

"Removes the file generated by profiling operations in this profile monitor,
 if the file still exists."

| f fileName |
(f := file) ifNotNil:[
  fileName := file pathName .
  f close .
  GsFile removeServerFile: fileName .
  file := nil .
].

]

{ #category : 'Updating' }
ProfMonitor >> removeResults [

"Releases results to aid garbage collection."

results ifNotNil:[
  results := nil .
  ].

]

{ #category : 'Reporting' }
ProfMonitor >> report [

"Formats and returns a string holding a report of the receiver's most
 recent profile run."

 ^ self reportDownTo: 0.03.

]

{ #category : 'Reporting' }
ProfMonitor >> reportAfterRun [

"Formats and returns a string holding a report of the receiver's most
 recent profile run."
 | res |
 self gatherResults ; removeFile .
 res := self reportDownTo: 0.03.
 self removeResults .
 ^ res

]

{ #category : 'Reporting' }
ProfMonitor >> reportAfterRunDownTo: tallyThreshold [

"Formats and returns a string holding a report of the receiver's most
 recent profile run."
 | res |
 self gatherResults ; removeFile .
 res := self reportDownTo: tallyThreshold.
 self removeResults .
 ^ res

]

{ #category : 'Reporting' }
ProfMonitor >> reportDownTo: thresholdArg [

"Formats and returns a string holding a report of the receiver's most recent
 profile run.  Stops reporting when a tally smaller than tally is
 encountered. "

 ^self reportDownTo: thresholdArg reports: self class defaultReports

]

{ #category : 'Reporting' }
ProfMonitor >> reportDownTo: thresholdArg reports: arrayOfReports [

"Formats and returns a string holding a report of the receiver's most recent
 profile run.  Stops reporting when a tally smaller than tally is encountered.

 thresholdArg should be a SmallInteger or a Float > 0 .
 A thresholdArg < 1.0 is interpreted as a percentage of the total
 samples, otherwise thresholdArg is intepreted as an absolute value.
 Report elements whose sample count is less than the threshold are
 omitted from the report."

  | saveFpe |
  results ifNil:[ self gatherResults ; removeFile ].
  results ifNil:[ ^ 'No profiling data available: check profiling parameters.'].
  ^ [ |rpt|
      saveFpe := FloatingPointError enabledExceptions .
      FloatingPointError enableExceptions: nil .
      self prepareForReportWithThreshold: thresholdArg.
      rpt := String new .
      (self dynamicInstVarAt: #overrunStr) ifNotNil:[:str | rpt add: str ; lf].

      arrayOfReports do: [:sym |
        sym == #samples ifTrue: [
	   rpt addAll: (self _samplingReport); lf; lf ].
        sym == #stackSamples ifTrue: [
	   rpt addAll: (self _stackReport); lf; lf ].
        (sym == #senders and: [(self dynamicInstVarAt: #totalTallies) > 0])  ifTrue: [
	   rpt addAll: (self _sendersReport); lf; lf ].
        (sym == #objCreation and: [ (traceObjCreation bitAnd: 1) ~~ 0 and:
	  [(self dynamicInstVarAt: #totalTallies) > 0]]) ifTrue: [
	   rpt addAll: (self _objCreationReport); lf; lf.].
      ].
      rpt
    ] ensure:[
      saveFpe size > 0 ifTrue:[ FloatingPointError enableExceptions: saveFpe].
    ]

]

{ #category : 'Accessing' }
ProfMonitor >> results [

"Returns the value of the instance variable results."

^results

]

{ #category : 'Updating' }
ProfMonitor >> results: newValue [

 "Modify the value of the instance variable results."

results:= newValue

]

{ #category : 'Monitoring' }
ProfMonitor >> resumeSampling [
  "if high resolution sampling in use, resumes sampling."
interval >= 10000000 ifTrue:[
  Error signal:'resume only supported with high resolution sampling'
].
self _zeroArgPrim: 5

]

{ #category : 'Monitoring' }
ProfMonitor >> runBlock: aBlock [

"Starts profiling, executes the block, and terminates profiling.
 Does not read the sample file.
 Use gatherResults followed by reporting methods to view results."

[ self startMonitoring.
  aBlock value.
] ensure:[
  self stopMonitoring .
].

]

{ #category : 'Accessing' }
ProfMonitor >> sampleDepth [

        ^sampleDepth

]

{ #category : 'Updating' }
ProfMonitor >> sampleDepth: newValue [

"Modify the value of the instance variable sampleDepth. Should be
 at least 20; small values will prune method call stacks and object
 creation stacks."

sampleDepth := newValue

]

{ #category : 'Accessing' }
ProfMonitor >> sampling [
  "Returns the type of sampling in use,  #real or #cpu ."

  ^ (traceObjCreation bitAnd: 2) == 0 ifTrue:[ #cpu ] ifFalse:[ #real ]

]

{ #category : 'Updating' }
ProfMonitor >> setOptions: anArray [
 "By default ProfMonitor analyzes execution time using cpu time .
  Alternatively, one of the following kinds of profiling can be selected.
      #objFaults #pageFaults #edenUsage #gcTime #objCreation .

  anArray is an Array of Strings or Symbols containing
    at most one of #objFaults #pageFaults #edenUsage #gcTime #objCreation .
  and optionally  #cpu or #real .

  The last #cpu or #real if any specifies real or cpu time sampling.
  #gcTime implies default #cpu  and is in units of milliseconds  .
  #objFaults and #pageFaults are in units of faults and imply default #real .
  #edenUsage  is in units of bytes and implies default #real .
  #objCreation implies default #cpu .
  "
  | sKind cnt mask defKind |
  cnt := 0 .
  mask := 0 .
  anArray do:[:x | | sym |
    sym := Symbol _existingWithAll: x .
    sym == #real ifTrue:[ sKind := #real ] ifFalse:[
    sym == #cpu ifTrue:[ sKind := #cpu ] ifFalse:[
    sym == #objFaults ifTrue:[ defKind := #real . mask := 16r4 . cnt := cnt + 1 ] ifFalse:[
    sym == #pageFaults ifTrue:[ defKind := #real . mask := 16r8 . cnt := cnt + 1 ] ifFalse:[
    sym == #edenUsage ifTrue:[ defKind := #real . mask := 16r10 . cnt := cnt + 1 ] ifFalse:[
    sym == #gcTime ifTrue:[ defKind := #cpu . mask := 16r20 . cnt := cnt + 1 ] ifFalse:[
    sym == #objCreation ifTrue:[ defKind := #cpu . mask := 16r1 . cnt := cnt + 1 ] ifFalse:[
      Error signal: 'unrecognized option ', x asString
    ]]]]]]].
  ].
  cnt > 1 ifTrue:[ Error signal:'too many options given' ].
  sKind ifNil:[ sKind := defKind ifNil:[ #cpu ]].
  sKind == #real ifTrue:[
    (System gemVersionAt: #osName) = 'Darwin' ifTrue:[ "clock_gettime not avail"
      Error signal:'Only #cpu profiling is supported on Darwin OS'.
    ].
    mask := mask bitOr: 16r2
  ].
  traceObjCreation := mask .

]

{ #category : 'Monitoring' }
ProfMonitor >> startMonitoring [

"Starts monitoring."
results ifNotNil:[ results := nil "force re-read of sample file"].
self _openFileAppend .
self _zeroArgPrim: 1 .  "enable and start monitoring in the virtual machine"

]

{ #category : 'Accessing' }
ProfMonitor >> startTime [

	^startTime

]

{ #category : 'Monitoring' }
ProfMonitor >> stopMonitoring [

"Stops monitoring. stores total number of samples written to file
  in the numSamples instVar."
| cnt delta |
cnt := self _zeroArgPrim: 2  . "stop monitoring in the virtual machine"
(delta := endTime - startTime) <= 1 ifTrue:[  "fix 41260"
  (OutOfRange new )
     name: 'CPU time used' min: 2 actual: delta ;
     details: 'code to be profiled must use more than 1ms of CPU or real time' ;
	 signal
].
numSamples ifNil:[ numSamples := 0 ].
numSamples := numSamples + cnt  .
self dynamicInstVarAt: #overrunStr put: (self _zeroArgPrim: 6).
file close .

]

{ #category : 'Monitoring' }
ProfMonitor >> suspendSampling [
  "if high resolution sampling in use, suspends sampling."
interval >= 10000000 ifTrue:[
  Error signal:'suspend only supported with high resolution sampling'
].
self _zeroArgPrim: 4

]

{ #category : 'Updating' }
ProfMonitor >> traceObjectCreation: aBoolean [

"Enable (aBoolean == true) or disable profiling of object creation.  The state
 change will take effect on the next invocation of ProfMonitor>>startMonitoring
 for the receiver."

aBoolean _validateClass: Boolean .
aBoolean ifTrue:[ traceObjCreation := traceObjCreation bitOr: 1 ]
         ifFalse:[ traceObjCreation := traceObjCreation bitAnd: 1 bitInvert ].
sampleDepth < 2 ifTrue:[ sampleDepth := 2 ].

]
