"
This class is the logical representation of an LDAP Directory Server to
 GemStone.  It contains the following instance variables which must be Strings
 unless otherwise noted:
   uri               - A Uniform Resource Identifier for LDAP conforming to
                       RFC 4516.
 Optional Arguments (may be nil):		       
   bindDN            - A string which is the distinguished name (DN) used to
                       bind to the server to perform lookups, or nil if anonymous
		       binds are to be used (no bindDN required).
   bindPW            - a ByteArray, which is the encrypted password for bindDN,
                       or nil if anonymous binds are to be used.
   baseDN            - Specifies the default base DN to use when performing ldap
                       operations.  The baseDN must be specified as a Distinguished
		       Name in LDAP format.
   tlsCaCert         - Specifies the file that contains certificates for all of
                       the Certificate Authorities the client will recognize.
   tlsCaCertDir      - Specifies the path of a directory that contains Certificate
                       Authority certificates in separate individual files. tlsCaCert is
		       always used before tlsCaCertDir.
   tlsCert           - Specifies the file that contains the client certificate.
   tlsKey            - Specifies the file that contains the private key
                       that matches the certificate stored in the tlsCert
		       instance variable. The private key must not have a passphrase.
   tlsReqCert        - A symbol which specifies what checks to perform on server certificates
                       in a TLS session, if any. The following symbols are recognized:
                         #never -   The client will not request or check any server certificate.
                         #allow -   The  server certificate is requested. If no certificate is provided,
			            the session proceeds normally. If a bad certificate is provided,
				    it will be ignored and the session proceeds normally.
                         #try -     The server certificate is requested. If no certificate is provided,
			            the session proceeds normally. If a bad certificate is provided,
				    the session is immediately terminated.
                         #demand -  The server certificate is requested. If no certificate is provided,
			            or a bad certificate is provided, the session is immediately
                                    terminated. This is the default setting.
				    
setuid WARNING: OpenLDAP does not read most environment variables nor home directory files
(such as $HOME/ldaprc) if the process is running in setuid mode. A process runs in setuid mode
if the real UNIX user id and the effective UNIX user id are not the same. If running in setuid
mode, the tls* inst vars should be set so that LDAP can locate the correct TLS credentials.
"
Class {
	#name : 'LdapDirectoryServer',
	#superclass : 'Object',
	#instVars : [
		'uri',
		'bindDN',
		'bindPW',
		'baseDN',
		'tlsCaCert',
		'tlsCaCertDir',
		'tlsCert',
		'tlsKey',
		'tlsReqCert'
	],
	#classVars : [
		'AllLdapDirectoryServers'
	],
	#gs_reservedoop : '156673',
	#category : 'System-LoginSupport'
}

{ #category : 'Private - Initializing' }
LdapDirectoryServer class >> _smartInitialize [
"This method should only be called once during filein or image upgrade."
  AllLdapDirectoryServers ifNil:[ | newArray |
      (Object _objectForOop: 156929) ifNotNil:[:o| 
         Error signal:'oop 156929 already exists as a ', o class name .
      ].
      newArray := Array new .
      newArray _unsafeSetOop: 156929  "OOP_AllLdapDirectoryServers".
      newArray objectSecurityPolicy: (Globals at:#DataCuratorObjectSecurityPolicy) .
      AllLdapDirectoryServers := newArray .
      ^ 'created'
  ] .
  ^ 'exists'

]

{ #category : 'Accessing' }
LdapDirectoryServer class >> allServers [
  ^ AllLdapDirectoryServers

]

{ #category : 'Instance Creation' }
LdapDirectoryServer class >> basicNewWithUri: uri bindDN: aBindDn password: password [
"Creates a new instance of the receiver, but does not add the new instance
 to the list of LDAP servers used to process login requests."

| result |
result := super new.
result uri: uri ;
       bindDN: aBindDn ;
       password: password .

^ result

]

{ #category : 'Instance Creation' }
LdapDirectoryServer class >> basicNewWithUri: uri bindDN: aBindDn password: password baseDN: baseDn tlsCaCert: caCert tlsCert: cert tlsKey: key tlsReqCert: aSymbol [
"Creates a new instance of the receiver but does not add it 
 the list of LdapDirectoryServer objects used to authorize logins."
 
| result |
result := super new.
result uri: uri ;
	bindDN: aBindDn ;
	password: password ;
	baseDN: baseDn ;
	tlsCaCert: caCert ;
	tlsCert: cert ;
	tlsKey: key ;
	tlsReqCert: aSymbol .
   
^ result
]

{ #category : 'Searching' }
LdapDirectoryServer class >> findServerWithUri: aUriString [
"Searches for an instance of LdapDirectoryServer with the given URI.
 Returns nil if the directory server was not found."

^ self allServers detect:[:e| e uri = aUriString ] ifNone:[ nil ]

]

{ #category : 'Illegal Operations' }
LdapDirectoryServer class >> new [

"Disallowed.  To create a new LdapDirectoryServer, use newWithUri:... instead."

self shouldNotImplement: #new

]

{ #category : 'Illegal Operations' }
LdapDirectoryServer class >> new: anInt [

"Disallowed.  To create a new LdapDirectoryServer, use newWithUri:... instead."
self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
LdapDirectoryServer class >> newWithUri: uri bindDN: aBindDn password: password [
"Creates a new instance of the receiver and adds the resulting object to
 the list of LdapDirectoryServer objects used to authorize logins."

| result |
result := self basicNewWithUri: uri
               bindDN: aBindDn
	       password: password .
self allServers add: result .
^ result

]

{ #category : 'Instance Creation' }
LdapDirectoryServer class >> newWithUri: uri bindDN: aBindDn password: password baseDN: baseDn tlsCaCert: caCert tlsCert: cert tlsKey: key tlsReqCert: aSymbol [
"Creates a new instance of the receiver and adds the resulting object to
 the list of LdapDirectoryServer objects used to authorize logins."
 
| result |
result := self basicNewWithUri: uri bindDN: aBindDn password: password baseDN: baseDn tlsCaCert: caCert tlsCert: cert tlsKey: key tlsReqCert: aSymbol .
self allServers add: result .	       
^ result
]

{ #category : 'Removing' }
LdapDirectoryServer class >> removeServerWithUri: aUriString [
"Removes an instance of LdapDirectoryServer with the given URI.
 Returns true if the directory server was removed, false if it
 could not be found."

| server |
server := self findServerWithUri: aUriString .
server isNil
  ifTrue:[ ^ false "not found" ] .

self allServers removeIdentical: server .
^ true

]

{ #category : 'Connection Testing' }
LdapDirectoryServer class >> testConnectionToServer: uri bindDN: aBindDn password: password [
"Attempts to perform a bind using aBindDn and password to the LDAP server specified by
 uri.  Returns true if the connection was successful, otherwise returns false."

 | result |
 "Create a temporary instance."
result := self basicNewWithUri: uri
               bindDN: aBindDn
	       password: password .
^ result testBind

]

{ #category : 'Connection Testing' }
LdapDirectoryServer class >> testConnectionToServer: uri bindDN: aBindDn password: password baseDN: baseDn tlsCaCert: caCert tlsCert: cert tlsKey: key tlsReqCert: aSymbol [
"Attempts to perform a bind using aBindDn and password to the LDAP server specified by
 uri.  Also sets the TLS credentials if those arguments are not nil.
Returns true if the connection was successful, otherwise returns false."
 
 | result |
 "Create a temporary instance."
result := self basicNewWithUri: uri
               bindDN: aBindDn
	       password: password 
	       baseDN: baseDn
	       tlsCaCert: caCert
	       tlsCert: cert
	       tlsKey: key 
	       tlsReqCert: aSymbol .
^ result testBind
]

{ #category : 'Private' }
LdapDirectoryServer >> _encryptPassword: aString [
"Encrypt aString into a ByteArray. Store the ByteArray in the bindPW inst var.
 Returns the receiver."

<primitive: 1008>
self _validateClass: LdapDirectoryServer .
aString _validateClass: String .
self _primitiveFailed: #_encryptPassword: args: { aString } .
self _uncontinuableError

]

{ #category : 'Private' }
LdapDirectoryServer >> _oneArgLdapPrim: opCode with: arg [

<primitive: 1009>
self _validateClass: LdapDirectoryServer .
self _primitiveFailed: #_oneArgLdapPrim:with: args: { opCode . arg} . 
self _uncontinuableError

]

{ #category : 'Private' }
LdapDirectoryServer >> _testBind [
^ self _oneArgLdapPrim: 0 with: nil
]

{ #category : 'Accessing' }
LdapDirectoryServer >> baseDN [
  ^ baseDN

]

{ #category : 'Updating' }
LdapDirectoryServer >> baseDN: newValue [
  baseDN := newValue copy

]

{ #category : 'Accessing' }
LdapDirectoryServer >> bindDN [
  ^ bindDN

]

{ #category : 'Updating' }
LdapDirectoryServer >> bindDN: aString [
  bindDN := aString copy

]

{ #category : 'Accessing' }
LdapDirectoryServer >> bindPW [
  ^ bindPW

]

{ #category : 'Updating' }
LdapDirectoryServer >> password: aString [
^ aString == nil
    ifTrue:[ nil ]
   ifFalse:[ self _encryptPassword: aString ]

]

{ #category : 'Private' }
LdapDirectoryServer >> testBind [
"Attempt to bind to the specified LDAP server. Returns true if successful, false if
 the password is invalid.  Raises an error if an LDAP error occurs."

| result |
result := self _testBind .
result class == String
  ifTrue:[ IOError signal: result ].
^ result

]

{ #category : 'Accessing' }
LdapDirectoryServer >> tlsCaCert [
  ^ tlsCaCert

]

{ #category : 'Updating' }
LdapDirectoryServer >> tlsCaCert: newValue [
  tlsCaCert := newValue copy

]

{ #category : 'Accessing' }
LdapDirectoryServer >> tlsCaCertDir [
  ^ tlsCaCertDir

]

{ #category : 'Updating' }
LdapDirectoryServer >> tlsCaCertDir: newValue [
  tlsCaCertDir := newValue copy

]

{ #category : 'Accessing' }
LdapDirectoryServer >> tlsCert [
  ^ tlsCert

]

{ #category : 'Updating' }
LdapDirectoryServer >> tlsCert: newValue [
  tlsCert := newValue copy

]

{ #category : 'Accessing' }
LdapDirectoryServer >> tlsKey [
  ^ tlsKey

]

{ #category : 'Updating' }
LdapDirectoryServer >> tlsKey: newValue [
  tlsKey := newValue copy

]

{ #category : 'Accessing' }
LdapDirectoryServer >> tlsReqCert [
  ^ tlsReqCert

]

{ #category : 'Updating' }
LdapDirectoryServer >> tlsReqCert: newValue [
"A symbol which specifies what checks to perform on server certificates
in a TLS session, if any. The following symbols are recognized:

	#never -   The client will not request or check any server certificate.
	#allow -   The  server certificate is requested. If no certificate is provided,
		          the session proceeds normally. If a bad certificate is provided,
			it will be ignored and the session proceeds normally.
	#try -	The server certificate is requested. If no certificate is provided,
			the session proceeds normally. If a bad certificate is provided,
			the session is immediately terminated.
	#demand -  The server certificate is requested. If no certificate is provided,
			or a bad certificate is provided, the session is immediately
			terminated. This is the default setting.
"
newValue isSymbol
	ifTrue:[  
		({ #never . #allow . #try . #demand } includesIdentical: newValue)
				ifFalse:[ ^ ArgumentError signal: 'Illegal tlsReqCert symbol' ].
	] ifFalse:[
		newValue isNil 
			ifFalse:[ ^ ArgumentError signal: 'Illegal object for tlsReqCert ' ] 
	].
	tlsReqCert := newValue 

]

{ #category : 'Accessing' }
LdapDirectoryServer >> uri [
  ^ uri

]

{ #category : 'Updating' }
LdapDirectoryServer >> uri: aString [
  uri := aString copy

]

{ #category : 'Password Validation' }
LdapDirectoryServer >> validatePassword: aPassword forUserId: aUserId withBaseDn: aBaseDn filterDn: aFilterDn [
"Use the receiver validate the password aPassword is valid for aUserId.  

 In order to validate the password, the complete distinguish name (DN) for 
 aUserId must be determined.  The DN can either be constructed from the aBaseDn
 pattern (explicit mode) or the DN may be resolved by searching the LDAP directory
 (search mode).  In explicit mode, aBaseDn must be a string that contains the string
 wildcard sequence '%s'.  GemStone will substitute '%s' with aUserId before
 doing the password validation.  aFilterDn must be nil in explicit mode.

 In search mode, aBaseDn is the search pattern used to resolve the DN and must
 NOT contain the string wildcard sequence '%s'.  In search mode, aFilterDn must
 be a string that contains the string wildcard sequence '%s'.  See the examples
 below.

 aUserId must be a string which represents the aUserId to be validated.
 aPassword must be a string which is the password for aUserId.

 Returns true if aPassword is the correct password for aUserId. Otherwise returns
 false if the password is incorrect or an error occurred while communicating
 with the LDAP server.  

 Setting the variable GS_DEBUG_LDAP=7 in the gem's environment will cause LDAP debugging
 information to be printed to stdout. Setting the variable GS_DEBUG_LDAP_DIR in the gem's
 environment will cause LDAP debugging information to be written to a new file in that
 directory.

 Example 1: Explicit mode

(LdapDirectoryServer basicNewWithUri: 'ldaps://myldap.mydomain.com'
 bindDN: nil password: nil baseDN: nil tlsCaCert: '/home/gemstone/cacert.pem'
 tlsCert: '/home/gemstone/cert.pem' 
tlsKey: '/home/gemstone/key.pem' tlsReqCert: #never)
validatePassword: 'swordfish' forUserId: 'DataCurator' 
withBaseDn: 'uid=%s,ou=Users,dc=mycompany,dc=com' filterDn: nil

 Example 2: Search mode with anonymous bind

(LdapDirectoryServer basicNewWithUri: 'ldaps://myldap.mydomain.com'
 bindDN: nil password: nil baseDN: nil tlsCaCert: '/home/gemstone/cacert.pem'
 tlsCert: '/home/gemstone/cert.pem' 
tlsKey: '/home/gemstone/key.pem' tlsReqCert: #never)
validatePassword: 'swordfish' forUserId: 'DataCurator' 
withBaseDn: 'ou=Users,dc=mycompany,dc=com' filterDn: '(uid=%s)'

Example 3: Search mode with authenticated bind

(LdapDirectoryServer basicNewWithUri: 'ldaps://myldap.mydomain.com'
 bindDN: 'LdapBindUser'  password: 'LdapBindPassword' baseDN: nil tlsCaCert: '/home/gemstone/cacert.pem'
 tlsCert: '/home/gemstone/cert.pem' 
tlsKey: '/home/gemstone/key.pem' tlsReqCert: #never)
validatePassword: 'swordfish' forUserId: 'DataCurator' 
withBaseDn: 'ou=Users,dc=mycompany,dc=com' filterDn: '(uid=%s)'
"

^ self _oneArgLdapPrim: 1 with: (Array with: aUserId with: aPassword with: aBaseDn with: aFilterDn)
]
