Extension { #name : 'JwtUserClaim' }

{ #category : 'Instance Creation' }
JwtUserClaim class >> new [

^ super new initialize

]

{ #category : 'Instance Creation' }
JwtUserClaim class >> newWildcardClaimWithJsonKey: keySym jsonKind: kindSym [

| newClaim |
newClaim := self new.
newClaim jsonKey: keySym ;
	jsonKind: kindSym ;
	acceptAllValues .
^ newClaim

]

{ #category : 'Instance Creation' }
JwtUserClaim class >> newWithJsonKey: keySym jsonKind: kindSym acceptedValues: anArray [

| newClaim |
newClaim := self new.
newClaim jsonKey: keySym ;
	jsonKind: kindSym ;
	acceptedValues: anArray  .
^ newClaim

]

{ #category : 'Constants' }
JwtUserClaim class >> validJsonKinds [

^ { #String . #Number . #ArrayOfStrings . #ArrayOfNumbers . #Boolean }

]

{ #category : 'Constants' }
JwtUserClaim class >> wildcard [

^ #*

]

{ #category : 'Updating' }
JwtUserClaim >> acceptAllValues [

"Causes the receiver to accept any value for the claim"
JwtSecurityData acceptAllValues: acceptedValues .
^ self

]

{ #category : 'Accessing' }
JwtUserClaim >> acceptedValues [
	^acceptedValues

]

{ #category : 'Updating' }
JwtUserClaim >> acceptedValues: aCollection [

"Sets the list of acceptedValues for the receiver to be aCollection, which is expected to be an Array of symbols
  or objects that understand the message #asSymbol ."
| idSet |
idSet := (aCollection collect:[:e| e asSymbol ]) asIdentitySet .
(idSet includesIdentical: self class wildcard)
	ifTrue:[ self acceptAllValues ]
	ifFalse:[ acceptedValues := idSet ].
^ self

]

{ #category : 'Updating' }
JwtUserClaim >> addAcceptedValue: anObject [

"Makes anObject an accepted value for the receiver. 
anObject must be a Symbol or an object that understands the message #asSymbol."

JwtSecurityData addAcceptedValue: anObject to: acceptedValues.
^ self

]

{ #category : 'Updating' }
JwtUserClaim >> addAllAcceptedValues: aCollection [

"Adds each object in aCollection to the receiver as an accepted value.
Each object must be a Symbol or an object that understands the message #asSymbol."

aCollection do:[:obj| JwtSecurityData addAcceptedValue: obj to: acceptedValues ] .
^ self

]

{ #category : 'Printing' }
JwtUserClaim >> asString [

^ 'a', self class name asString, ' for #', self jsonKey asString

]

{ #category : 'Initialize' }
JwtUserClaim >> initialize [

super initialize .
acceptedValues := IdentitySet new.
^ self


]

{ #category : 'Testing' }
JwtUserClaim >> isWildcard [

^ acceptedValues includesIdentical: self class wildcard

]

{ #category : 'Accessing' }
JwtUserClaim >> jsonKey [
	^jsonKey

]

{ #category : 'Updating' }
JwtUserClaim >> jsonKey: newValue [
	self validateJsonKey: newValue.
	jsonKey := newValue asSymbol .
	^ self

]

{ #category : 'Accessing' }
JwtUserClaim >> jsonKind [
	^jsonKind

]

{ #category : 'Updating' }
JwtUserClaim >> jsonKind: newValue [

	self validateJsonKind: newValue.
	jsonKind := newValue asSymbol .
	^ self

]

{ #category : 'Updating' }
JwtUserClaim >> objectSecurityPolicy: anObjectSecurityPolicy [

super objectSecurityPolicy: anObjectSecurityPolicy.
acceptedValues objectSecurityPolicy: anObjectSecurityPolicy.
^ self

]

{ #category : 'Printing' }
JwtUserClaim >> printConfigurationOn: ws [

ws tab ; nextPut: $# ; nextPutAll: self jsonKey asString.
JwtSecurityData printSet: self acceptedValues on: ws name: 'acceptedValues' .

]

{ #category : 'Printing' }
JwtUserClaim >> printOn: ws [

ws nextPutAll: self asString

]

{ #category : 'Validation' }
JwtUserClaim >> validate [

^ self 
	validateJsonKey ;
	validateJsonKind ;
	validateAcceptedValues

]

{ #category : 'Validation' }
JwtUserClaim >> validateAcceptedValues [

	^ self validateAcceptedValues: self acceptedValues

]

{ #category : 'Validation' }
JwtUserClaim >> validateAcceptedValues: vals [

^ JwtSecurityData validateIdentitySetOfSymbols: self acceptedValues name: #acceptedValues allowEmpty: false.

]

{ #category : 'Validation' }
JwtUserClaim >> validateJsonKey [

^ self validateJsonKey: self jsonKey

]

{ #category : 'Validation' }
JwtUserClaim >> validateJsonKey: aSymbol [

^ (aSymbol asSymbol == self class wildcard)
	ifTrue:[ ArgumentError signal: ('jsonKey may not be a wildcard') ]
	ifFalse:[ true ]

]

{ #category : 'Validation' }
JwtUserClaim >> validateJsonKind [

^ self validateJsonKind: self jsonKind

]

{ #category : 'Validation' }
JwtUserClaim >> validateJsonKind: aSymbol [

^ (self class validJsonKinds includesIdentical: aSymbol asSymbol)
	ifTrue:[ true ]
	ifFalse:[ ArgumentError signal: ('Invalid JSON kind ', aSymbol asString) ].

]
