Extension { #name : 'JwtSecurityData' }

{ #category : 'Accepted Values' }
JwtSecurityData class >> acceptAllValues: idSet [

^ self addAcceptedValue: self wildcard to: idSet

]

{ #category : 'Accepted Values' }
JwtSecurityData class >> addAcceptedValue: value to: idSet [

| wc hasWc |
wc := self wildcard.
hasWc := idSet includesIdentical: wc.

value == wc 
	ifTrue:[ "adding the wildcard"
 			hasWc 
				ifFalse:[ idSet removeAll: idSet ; add: wc ]
	] 
	ifFalse:[ "not adding wildcard"
  		hasWc ifTrue:[ idSet remove: wc ].
		idSet add: value asSymbol
].
^ idSet

]

{ #category : 'Examples' }
JwtSecurityData class >> jwtSecurityDataExample [

"Example of how to create a new UserProfile which uses JWTs for login authentication.
Should be run by DataCurator"

|jwtSecData claim userPro |

"Create a UserProfile if needed"
userPro := AllUsers userWithId: 'JwtUser' ifAbsent:[
  AllUsers addNewUserWithId: 'JwtUser' password: 'swordfish' ].

"Create JwtSecurityData"
jwtSecData := JwtSecurityData new.
jwtSecData 
	addAudience: 'JwtUser@gemtalksystems.com' ;
	addIssuer: 'https://accounts.google.com' ; 
	userIdKey: 'aud' ;
	addUserId: 'JwtUser@gemtalksystems.com' .

"Create and add claims"
claim := JwtUserClaim newWithJsonKey: #sub jsonKind: #String acceptedValues: { #'115156884418451143666' } .
jwtSecData addUserClaim: claim.
claim := JwtUserClaim newWithJsonKey: #azp jsonKind: #String acceptedValues: { #'115156884418451143666' } .
jwtSecData addUserClaim: claim.
claim := JwtUserClaim newWithJsonKey: #email jsonKind: #String acceptedValues:  { #'service-account@oauth2-test-436914.iam.gserviceaccount.com' } .
jwtSecData addUserClaim: claim.
claim := JwtUserClaim newWithJsonKey: #email_verified jsonKind: #Boolean acceptedValues: { true } .
jwtSecData addUserClaim: claim.
"Enable JWT authentication for the UserProfile using the JwtSecurityData object"
userPro enableJwtAuthenticationWith: jwtSecData .
"Commit the results"
^ System commit.


]

{ #category : 'Instance Creations' }
JwtSecurityData class >> new [

^ super new initialize

]

{ #category : 'Printing' }
JwtSecurityData class >> printSet: aSet on: aStream name: aString [

| first |
first := true.
aStream tab ; nextPutAll: aString asString ; nextPutAll: ': '.
aSet do:[:obj|
	first ifTrue:[ first := false ]
		ifFalse:[ aStream nextPutAll: ', ' ].
	aStream nextPutAll: obj asString.
].
aStream lf.

]

{ #category : 'Validation' }
JwtSecurityData class >> validateIdentitySetOfSymbols: obj name: aName allowEmpty: canBeEmpty [


(obj isEmpty and:[ canBeEmpty not ])
	ifTrue:[  ^ ArgumentError signal: ('Expected ', aName asString, ' to be non-empty IdentitySet') ].
(obj detect:[:each| each _isSymbol not ] ifNone:[ nil ]) 
	ifNotNil:[  ^ ArgumentError signal: ('Expected ', aName asString, ' to be an Array of Symbols') ].

^ true

]

{ #category : 'Constants' }
JwtSecurityData class >> wildcard [

^ #*

]

{ #category : 'Adding' }
JwtSecurityData >> addAudience: aSymbol [

"Adds aSymbol to the list of accepted audiences"
self class addAcceptedValue: aSymbol to: validAudiences .
^ self

]

{ #category : 'Adding' }
JwtSecurityData >> addIssuer: aSymbol [

"Adds aSymbol to the list of accepted issuers"
self class addAcceptedValue: aSymbol to: validIssuers .
^ self

]

{ #category : 'User Claims' }
JwtSecurityData >> addUserClaim: aJwtUserClaim [

"Adds a new user claim to the receiver. Raises an error if the receiver already has a claim with the same key."

aJwtUserClaim _validateClass: JwtUserClaim ; validate .
(self hasClaimWithKey: aJwtUserClaim jsonKey)
	ifTrue:[ ^ ArgumentError signal: 'Attempt to add duplicate user claim' ].

userClaims add: aJwtUserClaim .
^ self

]

{ #category : 'Adding' }
JwtSecurityData >> addUserId: aSymbol [

"Adds aSymbol to the list of accepted user ids"
self class addAcceptedValue: aSymbol to: validUserIds .
^ self

]

{ #category : 'Updating' }
JwtSecurityData >> allowAnyAudience [

| aud |
aud := self validAudiences .
aud removeAll: aud ; add: self wildcard .
^ self

]

{ #category : 'Updating' }
JwtSecurityData >> allowAnyIssuer [

| iss |
iss := self validIssuers .
iss removeAll: iss ; add: self wildcard .
^ self

]

{ #category : 'Updating' }
JwtSecurityData >> allowAnyUserId [

| uids |
uids := self validUserIds .
uids removeAll: uids ; add: self wildcard .
^ self

]

{ #category : 'Testing' }
JwtSecurityData >> allowsAnyAudience [

^ validAudiences includesIdentical: self wildcard

]

{ #category : 'Testing' }
JwtSecurityData >> allowsAnyIssuer [

^ validIssuers includesIdentical: self wildcard

]

{ #category : 'Testing' }
JwtSecurityData >> allowsAnyUserId [

^ validUserIds includesIdentical: self wildcard

]

{ #category : 'Printing' }
JwtSecurityData >> configurationReport [

| as |
as := AppendStream on: String new.
self printConfigurationOn: as.
^ as contents

]

{ #category : 'Testing' }
JwtSecurityData >> hasAnyWildcard [

"Returns true if any required JWT field is set to allow any value. The required JWT fields are audience, issuer, and userId."

^ (self allowsAnyUserId or:[ self allowsAnyIssuer]) or:[ self allowsAnyAudience ]

]

{ #category : 'User Claims' }
JwtSecurityData >> hasClaimWithKey: aSymbol [

"Answers true if the receiver has a claim with key aSymbol, false otherwise."
^ nil ~~ (self userClaimWithKey: aSymbol)

]

{ #category : 'Testing' }
JwtSecurityData >> hasUserClaims [

^ userClaims notEmpty

]

{ #category : 'Initialization' }
JwtSecurityData >> initialize [

userClaims := Array new .
validAudiences := IdentitySet new .
validIssuers := IdentitySet new .
validUserIds := IdentitySet new .
userIdKey := #aud .
^ self

]

{ #category : 'Updating' }
JwtSecurityData >> objectSecurityPolicy: anObjectSecurityPolicy [

super objectSecurityPolicy: anObjectSecurityPolicy.
validIssuers objectSecurityPolicy: anObjectSecurityPolicy.
validAudiences objectSecurityPolicy: anObjectSecurityPolicy.
validUserIds objectSecurityPolicy: anObjectSecurityPolicy.
userClaims ifNotNil:[
	userClaims objectSecurityPolicy: anObjectSecurityPolicy.
	userClaims do:[:e| e objectSecurityPolicy: anObjectSecurityPolicy ].
].
^ self

]

{ #category : 'Printing' }
JwtSecurityData >> printConfigurationOn: aStream [

| cls |
cls := self class.
aStream 
	nextPut: $a ; 
	nextPutAll: cls name asString ;
	nextPutAll: ' for userIdKey #' ;
	nextPutAll: self userIdKey asString ;
	lf.
cls 
	printSet: validIssuers on: aStream name: 'validIssuers' ;
	printSet: validAudiences on: aStream name: 'validAudiences' ;
	printSet: validUserIds on: aStream name: 'validUserIds' .
self printUserClaimsConfigurationOn: aStream.
^ aStream
	

]

{ #category : 'Printing' }
JwtSecurityData >> printUserClaimsConfigurationOn: aStream [

aStream nextPutAll: 'User Defined Claims:' ; lf.
self userClaims do:[:claim|
	claim printConfigurationOn: aStream
].
^ aStream

]

{ #category : 'User Claims' }
JwtSecurityData >> removeClaimWithKey: aSymbol [

"Removes the claim with key aSymbol from the receiver. Returns the removed claim on success or nil if the receiver does not have a claim with that key."
| claim |
claim := self userClaimWithKey: aSymbol.
claim ifNotNil:[ userClaims removeIdentical: claim ].
^ claim


]

{ #category : 'Accessing' }
JwtSecurityData >> userClaims [
	^userClaims

]

{ #category : 'Accessing' }
JwtSecurityData >> userClaimsSize [

^ userClaims size

]

{ #category : 'User Claims' }
JwtSecurityData >> userClaimWithKey: aSymbol [

"Returns aJwtUserClaim if the receiver has a claim with key aSymbol, otherwise returns nil."
^ self userClaims detect:[:e| e jsonKey == aSymbol ] ifNone:[ nil ]

]

{ #category : 'Accessing' }
JwtSecurityData >> userIdKey [
	^userIdKey

]

{ #category : 'Updating' }
JwtSecurityData >> userIdKey: aSymbol [

"Sets the name of the key in the JWT which references the userId.
At login time, the presented JWT credential must contain this key or the login will fail.
The wildcard character is disallowed. Returns the receiver."

| key |
key := aSymbol asSymbol.
key == self wildcard
  ifTrue:[ ^ ArgumentError signal: 'userIdKey may not be a wildcard' ].
userIdKey := key.
^ self

]

{ #category : 'Validation' }
JwtSecurityData >> validate [

"Validates the entire object and any user claims. Raises an error on validation failure or returns true on success."

self validateIssuers ;
	validateAudiences ;
	validateUserIdKey ;
	validateUserIds ;
	validateUserClaims .

^ true

]

{ #category : 'Validation' }
JwtSecurityData >> validateAudiences [

^ self class validateIdentitySetOfSymbols: self validIssuers name: #validAudiences allowEmpty: false

]

{ #category : 'Validation' }
JwtSecurityData >> validateIssuers [

^ self class validateIdentitySetOfSymbols: self validIssuers name: #validIssuers allowEmpty: false

]

{ #category : 'Validation' }
JwtSecurityData >> validateUserClaims [

| claims |
claims := self userClaims.
claims _validateClass: Array.
claims do:[:e| e _validateClass: JwtUserClaim ; validate ].
^ true

]

{ #category : 'Validation' }
JwtSecurityData >> validateUserIdKey [

^ self userIdKey _validateClass: Symbol

]

{ #category : 'Validation' }
JwtSecurityData >> validateUserIds [

^ self class validateIdentitySetOfSymbols: self validUserIds name: #validUserIds allowEmpty: false

]

{ #category : 'Accessing' }
JwtSecurityData >> validAudiences [
	^validAudiences

]

{ #category : 'Accessing' }
JwtSecurityData >> validIssuers [
	^validIssuers

]

{ #category : 'Accessing' }
JwtSecurityData >> validUserIds [
	^validUserIds

]

{ #category : 'Constants' }
JwtSecurityData >> wildcard [

^ self class wildcard

]
