Extension { #name : 'JsonWebToken' }

{ #category : 'JWT Example Helpers' }
JsonWebToken class >> example_addClaimsTo: jwt [

|  now |
now := System timeGmt  .
jwt 
"Add realistic header claims"
	keyId: '2oavsqEohBaTqJRU7GOIoJGhbVg' ;
"Add realistic payload claims"
	audience: 'GemTalk:test:jwt:key' ;
	issuer: 'oauth2@gemtalksystems.com' ;
	issuedAtTime: now asString ;
	notBeforeTime: now asString ;
	expirationTime: (now + 7200) asString .
^ jwt

]

{ #category : 'JWT Examples' }
JsonWebToken class >> example_createHmac256Jwt [

"JsonWebToken example_createHmac256Jwt"

| jwt |

"Create a new JWT and set it for hmac256"
jwt := JsonWebToken newForHmac256 .
"Add claims"
self example_addClaimsTo: jwt .
"sign the jwt with the secret key"
jwt signWithSecretKey: self example_secretKey .
^ jwt

]

{ #category : 'JWT Examples' }
JsonWebToken class >> example_createRsa256Jwt [

"JsonWebToken example_createRsa256Jwt"

| jwt |

"Create a new JWT and set it for rsa256"
jwt := JsonWebToken newForRsa256 .

"Add some claims"
self example_addClaimsTo: jwt.

"sign the jwt with the RSA private key"
jwt signWithPrivateKey: self example_privateKey.

^ jwt

]

{ #category : 'JWT Login Example' }
JsonWebToken class >> example_jwtLogin [

"Example of using an adhoc public key to perform a JWT login.
 Netldi must be running."

"JsonWebToken example_jwtLogin"


| session userPro jwt |
"Fetch and/or create a new UserProfile"
userPro := self loginExampleCreateUserProfile .
"Create a jwtString for login"
jwt := self loginExampleCreateJwtString .
"Setup a new external session"
session :=  GsTsExternalSession newDefault .
session
	username: userPro userId ;
	 jwtPassword: jwt ;
	setLoginDebug .

"Add public key valid public keys for login. Jwt was signed with the private key."
 System addJwtKey: self example_publicKey withId: self loginExampleJwtKeyId .
[ session login.
  session isLoggedIn 
	ifFalse:[ UserDefinedError signal: 'Login failed'].
] ensure:[
	session logout. "logout"
	System removeJwtKeyWithId: self loginExampleJwtKeyId . "Remove adhoc public key"
].
^ true
	

]

{ #category : 'JWT Example Helpers' }
JsonWebToken class >> example_privateKey [

^ GsTlsPrivateKey 
	newFromPemFile: '$GEMSTONE/examples/openssl/private/backup_sign_1_clientkey.pem' 
	withPassphraseFile: '$GEMSTONE/examples/openssl/private/backup_sign_1_client_passwd.txt' 

]

{ #category : 'JWT Example Helpers' }
JsonWebToken class >> example_publicKey [

^ self example_privateKey asPublicKey

]

{ #category : 'JWT Example Helpers' }
JsonWebToken class >> example_secretKey [

^ ByteArray fromBase64String: '/9XPlSmmzht9yglinpR9B4zXaiQHpfoDy5gMUz6CVHUbiW5WJQrDcZNf4zSKEryIU+b3saIa9jF44BX8YhVhMQ=='

]

{ #category : 'JWT Examples' }
JsonWebToken class >> example_verifyHmac256Jwt [

"JsonWebToken example_verifyHmac256Jwt"

| jwt jwtString anotherJwt |

jwt := self example_createHmac256Jwt .
"Create the JWT string"
jwtString := jwt asJwtString .
"Create a new JWT using the JWT string"
anotherJwt := JsonWebToken fromJwtString: jwtString .
"Verify the token with the secret key. Returns true"
^ anotherJwt verifySignatureWithSecretKey: self example_secretKey

]

{ #category : 'JWT Examples' }
JsonWebToken class >> example_verifyRsa256Jwt [

"JsonWebToken example_verifyRsa256Jwt"

| jwt jwtString anotherJwt |

"Create a signed RSA256 signed JWT"
jwt := self example_createRsa256Jwt . 

"Get the JWT as a JWT string"
jwtString := jwt asJwtString .

"Create a new JWT using the JWT string"
anotherJwt := JsonWebToken fromJwtString: jwtString .

"Verify the token using the public key. Returns true"
^ anotherJwt verifySignatureWithPublicKey: self example_publicKey

]

{ #category : 'JWT Examples' }
JsonWebToken class >> example_verifyRsa256JwtWithX509Certificate [

"JsonWebToken example_verifyRsa256JwtWithX509Certificate"

| jwt jwtString anotherJwt |

"Create a signed RSA256 signed JWT"
jwt := self example_createRsa256Jwt . 

"Get the JWT as a JWT string"
jwtString := jwt asJwtString .

"Create a new JWT using the JWT string"
anotherJwt := JsonWebToken fromJwtString: jwtString .

"Verify the token using the public key contained in the certificate. Returns true"
^ anotherJwt verifySignatureWithPublicKey: self example_x509Certificate asPublicKey

]

{ #category : 'JWT Example Helpers' }
JsonWebToken class >> example_x509Certificate [

^ GsX509Certificate 
	newFromPemFile: '$GEMSTONE/examples/openssl/certs/backup_sign_1_clientcert.pem'

]

{ #category : 'Instance Creation' }
JsonWebToken class >> fromJwtString: aJwtString [

^self new intializeFromJwtString: aJwtString

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleAudience [
	^ 'jwtloginexampleuser@gemtalksystems.com'

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleAuthorizedParty [

	^ '123456789'

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleCreateJwtString [

"Return signed JWT string to be used with the JWT login example"

| jwt now |
jwt := JsonWebToken newForRsa256 .
jwt audience: self loginExampleAudience ;
	issuer: self loginExampleIssuer ;
	keyId: self loginExampleJwtKeyId ;
	subject: self loginExampleSubject ;
	authorizedParty: self loginExampleAuthorizedParty ;
	issuedAtTime: (now := System timeGmt) ;
	expirationTime: now + 7200 ; " good for 2 hours"
	signWithPrivateKey: self example_privateKey .
^jwt asJwtString

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleCreateUserProfile [

"Return the example user profile. Create and commit if if needed."

^ AllUsers userWithId: self loginExampleGsUserId ifAbsent:[ |up |
  	up := AllUsers addNewUserWithId: self loginExampleGsUserId password: 'swordfish'.
	up enableJwtAuthenticationWith: self loginExampleJwtSecurityData .
	System commit.
	up
].

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleGsUserId [
	^ 'JwtExampleUser'

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleIssuer [
	^ 'https://accounts.gemtalksystems.com'

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleJwtKeyId [

^ 'C2C587F2718953AE8A4B37307C1641F3'

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleJwtSecurityData [

| jwtSec |
jwtSec := JwtSecurityData new.
jwtSec 
	addAudience: self loginExampleAudience ;
	addIssuer: self loginExampleIssuer ;
	addUserId: self loginExampleJwtUserId ;
	addUserClaim: (JwtUserClaim newWithJsonKey: #sub jsonKind: #String acceptedValues: { self loginExampleSubject }) ; 
	addUserClaim: (JwtUserClaim newWithJsonKey: #azp jsonKind: #String acceptedValues: { self loginExampleAuthorizedParty}) .
^ jwtSec

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleJwtUserId [
	^ 'jwtloginexampleuser@gemtalksystems.com'

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleSubject [

	^ '123456789'

]

{ #category : 'JWT Login Example Helpers' }
JsonWebToken class >> loginExampleUserId [
	^ 'jwtloginexampleuser@gemtalksystems.com'

]

{ #category : 'Instance Creation' }
JsonWebToken class >> new [

^ super new initialize

]

{ #category : 'Instance Creation' }
JsonWebToken class >> newForHmac256 [

"Answer a new instance initialized for Hmac256 shared key authentication."
^ self new initializeForHmac256

]

{ #category : 'Instance Creation' }
JsonWebToken class >> newForRsa256 [

"Answer a new instance initialized for Rsa256 public key authentication."

^ self new initializeForRsa256

]

{ #category : 'Accessing Header Claims' }
JsonWebToken >> algorithm [

^ self headerClaimAt: #alg

]

{ #category : 'Updating Header Claims' }
JsonWebToken >> algorithm: object [

^ self headerClaimAt: #alg put: object

]

{ #category : 'Accessing Claims' }
JsonWebToken >> allClaims [

^ self allHeaderClaims addAll: self allPayloadClaims ; yourself

]

{ #category : 'Accessing Header Claims' }
JsonWebToken >> allHeaderClaims [

^ header keys

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> allPayloadClaims [

^ payload keys

]

{ #category : 'Converting' }
JsonWebToken >> asJwtString [

"Answer the receiver represented as a JWT string. The result string has 3 parts, each encoded in base64url format and separated by a period (.)
Raises an ImproperOperation exception if the receiver is not signed."

self isSigned
  ifFalse:[ ^ ImproperOperation signal: 'Cannot create a JWT string for an unsigned JWT' ].

^ self signedString copy 
       add: $. ;
       addAll: self signatureAsBase64Url ;
       yourself

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> audience [

^ payload at: #aud

]

{ #category : 'Updating Payload Claims' }
JsonWebToken >> audience: anObject [

^ self payloadClaimAt: #aud put: anObject

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> authorizedParty [

^ payload at: #azp

]

{ #category : 'Updating Payload Claims' }
JsonWebToken >> authorizedParty: anObject [

^ self payloadClaimAt: #azp put: anObject

]

{ #category : 'Private' }
JsonWebToken >> buildSignedString [

"Builds the base64url string to be signed from the header and payload and stores the result in the signedString instance variable."

| tmp |
tmp := self headerAsBase64Url .
tmp add: $. ;
	addAll: self payloadAsBase64Url .
signedString := tmp.
^ tmp

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> expirationTime [

^ payload at: #exp

]

{ #category : 'Updating Payload Claims' }
JsonWebToken >> expirationTime: anObject [

^ self payloadClaimAt: #exp put: anObject

]

{ #category : 'Private' }
JsonWebToken >> header [
	^header

]

{ #category : 'Private' }
JsonWebToken >> header: newValue [
	header := newValue

]

{ #category : 'Converting' }
JsonWebToken >> headerAsBase64Url [

^ self headerAsJson asBase64UrlString

]

{ #category : 'Converting' }
JsonWebToken >> headerAsJson [

^ header asJson

]

{ #category : 'Accessing Header Claims' }
JsonWebToken >> headerClaimAt: aSymbol [

^ header at: aSymbol asSymbol

]

{ #category : 'Updating Header Claims' }
JsonWebToken >> headerClaimAt: aSymbol put: object [

"Add or update a claim with key aSymbol in the header."

header at: aSymbol asSymbol put: object .
^ self

]

{ #category : 'Initialize' }
JsonWebToken >> initialize [

super initialize .
header := SymbolDictionary new.
payload := SymbolDictionary new.
^ self

]

{ #category : 'Initialize' }
JsonWebToken >> initializeForHmac256 [

"Intialize the receiver to be signed as an HMAC-SHA-256 JWT"


^ self
	type: 'JWT' ;
	algorithm: 'HS256' ;
	yourself

]

{ #category : 'Initialize' }
JsonWebToken >> initializeForRsa256 [

"Intialize the receiver to be signed as an RSA-SHA-256 JWT"

^ self
	type: 'JWT' ;
	algorithm: 'RS256' ;
	yourself

]

{ #category : 'Initialize' }
JsonWebToken >> intializeFromJwtString: aJwtString [

"Initializes the receiver from by decoding aJwtString which must be a Base64Url encoded string representing a signed JWT.
Returns the receiver."

| rs base64 json jp dict |
rs := ReadByteStreamPortable on: aJwtString .
base64 := rs upTo: $. .
json := (ByteArray fromBase64UrlString: base64) bytesIntoString .
jp := JsonParser new.
dict := jp parse: json.
dict keysAndValuesDo:[:k :v| header at: k asSymbol put: v ].
base64 := rs upTo: $. .
signedString := aJwtString copyFrom: 1 to: (rs position - 1).
json := (ByteArray fromBase64UrlString: base64) bytesIntoString .
dict := jp parse: json.
dict keysAndValuesDo:[:k :v| payload at: k asSymbol put: v ].
signature := ByteArray fromBase64UrlString: rs upToEnd .
^ self

]

{ #category : 'Testing' }
JsonWebToken >> isExpired [

"Answer true if the receiver has expired, false if not"

^ System timeGmt > self expirationTime

]

{ #category : 'Testing' }
JsonWebToken >> isHmac256 [

"Answer true if the receiver uses an HMAC256 signature, false if not."

^ (self headerClaimAt: #alg) = 'HS256'

]

{ #category : 'Testing' }
JsonWebToken >> isRsa256 [

"Answer true if the receiver uses an RSA256 signature, false if not."
^ (self headerClaimAt: #alg) = 'RS256'

]

{ #category : 'Testing' }
JsonWebToken >> isSigned [

"Answer true if the receiver has been signed, false if not."

^ signature ~~ nil

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> issuedAtTime [

^ payload at: #iat

]

{ #category : 'Updating Payload Claims' }
JsonWebToken >> issuedAtTime: anObject [

^ self payloadClaimAt: #iat put: anObject

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> issuer [

^ payload at: #iss

]

{ #category : 'Updating Payload Claims' }
JsonWebToken >> issuer: anObject [

^ self payloadClaimAt: #iss put: anObject

]

{ #category : 'Accessing Header Claims' }
JsonWebToken >> keyId [

^ self headerClaimAt: #kid

]

{ #category : 'Updating Header Claims' }
JsonWebToken >> keyId: object [

^ self headerClaimAt: #kid put: object

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> notBeforeTime [

^ payload at: #nbf

]

{ #category : 'Updating Payload Claims' }
JsonWebToken >> notBeforeTime: anObject [

^ self payloadClaimAt: #nbf put: anObject

]

{ #category : 'Private' }
JsonWebToken >> payload [
	^payload

]

{ #category : 'Private' }
JsonWebToken >> payload: newValue [
	payload := newValue

]

{ #category : 'Converting' }
JsonWebToken >> payloadAsBase64Url [

^ self payloadAsJson asBase64UrlString

]

{ #category : 'Converting' }
JsonWebToken >> payloadAsJson [

^ payload asJson

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> payloadClaimAt: aSymbol [

^ payload at: aSymbol asSymbol

]

{ #category : 'Updating Payload Claims' }
JsonWebToken >> payloadClaimAt: aSymbol put: object [

"Add or update a claim with key aSymbol in the payload."

payload at: aSymbol asSymbol put: object .
^ self

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> secondsUntilExpiration [

"Answer a SmallInteger indicating the number of seconds until the receiver expires, or zero if it has already expired"

| exp now |
exp := self expirationTime .
now := System timeGmt.
^ now >= exp
	ifTrue:[ 0 ] "already expired"
	ifFalse:[ exp - now ]

]

{ #category : 'Private' }
JsonWebToken >> signature [
	^signature

]

{ #category : 'Private' }
JsonWebToken >> signature: newValue [
	signature := newValue

]

{ #category : 'Converting' }
JsonWebToken >> signatureAsBase64Url [

^ signature asBase64UrlString

]

{ #category : 'Private' }
JsonWebToken >> signedString [
	^signedString

]

{ #category : 'Private' }
JsonWebToken >> signedString: newValue [
	signedString := newValue

]

{ #category : 'Signing' }
JsonWebToken >> signWithPrivateKey: aGsTlsPrivateKey [

"Builds the string to sign from the receiver's header and payload and signs it
using aGsTlsPrivateKey. The signature ByteArray is stored in the signature 
instance vaiable.

aGsTlsPrivateKey must be an RSA private key."

| sig |
self isRsa256
 	ifFalse:[ ^ CryptoError signal: 'Illegal attempt to sign JWT with private key' ].
self validateBeforeSigning .
self buildSignedString.
sig := ByteArray new.
self signedString signWithSha256AndRsaPrivateKey: aGsTlsPrivateKey into: sig.
self signature: sig.
^ self

]

{ #category : 'Signing' }
JsonWebToken >> signWithSecretKey: aByteCollection [

"Builds the string to sign from the receiver's header and payload and signs it
using aByteCollection. The signature is stored in a new ByteArray which is stored in the signature 
instance vaiable."

| sig |
self isHmac256
 	ifFalse:[ ^ CryptoError signal: 'Illegal attempt to sign RSA JWT with a secret key' ].
self validateBeforeSigning .
self buildSignedString.
sig :=  self signedString asSha256HmacByteArrayWithKey: aByteCollection .
self signature: sig.
^ self

]

{ #category : 'Accessing Payload Claims' }
JsonWebToken >> subject [

^ payload at: #sub

]

{ #category : 'Updating Payload Claims' }
JsonWebToken >> subject: anObject [

^ self payloadClaimAt: #sub put: anObject

]

{ #category : 'Accessing Header Claims' }
JsonWebToken >> type [

^ self headerClaimAt: #typ

]

{ #category : 'Updating Header Claims' }
JsonWebToken >> type: object [

^ self headerClaimAt: #typ put: object

]

{ #category : 'Validation' }
JsonWebToken >> validateBeforeSigning [

"Validate the receiver has at least one header claim and one payload claim.
No validation for the presence of specific claims is done.
It is not an error if the receiver already has a signature.
Returns true on success or raises an Exception on error."

header isEmpty 
	ifTrue:[ ^ ImproperOperation signal: 'A least one header claim is required' ] .
payload isEmpty 
	ifTrue:[  ^ ImproperOperation signal: 'A least one payload claim is required' ] .
^true

]

{ #category : 'Verifying' }
JsonWebToken >> verifySignatureNoErrorWithPublicKey: aGsTlsPublicKey [

"Verifies the signature of the receiver using the public key.
aGsTlsPublicKey must be an instance of either GsTlsPublicKey or GsX509Certificate. 
Returns true if the signature matches the receiver or false if not."

^ [self verifySignatureWithPublicKey: aGsTlsPublicKey asPublicKey ] on: CryptoError do:[:ex| false ]


]

{ #category : 'Verifying' }
JsonWebToken >> verifySignatureNoErrorWithSecretKey: aByteCollection [

"Verifies the signature of the receiver using the secret key aByteCollection.
  Returns true if the signature matches the receiver, false if it does not."

| sig |
self isHmac256
 	ifFalse:[ ^ CryptoError signal: 'Illegal attempt to verify RSA signed JWT with a secret key' ].

sig := self signedString asSha256HmacByteArrayWithKey: aByteCollection .
^ sig = self signature
  

]

{ #category : 'Verifying' }
JsonWebToken >> verifySignatureWithPublicKey: aGsTlsPublicKey [

"Verifies the signature of the receiver using the public key.
  aGsTlsPublicKey must be an instance of either GsTlsPublicKey or GsX509Certificate.
  Returns true if the signature matches the receiver.
  Raises an exception if the signature does not match."

self isRsa256
 	ifFalse:[ ^ CryptoError signal: 'Illegal attempt to verify JWT with public key' ].
^ self signedString verifyWithSha256AndRsaPublicKey: aGsTlsPublicKey asPublicKey signature: self signature

]

{ #category : 'Verifying' }
JsonWebToken >> verifySignatureWithSecretKey: aByteCollection [

"Verifies the signature of the receiver using the secret key aByteCollection.
  Returns true if the signature matches the receiver.
  Raises an exception if the signature does not match."

| sig |
self isHmac256
 	ifFalse:[ ^ CryptoError signal: 'Illegal attempt to verify RSA signed JWT with a secret key' ].

sig := self signedString asSha256HmacByteArrayWithKey: aByteCollection .
^ sig = self signature
  ifTrue:[ true ]
  ifFalse:[ CryptoError signal: 'signature does not match' ].

]
