"
JsonParser is a recursive decent parser that parses JSON formatted strings.  
In versions 3.5-3.6.4, the JsonParser was a PetitParser class; this JsonParser is a 
smaller, faster implementation not related to PetitParser.

The older implementation is now named JsonPetitParser. If you have subclasses of 
JsonParser in earlier releases that depend on PetitParser behavior or variables, 
modify these to be subclasses of JsonPetitParser.

The following is a Json BNF from  https://www.json.org/

json
    element

value
    object
    array
    string
    number
    ""true""
    ""false""
    ""null""

object
    '{' ws '}'
    '{' members '}'

members
    member
    member ',' members

member
    ws string ws ':' element

array
    '[' ws ']'
    '[' elements ']'

elements
    element
    element ',' elements

element
    ws value ws

string
    '""' characters '""'

characters
    """"
    character characters

character
    '0020' . '10ffff' - '""' - '\'
    '\' escape

escape
    '""'
    '\'
    '/'
    'b'
    'n'
    'r'
    't'
    'u' hex hex hex hex

hex
    digit
    'A' . 'F'
    'a' . 'f'

number
    int frac exp

int
    digit
    onenine digits
    '-' digit
    '-' onenine digits

digits
    digit
    digit digits

digit
    '0'
    onenine

onenine
    '1' . '9'

frac
    """"
    '.' digits

exp
    """"
    'E' sign digits
    'e' sign digits

sign
    """"
    '+'
    '-'

ws
    """"
    '0009' ws
    '000a' ws
    '000d' ws
    '0020' ws
"
Class {
	#name : 'JsonParser',
	#superclass : 'Object',
	#instVars : [
		'stream',
		'linePosition',
		'lineNumber'
	],
	#category : 'ObjectSerialization'
}

{ #category : 'other' }
JsonParser class >> parse: aString [

	^self new
		initialize: aString;
		_value
]

{ #category : 'other' }
JsonParser >> _value [

	| char |
	char := self peekChar.
	char == ${	ifTrue: [^self object	].
	char == $[	ifTrue: [^self array	].
	char == $"	ifTrue: [^self string	].
	char == $t	ifTrue: [self assert: (stream next: 4) is: 'true'	. ^true	].
	char == $f	ifTrue: [self assert: (stream next: 5) is: 'false'	. ^false	].
	char == $n	ifTrue: [self assert: (stream next: 4) is: 'null'	. ^nil		].
	^self number

]

{ #category : 'other' }
JsonParser >> array [

	| array char |
	array := Array new.
	self assert: self nextChar is: $[.
	[
		(char := self peekChar) ifNil: [self error: 'Unterminated array!'].
		char ~~ $].
	] whileTrue: [ | v |
    v := self _value.
		array add: v.
		char := self peekChar.
		char == $, ifTrue: [
			self nextChar. 
			char := self peekChar.
		].
	].
	self assert: self nextChar is: $].
	^array

]

{ #category : 'other' }
JsonParser >> assert: actual is: expected [
	actual = expected ifTrue: [^self].
	Error signal: 'Invalid JSON, at ', self positionAsString,
        ' expected ' , expected printString,' actual ', actual printString
]

{ #category : 'other' }
JsonParser >> initialize: aString [

	stream := ReadByteStream on: aString.
  linePosition := 1 .
  lineNumber := 1 .
]

{ #category : 'other' }
JsonParser >> nextChar [

	| char |
	char := self peekChar.
	char ifNotNil: [stream next].
	^char

]

{ #category : 'other' }
JsonParser >> number [

	| char sign number fraction divisor exponent |
	char := self peekChar.
	char == $- ifTrue: [
		sign := -1.
		char := stream next; peek.
	] ifFalse: [
		sign := 1.
	].
	number := 0.
  char isDigit ifFalse:[ 
     Error signal: 'Invalid JSON, at ', self positionAsString,
          'expected a digit, got ', char printString .
  ].
	char == $0 ifTrue: [
		char := stream next; peek.
	] ifFalse: [
		[char ~~ nil and: [char isDigit]] whileTrue: [
			number := number * 10 + char codePoint - $0 codePoint.
			char := stream next; peek.
		].
	].
	divisor := 1.0.
	fraction := 0.
	char == $. ifTrue: [
		[
			char := stream next; peek.
			char notNil and: [char isDigit].
		] whileTrue: [
			fraction := fraction * 10 + char codePoint - $0 codePoint.
			divisor := divisor * 10.
			
		].
		number := number + (fraction / divisor).
	].
	number := number * sign.
	(char == $e or: [char == $E]) ifTrue: [
		char := stream next; peek.
		sign := 1.
		char == $- ifTrue: [sign := -1. char := stream next; peek] ifFalse: [
		char == $+ ifTrue: [char := stream next; peek]].
		exponent := 0.
		[char notNil and: [char isDigit]] whileTrue: [
			exponent := exponent * 10 +char codePoint - $0 codePoint.
			char := stream next; peek.
		].
		number := number * (10.0 raisedTo: exponent * sign).
	].
	^number

]

{ #category : 'other' }
JsonParser >> object [

	| char key object |
	object := Dictionary new.
	self assert: self nextChar is: ${.
	[
		(char := self peekChar) ifNil: [self error: 'Unterminated object!'].
		char ~~ $}.
	] whileTrue: [
		key := self string.
		self assert: self nextChar is: $:.
		object at: key put: self _value.
		char := self peekChar.
		char == $, ifTrue: [
			self nextChar.
			char := self peekChar.
		].
	].
	self nextChar.
	^object

]

{ #category : 'other' }
JsonParser >> parse: aString [

    ^self
        initialize: aString;
        _value
]

{ #category : 'other' }
JsonParser >> peekChar [
	| char isSep |
	[
		stream atEnd ifTrue: [^nil].
		char := stream peek.
			(isSep := char isSeparator) ifTrue:[
				 char codePoint == 10 ifTrue:[ 
         lineNumber := lineNumber + 1 .  
         linePosition := stream position + 2 .
       ].
    ].
    isSep
	] whileTrue: [stream next].
	^char

]

{ #category : 'other' }
JsonParser >> positionAsString [
  | pos |
  ^  'stream position ', (pos := stream position) asString, 
        ' (at ', (pos - linePosition + 1) asString , ' in line ', lineNumber asString, $)
]

{ #category : 'other' }
JsonParser >> string [
	"Began with double quotes character"

	| char str |
	str := String new.
	self assert: self nextChar is: $".
	[
		char := stream next.
		char ~~ $".
	] whileTrue: [
		char == $\ ifTrue: [
			char := stream next.
			char == $"	ifTrue: [ str add: $"	] ifFalse: [
			char == $\	ifTrue: [ str add: $\	] ifFalse: [
			char == $/	ifTrue: [ str add: $/	] ifFalse: [
			char == $b	ifTrue: [ str addCodePoint: 8	] ifFalse: [
			char == $f	ifTrue: [ str addCodePoint: 12	] ifFalse: [
			char == $n	ifTrue: [ str addCodePoint: 10	] ifFalse: [
			char == $r	ifTrue: [ str addCodePoint: 13	] ifFalse: [
			char == $t	ifTrue: [ str addCodePoint: 9	] ifFalse: [
			char == $u	ifTrue: [
				| code |
				code := '16r' , (stream next: 4).
				str add: (Character codePoint: code asNumber).
			]]]]]]]]]
		] ifFalse: [
			str add: char.
		].
	].
	^ str
]

{ #category : 'other' }
JsonParser >> test [
"
	GciJsonParser new test.
"

	| val |
	self initialize: 'true'.	self assert: self _value is: true.
	self initialize: 'false'.	self assert: self _value is: false.
	self initialize: 'null'.	self assert: self _value is: nil.
	"number"
	self initialize: '0'.			self assert: self _value is: 0.
	self initialize: '1'.			self assert: self _value class is: SmallInteger.
	self initialize: '123'.		self assert: self _value is: 123.
	self initialize: '-456'.		self assert: self _value is: -456.
	self initialize: '0.'.			self assert: self _value is: 0.
	self initialize: '0.0'.		self assert: self _value is: 0.
	self initialize: '0.25'.		self assert: self _value is: 0.25.
	self initialize: '0.25e2'.	self assert: self _value is: 25.
	self initialize: '25e-2'.	self assert: self _value is: 0.25.
	"string"
	self initialize: '"abc"'.		self assert: self _value is: 'abc'.
	self initialize: '"x\ty"'.	self assert: self _value is: 'x	y'.
	"array"
	self initialize: ' [ 1 , 2 ] '.	self assert: self _value is: #(1 2).
	"object"
	self initialize: '{"x": 1, "y": [2], "z": 3}'.
	val := self _value.
	self
		assert: val size is: 3;
		assert: (val at: 'x') is: 1;
		assert: (val at: 'y') is: #(2);
		assert: (val at: 'z') is: 3 .
]
