"
A IdentitySoftCollisionBucket is an IdentityCollisionBucket intended to
 store key/SoftReference pairs .
 key/SoftReference pairs whose SoftReferences have been cleared by 
 the in-memory garbage collector will be removed from a 
 IdentitySoftCollisionBucket during the next at:put: operation 
 following a garbage collection which cleared the references.  
 
 Cleared SoftReferences will also be removed when their key matches
 the argument key for a lookup operation .  The lookup code will
 not remove cleared SoftReferences for non-matching keys.

 Instances of IdentitySoftCollisionBucket are non-persistent and may
 not be committed to the repository.

Constraints:
	numElements: SmallInteger
	keyValueDictionary: Object
	lastCleanupCount: SmallInteger

instVar lastCleanupCount -- Private to the implementation of IdentitySoftCollisionBucket, used to manage
 removal of SoftReferences.
"
Class {
	#name : 'IdentitySoftCollisionBucket',
	#superclass : 'IdentityCollisionBucket',
	#instVars : [
		'lastCleanupCount'
	],
	#gs_options : [
		'disallowGciStore',
		'instancesNonPersistent'
	],
	#gs_reservedoop : '121089',
	#category : 'Collections-Internals'
}

{ #category : 'Instance creation' }
IdentitySoftCollisionBucket class >> new [

^ super new _initializeCleanup

]

{ #category : 'Cleanup' }
IdentitySoftCollisionBucket >> _cleanupReferences: okToRemoveBucket [
| currCount |
currCount := self _markSweepsThatClearedSoftRefsCount .
currCount = lastCleanupCount ifFalse:[ | lastKeyIdx startNumElem aKey |
  startNumElem := numElements .
  lastKeyIdx := (startNumElem + startNumElem) - 1 .
  lastKeyIdx _downTo: 1 by: 2 do:[ :idx | | aRef aVal |
    aRef := self _at: idx + 1.
    aVal := aRef _value  .
    aVal ifNil:[
      aKey ifNil:[ aKey := self _at: idx ].
      self removeFrom: idx  to: idx+ 1 . "remove key/softRef pair"
      numElements := numElements - 1 .
    ].
  ].
  lastCleanupCount := currCount .
  numElements == startNumElem ifFalse:[
    keyValueDictionary _bucketSizeChangeFrom: startNumElem to: numElements
		key: aKey bucket: self remove: okToRemoveBucket
  ].
].

]

{ #category : 'Private' }
IdentitySoftCollisionBucket >> _initializeCleanup [

  lastCleanupCount := 0

]

{ #category : 'Private' }
IdentitySoftCollisionBucket >> _markSweepsThatClearedSoftRefsCount [

"Returns OM.markSweepsThatClearedSoftRefsCount as a positive SmallInteger"

<primitive: 553>

self _primitiveFailed: #_markSweepsThatClearedSoftRefsCount

]

{ #category : 'Private' }
IdentitySoftCollisionBucket >> _referenceAt: idx [

" for the key stored at offset idx ,
  return the SofReference if softRef.value non-nil, otherwise
  remove the soft reference and associated key .
  Does not increment the use count of softRef .
"
| aSoftRef val |
aSoftRef := self _at: idx + 1 .
aSoftRef ~~ nil ifTrue:[
  val := aSoftRef _value .
  val == nil ifTrue:[ | oldN n aKey |
    aKey := self _at: idx .
    self removeFrom: idx to: idx + 1 . "remove key and SoftRef"
    oldN := numElements .
    n := oldN - 1 .
    n < 0 ifTrue:[ nil error:'numElements underflow in a SoftCollisionBucket'].
    numElements := n .
    keyValueDictionary _bucketSizeChangeFrom: oldN to: n key: aKey
                bucket: self remove: false .
    ^ nil .
  ] .
  ^ aSoftRef
].
^ nil

]

{ #category : 'Updating' }
IdentitySoftCollisionBucket >> at: aKey put: aValue [

"disallowed, should use  at:put:keyValDict_coll:  "

self shouldNotImplement: #at:put:

]

{ #category : 'Updating' }
IdentitySoftCollisionBucket >> at: aKey put: aValue keyValDict_coll: aKeyValDict [

"Stores the aKey/aValue pair in the receiver.
 Returns self size if this at:put: added a new key, 0 if this at:put:
 replaced a SoftReference for an existing  key .

 aValue is expected to be a SoftReference.

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

self _cleanupReferences: false .
^ super at: aKey put: aValue keyValDict_coll: aKeyValDict

]

{ #category : 'Cleanup' }
IdentitySoftCollisionBucket >> cleanupReferences [
  ^ self _cleanupReferences: true

]

{ #category : 'Accessing' }
IdentitySoftCollisionBucket >> referenceAt: aKey ifAbsent: aBlock [

"Returns the non-cleared SoftReference that corresponds to aKey.
 If no such key/SoftRef pair exists,
 returns the result of evaluating the zero-argument block aBlock."

  | index keyIndex aSoftRef |
  keyIndex := self searchForKey: aKey.
  keyIndex ~~ nil ifTrue: [
    index := (keyIndex + keyIndex) - 1 .
    aSoftRef:=  self _referenceAt: index .
    aSoftRef == nil ifTrue:[ ^ self _reportKeyNotFound: aKey with: aBlock ] .
    ^ aSoftRef
  ].
  ^ self _reportKeyNotFound: aKey with: aBlock

]

{ #category : 'Accessing' }
IdentitySoftCollisionBucket >> referenceAt: aKey otherwise: aValue [

"Returns the non-cleared SoftReference that corresponds to aKey.
 If no such key/SoftRef pair exists, returns aValue ."

  | keyIndex index aSoftRef |
  keyIndex := self searchForKey: aKey.
  keyIndex ~~ nil ifTrue:[
    index := (keyIndex + keyIndex) - 1 .
    aSoftRef :=  self _referenceAt: index .
    aSoftRef == nil ifTrue:[ ^ aValue ].
    ^ aSoftRef
  ].
  ^ aValue

]

{ #category : 'Removing' }
IdentitySoftCollisionBucket >> removeKey: aKey ifAbsent: aBlock [

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the result of evaluating the zero-argument block aBlock.

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

| res |
res := super removeKey: aKey ifAbsent: aBlock  .
self _cleanupReferences: true .
^ res

]

{ #category : 'Removing' }
IdentitySoftCollisionBucket >> removeKey: aKey otherwise: notFoundValue [

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the notFoundValue .

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

| res |
res := super removeKey: aKey otherwise: notFoundValue .
self _cleanupReferences: true .
^ res

]
