"
Abstract. Part of the internal implemention of hash tree collections.
Helps the collection in walking through nodes to find, add, or remove elements.
"
Class {
	#name : 'HtTreeWalker',
	#superclass : 'Object',
	#instVars : [
		'collection',
		'found',
		'value'
	],
	#category : 'Collections-Internals'
}

{ #category : 'instance creation' }
HtTreeWalker class >> forCollection: aCollection [
	^ self new
		collection: aCollection;
		yourself
]

{ #category : 'auditing' }
HtTreeWalker >> auditOnto: stream for: aCollection [
	"I'm dbTransient, so it is legit for all of my contents to be nil.
	But if non-nil, should be correct."

	| identifier |
	identifier := self class name , ' ' , self asOop printString , ' '.
	found
		ifNotNil: [ 
			found == false
				ifFalse: [ 
					stream
						nextPutAll:
								identifier , 'found is ' , found printString , ', should be false';
						lf ] ].
	collection
		ifNotNil: [ 
			collection == aCollection
				ifFalse: [ 
					stream
						nextPutAll:
								identifier , 'collection is ' , collection printString , ' with oop '
										, collection asOop printString
										, ' should be the collection I belong to';
						lf ] ].
	value
		ifNotNil: [ 
			stream
				nextPutAll:
						identifier , 'value is ' , value printString , ' with oop '
								, value asOop printString , ' should be nil';
				lf ]
]

{ #category : 'accessing' }
HtTreeWalker >> collection: aCollection [
	collection := aCollection
]

{ #category : 'accessing' }
HtTreeWalker >> found [
	^ found
]

{ #category : 'accessing' }
HtTreeWalker >> reset [
	found := false.
	value := nil
]

{ #category : 'private-query' }
HtTreeWalker >> searchInteriorNode: node forValueAt: key withHash: hash [
	| child childIndex maxIndex nextHash |
	childIndex := node lowestChildIndexForHash: hash.
	nextHash := hash.
	maxIndex := node maxChildIndex.
	[ nextHash = hash and: [ childIndex <= maxIndex and: [ found not ] ] ]
		whileTrue: [ 
			child := node at: childIndex.
			self searchNode: child forValueAt: key withHash: hash.
			nextHash := node at: childIndex + 1.
			childIndex := childIndex + 2 ]
]

{ #category : 'private-removing' }
HtTreeWalker >> searchInternalNode: node isFinalSibling: nodeIsFinal forRemoveKey: key withHash: hash [
	| child childIndex maxIndex nextHash childPercentFull |
	childIndex := node lowestChildIndexForHash: hash.
	nextHash := hash.
	maxIndex := node maxChildIndex.
	[ nextHash = hash and: [ childIndex <= maxIndex and: [ found not ] ] ]
		whileTrue: [ 
			child := node at: childIndex.
			nextHash := node at: childIndex + 1.

			childPercentFull := self
				searchNode: child
				isFinalSibling: (nodeIsFinal and: [ nextHash > hash ])
				forRemoveKey: key
				withHash: hash.
			childIndex := childIndex + 2 ].

	(found and: [ childPercentFull <= 40 ])
		ifTrue: [ 
			| childSibling "Child has gotten small. Try to consolidate it with a sibling." |
			"Try right sibling first."
			childIndex <= maxIndex
				ifTrue: [ 
					childSibling := node at: childIndex.
					childPercentFull + childSibling percentFull <= 85
						ifTrue: [ 
							child absorbRightSibling: childSibling.
							node removeLeftwardChildAtIndex: childIndex.
							^ node percentFull ] ].
			childIndex := childIndex - 4.
			childIndex >= 2
				ifTrue: [ 
					childSibling := node at: childIndex.
					childPercentFull + childSibling percentFull <= 85
						ifTrue: [ 
							child absorbLeftSibling: childSibling.
							node removeRightwardChildAtIndex: childIndex ] ] ].
	^ node percentFull
]

{ #category : 'private-query' }
HtTreeWalker >> searchLeafNode: node forValueAt: key withHash: hash [
	self subclassResponsibility
]

{ #category : 'private-removing' }
HtTreeWalker >> searchLeafNode: node isFinalSibling: nodeIsFinal forRemoveKey: key withHash: hash [
	self subclassResponsibility
]

{ #category : 'private-query' }
HtTreeWalker >> searchNode: node forValueAt: key withHash: hash [

	node isLeaf
		ifTrue: [ self searchLeafNode: node forValueAt: key withHash: hash ]
		ifFalse: [
		self searchInteriorNode: node forValueAt: key withHash: hash ]
]

{ #category : 'private-removing' }
HtTreeWalker >> searchNode: node isFinalSibling: nodeIsFinal forRemoveKey: key withHash: hash [
	"Answer the percent full (as an integer) that the  node is after removal"

	^ node isLeaf
		ifTrue: [ 
			self
				searchLeafNode: node
				isFinalSibling: nodeIsFinal
				forRemoveKey: key
				withHash: hash ]
		ifFalse: [ 
			self
				searchInternalNode: node
				isFinalSibling: nodeIsFinal
				forRemoveKey: key
				withHash: hash ]
]

{ #category : 'public' }
HtTreeWalker >> searchTree: rootNode forValueAt: key withHash: hash [
	found := false.
	self searchNode: rootNode forValueAt: key withHash: hash.
	^ found
]

{ #category : 'public' }
HtTreeWalker >> searchTree: rootNode removeKey: key withHash: hash [
	"Answer nil, or new root node if the tree is losing a level.
	Root node has no siblings, so is always the final sibling."

	found := false.
	self
		searchNode: rootNode
		isFinalSibling: true
		forRemoveKey: key
		withHash: hash.
	^ rootNode isDegenerate
		ifTrue: [ rootNode soleChild ]
		ifFalse: [ nil ]
]

{ #category : 'accessing' }
HtTreeWalker >> value [
	^value
]
