"
HtLeafNode (hash tree leaf node) is an abstract superclass for a 
page-sized hash table that uses open-addressing with linear probing. 
This gives good performance in modern memory hierarchies by 
needing to read fewer cache lines than alternative hash table 
architectures such as those that use buckets.

Instance variables:

tally		    The number of entries I contain
collection	The collection that I am part of

lowestHash	I am guaranteed to contain no keys whose permuted hash is less than this integer.
     A copy of the hash value to the left of the pointer to me in my parent node (if any).
					
highestHash		I am guaranteed to contain no keys whose permuted hash is greater than this integer.
     A copy of the hash value to the right of the pointer to me in my parent node (if any).

Cached constants -- these vary by subclass, but are the same for all instances of each subclass, and 
             are cached to avoid re-computing them all the time:
arraySize	   The number of indexed instance variables I contain
tableSize		 The number of entry slots I have in my table. Should be prime for good performance.
fillLine		 The number of entries which I can contain before being considered ""full"" and splitting.
"
Class {
	#name : 'HtLeafNode',
	#superclass : 'Array',
	#instVars : [
		'tableSize',
		'arraySize',
		'tally',
		'collection',
		'fillLine',
		'lowestHash',
		'highestHash'
	],
	#category : 'Collections-Internals'
}

{ #category : 'instance creation' }
HtLeafNode class >> forCollection: coll lowestHash: lowestHash highestHash: highestHash [
	^ (self new: self nodeBasicSize)
		initialize;
		collection: coll;
		lowestHash: lowestHash;
		highestHash: highestHash;
		yourself
]

{ #category : 'instance creation' }
HtLeafNode class >> nodeBasicSize [
	"To keep it in a page, limit is 2034 - self instSize
	For us, with seven named instvars, that is 2027."

	self subclassResponsibility
]

{ #category : 'walker access' }
HtLeafNode >> absorbLeftSibling: otherLeaf [
	"Copy all elements in otherLeaf to myself.
	otherLeaf must be the sibling adjacent to my left
	(lower hash values).
	Sender is responsible for assuring that I have sufficient room.
	otherLeaf is about to be discarded, don't bother to remove
	elements from it."

	self copyElementsFrom: otherLeaf.
	lowestHash := otherLeaf lowestHash
]

{ #category : 'walker access' }
HtLeafNode >> absorbRightSibling: otherLeaf [
	"Copy all elements in otherLeaf to myself.
	otherLeaf must be the sibling adjacent to my right
	(higher hash values).
	Sender is responsible for assuring that I have sufficient room.
	otherLeaf is about to be discarded, don't bother to remove
	elements from it."

	self copyElementsFrom: otherLeaf.
	highestHash := otherLeaf highestHash
]

{ #category : 'auditing' }
HtLeafNode >> auditOnto: stream for: aCollection lowestHash: parentLow highestHash: parentHigh [
	" Things to check:
	Are the constant instvars holding the correct constants?
		arraySize
		fillLine
		tableSize
	collection identical to aCollection?
	highestHash and lowestHash equal to parent ones passed in?
	Scan indexed instvars
		Is each key where it should be?
		Is each nil key accompanied by a nil value?
		Is the total number of non-nil key slots equal to tally?
"

	| identifier pairCount |
	identifier := 'LeafNode ' , self asOop printString , ' '.

	self _isLarge
		ifTrue: [ 
			stream
				nextPutAll: identifier , 'is a large object, should fit in one page.';
				lf ].

	arraySize = 2026 & (tableSize = 1013) & (fillLine = 759)
		ifFalse: [ 
			stream
				nextPutAll:
						identifier
								,
									'one or more constants are incorrect. Expected: arraySize = 2026, tableSize = 1013, fillLine = 759. Actual: arraySize =  '
								, arraySize printString , ' tableSize = ' , tableSize printString
								, ' fillLine = ' , fillLine printString;
				lf ].

	collection == aCollection
		ifFalse: [ 
			stream
				nextPutAll:
						identifier , 'collection is ' , collection printString , ' with oop '
								, collection asOop printString
								, ' should be the collection I belong to';
				lf ].

	highestHash = parentHigh
		ifFalse: [ 
			stream
				nextPutAll:
						identifier , 'highestHash should be equal to parent hash of '
								, parentHigh printString , ' but is ' , highestHash printString;
				lf ].

	lowestHash = parentLow
		ifFalse: [ 
			stream
				nextPutAll:
						identifier , 'lowestHash should be equal to parent hash of '
								, parentLow printString , ' but is ' , lowestHash printString;
				lf ].

	highestHash < lowestHash
		ifTrue: [ 
			stream
				nextPutAll:
						identifier , 'lowestHash ' , lowestHash printString
								, ' is greater than highestHash ' , highestHash printString;
				lf ].

	pairCount := 0.

	1 to: arraySize by: 2 do: [ :i | 
		| key value |
		key := self at: i.
		value := self at: i + 1.
		key
			ifNil: [ 
				value
					ifNotNil: [ 
						stream
							nextPutAll:
									identifier , 'nil key at index ' , i printString , ' has non-nil value '
											, value printString;
							lf ] ]
			ifNotNil: [ 
				| expectedIndex |
				pairCount := pairCount + 1.
				expectedIndex := self
					indexForKey: key
					withHash: (collection permutedHashOf: key).
				i = expectedIndex
					ifFalse: [ 
						stream
							nextPutAll:
									identifier , 'key at index ' , i printString , ' was expected to be at index '
											, expectedIndex printString;
							lf ] ] ].

	tally = pairCount
		ifFalse: [ 
			stream
				nextPutAll:
						identifier , 'tally is ' , tally printString
								, ' but the number of keys found was ' , pairCount printString;
				lf ].

	^ pairCount
]

{ #category : 'accessing' }
HtLeafNode >> collection: aCollection [
	collection := aCollection
]

{ #category : 'copying' }
HtLeafNode >> copyForCollection: aCollection [
	| copy |
	copy := self shallowCopy.
	copy collection: aCollection.
	^ copy
]

{ #category : 'accessing' }
HtLeafNode >> highestHash [
	^highestHash
]

{ #category : 'accessing' }
HtLeafNode >> highestHash: object [
	highestHash := object
]

{ #category : 'testing' }
HtLeafNode >> isDegenerate [
	"A leaf node, if it is the root, is allowed to be completely empty."

	^ false
]

{ #category : 'testing' }
HtLeafNode >> isFull [

	^ tally >= fillLine
]

{ #category : 'testing' }
HtLeafNode >> isLeaf [

	^ true
]

{ #category : 'accessing' }
HtLeafNode >> lowestHash [

	^ lowestHash
]

{ #category : 'accessing' }
HtLeafNode >> lowestHash: anObject [

	lowestHash := anObject
]

{ #category : 'accessing' }
HtLeafNode >> percentFull [

	^ tally * 100 // fillLine
]

{ #category : 'private' }
HtLeafNode >> replaceAllWith: aLeafNode [
	"Destroy any contents I may have, and replace them with those of aLeafNode.
	This does not touch any of my named instvars except tally."

	tally := aLeafNode tally.
	self
		replaceFrom: 1
		to: arraySize
		with: aLeafNode
		startingAt: 1
]

{ #category : 'accessing' }
HtLeafNode >> tally [

	^ tally
]
