"
GsX509Certificate encapsulates a TLS/SSL CA certificate per X.509.
"
Class {
	#name : 'GsX509Certificate',
	#superclass : 'GsTlsCredential',
	#category : 'Credentials'
}

{ #category : 'Class Membership' }
GsX509Certificate class >> isOpenSshClass [

^ false
]

{ #category : 'Instance Creation' }
GsX509Certificate class >> newFromPemFile: fileNameString [
"Reads data from the given file name in PEM format and creates a new instance
 of the receiver. If the PEM file contains multiple certificates, only
 the first is read.

 Raises an exception if the file is not in PEM format or if
 the type of object in the file does not match the receiver."

^ self tls3ArgPrim: 0 with: fileNameString with: nil with: 3

]

{ #category : 'Instance Creation' }
GsX509Certificate class >> newFromPemString: aPemString [
"Creates a new instance of the receiver based on the PEM string. If the PEM
 string contains multiple certificates, only the first is read.

 Raises an exception if the string is not in PEM format or if the type of object in
 the PEM string does not match the receiver."

^ self tls3ArgPrim: 1 with: aPemString with: nil with: 3

]

{ #category : 'Converting' }
GsX509Certificate >> asOpenSshKey [
  ^ self class speciesForOpenSshPublicKey newFromOpenSshString: self asPublicKey asOpenSshString
]

{ #category : 'Converting' }
GsX509Certificate >> asOpenSslKey [
  ^ self asPublicKey
]

{ #category : 'Converting' }
GsX509Certificate >> asPublicKey [
"Extract the public key from the receiver and return a new instance
 of the public key class for the receiver (default: GsTlsPublicKey)."

 ^ self publicKeySpecies fromCertificate: self

]

{ #category : 'Testing' }
GsX509Certificate >> canCreateDigitalSignatures [

^ false

]

{ #category : 'Testing' }
GsX509Certificate >> canVerifyDigitalSignatures [
^ self supportsDigitalSignatures

]

{ #category : 'Encrypting' }
GsX509Certificate >> encrypt: aByteObj [
"Encrypts aByteObj using the receiver which must be an RSA public key 
 or X509 certificate object. This method supports RSA keys only.
 Encryption is performed using the RSA-OAEP padding algorithm. 

 Only small amounts of data may be encrypted using asymmetric 
 keys. The maximum allowed size of aByteObj is dependent on the 
 the number of bits in the receiver's public key. For 2048 bit keys, 
 the maximum size is 214 bytes. For 4096 bit keys, the maximum size is
 470 bytes.

 aByteObj must be an instance of a byte class, typically ByteArray.
 Byte objects with character sizes greater than 1 are not supported.

 Returns a new instance of ByteArray containing the encrypted data.
 Raises an exception on error."

^ self tls1ArgInstPrim: 3 with: aByteObj
]

{ #category : 'Testing' }
GsX509Certificate >> isPrivateKey [
 ^ false

]

{ #category : 'Testing' }
GsX509Certificate >> isPublicKey [
 ^ false

]

{ #category : 'Testing' }
GsX509Certificate >> isSelfSigned [

"Answer a Boolean indicating if the receiver is a self-signed certificate."

^self subjectName = self issuerName

]

{ #category : 'Accessing' }
GsX509Certificate >> issuerName [

"Returns a string representing the issuer common name of the receiver."

^ self tls0ArgInstPrim: 10
]

{ #category : 'Testing' }
GsX509Certificate >> isValidNow [

"Answer a Boolean indicating if the receiver is valid at this point in time,
 that is the current time falls within the window between the receiver's 'not before'
 and 'not after' times."

| gmtNow |
gmtNow := System timeGmt .
^ (gmtNow >= self notBeforeTimeGmtSeconds) and:[ gmtNow <= self notAfterTimeGmtSeconds ]

]

{ #category : 'Testing' }
GsX509Certificate >> isX509Certificate [
 ^ true

]

{ #category : 'Accessing' }
GsX509Certificate >> notAfterTime [

"Returns a SmallDateAndTime representing the 'not after' time of the receiver in GMT"

 ^ DateAndTime posixSeconds: self notAfterTimeGmtSeconds offset: Duration zero
]

{ #category : 'Accessing' }
GsX509Certificate >> notAfterTimeGmtSeconds [

"Returns a SmallInteger representing 'not after' time of the receiver 
 expressed as the number of seconds since 00:00:00UTC January 1, 1970."
 
^ self tls0ArgInstPrim: 12
]

{ #category : 'Accessing' }
GsX509Certificate >> notBeforeTime [

"Returns a SmallDateAndTime representing the 'not before' time of the receiver in GMT"

 ^ DateAndTime posixSeconds: self notBeforeTimeGmtSeconds offset: Duration zero
]

{ #category : 'Accessing' }
GsX509Certificate >> notBeforeTimeGmtSeconds [

"Returns a SmallInteger representing 'not before' time of the receiver 
 expressed as the number of seconds since 00:00:00UTC January 1, 1970."
 
 ^ self tls0ArgInstPrim: 11
]

{ #category : 'Converting' }
GsX509Certificate >> publicKeySpecies [
 ^ GsTlsPublicKey

]

{ #category : 'Accessing' }
GsX509Certificate >> subjectAlternateNames [

"Returns an Array of Strings representing the contents of the subject alternate
 extension contained in the receiver or an empty array if the receiver does not
 contain the extension."

^ self tls0ArgInstPrim: 13
]

{ #category : 'Accessing' }
GsX509Certificate >> subjectName [

"Returns a string representing the subject common name of the receiver."

^ self tls0ArgInstPrim: 9
]
