"
A GsSshSocket represents a SSH socket connection, allowing you to connect to another host using 
SSH and perform commands. 

To create a SSH connection, create a client socket using methods inherited from GsSocket, and 
connect using sshConnect. 

The default userId is the userId of the current session. If passwordless login is setup to the 
remote host, you do not need to specify the password or private key.  

  sshSock := GsSshSocket newClient.
  sshSock connectToHost: <aHostnameOrIP> timeoutMs: 2000.
  ""depending on configuration""  sshSock userId: <userName>.
  ""depending on configuration""  sshSock privateKey: <aTlsPrivateKey>. 
  ""depending on configuration""  sshSock privateKey: <aGsSshPrivateKey>. 
  ""depending on configuration""  sshSock password: <passwordString>. 
  ""depending on configuration""  sshSock disableHostAuthentication .
  sshSock sshConnect.

You can then execute command using executeRemoteCommand:, or other methods to perform non-blocking 
query. Close the socket when you no longer need the connection. 

   result := sshSock executeRemoteCommand: 'ulimit -a'.
   sshSock close.
"
Class {
	#name : 'GsSshSocket',
	#superclass : 'GsSignalingSocket',
	#gs_options : [
		'noInheritOptions'
	],
	#category : 'OSAccess-Sockets'
}

{ #category : 'Private' }
GsSshSocket class >> _oneArgSshPrim: opcode with: arg1 [
"GsSshSocket opCodes
opcode	method
1		instance method: hostAuthenticationEnabled:
2		instance method: password:
3		instance method: privateKey:
4		instance method: executeRemoteCommand:
5		instance method: makeBlocking / makeNonBlocking
6		instance method: nbExecuteRemoteCommand:
100		class method enableTraceFileInDirectory:
101		class method setSshTraceLevel:

GsSftpSocket opCodes
50		instance method: contentsOfRemoteDirectory
51		instance method: removeRemoteDirectory:
52		instance method: contentsAndStatDetailsOfRemoteDirectory:
53		instance method: removeRemoteFile:
54		instance method: stat:
55		instance method: lstat:
"
<primitive: 903>
^ self _primitiveFailed: #_oneArgSshPrim:with: args: { opcode }

]

{ #category : 'Private' }
GsSshSocket class >> _zeroArgSshPrim: opcode [

"GsSshSocket opCodes
opcode  method
1		instance method: initializeAsClient
2		instance method: initializeAfterConnect
3		instance method _sshConnect
4		instance method hasSshConnectInProgress
5		instance method _sshClose
6		instance method nbRemoteCommandResult
7		instance method: hasCommandInProgress
8		instance method isBlocking
9		instance method nbRemoteCommandResultReady
100		class method disableTraceFile
101 	class method getSshLogLevel
102		class method libSshVersion
103		class method traceFileName

GsSftpSocket opCodes
opcode  method
50		instance method: currentRemoteDirectory

GsSftpRemoteFile opCodes
opcode  method
70		instance method: close
"
<primitive: 902>
^ self _primitiveFailed: #_zeroArgSshPrim: args: { opcode }

]

{ #category : 'Instance Creation' }
GsSshSocket class >> basicNew [

"disallowed"

self shouldNotImplement: #basicNew

]

{ #category : 'Defaults' }
GsSshSocket class >> defaultPortNumber [
"Default ssh port number is 22"
	^22

]

{ #category : 'Debugging' }
GsSshSocket class >> disableTraceFile [

"Disables tracing previosly enabled by the enableTraceFileInDirectory: method.
Returns true."

^ self _zeroArgSshPrim: 100

]

{ #category : 'Debugging' }
GsSshSocket class >> enableTraceFileInDirectory: aDirectory [

"Creates a text file in aDirectory and writes logging information to that file.
The logging level may be set by setLogLevel: class method.
Returns true on success or raises an error if the diretory does not exist"

^ self _oneArgSshPrim: 100 with: aDirectory

]

{ #category : 'Examples' }
GsSshSocket class >> exampleHost [
"Answer a string containing the host domain name of the test ssh server maintained by GemTalk."
^ 'ssh-test.gemtalksystems.com'

]

{ #category : 'Examples' }
GsSshSocket class >> exampleOpenSshPrivateKey [
"Return aGsSshPrivateKey that may be used to access the ssh test server at ssh-test.gemtalksystems.com"

^GsSshPrivateKey newFromOpenSshString: self examplePrivateKeyAsOpenSshString

]

{ #category : 'Examples' }
GsSshSocket class >> examplePassword [
"Answer a string containing the host password of the test ssh server maintained by GemTalk."

^ 'ymPEU9Kxajkbvzx'

]

{ #category : 'Examples' }
GsSshSocket class >> examplePrivateKey [
"Return aGsTlsPrivateKey that may be used to access the ssh test server at ssh-test.gemtalksystems.com"

^GsTlsPrivateKey newFromPemString: self examplePrivateKeyAsPem

]

{ #category : 'Examples' }
GsSshSocket class >> examplePrivateKeyAsOpenSshString [

"Private key for the ssh test server at ssh-test.gemtalksystems.com in OpenSSH format"
^ 
'-----BEGIN OPENSSH PRIVATE KEY-----
b3BlbnNzaC1rZXktdjEAAAAABG5vbmUAAAAEbm9uZQAAAAAAAAABAAABFwAAAAdzc2gtcn
NhAAAAAwEAAQAAAQEBVeSnL412Ivvpl9LLjmDHXW6SCyxcET4HdF3gomjgCiuWGMOYcYtH
dQeJJ2MuPp8VqvTr6MQ7jpN/SGYrwEaXjdM82wZE4qjq/saGEHZstsi8Hs1cBZWhIaSHc7
JLUiWQNX3jEAX5otxVO3eN2NfHjJkdx6sVH4gHMAfhtLajo+DGdWalx148tTD4rWotVWLH
2s3PDs8s/ewC6idBzV9E+2qUpbshKkBPTYFEUkHK/s+1KpCOAynBOWTNt3MVp4KNhWI43/
68B5JEJA8Bh15WsPr7EOJhgaUsBE6UI9rFyAHhWnlpc1cF9Pl5YI5MZe8DmTwo2GyYliCO
/eBWJvny6wAAA8DnxJPA58STwAAAAAdzc2gtcnNhAAABAQFV5KcvjXYi++mX0suOYMddbp
ILLFwRPgd0XeCiaOAKK5YYw5hxi0d1B4knYy4+nxWq9OvoxDuOk39IZivARpeN0zzbBkTi
qOr+xoYQdmy2yLwezVwFlaEhpIdzsktSJZA1feMQBfmi3FU7d43Y18eMmR3HqxUfiAcwB+
G0tqOj4MZ1ZqXHXjy1MPitai1VYsfazc8Ozyz97ALqJ0HNX0T7apSluyEqQE9NgURSQcr+
z7UqkI4DKcE5ZM23cxWngo2FYjjf/rwHkkQkDwGHXlaw+vsQ4mGBpSwETpQj2sXIAeFaeW
lzVwX0+Xlgjkxl7wOZPCjYbJiWII794FYm+fLrAAAAAwEAAQAAAQEAizMkarUC422DhwAZ
RqfapAzPw2LVPWPu7w8F1bozdZCXdQ+18ozNlDV0PygffPmSfd9oaYXz5bHiAd0vdQKI1A
KsZVShGVPDEeZMUUmWK7mA9l2QWOm6CBOP3qg6CIEovM67cxurrwZcYXDkvOPl6DWzLUdX
u1XL719WIxi1eZOV6iFvM9e+e4pvAQSksMoWUY611oq1+ilU76NPjvB3E3Qvy3CD0I9K1X
mmVZb4gd/tOH6ZchWmQFb6qO6oMvQ+ySEhWlWRgJ09ZdZ7YeHDSf2KTqISJeLf22kwzxCD
Ggt3v3RiHMyaJD2P61sIOff614OeUBT4AAiLTR6qHYOBmQAAAIAztqwB1/s118RwMfWYE3
ohshPApJDmwvu729bNm3VD15ZrrPpZELmNN1hUM5mfSY/VgIEst3ER4IF89Ue/GlQCnZH8
19Dd8q/av6lRuyHwcow5A6iwj8tSHa1iUmETBSgT8lczBChGZZtXXUpswTdKKZMYZMEm+P
65hJaVa6qIJwAAAIEBv/PhCa2EwYreNg2GmCFK2+8ypOS4+MV3Os5QOyBim+K+XKtoUlrQ
GntrOiyc1KOCMgW52KTcJ2iwQlEDwy1tybDSZJ6sQ9+MDRGBUQ343aE+/lZZITMHvwPCVJ
5fo27z2JRE0tBKg52gl+rZc/zulmpWpg9mcaThIblgq4yUz/cAAACBAMNjX7M/ukckrFNl
Qz+mBppnCMOhLF77lo1TapTx0KIJL29jMaEB2rniP195kiNCNcGd5ipJD5JU+itJfIDUby
63AM4Nix1YoW3CG59pkI73m1NywkVr0bnPWUW+3ED+6Z3xxWWNNyLraBOF0L/DSjxkb9mn
QAD/YmAAizme+Z+tAAAAAAECAwQFBgcICQo=
-----END OPENSSH PRIVATE KEY-----'

]

{ #category : 'Examples' }
GsSshSocket class >> examplePrivateKeyAsPem [

"Private key for the ssh test server at ssh-test.gemtalksystems.com"
^
'-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEBVeSnL412Ivvpl9LLjmDHXW6SCyxcET4HdF3gomjgCiuWGMOY
cYtHdQeJJ2MuPp8VqvTr6MQ7jpN/SGYrwEaXjdM82wZE4qjq/saGEHZstsi8Hs1c
BZWhIaSHc7JLUiWQNX3jEAX5otxVO3eN2NfHjJkdx6sVH4gHMAfhtLajo+DGdWal
x148tTD4rWotVWLH2s3PDs8s/ewC6idBzV9E+2qUpbshKkBPTYFEUkHK/s+1KpCO
AynBOWTNt3MVp4KNhWI43/68B5JEJA8Bh15WsPr7EOJhgaUsBE6UI9rFyAHhWnlp
c1cF9Pl5YI5MZe8DmTwo2GyYliCO/eBWJvny6wIDAQABAoIBAQCLMyRqtQLjbYOH
ABlGp9qkDM/DYtU9Y+7vDwXVujN1kJd1D7XyjM2UNXQ/KB98+ZJ932hphfPlseIB
3S91AojUAqxlVKEZU8MR5kxRSZYruYD2XZBY6boIE4/eqDoIgSi8zrtzG6uvBlxh
cOS84+XoNbMtR1e7VcvvX1YjGLV5k5XqIW8z1757im8BBKSwyhZRjrXWirX6KVTv
o0+O8HcTdC/LcIPQj0rVeaZVlviB3+04fplyFaZAVvqo7qgy9D7JISFaVZGAnT1l
1nth4cNJ/YpOohIl4t/baTDPEIMaC3e/dGIczJokPY/rWwg59/rXg55QFPgACItN
Hqodg4GZAoGBAb/z4QmthMGK3jYNhpghStvvMqTkuPjFdzrOUDsgYpvivlyraFJa
0Bp7azosnNSjgjIFudik3CdosEJRA8Mtbcmw0mSerEPfjA0RgVEN+N2hPv5WWSEz
B78DwlSeX6Nu89iURNLQSoOdoJfq2XP87pZqVqYPZnGk4SG5YKuMlM/3AoGBAMNj
X7M/ukckrFNlQz+mBppnCMOhLF77lo1TapTx0KIJL29jMaEB2rniP195kiNCNcGd
5ipJD5JU+itJfIDUby63AM4Nix1YoW3CG59pkI73m1NywkVr0bnPWUW+3ED+6Z3x
xWWNNyLraBOF0L/DSjxkb9mnQAD/YmAAizme+Z+tAoGBAb88Jroa3CjAFQuyhWbu
Fmdvcgjfsy3tORUlV0UxGEK7J4QuPoG62XsXLf9u+0Xx2dNHlD2Qm51dEF2ltTPw
72QnfmenZCZ/0rxZddsPMCFXFCWq4GIdKOa1Qhhp5uKtrBfYML6p5ztw7R3ABEuh
hDP5B3nUdluQWpXpF3MvcIQRAoGAckTT20kR8DmKbttyEO9QPUy023SPNp181vpK
AwHJOnqUu1gP1lH3UW74ESZQST6Xobxut5wy9ymrhVVc8xtKSs/MMLXK/kavYjl7
Xiem23YD6THcXC1KYpjZZOjSd1Cd4DldcJ69+DYkkAZap0vXRqqWn3wo+mxhZDPt
OK2436ECgYAztqwB1/s118RwMfWYE3ohshPApJDmwvu729bNm3VD15ZrrPpZELmN
N1hUM5mfSY/VgIEst3ER4IF89Ue/GlQCnZH819Dd8q/av6lRuyHwcow5A6iwj8tS
Ha1iUmETBSgT8lczBChGZZtXXUpswTdKKZMYZMEm+P65hJaVa6qIJw==
-----END RSA PRIVATE KEY-----'

]

{ #category : 'Examples' }
GsSshSocket class >> exampleUserId [
"Answer a string containing the user ID for the test ssh server maintained by GemTalk."

^ 'sshtest'

]

{ #category : 'Debugging' }
GsSshSocket class >> getSshLogLevel [

"Returns a symbol representing the global logging level.  Valid levels are:
     #SSH_LOG_NOLOG -no logging
     #SSH_LOG_WARNING -only warnings
     #SSH_LOG_PROTOCOL -high level protocol information
     #SSH_LOG_PACKET -lower level protocol infomations, packet level
     #SSH_LOG_FUNCTIONS -every function path"

^ self _zeroArgSshPrim: 101

]

{ #category : 'Connection Testing' }
GsSshSocket class >> hostIsAvailableForSsh: aHostOrIp [
"Answer a boolean indicating if host aHostOrIp is accepting connections on the default ssh port (22).
The connection attempt will block for a 1 second."

^self hostIsAvailableForSsh: aHostOrIp atPort: self defaultPortNumber withTimeout: 1000

]

{ #category : 'Connection Testing' }
GsSshSocket class >> hostIsAvailableForSsh: aHostOrIp atPort: portNum withTimeout: timeoutMs [

"Answer a boolean indicating if host aHostOrIp is accepting connections on port portNum.
The connection attempt will block for a maximum of timeoutMs."

|sshSock |
sshSock := self newClient.
^ [
 [ sshSock connectTo: portNum on: aHostOrIp timeoutMs: timeoutMs ] on: SocketError do:[:ex|  false ]
] ensure:[ sshSock close ]

]

{ #category : 'Connection Testing' }
GsSshSocket class >> hostIsAvailableForSsh: aHostOrIp withTimeout: timeoutMs [

"Answer a boolean indicating if host aHostOrIp is accepting connections on the default ssh port (22).
The connection attempt will block for a maximum of timeoutMs."

^self hostIsAvailableForSsh: aHostOrIp atPort: self defaultPortNumber withTimeout: timeoutMs

]

{ #category : 'Version' }
GsSshSocket class >> libSshVersion [
"Answer a string representing the version of the libssh library"

^ self _zeroArgSshPrim: 102

]

{ #category : 'Examples' }
GsSshSocket class >> nbSshClientExample1 [

"Example of remote ssh command using non-blocking protocol with userId / password authentication.

GsSshSocket nbSshClientExample1"

^ self nbSshClientExampleToHost: self exampleHost userId: self exampleUserId password: self examplePassword

]

{ #category : 'Examples' }
GsSshSocket class >> nbSshClientExample2 [

"Example of remote ssh command using non-blocking protocol with PKI authentication.

GsSshSocket nbSshClientExample2"

^ self nbSshClientExampleToHost: self exampleHost withUserId: self exampleUserId privateKey: self examplePrivateKey

]

{ #category : 'Examples' }
GsSshSocket class >> nbSshClientExample3 [

"Example of remote ssh command using non-blocking protocol with PKI authentication.

GsSshSocket nbSshClientExample3"

^ self nbSshClientExampleToHost: self exampleHost withUserId: self exampleUserId privateKey: self exampleOpenSshPrivateKey

]

{ #category : 'Examples' }
GsSshSocket class >> nbSshClientExampleToHost: aHostOrIp userId: id password: pw [
"Non-blocking ssh example. All ssh methods that start with 'nb' are non-blocking"

|sshSock result timeoutMs |
"Uncomment to generate trace file in /tmp"
"GsSshSocket enableTraceFileInDirectory: '/tmp' ."

"Create a new instance and do a normal socket connect"
sshSock := GsSshSocket newClient.
[sshSock connectTo: 22 on: aHostOrIp] onException: (SocketError, SshSocketError) do:[:ex|
	sshSock close.
	ex pass
].

"Set credentials. This step is not needed if doing passwordless ssh login with a public key in ~/.ssh directory"
sshSock userId: id ; password: pw .

"Disable authenticating the remote host"
sshSock disableHostAuthentication .

"ssh connect and wait for connection to finish for up to 10 seconds"
timeoutMs := 10000 .
sshSock nbSshConnectTimeout: timeoutMs .

"Start a remote command"
sshSock  nbExecuteRemoteCommand: 'ulimit -a'.
"Wait for remote command to finish"
sshSock nbRemoteCommandResultReadyWithin: timeoutMs .
result := sshSock nbRemoteCommandResult .
sshSock close.
^ result

]

{ #category : 'Examples' }
GsSshSocket class >> nbSshClientExampleToHost: aHostOrIp withUserId: aUserId privateKey: aTlsPrivateKey [
"Non-blocking ssh example. All ssh methods that start with 'nb' are non-blocking"

|sshSock result timeoutMs |
"Uncomment to generate trace file in /tmp"
"GsSshSocket enableTraceFileInDirectory: '/tmp' ."

"Create a new instance and do a normal socket connect"
sshSock := GsSshSocket newClient.
[sshSock connectTo: 22 on: aHostOrIp] onException: (SocketError, SshSocketError) do:[:ex|
	sshSock close.
	ex pass
].

"Set credentials. This step is not needed if doing passwordless ssh login with a public key in ~/.ssh directory"
sshSock userId: aUserId ; privateKey: aTlsPrivateKey .

"Disable authenticating the remote host"
sshSock disableHostAuthentication .

timeoutMs := 10000 .
"ssh connect and wait for connection to finish"
sshSock nbSshConnectTimeout: timeoutMs .

"Start a remote command"
sshSock  nbExecuteRemoteCommand: 'ulimit -a'.
"Wait for remote command to finish"
sshSock nbRemoteCommandResultReadyWithin: timeoutMs .
result := sshSock nbRemoteCommandResult .
sshSock close.
^ result

]

{ #category : 'Instance Creation' }
GsSshSocket class >> new [
  self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
GsSshSocket class >> new: aSize [
  self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
GsSshSocket class >> newClient [

"Creates a new instance and initializes it to act as a AF_INET client socket.
 No connection is made."

^ super new initializeAsClient

]

{ #category : 'Instance Creation' }
GsSshSocket class >> newClientFromGsSocket: aGsSocket [

	self shouldNotImplement: #newClientFromGsSocket:

]

{ #category : 'Instance Creation' }
GsSshSocket class >> newClientIpv6 [

"Creates a new instance and initializes it to act as a AF_INET6 client socket.
 No connection is made."

^ super newIpv6 initializeAsClient

]

{ #category : 'Instance Creation' }
GsSshSocket class >> newClientIpv6FromGsSocket: aGsSocket [

self shouldNotImplement: #newClientIpv6FromGsSocket:

]

{ #category : 'Debugging' }
GsSshSocket class >> setTraceFileLevel: aSymbol [

"Sets the global ssh logging level.  Valid levels are:
     #SSH_LOG_NOLOG -no logging
     #SSH_LOG_WARNING -only warnings
     #SSH_LOG_PROTOCOL -high level protocol information
     #SSH_LOG_PACKET -lower level protocol infomations, packet level
     #SSH_LOG_FUNCTIONS -every function path

Log data are written to the trace file if a trace file is active (see the #enableTraceFileInDirectory: method).
If a trace file is not active, log data are written to stdout.
Returns true on success or raises an exception if aSymbol is not a valid ssh logging level"

^ self _oneArgSshPrim: 101 with: aSymbol

]

{ #category : 'Examples' }
GsSshSocket class >> sshClientExample1 [

"Example of an remote ssh command using blocking protocol with userId / password authentication.

GsSshSocket sshClientExample1"

^ self sshClientExampleToHost: self exampleHost userId: self exampleUserId password: self examplePassword

]

{ #category : 'Examples' }
GsSshSocket class >> sshClientExample2 [

"Example of an remote ssh command using blocking protocol with PKI authentication.

GsSshSocket sshClientExample2"

^ self sshClientExampleToHost: self exampleHost userId: self exampleUserId privateKey: self examplePrivateKey

]

{ #category : 'Examples' }
GsSshSocket class >> sshClientExample3 [

"Example of an remote ssh command using blocking protocol with PKI authentication.

GsSshSocket sshClientExample3"

^ self sshClientExampleToHost: self exampleHost userId: self exampleUserId privateKey: self exampleOpenSshPrivateKey

]

{ #category : 'Examples' }
GsSshSocket class >> sshClientExampleToHost: aHostOrIp userId: id password: pw [

"Blocking ssh example. All ssh methods below block until completion"

|sshSock result |
"Uncomment to generate trace file in /tmp"
"GsSshSocket enableTraceFileInDirectory: '/tmp' ."
sshSock := GsSshSocket newClient.
[sshSock connectTo: 22 on: aHostOrIp] onException: (SocketError, SshSocketError) do:[:ex|
	sshSock close.
	ex pass
].

"Set userId / password for test server"
sshSock userId: id ; password: pw .

"Disable authenticating the remote host"
sshSock disableHostAuthentication .

sshSock sshConnect .
result := sshSock  executeRemoteCommand: 'ulimit -a'.
sshSock close.
^ result

]

{ #category : 'Examples' }
GsSshSocket class >> sshClientExampleToHost: aHostOrIp userId: id privateKey: aGsTlsPrivatreKey [

"Blocking ssh example. All ssh methods below block until completion"

|sshSock result |
"Uncomment to generate trace file in /tmp"
"GsSshSocket enableTraceFileInDirectory: '/tmp' ."
sshSock := GsSshSocket newClient.
[sshSock connectTo: 22 on: aHostOrIp] onException: (SocketError, SshSocketError) do:[:ex|
	sshSock close.
	ex pass
].
"Set userId / password for test server"
sshSock userId: id ; privateKey: aGsTlsPrivatreKey .

"Disable authenticating the remote host"
sshSock disableHostAuthentication .

sshSock sshConnect .
result := sshSock  executeRemoteCommand: 'ulimit -a'.
sshSock close.
^ result

]

{ #category : 'Debugging' }
GsSshSocket class >> traceFileName [
"Return a string representing the file name of the trace file or nil if the trace file is not active."

^ self _zeroArgSshPrim: 103

]

{ #category : 'Private' }
GsSshSocket >> _hostAuthenticationEnabled: aBooleanOrNil [
"Enables or disables or queries the state of host authentication (authentication of the server's public key).
Arguments true or false enable or disable host authentication respecitvely and return the receiver.
An argument of nil returns a Boolean indicating the current state of host authentication.

Host authentication is enabled by default, meaning the host's public key must appear in known_hosts file.
Host authentication is performed as part of the sshConnect process.

Returns the receiver or raises an exception on error.
Warning: disabling host authentication exposes the session to man in the middle attacks."

	^self _oneArgSshPrim: 1 with: aBooleanOrNil

]

{ #category : 'Private' }
GsSshSocket >> _oneArgSshPrim: opcode with: arg1 [

"GsSshSocket opCodes
opcode	method
1		instance method: hostAuthenticationEnabled:
2		instance method: password:
3		instance method: privateKey:
4		instance method: executeRemoteCommand:
5		instance method: makeBlocking / makeNonBlocking
6		instance method: nbExecuteRemoteCommand:
100		class method enableTraceFileInDirectory:
101		class method setSshTraceLevel:

GsSftpSocket opCodes
51		instance method: removeRemoteDirectory:
53		instance method: removeRemoteFile:
54		instance method: stat:
55		instance method: lstat:

GsSftpRemoteFile opCodes:
70		instance method: position:
"
<primitive: 903>
^ self _primitiveFailed: #_oneArgSshPrim:with: args: { opcode }

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> _readInto: aString startingAt: anOffset maxBytes: numBytes [

^ self shouldNotImplement: #_readInto:startingAt:maxBytes:

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> _recvfrom: maxBytes [

^ self shouldNotImplement: #_recvfrom:

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> _sendUdp: aString startingAt: anOffset to: hostName port: portNum [

^ self shouldNotImplement: #_sendUdp:startingAt:to:port:

]

{ #category : 'Private' }
GsSshSocket >> _sshClose [

"Private.
Close the socket and free the internal ssh structures but not the GsSocket structures.
Call close instead of this method. Returns the receiver."

^ self _zeroArgSshPrim: 5

]

{ #category : 'Private' }
GsSshSocket >> _sshConnect [

"Connects the receiver to the ssh server.
 Returns true on success, false if the connection is in progress on a nonblocking socket, or raises an exception on error.
If the method returns false, the application should wait for the socket to become read-ready and then send
the message again."

^self _zeroArgSshPrim: 3

]

{ #category : 'Private' }
GsSshSocket >> _twoArgSshPrim: opcode with: arg1 with: arg2 [
"
GsSshSocket opCodes
opcode  function
1	instance methods: sshOptionAt: / sshOptionAt:put:

GsSftpSocket opCodes
50	instance method: createRemoteDirectory:mode:
51	instance method: renameRemoteFile: to:
52	instance method: contentsAndStatDetailsOfRemoteDirectory:withPattern:
53	instance method: contentsOfRemoteDirectorywithPattern:

GsSftpRemoteFile opCodes
70	instance method: read:into:
71	instance method: write:from:
"

<primitive: 904>
^ self _primitiveFailed: #_twoArgSshPrim:with:with: args: {opcode}

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> _write: aByteObject startingAt: anOffset ofSize: numBytes [

^ self shouldNotImplement: #_write:startingAt:ofSize:

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> _writev: numBuffers specs: specArray byteOffset: startOffset [

^ self shouldNotImplement: #_writev:specs:byteOffset:

]

{ #category : 'Private' }
GsSshSocket >> _zeroArgSshPrim: opcode [

"GsSshSocket opCodes
opcode  method
1		instance method: initializeAsClient
2		instance method: initializeAfterConnect
3		instance method _sshConnect
4		instance method hasSshConnectInProgress
5		instance method _sshClose
6		instance method nbRemoteCommandResult
7		instance method: hasCommandInProgress
8		instance method isBlocking
9		instance method nbRemoteCommandResultReady
100		class method disableTraceFile
101 	class method getSshLogLevel
102		class method libSshVersion
103		class method traceFileName

GsSftpSocket opCodes
opcode  method
50		instance method: currentRemoteDirectory

GsSftpRemoteFile opCodes
opcode  method
70		instance method: close
71		instance method: isOpen
72		instance method: fstat
73		instance method: isReadable
74		instance method: isWriteable
75		instance method: isAppendable
76		instance method: position
"
<primitive: 902>
^ self _primitiveFailed: #_zeroArgSshPrim: args: { opcode }

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> accept [

"Server operations are not supported"
^self shouldNotImplement: #accept

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> acceptTimeoutMs: timeoutMs [

"Server operations are not supported"
^self shouldNotImplement: #acceptTimeoutMs:

]

{ #category : 'Socket Operations' }
GsSshSocket >> close [
"Closes the ssh connection and the underlying GsSocket socket and frees the associated data structures.
Returns the receiver.
Closing a GsSshSocket more than once has no effect and does not raise an exception."

super close.
^ self _sshClose

]

{ #category : 'Client Operations' }
GsSshSocket >> connectTo: portNumber on: aHost timeoutMs: timeoutMs [

"Connect the receiver to the server socket identified by portNumber and aHost.
aHost may be the name of the host or its numeric address,
or aHost == -1 for <broadcase> , or aHost == nil for IN6ADDR_ANY_INIT .
portNumber maybe either a SmallInteger, or the String name of a service.
timeoutMs is a SmallInteger specifying maximum time to wait for the
connection to complete, or -1 to wait indefinitely.
If aHost is the name of a host, connect is attempted on IPv4 first
if getaddrinfo returns any IPv4 addresses for aHost, then attempted
on IPv6.
Returns true if the connection succeeded otherwise signals an Error.

Note: This method initiates the socket-level TCP/IP connection only, not the ssh connection.
To initiate the ssh connection, use the #sshConnect method.
 "

(super connectTo: portNumber on: aHost timeoutMs: timeoutMs)
  ifTrue:[ self initializeAfterConnect: aHost  ] .
^ true

]

{ #category : 'Client Operations' }
GsSshSocket >> connectToHost: aHost timeoutMs: timeoutMs [

"Connect the receiver to the default server ssh port (22) on the server aHost.
See the method connectTo:on:timeoutMs: for more details. "

(super connectTo: self class defaultPortNumber on: aHost timeoutMs: timeoutMs)
  ifTrue:[ self initializeAfterConnect: aHost ] .
^ true

]

{ #category : 'Updating' }
GsSshSocket >> disableHostAuthentication [
"Disables authentication of the host server's public key.
Host authentication is enabled by default, meaning the host's public key must appear in known_hosts file.
Returns the receiver or raises an exception on error.

*** WARNING ***
Disabling host authentication exposes the connection to man in the middle attacks.
Do not disable host authentication unless you are certain the server host can be trusted."

^ self _hostAuthenticationEnabled: false

]

{ #category : 'Debugging' }
GsSshSocket >> disableTracing [
"Disables tracing on the receiver"
^ self sshOptionAt: #SSH_OPTIONS_LOG_VERBOSITY put: #SSH_LOG_NOLOG

]

{ #category : 'Updating' }
GsSshSocket >> enableHostAuthentication [
"Enables authentication of the host server's public key.
Host authentication is enabled by default, meaning the host's public key must appear in known_hosts file.
Returns the receiver or raises an exception on error."

^ self _hostAuthenticationEnabled: true

]

{ #category : 'Debugging' }
GsSshSocket >> enableTracing [

"Enables tracing on the receiver at the ssh protocol level"

^ self sshOptionAt: #SSH_OPTIONS_LOG_VERBOSITY put:  #SSH_LOG_PROTOCOL

]

{ #category : 'Client Operations' }
GsSshSocket >> executeRemoteCommand: aString [
"Places the receiver in blocking mode and tells ssh to execute aString on the server.
Blocks the main thread until the command is complete.
Returns a String representing is the result of the remote command. If the remote command succeeds but returns no output, an empty string is returned.
Raises an exception if the remote command is interrupted by a signal or if the remote command returns a non-zero exit status.
Also raises an exception if an error occurs.
"

^ self makeBlocking ; _oneArgSshPrim: 4 with: aString

]

{ #category : 'Testing' }
GsSshSocket >> hasNbCommandInProgress [

"Returns a boolean indicating if there is a non-blocking command in progress."

^self _zeroArgSshPrim: 7

]

{ #category : 'Testing' }
GsSshSocket >> hasNbSshConnectInProgress [
"Returns a Boolean which indicates if the receiver has a non-blocking sshConnect operation in progress."

^ self _zeroArgSshPrim: 4

]

{ #category : 'Client Operations' }
GsSshSocket >> hostAuthenticationEnabled: aBoolean [
"Enables or disables authentication of the server's public key.
Host authentication is enabled by default, meaning the host's public key must appear in known_hosts file.

Returns the receiver or raises an exception on error.
Warning: disabling host authentication exposes the session to man in the middle attacks."

^ self _oneArgSshPrim: 1 with: aBoolean

]

{ #category : 'Private' }
GsSshSocket >> initializeAfterConnect: host [

"Performs ssh initialization after the underlying socket has been connected.
Returns the receiver"

self sshOptionAt: #SSH_OPTIONS_HOST put: host .
^ self _zeroArgSshPrim: 2

]

{ #category : 'Private' }
GsSshSocket >> initializeAsClient [

"Returns the receiver"
^ self _zeroArgSshPrim: 1

]

{ #category : 'Testing' }
GsSshSocket >> isBlocking [
"Returns a Boolean indicating if the receiver is in blocking mode."

^ self _zeroArgSshPrim: 8

]

{ #category : 'Testing' }
GsSshSocket >> isHostAuthenticationEnabled [
"Answers a Boolean indicating if authentication of the server host's public key is performed as part of the sshConnect process."

^ self _hostAuthenticationEnabled: nil

]

{ #category : 'Testing' }
GsSshSocket >> isNonBlocking [
"Returns a Boolean indicating if the receiver is in non-blocking mode."

^ self isBlocking not

]

{ #category : 'Updating' }
GsSshSocket >> makeBlocking [

"Puts the receiver in blocking mode if it is not.
The blocking mode of a GsSshSocket may not be modified
if a non-blocking remote command or sshConnect operation is in progress.
An exception will be raised in this case."

self isBlocking ifFalse:[ self _oneArgSshPrim: 5 with: true ] .
^ self

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> makeListener: queueLength [

"Server operations are not supported"
^self shouldNotImplement: #makeListener:

]

{ #category : 'Updating' }
GsSshSocket >> makeNonBlocking [
"Puts the receiver in non-blocking mode if it is not.
The blocking mode of a GsSshSocket may not be modified
if a non-blocking remote command or sshConnect operation is in progress.
An exception will be raised in this case."

self isNonBlocking ifFalse:[ self _oneArgSshPrim: 5 with: false ] .
^ self

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> makeServer [

"Server operations are not supported"
^self shouldNotImplement: #makeServer

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> makeServer: queueLength [

"Server operations are not supported"
^self shouldNotImplement: #makeServer:

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> makeServer: queueLength atPort: portNum atAddress: address [

"Server operations are not supported"
^self shouldNotImplement: #makeServer:atPort:atAddress:

]

{ #category : 'Client Operations (Non-blocking)' }
GsSshSocket >> nbExecuteRemoteCommand: aString [
"Places the receiver in blocking mode and tells ssh to execute aString on the server.

Only one remote command may be in progress at any time. Attempting to start
a second remote command before the first has completed raises an error.

Use the nbRemoteCommandStatus to query the progress of the remote command
and the nbRemoteCommandResult to complete the remote command and return
the result as a String object.

Returns false to indicate the remote command was started but has not yet finished executing.
The #nbRemoteCommandResult method should be used to obtain the result.
Raises an exception if an error occurs."

^ self makeNonBlocking ; _oneArgSshPrim: 6 with: aString

]

{ #category : 'Client Operations (Non-blocking)' }
GsSshSocket >> nbRemoteCommandResult [

"Attempts to completes a remote command in progress. If the command has completed,
return a String representing the result of the command. Remote commands that complete successfully
but generate no output will result in an empty string.
Returns false if the remote command has not yet completed.
Raises an exception if no remote command is in progress. Also raises an
exception if the remote command fails (exit status is not zero) or if the remote command
was interrupted by a signal.

The readWillNotBlockWithin: method may be used to determine when the remote command has
made progress and this method should be run again. Note that this method may need to be repeated several times
after readWillNotBlockWithin: returns true because performing the remote command may
involve several steps internally."

^self _zeroArgSshPrim: 6

]

{ #category : 'Client Operations (Non-blocking)' }
GsSshSocket >> nbRemoteCommandResultReady [
"Attempts to advance the progress of a remote command in progress.
Returns true if the remote command has completed. Use the nbRemoteCommandResult method to obtain the result.
Returns false if the remote command has not yet completed
Raises an exception if no remote command is in progress. Also raises an
exception if the remote command fails (exit status is not zero) or if the remote command
was interrupted by a signal.

The readWillNotBlockWithin: method may be used to determine when the remote command has
made progress and this method should be run again. Note that this method may need to be repeated several times
after readWillNotBlockWithin: returns true because performing the remote command may
involve several steps internally."

^self _zeroArgSshPrim: 9

]

{ #category : 'Client Operations (Non-blocking)' }
GsSshSocket >> nbRemoteCommandResultReadyWithin: timeoutMs [

"Attempts to advance the progress of a remote command in progress.
Returns true if the remote command has completed. Use the nbRemoteCommandResult method to obtain the result.
Raises an exception if the remote command has not completed within timeoutMs. Also
raises an exception if no remote command is in progress or if the remote command fails
(exit status is not zero) or if the remote command was interrupted by a signal."

| endTime failed sys |
sys := System .
endTime := sys _timeMs + timeoutMs .
failed := false .
[ failed ] whileFalse: [ 	| timeLeftMs |
	self nbRemoteCommandResultReady ifTrue:[ ^ true ].
	timeLeftMs := endTime  - sys _timeMs .
	timeLeftMs > 0
		ifTrue:[ failed := true ~~ (self readWillNotBlockWithin: timeLeftMs) ]
		ifFalse:[ failed := true ].
].

^ SshSocketError signal: ('nbRemoteCommandResultReadyWithin: timed out after ', timeoutMs asString, ' ms ')

]

{ #category : 'Client Operations (Non-blocking)' }
GsSshSocket >> nbSshConnect [

"Places the receiver in non-blocking mode and connects to the ssh server.
Returns true on success and false if the operation is in progress and must be retried.
Raises an exception on error."

self makeNonBlocking .
^ self _sshConnect .

]

{ #category : 'Client Operations (Non-blocking)' }
GsSshSocket >> nbSshConnectTimeout: timeoutMs [

"Places the receiver in non-blocking mode and connects to the ssh server.
Returns true on success and false if the operation is in progress and must be retried.
Raises an exception on error."

|  endTime failed sys |
self makeNonBlocking .
sys := System .
endTime := sys _timeMs + timeoutMs .
failed := false .
[ failed ] whileFalse: [ 	| timeLeftMs |
	self nbSshConnect ifTrue:[ ^ true ].
	timeLeftMs := endTime  - sys _timeMs .
	timeLeftMs > 0
		ifTrue:[ failed := true ~~ (self readWillNotBlockWithin: timeLeftMs) ]
		ifFalse:[ failed := true ].
].

^ SshSocketError signal: ('nbSshConnectTimeout timed out after ', timeoutMs asString, ' ms ')

]

{ #category : 'Client Operations' }
GsSshSocket >> password: aString [

"Forces authentication by user ID / password and sets the password to use for authentication.
Returns the receiver or raises an error if aString is not an instance of String or if aString is empty"

^ self _oneArgSshPrim: 2 with: aString

]

{ #category : 'Client Operations' }
GsSshSocket >> privateKey: aPrivateKey [
"Tells ssh to use the specified private key to authenticate the user.
 aPrivate key must be an instance of either GsTlsPrivateKey or GsSshPrivateKey."

^ self _oneArgSshPrim: 3 with: aPrivateKey

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> recvfrom: maxBytes [

"UDP operations are not supported"
^self shouldNotImplement: #recvfrom:

]

{ #category : 'Accessing' }
GsSshSocket >> remoteHost [

^ self sshOptionAt:  #SSH_OPTIONS_HOST

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> sendUdp: aString flags: flagsInt toHost: hostName port: aPort [

"UDP operations are not supported"
^self shouldNotImplement: #sendUdp:flags:toHost:port:

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> shutdownReading [

^ self shouldNotImplement: #shutdownReading

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> shutdownReadingAndWriting [

^ self shouldNotImplement: #shutdownReadingAndWriting

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> shutdownWriting [

^ self shouldNotImplement: #shutdownWriting

]

{ #category : 'Error Reporting' }
GsSshSocket >> signalError [
  ^ SshSocketError signal: 'no details'

]

{ #category : 'Error Reporting' }
GsSshSocket >> signalError: aString [
  ^ SshSocketError signal: aString

]

{ #category : 'Unsupported Operations' }
GsSshSocket >> speciesForAccept [

"Server operations are not supported"
^self shouldNotImplement: #speciesForAccept

]

{ #category : 'Client Operations' }
GsSshSocket >> sshConnect [

"Places the receiver in blocking mode and connects the receiver to the ssh server.
 Returns true on success or raises an exception on error."

^ self makeBlocking ; _sshConnect

]

{ #category : 'Configuration' }
GsSshSocket >> sshOptionAt: optionName [
"Gets the value of the specified option.
 Not all setable options are accessible by this method. This is a limitation of the underlying libssh package.
 Returns string showing the value of the specified option or an empty string if the option has not yet been set,
 except for #SSH_OPTIONS_PORT, which returns a SmallInteger.

 Raises an exception if an error occurred.

 optionName may be any of the following symbols:

 #SSH_OPTIONS_FD - The file descriptor of the ssh socket or -1 if it has not been set yet.
 #SSH_OPTIONS_HOST - The hostname or ip address of the peer.
 #SSH_OPTIONS_PORT - The port to connect to.
 #SSH_OPTIONS_USER - The username for authentication.
 #SSH_OPTIONS_KNOWNHOSTS - The known hosts file name.
 #SSH_OPTIONS_GLOBAL_KNOWNHOSTS - The global known hosts file name.
 #SSH_OPTIONS_PROXYCOMMAND - Get the command to be executed in order to connect to the server.
"

^ self _twoArgSshPrim: 1 with: optionName with: nil

]

{ #category : 'Configuration' }
GsSshSocket >> sshOptionAt: optionName put: value [

"Sets the value of the specified option.
 Returns the receiver or an exception if an error occurred.
 optionName can be any of the following symbols:

 #SSH_OPTIONS_HOST - The hostname or ip address to connect to
  Kind: String, Default: none (must be specified)

 #SSH_OPTIONS_PORT - The port to connect to.
  Kind: SmallInteger, Default: 22, Min: 1, Max: 65535

 #SSH_OPTIONS_FD - The file descriptor to use.
  Kind: SmallInteger, Default: assigned at instance creation.

 #SSH_OPTIONS_BINDADDR - The address to bind the client to.
  Kind: String, Default: none (assigned by operating system).

 #SSH_OPTIONS_USER - The username for authentication.
  Kind: String, Default: UNIX user id.

 #SSH_OPTIONS_SSH_DIR - Set the ssh directory.
  The ssh directory is used for files like known_hosts and identity (private and
  public key). It may include '%s' which will be replaced by the user home
  directory.
  Kind: String, Default: ~/.ssh

 #SSH_OPTIONS_KNOWNHOSTS - Set the known hosts file name.
  The known hosts file is used to certify remote hosts are genuine. It may
  include '%d' which will be replaced by the user home directory.
  Kind: String, Default: ~/.ssh/known_hosts.

 #SSH_OPTIONS_GLOBAL_KNOWNHOSTS - Set the global known hosts file name.
  The known hosts file is used to certify remote hosts are genuine.
  Kind: String, Default: /etc/ssh/ssh_known_hosts

 #SSH_OPTIONS_ADD_IDENTITY - Add a new identity file name to the identity list.
  The identity used to authenticate with public key will be prepended to the
  list. It may include '%s' which will be replaced by the user home directory.
  Kind: String, Defaults: ~/.ssh/id_rsa  ~/.ssh/id_dsa

 #SSH_OPTIONS_TIMEOUT - Set a timeout for the connection in seconds.
  -1 means wait forever.
  Kind: SmallInteger, Default: 10, Min: -1, Max: 4294967294

 #SSH_OPTIONS_TIMEOUT_USEC - Set a timeout for the connection in microseconds.
  -1 means wait forever.
  Kind: SmallInteger, Default: 10000000, Min: -1, Max: 4294967294

 #SSH_OPTIONS_LOG_VERBOSITY - Set the session logging verbosity.
  The verbosity of the log messages. Every log smaller or equal to verbosity will
  be shown. Log data are written to the trace file if a trace file is active
  (see GsSshSocket (c) enableTraceFileInDirectory: method). If the trace
  file is not active, log data are written to stdout.

  Possible values:
     #SSH_LOG_NOLOG -no logging
     #SSH_LOG_WARNING -only warnings
     #SSH_LOG_PROTOCOL -high level protocol information
     #SSH_LOG_PACKET -lower level protocol infomations, packet level
     #SSH_LOG_FUNCTIONS -every function path
  Kind: Symbol, Default: #SSH_LOG_NOLOG

 #SSH_OPTIONS_COMPRESSION_C_S - Enable/disable the zlib compression for sending
  from client to server.
  Kind: Boolean, Default: false (compression disabled)

 #SSH_OPTIONS_COMPRESSION_S_C - Enable/disable the zlib compression for sending
  from server to client.
  Kind: Boolean, Default: false (compression disabled)

 #SSH_OPTIONS_COMPRESSION - Enable/disable compression to use for both directions
  communication (client->server and server->client).
  Kind: Boolean, Default: false

 #SSH_OPTIONS_COMPRESSION_LEVEL - Set the zlib compression level to use for
  zlib functions. 1 means fastest and lowest compression. 9 means slowest and
  highest compression. Compression must also be enabled via one of the
  SSH_OPTIONS_COMPRESSSION* options otherwise this option has no effect.
  Kind: SmallInteger, Default: 7

 #SSH_OPTIONS_STRICTHOSTKEYCHECK - If this option is set to true, ssh will never
  automatically add host keys to the ~/.ssh/known_hosts file, and refuses to
  connect to hosts whose host key has changed. This provides maximum  protection
  against trojan horse attacks, though it can be annoying when the
  /etc/ssh/ssh_known_hosts file	is poorly maintained or	when connections to new
  hosts are frequently made. This option forces the user to manually add all new
  hosts. If this option is set to false, ssh will automatically add new host keys
  to the user known hosts files.
  Kind: Boolean, Default: true

 #SSH_OPTIONS_PROXYCOMMAND - Set the command to be executed in order to connect to
  server.
  Kind: String, Default: none

 #SSH_OPTIONS_PUBKEY_AUTH
  Set it if pubkey authentication should be used.
  Kind: Boolean, Default: true

 #SSH_OPTIONS_NODELAY
  Set it to disable Nagle's Algorithm (TCP_NODELAY) on the session socket.
  Kind: Boolean, Default: true

 #SSH_OPTIONS_PROCESS_CONFIG
  Set it to false to disable automatic processing of per-user
  and system-wide OpenSSH configuration files. LibSSH
  automatically uses these configuration files unless
  you provide it with this option or with different file.
  Kind: Boolean, Default: false

 #SSH_OPTIONS_REKEY_DATA
  Set the data limit that can be transferred with a single
  key in bytes. RFC 4253 Section 9 recommends 1GB of data, while
  RFC 4344 provides more specific restrictions, that are applied
  automatically. When specified, the lower value will be used.
  Kind: SmallInteger, Default: 1073741824

 #SSH_OPTIONS_REKEY_TIME
  Set the time limit for a session before intializing a rekey
  in seconds. RFC 4253 Section 9 recommends one hour.
  Kind: SmallInteger, Default: 3600"

^ self _twoArgSshPrim: 1 with: optionName with: value

]

{ #category : 'Client Operations' }
GsSshSocket >> userId: aString [

"Sets the user ID to use for authentication.
For password authentication, the #password: method must also be called.
For public key authentication, credentials are obtained from the home directory of the specified user ID.
By default, the user ID running the gem process is used."

^ self sshOptionAt: #SSH_OPTIONS_USER put: aString

]
