Extension { #name : 'GsSingleRefPathResult' }

{ #category : 'Instance Creation' }
GsSingleRefPathResult class >> newForSearchObject: anObj isDead: aBoolean path: anArray [

"Creates a new GsSingleRefPathResult object for the given object. The
complete reference path is contained in anArray.  The first element of
anArray must be a member of the limit set and the last element must
be the search object.

Raises an error if the reference path is not valid for the given search oop."

|result|
result := self new.
result searchOop: anObj.
result isDead: aBoolean.
result addAll: anArray.
result validate.
^result

]

{ #category : 'Accessing' }
GsSingleRefPathResult >> isDead [
^isDead

]

{ #category : 'Updating' }
GsSingleRefPathResult >> isDead: newValue [
isDead := newValue

]

{ #category : 'Printing' }
GsSingleRefPathResult >> logReferencePath [

"Writes the complete reference path to stdout.  For an RPC gem, the path is
written to the gem log file.  For linked topaz session, the path is written
to the screen."

GsSingleRefPathFinder printMessageToLog: (self resultStringWithTabs: 1)
                      includeTime: false .
^ self

]

{ #category : 'Printing' }
GsSingleRefPathResult >> printHeaderOn: ws [

^self printHeaderOn: ws tabs: 0

]

{ #category : 'Printing' }
GsSingleRefPathResult >> printHeaderOn: ws tabs: anInt [

	anInt timesRepeat: [ws tab].
	ws
		nextPutAll: 'Reference path for search oop ';
		nextPutAll: searchOop asOop asString;
		space;
		nextPut: $(;
		nextPutAll: searchOop class name asString;
		nextPut: $);
		lf.
	^ws

]

{ #category : 'Printing' }
GsSingleRefPathResult >> printReferencePathOn: ws [

^ self printReferencePathOn: ws tabs: 0

]

{ #category : 'Printing' }
GsSingleRefPathResult >> printReferencePathOn: ws tabs: anInt [

	isDead ifTrue:[
    anInt + 1 timesRepeat: [ws tab].
	  ws nextPutAll: 'Object is dead.  No reference path found';
				lf
  ] ifFalse:[
    1 to: self size do: [:n | | obj cls |
			obj := self at: n.
			anInt + 1 timesRepeat: [ws tab].
			ws nextPutAll: n asString;
				 space: 3;
				 nextPutAll: obj asOop asString;
				 space;
				 nextPut: $(;
				 nextPutAll: obj class name asString;
				 nextPut: $) .
      (cls := obj class) == SymbolAssociation ifTrue:[
        ws nextPutAll:'  key ', obj key printString .
      ].
      cls == SymbolDictionary ifTrue:[ ws nextPutAll:'  name ', obj name printString .  ].     
      cls == UserProfile ifTrue:[ ws nextPutAll:'  userId ', obj userId printString .  ].     
      obj isMeta ifTrue:[ ws nextPutAll: '  ', obj name ].
			ws lf
    ]
  ].
	^ws

]

{ #category : 'Printing' }
GsSingleRefPathResult >> printResultsToGsFile: aGsFile [

aGsFile nextPutAll: self resultString

]

{ #category : 'Printing' }
GsSingleRefPathResult >> resultString [

^self resultStringWithTabs: 0

]

{ #category : 'Printing' }
GsSingleRefPathResult >> resultStringWithTabs: anInt [

|ws|
ws := AppendStream on: String new.
self printHeaderOn: ws tabs: anInt ;
printReferencePathOn: ws tabs: anInt.
^ws contents

]

{ #category : 'Accessing' }
GsSingleRefPathResult >> searchOop [
^searchOop

]

{ #category : 'Updating' }
GsSingleRefPathResult >> searchOop: newValue [
searchOop := newValue

]

{ #category : 'Validating' }
GsSingleRefPathResult >> validate [

^ self validateWithLimitSet: nil

]

{ #category : 'Validating' }
GsSingleRefPathResult >> validateWithLimitSet: aLimitSet [
"Validates the receiver contains a complete and correct reference path for
searchOop using the given limit set.  If aLimitSet is nil, then the default
limit set is used.

Raises an error if a problem is found, otherwise returns the receiver."

| myLimitSet |
isDead ifTrue: [^self].
myLimitSet := aLimitSet ifNil:[  
   searchOop isClass ifTrue:[ GsSingleRefPathFinder buildLimitSetForRefPathScanForClass ] 
                    ifFalse:[ SystemRepository buildLimitSetForRefPathScan ]
] ifNotNil: [ aLimitSet].
self size == 0 ifTrue: [self halt: 'Receiver is empty'].
(myLimitSet includes: self first)
  ifFalse: [self halt: 'First object is not a limit object'].
self last == searchOop
  ifFalse: [self halt: 'Last element of reference path is not the search object'].
2 to: self size do:[:n | | child parent |
  parent := self at: n - 1.
  child := self at: n.
  (((GsBitmap with: parent) primReferencedObjects) includes: child)
    ifFalse: [self halt: 'parent object does not reference child object']].
^self

]
