Extension { #name : 'GsSingleRefPathFinder' }

{ #category : 'Primitive' }
GsSingleRefPathFinder class >> _refPathDoScanForParents: searchObjs excludeParentRefs: excludeOops onlySearchObjs: aBoolean [

"Scans the data pages found in the refPathSetup method to find the parent oops.
 If onlySearchObjs is true, then data for only the search objects is collected,
 otherwise data for all of the objects in the repository is collected and any
 object may be queried and the oop of a single parent and whether it has
 additional parents is returned. For each of the objects in the searchObjs this
 method saves all of the parent references. If a parent object is in the
 excludeOops, it is not included in the parent set.

 The searchObjs and excludeOops arguments may be passed an Array or a GsBitmap.

 Elements of both the searchObjs and the excludeOops must be either committed
 non-special objects, or  SmallIntegers;  SmallIntegers must be objectIds of
 committed non-special objects and can be resolved as
 (Object _objectForOop: anObjectId).

 When the scan is complete the parentsOf or findReferencePath methods can be
 used to get the information captured for any object. "

<primitive: 894>
^ self _primitiveFailed:
    #_refPathDoScanForParents:excludeParentRefs:onlySearchObjs:
    args: {  searchObjs . excludeOops . aBoolean }

]

{ #category : 'Primitive' }
GsSingleRefPathFinder class >> _refPathSetupScanWithMaxThreads: maxThreads waitForLock: lockWaitTime
    pageBufSize: aBufSize percentCpuLimit: percentCpu [

"Sets up the current session for performing multi threaded scans of the
 repository for finding reference paths to objects.  Performs an abort
 and leaves the session in a transaction.

 See _scanPomWithMaxThreads for definitions of maxThreads, waitTimeSeconds,
 aBufSize, cpuPersent.
 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 A GciHardBreak during this method will terminate the session."

<primitive: 527>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger ; _validateMin: 1 max: maxInt .
lockWaitTime _validateClass: SmallInteger ; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger;  _validateIsPowerOf2 .
self _validatePercentage: percentCpu .
^ self _primitiveFailed:
    #_refPathSetupScanWithMaxThreads:waitForLock:pageBufSize:percentCpuLimit:
    args: {  maxThreads . lockWaitTime . aBufSize . percentCpu }

]

{ #category : 'Private' }
GsSingleRefPathFinder class >> buildLimitSetForRefPathScanForClass [
  | symListBlock set |
  set := IdentitySet new .
  symListBlock := [:eachDict|
    (set includes: eachDict) ifFalse:[
      set add: eachDict.
    ].
  ].
  set add: AllUsers .
  AllUsers do:[:aUserPro|
    set add: aUserPro .
    aUserPro == System myUserProfile ifTrue:[
      aUserPro symbolList do: symListBlock
    ].
  ].
  set add: (Globals at: #SharedDependencyLists).
  set add: ((AllUsers userWithId:'SymbolUser') resolveSymbol: #AllSymbols) value .
  ^ set asArray
]

{ #category : 'Defaults' }
GsSingleRefPathFinder class >> defaultLimitSetDescendantsMaxLevels [

"Default number of levels that may be traversed to seed the limit set."
 ^SmallInteger maximumValue "effectively infinite"

]

{ #category : 'Defaults' }
GsSingleRefPathFinder class >> defaultLimitSetDescendantsMaxObjs [

"Maximum size of a limit set descendant generation in number of objects."
^ 1000000

]

{ #category : 'Defaults' }
GsSingleRefPathFinder class >> defaultLockWaitTime [

"Default number of seconds to wait for the garbage collection lock expressed in seconds."
^ 120

]

{ #category : 'Defaults' }
GsSingleRefPathFinder class >> defaultMaxThreads [

"Default number of threads to use to scan the repository."
^ SystemRepository _aggressiveMaxThreadCount

]

{ #category : 'Defaults' }
GsSingleRefPathFinder class >> defaultPageBufferSize [

"Default number of 16 KB disk pages for each thread to buffer.  Must be a power of 2."
^ 8

]

{ #category : 'Defaults' }
GsSingleRefPathFinder class >> defaultPercentCpuLimit [

"Default maximum percentage of total CPU that the all threads in the scan may consume."
^ 95 "percent of total CPU"

]

{ #category : 'Defaults' }
GsSingleRefPathFinder class >> defaultPrintToLog [

"Default value for printing output to stdout of the process."
^ 1

]

{ #category : 'Instance Creation' }
GsSingleRefPathFinder class >> newForSearchObjects: anArray [
"Creates a new instance with default settings."
^self
  newForSearchObjects: anArray
  withMaxThreads: nil
  waitForLock: nil
  pageBufSize: nil
  percentCpuLimit: nil
  maxLimitSetDescendantObjs: nil
  maxLimitSetDescendantLevels: nil
  printToLog: nil 

]

{ #category : 'Instance Creation' }
GsSingleRefPathFinder class >> newForSearchObjects: anArray withMaxThreads: threads waitForLock: seconds pageBufSize: pages percentCpuLimit: percent
maxLimitSetDescendantObjs: maxChildObjs maxLimitSetDescendantLevels: maxChildLevels printToLog: logArg [

"Creates and configures a new instance of the receiver.  
 The first argument must be an array of committed disk objects.
 The logBoolean argument may be Boolean or nil or a SmallInteger.   
 All other arguments must be a SmallInteger or nil.
 A nil argument indicates the default value for the argument is to be used.

 The pages argument must be nil or a SmallInteger which is a power of 2."

| result |
result := self new.
result
  maxThreads: (threads ifNil:[self defaultMaxThreads] ifNotNil:[threads]) ;
  lockWaitTime: (seconds ifNil:[self defaultLockWaitTime] ifNotNil:[seconds]) ;
  pageBufferSize: (pages ifNil:[self defaultPageBufferSize] ifNotNil:[pages]) ;
  percentCpuLimit: (percent ifNil:[self defaultPercentCpuLimit] ifNotNil:[percent]) ;
  maxLimitSetDescendantObjs: (maxChildObjs ifNil:[self defaultLimitSetDescendantsMaxObjs] ifNotNil:[maxChildObjs]) ;
  maxLimitSetDescendantLevels: (maxChildLevels ifNil:[self defaultLimitSetDescendantsMaxLevels] ifNotNil:[maxChildLevels]) ;
  printToLog: (logArg ifNil:[self defaultPrintToLog] ifNotNil:[ logArg]) ;
  initializeForSearchObjects: anArray.
^ result

]

{ #category : 'Logging' }
GsSingleRefPathFinder class >> printMessageToLog: aString [

"Print a message to stdout and include a timestamp."
^ self printMessageToLog: aString includeTime: true

]

{ #category : 'Logging' }
GsSingleRefPathFinder class >> printMessageToLog: aString includeTime: aBoolean [

aBoolean ifTrue:[ | msg |
  msg := String withAll: '['.
  msg
    addAll: DateTime now asString;
    addAll: '] ';
    addAll: aString;
    add: Character lf.
    GsFile gciLogServer: msg]
ifFalse: [GsFile gciLogServer: aString].
^ self

]

{ #category : 'Primitive' }
GsSingleRefPathFinder class >> refPathCleanup [

"Logs out the slave sessions and cleans up the saved state for the
 multi threaded reference path scans."

System _zeroArgPrim: 179.

]

{ #category : 'Primitive' }
GsSingleRefPathFinder class >> scanForParents: anArrayOfBitmaps [

"Scans the data pages found in the refPathSetup method to find the parent oops
 for each group of child oops.

 anArrayOfBitmaps is an array of non-empty GsBitmap instances containing child
 oops to scan for. The argument array and its GsBitmap elements are not
 modified by this method.

 This method may be called multiple times.  The size of the anArrayOfBitmaps
 argument may be different each time.  It is an error if any of the GsBitmap
 elements are empty.

 Returns an Array of GsBitmap instances containing the parent oops of the
 objects contained in the argument bitmaps. The result array will be the
 same size and in the same order as the argument array."

<primitive: 597>
^ self _primitiveFailed: #scanForParents: args: { anArrayOfBitmaps }

]

{ #category : 'Adding' }
GsSingleRefPathFinder >> addSearchObject: anObject [

"Adds anObject to list of objects for which reference paths are to be found."

|newSearch|
newSearch := GsSingleRefPathFinderForObject newForSearchObject: anObject
                                            refPathFinder: self .
allSearches add: newSearch .
numSearchesActive := numSearchesActive + 1 .
^ self

]

{ #category : 'Searching' }
GsSingleRefPathFinder >> allActiveSearches [
"Answer an array of GsSingleRefPathFinderForObject that are not finished."
^ allSearches select:[:e| e completed not]

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> allSearches [
^allSearches

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> allSearches: newValue [
allSearches := newValue

]

{ #category : 'Initialization' }
GsSingleRefPathFinder >> basicInit [

allSearches := Array new .
numSearchesActive := 0 .
numPassesDone := 0 .
otScanDone := false .
^ self

]

{ #category : 'Limit Set' }
GsSingleRefPathFinder >> buildLimitSet [

"Collect descendants of the default limit set and add them to the limitObjects
 array.  Stop collecting descendants when the size of a generation exceeds
 maxLimitSetDescendantObjs or the depth of the traversal exceeds
 maxLimitSetDescendantLevels."

| searchForClass limitSet |
self printTimedOpStartMessageToLog: 'Starting build of limit set and descendants'.
searchForClass := (self dynamicInstVarAt: #searchForClass ) ~~ nil .
limitSet := searchForClass ifTrue:[ self class buildLimitSetForRefPathScanForClass ]
                          ifFalse:[ SystemRepository buildLimitSetForRefPathScan ].
limitObjects := Array with: (GsBitmap withAll: limitSet ) .
self buildLimitSetDescendants .
self printTimedOpEndMessageToLog: 'Finished build of limit set and descendants'.
self logLimitSetSizes .
^ self

]

{ #category : 'Limit Set' }
GsSingleRefPathFinder >> buildLimitSetDescendants [

"Collect descendants of the default limit set and add them to the limitObjects
 array.  Stop collecting descendants when the size of a generation exceeds
 maxLimitSetDescendantObjs or the depth of the traversal exceeds
 maxLimitSetDescendantLevels."

| parents alreadySeen done levels |
(maxLimitSetDescendantObjs <= 0 or:[maxLimitSetDescendantLevels <= 0])
  ifTrue:[ ^ self ]. "Limit set descendants is disabled"

parents := self defaultLimitSet .
alreadySeen := parents copy.
done := false.
levels := 0.
[done] whileFalse:[ |children|
  levels := levels + 1.
  children := parents primReferencedObjects .
  children removeAll: alreadySeen.
  children isEmpty
    ifTrue:[ done := true]
    ifFalse:[
      limitObjects add: children.
      alreadySeen addAll: children.
      done := (children size > maxLimitSetDescendantObjs) or:[ levels >= maxLimitSetDescendantLevels ].
      parents := children.
    ].
].
^ self

]

{ #category : 'Results' }
GsSingleRefPathFinder >> buildResultObjects [
"Returns an Array of GsSingleRefPathResult objects"

^ allSearches collect:[:e| e buildResultObject]

]

{ #category : 'Scanning' }
GsSingleRefPathFinder >> completedOneSearch [
numSearchesActive := numSearchesActive - 1.

]

{ #category : 'Limit Set' }
GsSingleRefPathFinder >> defaultLimitSet [

"Return first element in the limitObjects array."

^ limitObjects first

]

{ #category : 'Limit Set' }
GsSingleRefPathFinder >> handleLimitSetDescendantsWithSearchObject: anObj [

| aSearch|
aSearch := self searchForObject: anObj.
aSearch handleLimitSetDescendantsWithSearchObject.
^ self

]

{ #category : 'Limit Set' }
GsSingleRefPathFinder >> handleSearchObjectsInLimitSetDescendants [
| intersection |

intersection := self searchObjectsInLimitSetDescendants.
intersection do:[:eachObj|
  self handleLimitSetDescendantsWithSearchObject: eachObj ].
^ self

]

{ #category : 'Initialization' }
GsSingleRefPathFinder >> initializeForSearchObjects: anArray [

self basicInit .
self validateSearchObjects: anArray .
searchObjects := anArray asGsBitmap .
anArray do:[:e| self addSearchObject: e] .
^ self

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> limitObjects [
^limitObjects

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> limitObjects: newValue [
limitObjects := newValue

]

{ #category : 'Limit Set' }
GsSingleRefPathFinder >> limitSetDescendants [
"Answer a GsBitmap containing all descendants of the base limit set, but not
 the base limit set itself."

| result |
result := GsBitmap new.
"First element is the base limit set.  Skip that one."
2 to: limitObjects size do: [:n | result addAll: (limitObjects at: n)].
^result

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> lockWaitTime [
^lockWaitTime

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> lockWaitTime: newValue [
lockWaitTime := newValue

]

{ #category : 'Logging' }
GsSingleRefPathFinder >> logEndOfScan [

printToLog > 0 ifTrue:[ | msg |
  msg := String withAll: 'Finished scan '.
  msg addAll: numPassesDone asString.
  self printTimedOpEndMessageToLog: msg]
]

{ #category : 'Logging' }
GsSingleRefPathFinder >> logFinished [

printToLog > 0 ifTrue:[ self printMessageToLog: 'Finished scans' ].
]

{ #category : 'Logging' }
GsSingleRefPathFinder >> logLimitSetSizes [

printToLog > 2 ifTrue:[ | ws |
  ws := AppendStream on: String new.
  ws
    nextPutAll: 'Sizes of limit set descendants by generation:';
    lf.
  1 to: limitObjects size do:[:n |
    ws tab;
      nextPutAll: n asString;
      space: 2;
      nextPutAll: (limitObjects at: n) size asString;
      lf].
  self class printMessageToLog: ws contents includeTime: false].
^self

]

{ #category : 'Logging' }
GsSingleRefPathFinder >> logStartOfScan [

printToLog > 0 ifTrue: [| msg numCompleted numSearchObjs |
  numSearchObjs := searchObjects size .
  numCompleted := numSearchObjs - numSearchesActive .
  msg := String withAll: 'Starting scan '.
  msg addAll: numPassesDone asString ;
    addAll: '. Search object summary: ';
    addAll: numSearchObjs asString ;
    addAll: ' total,  ' ;
    addAll: numSearchesActive asString ;
    addAll: ' active,  ';
    addAll: numCompleted asString ;
    addAll: ' completed'.
  self printTimedOpStartMessageToLog: msg]
  
]

{ #category : 'Logging' }
GsSingleRefPathFinder >> logStartup [

printToLog > 0 ifTrue:[
    self printMessageToLog:
      'Starting find one reference path scan for the following objects:';
    printSearchObjectsToLog]

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> maxLimitSetDescendantLevels [
^ maxLimitSetDescendantLevels

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> maxLimitSetDescendantLevels: newValue [
maxLimitSetDescendantLevels := newValue

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> maxLimitSetDescendantObjs [
^ maxLimitSetDescendantObjs

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> maxLimitSetDescendantObjs: newValue [
maxLimitSetDescendantObjs := newValue

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> maxThreads [
^maxThreads

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> maxThreads: newValue [
maxThreads := newValue

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> numPassesDone [
^numPassesDone

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> numPassesDone: newValue [
numPassesDone := newValue

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> numSearchesActive [
^numSearchesActive

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> numSearchesActive: newValue [
numSearchesActive := newValue

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> opStartTime [
^opStartTime

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> opStartTime: newValue [
opStartTime := newValue

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> otScanDone [
^otScanDone

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> otScanDone: newValue [
otScanDone := newValue

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> pageBufferSize [
^pageBufferSize

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> pageBufferSize: newValue [

"Value must be a power of 2"
newValue _validateIsPowerOf2 .
pageBufferSize := newValue

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> percentCpuLimit [
^percentCpuLimit

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> percentCpuLimit: newValue [
percentCpuLimit := newValue

]

{ #category : 'Logging' }
GsSingleRefPathFinder >> printMessageToLog: aString [

printToLog > 0 ifTrue: [self class printMessageToLog: aString].
^ self

]

{ #category : 'Logging' }
GsSingleRefPathFinder >> printSearchObjectsToLog [

printToLog > 1 ifTrue:[| oopList classNameList msg |
  msg := String new.
  oopList := allSearches collect: [:e | e searchOop asOop asString].
  classNameList := allSearches
  collect: [:e | e searchOop class name asString].
  1 to: oopList size do:[:n |
    msg
    addAll: '   ';
    addAll: n asString;
    addAll: '  ';
    addAll: (oopList at: n);
    addAll: ' (';
    addAll: (classNameList at: n);
    addAll: ')';
    add: Character lf].
  GsFile gciLogServer: msg]

]

{ #category : 'Logging' }
GsSingleRefPathFinder >> printTimedOpEndMessageToLog: aString [


printToLog > 1 ifTrue: [ |seconds msg|
  seconds := (System timeGmt2005 - opStartTime) asString .
  msg := String withAll: aString .
  msg addAll: ' in '; addAll: seconds ; addAll: ' seconds'.
  self class printMessageToLog: msg].
^ self

]

{ #category : 'Logging' }
GsSingleRefPathFinder >> printTimedOpStartMessageToLog: aString [

printToLog > 1 ifTrue: [
  opStartTime := System timeGmt2005 .
  self class printMessageToLog: aString].
^ self

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> printToLog [
^printToLog

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> printToLog: newValue [
  newValue == true ifTrue:[ printToLog := 1. ^ self] .
  newValue == false ifTrue:[ printToLog := 0 . ^ self].
  newValue ifNil:[ printToLog := 0 . ^ self].
  newValue _validateClass: SmallInteger .
  printToLog := newValue
]

{ #category : 'Scanning' }
GsSingleRefPathFinder >> runOneScan [

| searches childrenBitmaps parentBitmaps |
self scanObjectTable .
searches := self allActiveSearches.
searches do:[:e| e updateSearchOopsUnion].
childrenBitmaps := searches collect:[:e| e childrenToFind ].
numPassesDone := numPassesDone + 1.
self logStartOfScan.
parentBitmaps := GsSingleRefPathFinder scanForParents: childrenBitmaps.
self logEndOfScan.
1 to: searches size do:[:n|
  (searches at: n) processResultsOfScan: (parentBitmaps at: n)
].

]

{ #category : 'Scanning' }
GsSingleRefPathFinder >> runScan [
 [
   self logStartup.
   self buildLimitSet .
   self validateSearchObjectsAreNotLimitObjects .
   self handleSearchObjectsInLimitSetDescendants .
   scanStartTime := System timeGmt2005 .
   [numSearchesActive > 0] whileTrue:[ self runOneScan].
   self logFinished .
 ] ensure:[
   GsSingleRefPathFinder refPathCleanup
 ]
]

{ #category : 'Results' }
GsSingleRefPathFinder >> scanAndReport [
"Run the scan, build the results, and collect the path strings
 for each result.  Return a string."

| str |
str := String new.
self runScan buildResultObjects
   do:[:e| str add: e resultString].
^str

]

{ #category : 'Scanning' }
GsSingleRefPathFinder >> scanObjectTable [

otScanDone ifFalse:[
  self printTimedOpStartMessageToLog: 'Starting object table scan'.
  GsSingleRefPathFinder
    _refPathSetupScanWithMaxThreads: maxThreads
    waitForLock: lockWaitTime
    pageBufSize: pageBufferSize
    percentCpuLimit: percentCpuLimit.
  self printTimedOpEndMessageToLog: 'Finished object table scan'.
  otScanDone := true].
^self

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> scanStartTime [
^scanStartTime

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> scanStartTime: newValue [
scanStartTime := newValue

]

{ #category : 'Searching' }
GsSingleRefPathFinder >> searchForObject: anObject [
"Find the GsSingleRefPathFinderForObject instance for anObject."
^ allSearches detect:[:e| e searchOop == anObject ]

]

{ #category : 'Accessing' }
GsSingleRefPathFinder >> searchObjects [
^searchObjects

]

{ #category : 'Updating' }
GsSingleRefPathFinder >> searchObjects: newValue [
  "Warning, limit set might need recompuation if you use this method"
searchObjects := newValue

]

{ #category : 'Limit Set' }
GsSingleRefPathFinder >> searchObjectsInLimitSetDescendants [
"Answer a GsBitmap containing any search objects that are present in the
 descendants of the limit set."

^ self limitSetDescendants * searchObjects

]

{ #category : 'Validation' }
GsSingleRefPathFinder >> validateSearchObjects: anArray [

"Check input array to ensure all objects are committed disk objects.
 Special objects and uncommitted objects are not allowed."

1 to: anArray size do: [:i| | obj |
  obj := anArray at: i .
  (obj isSpecial or: [ obj isCommitted not ]) ifTrue:
    [ obj _error: #rtErrSpecialOrNotCommitted ].
  obj isClass ifTrue:[ self dynamicInstVarAt: #searchForClass put: true ].
].
^ self

]

{ #category : 'Validation' }
GsSingleRefPathFinder >> validateSearchObjectsAreNotLimitObjects [
"Checks to ensure that none of the search objects appear in the default limit
 set.  Raises an exception if any objects satisfy that condition."

| searchOopsInLimitSet|
searchOopsInLimitSet := self defaultLimitSet * searchObjects .
searchOopsInLimitSet isEmpty ifFalse:[
    ArgumentError signal: 'Search objects also appear in the limit set , oops: ',
        searchOopsInLimitSet _asArrayOfOops printString 
].
^ self

]
