"
GsSftpRemoteFile supports creating, reading and writing remote files via an instance of GsSftpSocket. 
You must have an established connection on the GsSftpSocket before creating the instance of 
GsSftpRemoteFile. Instances are created usng the GsSftpSocket as an argument.  

instance variables
  sftpSocket -- the instance of GsSftpSocket through which the file is accessed.
  removeFileName -- the name of the remote file

For example, to download a remote file to a local file using SFTP:

sshFtpSock := GsSftpSocket newClient.
sshFtpSock connectToHost: hostnameOrIP timeoutMs: 2000.
sshFtpSock userId: userName; password: userPassword.
sshFtpSock disableHostAuthentication.
sshFtpSock sshConnect.
remoteSftpFile := GsSftpRemoteFile 
    openRemoteFileReadOnly: remoteFileName 
    withSftpSocket: sftpSock.
localFile := GsFile openWriteOnServer: localFileName.
bytes := remoteSftpFile readAllInto: localFile.
localFile close.
sftpFile close.
"
Class {
	#name : 'GsSftpRemoteFile',
	#superclass : 'Object',
	#instVars : [
		'sftpSocket',
		'remoteFileName'
	],
	#gs_reservedoop : '250625',
	#category : 'OSAccess-Sockets'
}

{ #category : 'Private' }
GsSftpRemoteFile class >> _fiveArgSftpRemoteFilePrim: opcode with: arg1 with: arg2 with: arg3 with: arg4 with: arg5 [

"opcode function
1		instance method:	readFromOffset: to: into: startingAt:
2		instance method:	writeToOffset: from: startingAt: endingAt:
100		class method: 		openRemoteFile:mode:permissions:errorIfExists:withSftpSocket:

"
<primitive: 905>

^self _primitiveFailed: #_fiveArgSftpRemoteFilePrim:with:with:with:with:with: args: {opcode}

]

{ #category : 'Instance Creation' }
GsSftpRemoteFile class >> createNewRemoteFile: fileName withSftpSocket: aGsSftpSocket [

"Creates a new remote file for writing with default permissions on a remote host using a connected instance of GsSftpSocket.
Raises an error if the remote file already exists.
Returns a new instance of the receiver or raises an exception on error."

^self openRemoteFile: fileName mode: 'w' permissions: nil errorIfExists: true withSftpSocket: aGsSftpSocket

]

{ #category : 'Instance Creation' }
GsSftpRemoteFile class >> createOrOverwriteRemoteFile: fileName withSftpSocket: aGsSftpSocket [

"Opens or creates a remote file with default permissions for writing on a remote host using a connected instance of GsSftpSocket.
If the file already exists it is overwritten. If the file does not exist it will be created.

Returns a new instance of the receiver or raises an exception on error."

^self openRemoteFile: fileName mode: 'w' permissions: nil errorIfExists: false withSftpSocket: aGsSftpSocket

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> createRandomRemoteFileFromSftpSocket: sftpSock forSelector: aSelector [

"Create a new remote file with a random file name and upload 64 K of random data to it.
Answer the new remote file name."

|  sftpFile remoteFn ba |
remoteFn := self randomRemoteFileNameForSelector: aSelector .
"Create a new write-only sftpFile instance using the sftp socket we created."
ba := ByteArray withRandomBytes: 65536 .
sftpFile := GsSftpRemoteFile createNewRemoteFile: remoteFn withSftpSocket: sftpSock .
sftpFile writeAllFrom: ba .
sftpFile close. "Close remote file"
^ remoteFn

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> downloadToLocalFileExample [
"This example will download a random file from the sftp server to the local /tmp directory."

"GsSftpRemoteFile downloadToLocalFileExample"

| sftpSock sftpFile gsFile  localFn bytes anyFile sel |
sel := (GsProcess methodAt: 1 ) selector .
"Common example method to create the sftpSocket"
sftpSock := self getSftpSocketExample .
anyFile := self createRandomRemoteFileFromSftpSocket: sftpSock  forSelector: sel.
localFn := '/tmp/', anyFile .
"Create a new read-only sftpFile instance using the sftp socket we created.
We will download the file named anyFile on ssh-test to local file /tmp/anyFile"
sftpFile := self openRemoteFileReadOnly: anyFile withSftpSocket: sftpSock .

"Open a new GsFile instance where we will write the data. Delete file if it's already there"
gsFile := GsFile removeServerFile: localFn ; openWriteOnServer: localFn .
"Download the entire file"
bytes := sftpFile readAllInto: gsFile .
gsFile close.
sftpFile close.
sftpSock removeRemoteFile: anyFile ; close.
^ 'success: ', bytes asString , ' bytes were downloaded to ', localFn

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> downloadToObjectExample [
"This example downloads the first 64K of a random remote file into a ByteArray object."

"GsSftpRemoteFile downloadToObjectExample"

| sftpSock sftpFile  bytes byteArray anyFile sel |
sel := (GsProcess methodAt: 1 ) selector .
"Common example method to create the sftpSocket"
sftpSock := self getSftpSocketExample .
anyFile :=self createRandomRemoteFileFromSftpSocket: sftpSock  forSelector: sel.
"Create a new read-only sftpFile instance using the sftp socket we created."
sftpFile := self openRemoteFileReadOnly: anyFile withSftpSocket: sftpSock .
byteArray := ByteArray new.

"Download the first 64K of the file into the ByteArray"
bytes := sftpFile readAllInto: byteArray .
sftpFile close.
sftpSock removeRemoteFile: anyFile ; close.
^ Array with: ('success: ', bytes asString , ' bytes were downloaded to a ByteArray ') with: byteArray

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> getSftpSocketExample [

"GsSftpRemoteFile getSftpSocketExample"

| sftpSock host |
host :=  GsSshSocket exampleHost .

"Uncomment next line to turn on tracing to a log file in /tmp"
"GsSftpSocket enableTraceFileInDirectory: '/tmp' ."
"Get a GsSftpSocket connection to the test sftp server on ssh port 22"
sftpSock := GsSftpSocket newClient.
[sftpSock connectTo: 22 on: host] onException: (SocketError, SshSocketError) do:[:ex|
	sftpSock close.
	ex pass
].
"Set userId / password for test server"
sftpSock userId: GsSshSocket exampleUserId ; password: GsSshSocket examplePassword   .
"Disable authenticating the remote host"
sftpSock disableHostAuthentication .
"Connect and perform ssh handshake. If this fails, check firewall settings"
sftpSock sshConnect .
^sftpSock

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> lstatExample [
"This example does an lstat operation on a random remote file on the sftp server and returns an instance of GsFileStat."
"GsSftpRemoteFile lstatExample"

| sftpSock anyFile stat  |
"Common example method to create the sftpSocket"
sftpSock := self getSftpSocketExample .
anyFile := (sftpSock contentsOfRemoteDirectoryNoDotFiles: '.') first.
stat := sftpSock lstat: anyFile .
sftpSock close.
^ stat

]

{ #category : 'Instance Creation' }
GsSftpRemoteFile class >> new [
  self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
GsSftpRemoteFile class >> new: aSize [
  self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
GsSftpRemoteFile class >> openRemoteFile: fileName mode: openMode permissions: permInt errorIfExists: aBoolean withSftpSocket: aGsSftpSocket [

"Opens or creates a remote file on a remote host using a connected instance of GsSftpSocket.
The fileName argument  is name of the file on the remote host.  The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+' '.  The mode has the same meaning as it does for the
 C library function, fopen().

For modes that create a new file, permInt must be a SmallInteger between 0 and 8r777,
or nil which causes the default file permissions of 8r770 to be used.

If aBoolean is true and the mode is a write or append mode, an error will be raised if the file already exists.
If aBoolean is false and the mode is a write or append mode, the file will be overwritten if it already exists.
For read modes, aBoolean is ignored and an error will be raised if the file does not exist.

Returns a new instance of the receiver or raises an exception on error."

| result |
result := self _fiveArgSftpRemoteFilePrim: 100 with: fileName with: openMode with: permInt with: aBoolean with: aGsSftpSocket .
result remoteFileName: fileName copy ;  initializeWith: aGsSftpSocket .
^result

]

{ #category : 'Instance Creation' }
GsSftpRemoteFile class >> openRemoteFileAppend: fileName withSftpSocket: aGsSftpSocket [

"Opens a remote file for appending (writing at the end) on a remote host using a connected instance of GsSftpSocket.
Creates the file if it does not exist.
Returns a new instance of the receiver or raises an exception on error."

^self openRemoteFile: fileName mode: 'a' permissions: nil errorIfExists: false withSftpSocket: aGsSftpSocket

]

{ #category : 'Instance Creation' }
GsSftpRemoteFile class >> openRemoteFileReadOnly: fileName withSftpSocket: aGsSftpSocket [

"Opens a remote file for reading on a remote host using a connected instance of GsSftpSocket.
Raises an error if the remote file does not exist.
Returns a new instance of the receiver or raises an exception on error."

^self openRemoteFile: fileName mode: 'r' permissions: nil errorIfExists: false withSftpSocket: aGsSftpSocket

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> randomRemoteFileNameForSelector: aSelector [

"Create a new remote file with a random file name including aSelector"

^ aSelector asString, '_', GsUuidV4 new asString, '.rnd' . "generate random remote filename which includes the callers selector"

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> statExample [
"This example does an stat operation on a random remote file on the sftp server and returns an instance of GsFileStat."
"GsSftpRemoteFile statExample"

| sftpSock anyFile stat  |
"Common example method to create the sftpSocket"
sftpSock := self getSftpSocketExample .
anyFile := (sftpSock contentsOfRemoteDirectoryNoDotFiles: '.') first.
stat := sftpSock stat: anyFile .
sftpSock close.
^ stat

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> uploadObjectToRemoteFileAndRenameExample [
"This example uploads the data contained in a ByteArray to a new file on the sftp server and then renames the file"

"GsSftpRemoteFile uploadObjectToRemoteFileAndRenameExample"

| sftpSock sftpFile bytes remoteFn randomData newRemoteFn sel |
sel := (GsProcess methodAt: 1 ) selector .
"Common example method to create the sftpSocket"
sftpSock := self getSftpSocketExample .

remoteFn := self randomRemoteFileNameForSelector: sel . "generate random remote filename"
"Create a new write-only sftpFile instance using the sftp socket we created."
sftpFile := self createNewRemoteFile: remoteFn withSftpSocket: sftpSock .
"Create a ByteArray of random data"
randomData := ByteArray withRandomBytes: 65536 .

"Upload the entire ByteArray to the file"
bytes := sftpFile writeAllFrom: randomData .
sftpFile close.
"Cleanup: Tell the sftpSocket to delete the remote file"
newRemoteFn := GsUuidV4 new asString . "generate random new remote filename"
sftpSock renameRemoteFile: remoteFn to: newRemoteFn .
sftpSock removeRemoteFile: newRemoteFn ; close.
^ 'success: ', bytes asString , ' bytes were uploaded to ', remoteFn , ' and renamed to ', newRemoteFn

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> uploadObjectToRemoteFileExample [
"This example uploads the data contained in a ByteArray to a new file on the sftp server"

"GsSftpRemoteFile uploadObjectToRemoteFileExample"

| sftpSock sftpFile bytes remoteFn randomData sel |
sel := (GsProcess methodAt: 1 ) selector .
"Common example method to create the sftpSocket"
sftpSock := self getSftpSocketExample .

remoteFn := self randomRemoteFileNameForSelector: sel . "generate random remote filename"
"Create a new write-only sftpFile instance using the sftp socket we created."
sftpFile := self createNewRemoteFile: remoteFn withSftpSocket: sftpSock .
"Create a ByteArray of random data"
randomData := ByteArray withRandomBytes: 65536 .

"Upload the entire ByteArray to the file"
bytes := sftpFile writeAllFrom: randomData  .
sftpFile close.
"Cleanup: Tell the sftpSocket to delete the remote file"
sftpSock removeRemoteFile: remoteFn ; close .
^ 'success: ', bytes asString , ' bytes were uploaded to ', remoteFn

]

{ #category : 'Examples' }
GsSftpRemoteFile class >> uploadToRemoteFileExample [
"This example uploads the file $GEMSTONE/bin/intiail.config to the sftp server under a new random file name"

"GsSftpRemoteFile uploadToRemoteFileExample"

| sftpSock sftpFile gsFile localFn bytes remoteFn gs sel |
sel := (GsProcess methodAt: 1 ) selector .
"Common example method to create the sftpSocket"
sftpSock := self getSftpSocketExample .
gs := System gemEnvironmentVariable: 'GEMSTONE'.
localFn := gs, '/bin/initial.config' .
remoteFn := self randomRemoteFileNameForSelector: sel .
"Create a new write-only sftpFile instance using the sftp socket we created."
sftpFile := self  createNewRemoteFile: remoteFn withSftpSocket: sftpSock .
"Open a new GsFile instance where we will read the data from."
gsFile := GsFile openReadOnServer: localFn .
gsFile ifNil:[ self halt: ('Cannot find local file ', localFn) ].
"Upload the entire file"
bytes := sftpFile writeAllFrom: gsFile .
gsFile close. "Close source file"
sftpFile close. "Close remote file"
sftpSock removeRemoteFile: remoteFn ; close.
^ 'success: ', bytes asString , ' bytes were uploaded to ', remoteFn

]

{ #category : 'Private' }
GsSftpRemoteFile >> _fiveArgSftpRemoteFilePrim: opcode with: arg1 with: arg2 with: arg3 with: arg4 with: arg5 [

"opcode function
1		instance method:	readFromOffset: to: into: startingAt:
2		instance method:	writeToOffset: from: startingAt: endingAt:
100		class method: 		openRemoteFile:mode:permissions:errorIfExists:withSftpSocket:

"
<primitive: 905>

^self _primitiveFailed: #_fiveArgSftpRemoteFilePrim:with:with:with:with:with: args: {opcode}

]

{ #category : 'Private' }
GsSftpRemoteFile >> _oneArgSftpRemoteFilePrim: opcode with: arg1 [

"GsSshSocket opCodes
opcode	method
1		instance method: hostAuthenticationEnabled:
2		instance method: password:
3		instance method: privateKey:
4		instance method: executeRemoteCommand:
5		instance method: makeBlocking / makeNonBlocking
6		instance method: nbExecuteRemoteCommand:
100		class method enableTraceFileInDirectory:
101		class method setSshTraceLevel:

GsSftpSocket opCodes
51		instance method: removeRemoteDirectory:
53		instance method: removeRemoteFile:
54		instance method: stat:
55		instance method: lstat:

GsSftpRemoteFile opCodes:
70		instance method: position:
"
<primitive: 903>
^ self _primitiveFailed: #_oneArgSftpRemoteFilePrim:with: args: { opcode }

]

{ #category : 'Private' }
GsSftpRemoteFile >> _twoArgSftpRemoteFilePrim: opcode with: arg1 with: arg2 [
"
GsSshSocket opCodes
opcode  function
1	instance methods: sshOptionAt: / sshOptionAt:put:

GsSftpSocket opCodes
50	instance method: createRemoteDirectory:mode:
51	instance method: renameRemoteFile: to:
52	instance method: contentsAndStatDetailsOfRemoteDirectory:withPattern:
53	instance method: contentsOfRemoteDirectorywithPattern:

GsSftpRemoteFile opCodes
70	instance method: read:into:
71	instance method: write:from:
"

<primitive: 904>
^ self _primitiveFailed: #_twoArgSftpRemoteFilePrim:with:with: args: {opcode}

]

{ #category : 'Private' }
GsSftpRemoteFile >> _zeroArgSftpRemoteFilePrim: opCode [

"GsSshSocket opCodes
opcode  method
1		instance method: initializeAsClient
2		instance method: initializeAfterConnect
3		instance method _sshConnect
4		instance method hasSshConnectInProgress
5		instance method _sshClose
6		instance method nbRemoteCommandResult
7		instance method: hasCommandInProgress
8		instance method isBlocking
9		instance method nbRemoteCommandResultReady
100		class method disableTraceFile
101 	class method getSshLogLevel
102		class method libSshVersion
103		class method traceFileName

GsSftpSocket opCodes
opcode  method
50		instance method: currentRemoteDirectory

GsSftpRemoteFile opCodes
opcode  method
70		instance method: close
71		instance method: isOpen
72		instance method: fstat
73		instance method: isReadable
74		instance method: isWriteable
75		instance method: isAppendable
76		instance method: position
"

<primitive: 902>
^ self _primitiveFailed: #_zeroArgSftpRemoteFilePrim: args: { opCode }

]

{ #category : 'Assertions' }
GsSftpRemoteFile >> assertOpen [

^ self isOpen
	ifTrue: [ self ]
	ifFalse:[ SshSocketError signal: 'Attempted operation on a remote file which is not open']

]

{ #category : 'Assertions' }
GsSftpRemoteFile >> assertReadable [

^ self isReadable
	ifTrue: [ self ]
	ifFalse:[ SshSocketError signal: 'Attempt to read a remote file not opened for reading']

]

{ #category : 'Testing' }
GsSftpRemoteFile >> atEnd [
"Returns true if the receiver is currently positioned at the end of its
 file, false if not. Raises an exception if the receiver is not open or if an error occurs."

self assertOpen ; assertReadable.
^ self position >= self size

]

{ #category : 'Closing' }
GsSftpRemoteFile >> close [

self _zeroArgSftpRemoteFilePrim: 70 .
sftpSocket _removeReference: self .
sftpSocket := nil.
^ self

]

{ #category : 'File Operations' }
GsSftpRemoteFile >> fstat [

"Return an instance of GsFileStat describing the remote file.
Raise an exception if the file is not open or cannot be accessed."

^ self _zeroArgSftpRemoteFilePrim: 72

]

{ #category : 'Private' }
GsSftpRemoteFile >> initializeWith: aGsSftpSocket [

"primitive stores aGsSftpSocket in inst var aGsSftpSocket"
aGsSftpSocket _addReference: self .
^ self

]

{ #category : 'Testing' }
GsSftpRemoteFile >> isAppendable [

"Answer a Boolean indicating if the receiver is open for appending"

^ self _zeroArgSftpRemoteFilePrim: 75

]

{ #category : 'Testing' }
GsSftpRemoteFile >> isOpen [

"Answer a Boolean indicating if the receiver is open"

^ self _zeroArgSftpRemoteFilePrim: 71

]

{ #category : 'Testing' }
GsSftpRemoteFile >> isReadable [

"Answer a Boolean indicating if the receiver is open for reading"

^ self _zeroArgSftpRemoteFilePrim: 73

]

{ #category : 'Testing' }
GsSftpRemoteFile >> isWritable [

"Answer a Boolean indicating if the receiver is open for writing"

^ self _zeroArgSftpRemoteFilePrim: 74

]

{ #category : 'Reading' }
GsSftpRemoteFile >> peek [

"Returns the a Character representing the next byte in the receiver without advancing
the file pointer. The receiver must be open for reading. Returns nil if the receiver is at the end
of file. Raises an exception on error."

|ba|
^ (1 == (self peek: 1 into: (ba := ByteArray new: 1)))
	ifTrue:[ ba charAt: 1 ]
	ifFalse:[ nil ]

]

{ #category : 'Reading' }
GsSftpRemoteFile >> peek: amount into: aByteObject [

"Reads the next amount bytes from the receiver and stores into aByteObject without advancing the receiver's file pointer.
The receiver must be open for reading. Returns the number of bytes stored in the receiver, which may be less than amount
if the operation would read beyond the end of the receiver. Raises an exception on error."

|pos result |
pos := self position .
result := self read: amount into: aByteObject .
self position: pos.
^ result

]

{ #category : 'Positioning' }
GsSftpRemoteFile >> position [

"Answer the current position in the remote file. The position is the zero-based offset from the first byte in the remote file,
i.e. the beginning of the file has a position of 0."

^ self _zeroArgSftpRemoteFilePrim: 76

]

{ #category : 'Positioning' }
GsSftpRemoteFile >> position: anInt [

"Changes the position of file pointer of the receiver to be anInt. The position is the zero-based offset from the first byte in the remote file,
i.e. the beginning of the file has a position of 0.
Raises an error if anInt would change the position to be beyond the end of the remote file.
Returns the receiver."

^self _oneArgSftpRemoteFilePrim: 70 with: anInt

]

{ #category : 'Reading' }
GsSftpRemoteFile >> read: amount into: byteObjOrGsFile [

"Reads up to the given number of bytes into byteObjOrGsFile starting at the current file position of the receiver.
The receiver must be open for reading. If byteObjOrGsFile is a byte object, the data read is stored starting at index 1.
If byteObjOrGsFile is an instance of GsFile, the data read is stored into the GsFile as if the method #write:from: method
had been called. The GsFile must be opened for writing or appending.

Returns the number of bytes stored into byteObjOrGsFile."

^self _twoArgSftpRemoteFilePrim: 70 with: amount with: byteObjOrGsFile

]

{ #category : 'Reading' }
GsSftpRemoteFile >> readAllInto: byteObjOrGsFile [

"Reads the contents of the receiver into byteObjOrGsFile. Reading begins at the current position of the reeiver's file pointer.
The receiver must be open for reading. If byteObjOrGsFile is a byte object, the data read is stored starting at index 1.
If byteObjOrGsFile is an instance of GsFile, the data read is stored into the GsFile as if the method #write:from: method
had been called. The GsFile must be opened for writing or appending.

Returns the number of bytes stored into byteObjOrGsFile."

^self read: -1 into: byteObjOrGsFile

]

{ #category : 'Accessing' }
GsSftpRemoteFile >> remoteFileName [
	^remoteFileName

]

{ #category : 'Updating' }
GsSftpRemoteFile >> remoteFileName: newValue [
	remoteFileName := newValue

]

{ #category : 'Accessing' }
GsSftpRemoteFile >> size [

"Answer a SmallInteger indicating the size of the remote file in bytes.
Raises an error if the receiver is not open."

^ self fstat size

]

{ #category : 'Writing' }
GsSftpRemoteFile >> write: amount from: byteObjOrGsFile [

"Writes the given number of bytes from byteObjOrGsFile to the receiver which must be open for writing or appending.
If the receiver is open for appending, the write begins at the end of the file. Otherwise the write begins at the receiver's current
file pointer position.

byteObjOrGsFile must be either an instance of a ByteClass or an instance of GsFile open for reading.
Returns the number of bytes written to the receiver."

^self _twoArgSftpRemoteFilePrim: 71 with: amount with: byteObjOrGsFile

]

{ #category : 'Writing' }
GsSftpRemoteFile >> writeAllFrom: byteObjOrGsFile [

"Writes the contents of byteObjOrGsFile into the receiver. The receiver must be open for writing or appending.
If the receiver is open for appending, the write begins at the end of the file. Otherwise the write begins at the receiver's current
file pointer position.

byteObjOrGsFile must be either an instance of a ByteClass or an instance of GsFile open for reading.
If byteObjOrGsFile is a GsFile, reading begins at the current position of the file pointer.
Returns the number of bytes written to the receiver."

^self write: -1 from: byteObjOrGsFile

]
