"
GsSecureSocket provides the means for creating and binding TLS TCP sockets
 through the operating system of the machine that is running the session's Gem process, and
 for communicating across those sockets.  Methods that block GemStone Smalltalk
 until the socket operation completes are interruptable by a hard break.  (You
 can get a hard break in Topaz by pressing the control-C key twice.  You can get
 a hard break in GemBuilder for C by calling the GciHardBreak function, and in
 GemBuilder for Smalltalk by calling the corresponding hard break method.)

 Like instances of GsSocket, instances of GsSecureSocket automatically have
 their C state closed when the instance is garbage collected or 
 when a persistent instance drops out of memory.

 SSLv2 and SSLv3 connections are not supported because such connections are known to 
 be insecure. Only and TLSv1.x connections are currently supported.

 To create a secure connection, a regular socket connection must first be established.
 Use the methods inherited from GsSocket to establish a socket connection.

                              Warning:
    Do not retain an instance of GsSecureSocket from one session to another.
    Instances of GsSecureSocket are intended to exist only within a given GemStone
    session.  GsSecureSockets that are used across sessions always generate an error.

 All instVars of GsSecureSocket are private, for use by the implementation
 of socket methods, and for use by the ProcessorScheduler only.

Constraints:
	fileDescriptor: SmallInteger
	lineNumber: SmallInteger
	readWaiters: Object
	writeWaiters: Object
	readyEvents: SmallInteger
	pollArrayOfs: SmallInteger
	interrupting: Object
"
Class {
	#name : 'GsSecureSocket',
	#superclass : 'GsSignalingSocket',
	#gs_reservedoop : '248833',
	#category : 'OSAccess-Sockets'
}

{ #category : 'Private' }
GsSecureSocket class >> _newClientNoSslState [

"Used in hostagent code.
 Creates a new instance with no connection and no ssl state and a AF_INET client socket."

^ super new 
]

{ #category : 'Private' }
GsSecureSocket class >> _oneArgSslPrim: opcode with: arg [

"opcode  function
   1        class method: useCACertificateFileForClients:
   2        class method: useCACertificateFileForServers:
   3        class method: setClientCipherListFromString:
   4        class method: setServerCipherListFromString:
   5     instance method: setCipherListFromString:
   6        class method: setCertificateVerificationOptionsForServer:
   7     instance method: setCertificateVerificationOptions:
   8        class method: useCACertificateDirectoryForClients:
   9        class method: useCACertificateDirectoryForServers:
  10     instance method: initializeAsServerFromGsSocket:
  11     instance method: initializeAsClientFromGsSocket:
  12     instance method: _installSsl:
  13     instance method: setServerNameIndication:
  14     instance method: setPreSharedKey:
  15     instance method: tlsMinVersion:
  16     instance method: tlsMaxVersion:
  17:       class method: tlsServerMinVersion:
  18:       class method: tlsServerMaxVersion:
  19:       class method: tlsClientMinVersion:
  20:       class method: tlsClientMaxVersion:
  21     instance method: setExpectedHost:
  22     instance method: addExpectedHost:
  23     instance method: setExpectedHostFlags:
"

<primitive: 910>
^ self _primitiveFailed: #_oneArgSslPrim:with: args: { opcode . arg }

]

{ #category : 'Private' }
GsSecureSocket class >> _threeArgSslPrim: opcode with: arg1 with: arg2 with: arg3 [

"opcode  function
   1     instance method: _write:startingAt:ofSize:
   2     instance method: _readInto:startingAt:maxBytes:
   3        class method: useServerCertificate: withPrivateKey: privateKeyPassphrase:
   4        class method: useClientCertificate: withPrivateKey: privateKeyPassphrase:
   5     instance method: useCertificate: withPrivateKey: privateKeyPassphrase:
   6:       class method: useServerCertificateFile:withPrivateKeyFile:privateKeyPassphrase:
   7:       class method: useClientCertificateFile:withPrivateKeyFile:privateKeyPassphrase:
   8:    instance method: useCertificateFile:withPrivateKeyFile::privateKeyPassphrase:
"
<primitive: 909>
^ self _primitiveFailed: #_threeArgSslPrim:with:with:with:
  args: { opcode . arg1 . arg2 . arg3 }

]

{ #category : 'Private' }
GsSecureSocket class >> _zeroArgSslPrim: opcode [

"opcode  function
   1     instance method: secureConnect
   2     instance method: secureAccept
   3     instance method: hasSecureConnection
   4     instance method: secureClose
   5     instance method: fetchLastIoErrorString
   6        class method: fetchErrorStringArray
   7        class method: clearErrorQueue
   8        class method: disableCertificateVerificationOnClient
   9        class method: enableCertificateVerificationOnClient
  10        class method: disableCertificateVerificationOnServer
  11        class method: enableCertificateVerificationOnServer
  12     instance method: initializeAsClient
  13     instance method: initializeAsServer
  14        class method: fetchLastCertificateVerificationErrorForServer
  15        class method: fetchLastCertificateVerificationErrorForClient
  16     instance method: fetchCipherDescription
  17     instance method: disableCertificateVerification
  18     instance method: enableCertificateVerification
  19        class method: sslLibraryVersionString
  20     instance method: _peek
  21     instance method: certificateVerificationEnabled
  22        class method: certificateVerificationEnabledOnClient
  23        class method: certificateVerificationEnabledOnServer
  24     instance method: fetchCertificateVerificationOptions
  25        class method: fetchCertificateVerificationOptionsForServer
  26     instance method:  _noFreeSslOnGc
  27     instance method: getServerNameIndication
  28     instance method: tlsMinVersion
  29     instance method: tlsMaxVersion
  30        class method: tlsServerMinVersion
  31        class method: tlsServerMaxVersion
  32        class method: tlsClientMinVersion
  33        class method: tlsClientMaxVersion
  34     instance method: tlsActualVersion
  35     instance method: peerCertificate
  36     instance method: peerCertificateChain
  37     instance method: matchedPeerName
"

<primitive: 908>
^ self _primitiveFailed: #_zeroArgSslPrim: args: { opcode }

]

{ #category : 'Examples' }
GsSecureSocket class >> anonymousTlsClientExample: logToGciClient usingPort: portNum on: host [

| sslClient data dataBuffer |

"Anonymous TLS means encrypting data over the socket connection without verifying
 the client or the server's identity. Because identities are not verified, certificates 
 are not required to establish a connection.

 WARNING: anonymous TLS connections are exposed to man-in-the-middle attacks and 
 are therefore NOT recommended for most applications."

"Setup a normal socket connection first."

"Create a new SSL client socket"
sslClient:= self newClient.

[ | failBlock bytesRead dataLength gotEof |

  failBlock := [:sock :shouldLog| | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do:[:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Now connect to the server to get a normal socket connection.
   On error, close the socket and pass to the next exception handler."
  (sslClient connectTo: portNum on: host timeoutMs: 3000)
    ifTrue:[ GsFile gciLog: 'regular socket connect completed OK.'
                    onClient: logToGciClient ] .

  "uncomment next line to put the socket in blocking mode."
  "sslClient makeBlocking."

  "Enable all ciphers that do not require authentication. 
	-ALL - remove all ciphers from the list but allow adding back later.
	aNULL - all ciphers that do not require authentication.
	@STRENGTH - sort ciphers by strength.
	@SECLEVEL=0 - enable low security ciphers. Required to use ciphers that do not authenticate."
  (sslClient setCipherListFromString: '-ALL:aNULL:@STRENGTH:@SECLEVEL=0')
    ifFalse:[ GsFile gciLog: 'setCipherListFromString: failed'
                    onClient: logToGciClient
    ].

  "Attempt to establish a secure connection to the SSL server..."
  ([sslClient secureConnect]
    on: SocketError
    do:[:ex| |sslError|
      GsFile gciLog: 'secureConnect failed' onClient: logToGciClient.
      sslError := sslClient fetchLastIoErrorString .
      sslError ifNil: [
         GsFile gciLog: 'nil result from fetchLastIoErrorString'
                onClient: logToGciClient
          ] ifNotNil: [
          GsFile gciLog: ('SSL error: =>' , sslError) onClient: logToGciClient].
	  ex pass.
    ])
    ifTrue:[ GsFile gciLog: 'Secure connection established'
                    onClient: logToGciClient].


  "We have a secure connection to the server if we get here."
  GsFile gciLog: ('Current cipher in use is: ', sslClient fetchCipherDescription)
         onClient: logToGciClient.

  "Read length from the client"
  GsFile gciLog: 'Starting read' onClient: logToGciClient.

  dataLength := sslClient read: 8.
  (dataLength == nil or: [dataLength size < 8])
       ifTrue: [GsFile gciLog: 'read of 8 bytes of length failed'
  				onClient: logToGciClient .
  	     		failBlock value: sslClient value: logToGciClient]
       ifFalse: [GsFile gciLog: 'read 8 bytes of length, encoded ', dataLength
  				onClient: logToGciClient].
  dataLength := Integer fromHexString: dataLength.
  GsFile gciLog: 'Data length should be ', dataLength asString
  	onClient: logToGciClient.

  dataBuffer := String new.
  gotEof := false .
  [(dataBuffer size < (dataLength + 10)) and:[ gotEof not]] whileTrue: [
     bytesRead := sslClient read: 16272 into: dataBuffer startingAt: dataBuffer size + 1.
     bytesRead ifNil:[
       GsFile gciLog: 'read failed' onClient: logToGciClient .
              failBlock value: sslClient  value: logToGciClient
     ] ifNotNil:[
        bytesRead == 0 ifTrue:[ gotEof := true ]
          ifFalse:[ GsFile gciLog: 'Read ', bytesRead asString, ' bytes from peer.'
                    onClient: logToGciClient ].
     ].
  ].
  gotEof ifFalse:[
    GsFile gciLog: 'did not get EOF' onClient: logToGciClient.
    failBlock value: sslClient value: logToGciClient.
  ].
  dataBuffer size ~= dataLength ifTrue: [
    GsFile gciLog: 'Read failed ' onClient: logToGciClient.
    failBlock value: sslClient value: logToGciClient.
  ].
  GsFile gciLog: 'Finished reading, read ', dataBuffer size asString, ' bytes from peer'
  	onClient: logToGciClient .

] on: SocketError do:[:ex| sslClient close. ex pass ].

sslClient close.

data := (PassiveObject newWithContents: dataBuffer) activate.
GsFile gciLog: 'Result is ', data class asString, ' of size ', data size asString
     onClient: logToGciClient.
^true
]

{ #category : 'Examples' }
GsSecureSocket class >> anonymousTlsServerExample: logToGciClient usingPort: portNum on: host [


"Anonymous TLS means encrypting data over the socket connection without verifying
 the client or the server's identity. Because identities are not verified, certificates 
 are not required to establish a connection.

 WARNING: anonymous TLS connections are exposed to man-in-the-middle attacks and 
 are therefore NOT recommended for most applications."

| listener sslServer |

[
  | failBlock bigStr result data dataSizeString |

  failBlock := [:sock :shouldLog | | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do: [:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Create a new SSL server socket."
  listener := self newServer.

  "Make it listen for the client."
  (listener makeServerAtPort: portNum) ifNil: [ | err |
  	err := listener lastErrorString.
  	listener close.
  	SocketError signal: 'makeServerAtPort failed, ', err asString.
   ].

  listener port == portNum ifFalse:[ SocketError signal: 'bad port number ', listener port asString ] .

  GsFile gciLog: 'Waiting for GsSecureSocket clientExample to connect on port ', portNum asString 
  	onClient: logToGciClient .
  [listener readWillNotBlockWithin: 5000] whileFalse: [
  	GsFile gciLog: 'waiting...' onClient: logToGciClient
    	].

  GsFile gciLog: 'Client connected.' onClient: logToGciClient .

  "Do the normal accept to establish a standard socket connection."
  "sslServer is our connection to the client."
  sslServer := listener accept .

  GsFile gciLog: 'Normal accept completed OK.' onClient: logToGciClient .

  "uncomment next to put the socket in blocking mode."
  "sslServer makeBlocking."

  "sslServer has only a normal connection to the peer.  If will have a secure
   connection if secureAccept succeeds."


  "Enable all ciphers that do not require authentication. 
	-ALL - remove all ciphers from the list but allow adding back later.
	aNULL - all ciphers that do not require authentication.
	@STRENGTH - sort ciphers by strength.
	@SECLEVEL=0 - enable low security ciphers. Required to use ciphers that do not authenticate."

  (sslServer setCipherListFromString: '-ALL:aNULL:@STRENGTH:@SECLEVEL=0')
    ifFalse:[ GsFile gciLog: 'setCipherListFromString: failed'  onClient: logToGciClient  ].

  ([sslServer secureAccept] on: SocketError do:[:ex| | sslError |
    GsFile gciLog: 'secureAccept failed.'  onClient: logToGciClient.
    sslError := sslServer fetchLastIoErrorString .
    sslError ifNil: [
        GsFile gciLog: 'nil result from fetchLastIoErrorString'  onClient: logToGciClient
    ] ifNotNil:[
        GsFile gciLog: ('SSL error: =>' , sslError)
           onClient: logToGciClient.].
	ex pass
    ])
      ifTrue:[ GsFile gciLog: 'Secure connection established.'
                      onClient: logToGciClient ].

  "If we get here, we have a secure connection to the client."

  GsFile gciLog: ('Current cipher in use is: ', sslServer fetchCipherDescription)
         onClient: logToGciClient.

  "Build data to send"
  data := Array new.
  4096 timesRepeat: [ data add: (String withAll: '0123456789ABCDEF') ].
  bigStr := String new .
  PassiveObject passivate: data toStream: (WriteStream on: bigStr).
  dataSizeString := bigStr size asHexStringWithLength: 8.
  dataSizeString size > 8 ifTrue:
      [UserDefinedError signal: 'Protocol Error, attempt to send more than 4294967295 bytes'].

  GsFile gciLog: 'Sending data length to client: ', bigStr size asString,
      ' bytes, encoded as ', dataSizeString onClient: logToGciClient.

  result := sslServer write: dataSizeString.
  result == dataSizeString size
    ifTrue: [ GsFile gciLog: 'Finished sending length to peer'
  		onClient: logToGciClient ]
    ifFalse: [ GsFile gciLog: 'write of length bytes failed, result was ', result asString
                  onClient: logToGciClient .
              failBlock value: sslServer value: logToGciClient.
  		].

  GsFile gciLog: 'Sending ', bigStr size asString, ' bytes to client'
         onClient: logToGciClient.
  result := sslServer write: bigStr .
  result == bigStr size
    ifTrue:[ GsFile gciLog: ('Finished sending ', bigStr size asString, ' bytes to peer')
                     onClient: logToGciClient ]
    ifFalse:[ GsFile gciLog: ('write failed, result was ', result asString)
                     onClient: logToGciClient .
              failBlock value: sslServer value: logToGciClient.
              ].
] on: SocketError do:[:ex|
        sslServer ifNotNil:[ sslServer close ].
        listener ifNotNil:[ listener close ].
        ex pass
].

"close method will close the SSL connection and underlying socket."
sslServer close.
listener close .
^ true
]

{ #category : 'Instance Creation' }
GsSecureSocket class >> basicNew [

"disallowed"

self shouldNotImplement: #basicNew

]

{ #category : 'Querying' }
GsSecureSocket class >> certificateVerificationEnabledOnClient [

"Answer true if certificate verification is enabled for client sockets.
 Otherwise answer false."

^ self _zeroArgSslPrim: 22

]

{ #category : 'Querying' }
GsSecureSocket class >> certificateVerificationEnabledOnServer [

"Answer true if certificate verification is enabled for server sockets.
 Otherwise answer false."

^ self _zeroArgSslPrim: 23

]

{ #category : 'Error Handling' }
GsSecureSocket class >> clearErrorQueue [

"Clears all errors from the error queue for all GsSecureSocket instances.
 Returns the receiver."

^ self _zeroArgSslPrim: 7

]

{ #category : 'Examples' }
GsSecureSocket class >> clientExample [

"Setup socket using class methods"
^ self clientExample: true usingPort: 57785 on: 'localhost'

]

{ #category : 'Examples' }
GsSecureSocket class >> clientExample: logToGciClient usingPort: portNum on: host [

| sslClient data dataBuffer |

"Setup a normal socket connection first."

"Create an new SSL client socket"
sslClient:= self newClient.

[ | failBlock bytesRead dataLength gotEof |

  failBlock := [:sock :shouldLog| | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do:[:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Now connect to the server to get a normal socket connection.
   On error, close the socket and pass to the next exception handler."
  (sslClient connectTo: portNum on: host timeoutMs: 3000)
    ifTrue:[ GsFile gciLog: 'regular socket connect completed OK.'
                    onClient: logToGciClient ] .

  "uncomment next line to put the socket in blocking mode."
  "sslClient makeBlocking."

  "Specify the file which contains a list of trusted CA certificates.
   These will be used to validate the certificate presented by the peer
   during the SSL handshake.  The file must be in PEM format."
  GsSecureSocket useCACertificateFileForClients:
        '$GEMSTONE/examples/openssl/certs/cacert.pem'.

  "Tell SSL that all clients are to drop the connection unless
   the certificate presented by the server is authenticated."
  GsSecureSocket enableCertificateVerificationOnClient .

  "Set the cipher list for client SSL sockets.
   Use all ciphers except NULL ciphers and anonymous Diffie-Hellman
   and sort by strength."
  (GsSecureSocket setClientCipherListFromString: 'ALL:!ADH:@STRENGTH')
    ifFalse:[ GsFile gciLog: 'setClientCipherListFromString: failed'
                    onClient: logToGciClient
    ].

  "Attempt to establish a secure connection to the SSL server..."
  ([sslClient secureConnect]
    on: SocketError
    do:[:ex| |certError|
      GsFile gciLog: 'secureConnect failed' onClient: logToGciClient.
      certError := GsSecureSocket fetchLastCertificateVerificationErrorForClient .
      certError ifNil: [
         GsFile gciLog: 'nil result from fetchLastCertificateVerificationErrorForClient'
                onClient: logToGciClient
          ] ifNotNil: [
          GsFile gciLog: ('Cert error: =>' , certError) onClient: logToGciClient].
	  ex pass.
    ])
    ifTrue:[ GsFile gciLog: 'Secure connection established'
                    onClient: logToGciClient].


  "We have a secure connection to the server if we get here."
  GsFile gciLog: ('Current cipher in use is: ', sslClient fetchCipherDescription)
         onClient: logToGciClient.

  "Read length from the client"
  GsFile gciLog: 'Starting read' onClient: logToGciClient.

  dataLength := sslClient read: 8.
  (dataLength == nil or: [dataLength size < 8])
       ifTrue: [GsFile gciLog: 'read of 8 bytes of length failed'
  				onClient: logToGciClient .
  	     		failBlock value: sslClient value: logToGciClient]
       ifFalse: [GsFile gciLog: 'read 8 bytes of length, encoded ', dataLength
  				onClient: logToGciClient].
  dataLength := Integer fromHexString: dataLength.
  GsFile gciLog: 'Data length should be ', dataLength asString
  	onClient: logToGciClient.

  dataBuffer := String new.
  gotEof := false .
  [(dataBuffer size < (dataLength + 10)) and:[ gotEof not]] whileTrue: [
     bytesRead := sslClient read: 16272 into: dataBuffer startingAt: dataBuffer size + 1.
     bytesRead ifNil:[
       GsFile gciLog: 'read failed' onClient: logToGciClient .
              failBlock value: sslClient  value: logToGciClient
     ] ifNotNil:[
        bytesRead == 0 ifTrue:[ gotEof := true ]
          ifFalse:[ GsFile gciLog: 'Read ', bytesRead asString, ' bytes from peer.'
                    onClient: logToGciClient ].
     ].
  ].
  gotEof ifFalse:[
    GsFile gciLog: 'did not get EOF' onClient: logToGciClient.
    failBlock value: sslClient value: logToGciClient.
  ].
  dataBuffer size ~= dataLength ifTrue: [
    GsFile gciLog: 'Read failed ' onClient: logToGciClient.
    failBlock value: sslClient value: logToGciClient.
  ].
  GsFile gciLog: 'Finished reading, read ', dataBuffer size asString, ' bytes from peer'
  	onClient: logToGciClient .

] on: SocketError do:[:ex| sslClient close. ex pass ].

sslClient close.

data := (PassiveObject newWithContents: dataBuffer) activate.
GsFile gciLog: 'Result is ', data class asString, ' of size ', data size asString
     onClient: logToGciClient.
^true

]

{ #category : 'Examples' }
GsSecureSocket class >> clientExample2 [

"Setup socket using instances methods"
^ self clientExample2: true usingPort: 57785 on: 'localhost'

]

{ #category : 'Examples' }
GsSecureSocket class >> clientExample2: logToGciClient usingPort: portNum on: host [

"This version uses GsSecureSocket instance methods for setup rather
 than class methods."

| sslClient data dataBuffer |


"Create an new SSL client socket"
sslClient:= self newClient.

[ | failBlock bytesRead dataLength gotEof |

  failBlock := [:sock :shouldLog| | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do:[:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Specify the file which contains a list of trusted CA certificates.
   These will be used to validate the certificate presented by the peer
   during the SSL handshake.  The file must be in PEM format.
   Currently, there is no easy way to specify the CA list from a String.
   This must be done before creating the instance of GsSecureSocket! "
  GsSecureSocket useCACertificateFileForClients:
        '$GEMSTONE/examples/openssl/certs/cacert.pem'.

  "Connect to the server to get a normal socket connection.
   On error, close the socket and pass to the next exception handler."
  (sslClient connectTo: portNum on: host timeoutMs: 3000)
    ifTrue:[ GsFile gciLog: 'regular socket connect completed OK.'
                    onClient: logToGciClient ] .

  "uncomment next line to put the socket in blocking mode."
  "sslClient makeBlocking."

  "Tell SSL that this SSL client is to drop the connection unless
   the certificate presented by the server is authenticated."
  sslClient enableCertificateVerification .

  "Set the cipher list for this SSL socket only.
   Use all ciphers except NULL ciphers and anonymous Diffie-Hellman
   and sort by strength."
  sslClient setCipherListFromString: 'ALL:!ADH:@STRENGTH'.

  "Attempt to establish a secure connection to the SSL server..."
  ([sslClient secureConnect]
    on: SocketError
    do:[:ex| |certError|
      GsFile gciLog: 'secureConnect failed' onClient: logToGciClient.
      certError := GsSecureSocket fetchLastCertificateVerificationErrorForClient .
      certError ifNil: [
         GsFile gciLog: 'nil result from fetchLastCertificateVerificationErrorForClient'
                onClient: logToGciClient
          ] ifNotNil: [
          GsFile gciLog: ('Cert error: =>' , certError) onClient: logToGciClient].
      ex pass.
    ])
    ifTrue:[ GsFile gciLog: 'Secure connection established'
                    onClient: logToGciClient].

  "We have a secure connection to the server if we get here."
  GsFile gciLog: ('Current cipher in use is: ', sslClient fetchCipherDescription)
         onClient: logToGciClient.

  "Read length from the client"
  GsFile gciLog: 'Starting read' onClient: logToGciClient.

  dataLength := sslClient read: 8.
  (dataLength == nil or: [dataLength size < 8])
       ifTrue: [GsFile gciLog: 'read of 8 bytes of length failed'
  				onClient: logToGciClient .
  	     		failBlock value: sslClient value: logToGciClient]
       ifFalse: [GsFile gciLog: 'read 8 bytes of length, encoded ', dataLength
  				onClient: logToGciClient].
  dataLength := Integer fromHexString: dataLength.
  GsFile gciLog: 'Data length should be ', dataLength asString
  	onClient: logToGciClient.

  dataBuffer := String new.
  gotEof := false .
  [(dataBuffer size < (dataLength + 10)) and:[ gotEof not]] whileTrue: [
     bytesRead := sslClient read: 16272 into: dataBuffer startingAt: dataBuffer size + 1.
     bytesRead ifNil:[
       GsFile gciLog: 'read failed' onClient: logToGciClient .
              failBlock value: sslClient  value: logToGciClient
     ] ifNotNil:[
        bytesRead == 0 ifTrue:[ gotEof := true ]
          ifFalse:[ GsFile gciLog: 'Read ', bytesRead asString, ' bytes from peer.'
                    onClient: logToGciClient ].
     ].
  ].
  gotEof ifFalse:[
    GsFile gciLog: 'did not get EOF' onClient: logToGciClient.
    failBlock value: sslClient value: logToGciClient.
  ].
  dataBuffer size ~= dataLength ifTrue: [
    GsFile gciLog: 'Read failed ' onClient: logToGciClient.
    failBlock value: sslClient value: logToGciClient.
  ].
  GsFile gciLog: 'Finished reading, read ', dataBuffer size asString, ' bytes from peer'
  	onClient: logToGciClient .

] on: SocketError do:[:ex| sslClient close. ex pass ].

sslClient close.

data := (PassiveObject newWithContents: dataBuffer) activate.
GsFile gciLog: 'Result is ', data class asString, ' of size ', data size asString
     onClient: logToGciClient.
^true

]

{ #category : 'Peer Authentication' }
GsSecureSocket class >> disableCertificateVerificationOnClient [
"Directs new client sockets ignore invalid server certificates.  The connection
 will not fail if the server certificate is found to be invalid.

 The client always requests a certificate from the server.  By default,
 the connection will fail if the certificate from the server is invalid.

 The effect is valid for the current session only;
 the affected SSL state is not committed to the repository.

 Has no effect on instances created before the method was called.

 Returns the receiver."

^ self _zeroArgSslPrim: 8

]

{ #category : 'Peer Authentication' }
GsSecureSocket class >> disableCertificateVerificationOnServer [
"Directs server sockets to not request a certificate from the client.
 This is the default mode for server sockets.

 The effect is valid for the current session only;
 the affected SSL state is not committed to the repository.

 Has no effect on instances created before the method was called.

 Returns the receiver."

^ self _zeroArgSslPrim: 10

]

{ #category : 'Peer Authentication' }
GsSecureSocket class >> enableCertificateVerificationOnClient [
"Directs client sockets to verify the certificate from the server.
 The connection will fail if the server does not provide a valid
 certificate to the client.

 The client always requests a certificate from the server.

 This is the default mode for client sockets.

 The effect is valid for the current session only;
 the affected SSL state is not committed to the repository.

 Has no effect on instances created before the method was called.

 Returns the receiver."

^ self _zeroArgSslPrim: 9

]

{ #category : 'Peer Authentication' }
GsSecureSocket class >> enableCertificateVerificationOnServer [
"Directs server sockets to request and verify a certificate from the client.
 The connection will fail if the client does not provide a certificate or
 the certificate is found to be invalid.

 The effect is valid for the current session only;
 the affected SSL state is not committed to the repository.

 Has no effect on instances created before the method was called.

 By default, servers do not request certificates from clients.

 Returns the receiver."

^ self _zeroArgSslPrim: 11

]

{ #category : 'Querying' }
GsSecureSocket class >> fetchCertificateVerificationOptionsForServer [

"Answers an Array of Symbols which represent the certificate verification
 options used by server sockets.

 The supported server options are:

 #SSL_VERIFY_FAIL_IF_NO_PEER_CERT
    if the client did not return a certificate, the TLS/SSL handshake is
    immediately terminated with a 'handshake failure' alert.

 #SSL_VERIFY_CLIENT_ONCE
    only request a client certificate on the initial TLS/SSL handshake.
    Do not ask for a client certificate again in case of a renegotiation.

 Refer to the OpenSSL documentation for more information about these options."

^ self _zeroArgSslPrim: 25

]

{ #category : 'Error Handling' }
GsSecureSocket class >> fetchErrorStringArray [

"Returns an Array of error strings generated by the OpenSSL package.
 The errors returned are cleared from the SSL error queue.  The array
 is ordered from oldest to newest error."

^ self _zeroArgSslPrim: 6

]

{ #category : 'Error Handling' }
GsSecureSocket class >> fetchLastCertificateVerificationErrorForClient [

"Fetches and clears a string representing the last certificate verification
 error logged by a client SSL socket.  Returns nil if no such error has
 occurred."

^ self _zeroArgSslPrim: 15

]

{ #category : 'Error Handling' }
GsSecureSocket class >> fetchLastCertificateVerificationErrorForServer [

"Fetches and clears a string representing the last certificate verification
 error logged by a server SSL socket.  Returns nil if no such error has
 occurred."

^ self _zeroArgSslPrim: 14

]

{ #category : 'Examples' }
GsSecureSocket class >> getPasswordFromFile: aString [
| gsf result |

gsf := GsFile openReadOnServer: aString .
gsf ifNil:[ SocketError signal: ('Could not open password file ', aString) ] .
result := gsf nextLine trimWhiteSpace .
gsf close.
 ^ result

]

{ #category : 'Examples' }
GsSecureSocket class >> httpsClientExample [

"Connect to the google https web server on port 443 in blocking mode and
 perform a simple GET request.  Full verification of the server certificate
 is performed.  Returns true on success"

^ self httpsClientExampleForHost: 'www.google.com'
       withSniName: 'google.com'
       blocking: true

]

{ #category : 'Examples' }
GsSecureSocket class >> httpsClientExampleForHost: hostName withSniName: sniName [

  ^ self  httpsClientExampleForHost: hostName withSniName: sniName blocking: true
]

{ #category : 'Examples' }
GsSecureSocket class >> httpsClientExampleForHost: hostName withSniName: sniName blocking: bool [

"Connect to an https web server on port 443 at the given host and
 perform a simple GET request.  Full verification of the server certificate
 is performed.  Self-signed certificates will fail verification."

| sslClient sni |

[
  | failBlock portNum logToGciClient request bytesSent responseString caCert |
  
  portNum := 443. "https port"
  logToGciClient := true.
  failBlock := [:sock :shouldLog| | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do:[:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].
  
  "Specify the file or directory which contains a list of trusted CA certificates.
   These will be used to validate the certificate presented by the peer
   during the SSL handshake.  The files in the directory must be in PEM
   format and the file names must be the hash value of the CA subject name.
   See https://www.openssl.org/docs/man1.0.2/ssl/SSL_CTX_load_verify_locations.html
   for more information.
  
   The file or directory must be specified before creating the instance of GsSecureSocket."
  
  caCert := GsSecureSocket setCaCertLocation .
  caCert ifNil:[  GsFile gciLog:
                     'Warning: Unable to find CA cert file on this host. Disabling cert verification'
		     onClient: false ]
       ifNotNil:[ GsFile gciLog: ('Using CA cert file ', caCert) onClient: false ].
       
  
  "Create an new SSL client socket"
  sslClient:= self newClient.
  
  "Now connect to the server to get a normal socket connection."
  sslClient connectTo: portNum on: hostName timeoutMs: 3000 .

  "Put the socket in the requested blocking mode."
  bool ifTrue:[  sslClient makeBlocking ] ifFalse:[ sslClient makeNonBlocking ].
  
  "Tell SSL that this SSL client is to drop the connection unless
   the certificate presented by the server is authenticated."
  sslClient enableCertificateVerification .
  
  "Set the cipher list for this SSL socket only.
   Use all ciphers except NULL ciphers and anonymous Diffie-Hellman
   and sort by strength."
  sslClient setCipherListFromString: 'ALL:!ADH:@STRENGTH' .
  
  "If we have an SNI hostname, set it now"
  sniName ifNotNil:[ (sslClient setServerNameIndication: sniName)
                        ifFalse:[sslClient signalError: 'Error setting SNI hostname']
  	] .
  sni := sslClient getServerNameIndication .
  GsFile gciLog: ('sni name returned by getServerNameIndication is ', sni asString)
         onClient: false .
  
  "Attempt to establish a secure connection to the SSL server..."
  ([sslClient secureConnect]
    on: SocketError
    do:[:ex| |certError|
      GsFile gciLog: 'secureConnect failed' onClient: logToGciClient.
      certError := GsSecureSocket fetchLastCertificateVerificationErrorForClient .
      certError ifNil: [  
         GsFile gciLog: 'nil result from fetchLastCertificateVerificationErrorForClient' 
                onClient: logToGciClient
          ] ifNotNil: [
          GsFile gciLog: ('Cert error: =>' , certError) onClient: logToGciClient].
	  ex pass.
    ])
    ifTrue:[ GsFile gciLog: 'Secure connection established'
                    onClient: logToGciClient].

  "We have a secure connection to the server if we get here."
  
  "Get the cipher used and print it"
  GsFile gciLog: ('Current cipher in use is: ', sslClient fetchCipherDescription)
         onClient: logToGciClient.
  
  "Build a simple http 1.1 GET request string.  Terminate with a blank line"
  request := String withAll: 'GET / HTTP/1.1'.
  request add: Character cr ;
          add: Character lf ;
  	addAll: 'Host: ';
  	addAll: hostName ;
          add: Character cr ;
          add: Character lf ;
          add: Character cr ;
  	add: Character lf.
  
  GsFile gciLog: 'Sending a ', request size asString, ' byte request to client: '
       onClient: logToGciClient.
  GsFile gciLog: request onClient: logToGciClient.
  
  "Send the GET request to the web server"
  bytesSent := sslClient write: request.
  bytesSent == request size
    ifTrue: [ GsFile gciLog: 'Finished sending request to peer' 
  		onClient: logToGciClient ]
    ifFalse: [ GsFile gciLog: 'write of request failed, result was ', bytesSent asString,
    ' expected ', request size asString
                  onClient: logToGciClient .
              failBlock value: sslClient value: logToGciClient.
  ].
  
  "Get the response from the web server and print it"
  GsFile gciLog: 'Waiting for response from server...' onClient: logToGciClient.
  responseString := sslClient read: 2048 .
  GsFile gciLog: 'Finished reading ', responseString size asString, ' bytes from server.'
  		onClient: logToGciClient.
  GsFile gciLog: responseString onClient: logToGciClient.

] on: SocketError do:[:ex|
   sslClient ifNotNil:[ sslClient close ].
   ex pass
].   
"All done! Close the connection and return"
sslClient close.
^true

]

{ #category : 'Examples' }
GsSecureSocket class >> httpsClientExampleNB [

"Connect to the google https web server on port 443 in nonblocking mode and
 perform a simple GET request.  Full verification of the server certificate
 is performed.  Returns true on success"

^ self httpsClientExampleForHost: 'www.google.com'
       withSniName: 'google.com'
       blocking: false
]

{ #category : 'Examples' }
GsSecureSocket class >> httpsSniClientExample [

"Example to connect to a webserver that requires SNI in blocking mode.
 Not using SNI (server name == nil) will fail to connect.

 Returns true on success"

^ self httpsClientExampleForHost: 'chrismeller.com'
       withSniName: 'chrismeller.com'
       blocking: true

]

{ #category : 'Examples' }
GsSecureSocket class >> httpsSniClientExampleNB [

"Example to connect to a webserver that requires SNI in nonblocking mode. 
 Not using SNI (server name == nil) will fail to connect.

 Returns true on success"

^ self httpsClientExampleForHost: 'chrismeller.com'
       withSniName: 'chrismeller.com'
       blocking: false

]

{ #category : 'Error Handling' }
GsSecureSocket class >> lastErrorString [

  "Provided for compatibility with GsSocket.
   Returns a string describing the elements in the SSL error queue,
   and clears that queue.  Returns nil if no error is available"

| arr str |
arr := self fetchErrorStringArray .
arr size > 0 ifTrue:[
  str := String new .
  1 to: arr size do:[:j | str add: (arr at: j); add: '; ' ].
] ifFalse:[
  str := super lastErrorString .
].
^ str

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> new [
  self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> new: aSize [
  self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> newClient [

"Creates a new instance and initializes it to act as a AF_INET client socket.
 No connection is made."

^ super new initializeAsClient

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> newClientFromGsSocket: aGsSocket [

"Creates a new instance and initializes it to act as an SSL client socket.
 Any existing (non-SSL) connection owned by aGsSocket is transferred to the
 new instance and aGsSocket is effectively closed. No SSL connection is made."

^ super new initializeAsClientFromGsSocket: aGsSocket

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> newClientIpv6 [

"Creates a new instance and initializes it to act as a AF_INET6 client socket.
 No connection is made."

^ super newIpv6 initializeAsClient

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> newClientIpv6FromGsSocket: aGsSocket [

"Creates a new IPv6 instance and initializes it to act as an SSL client socket.
 Any existing (non-SSL) connection owned by a GsSocket is transferred to the
 new instance and aGsSocket is effectively closed. No SSL connection is made."

^ super newIpv6 initializeAsClientFromGsSocket: aGsSocket

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> newServer [

"Creates a new instance and initializes it to act as a AF_INET server socket.
 No connection is made."

^ super new initializeAsServer

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> newServerFromGsSocket: aGsSocket [

"Creates a new instance and initializes it to act as an SSL server socket.
 Any existing (non-SSL) connection owned by aGsSocket is transferred to the
 new instance and aGsSocket is effectively closed. No SSL connection is made."

^ super new initializeAsServerFromGsSocket: aGsSocket

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> newServerIpv6 [

"Creates a new instance and initializes it to act as a AF_INET6 server socket.
 No connection is made."

^ super newIpv6 initializeAsServer

]

{ #category : 'Instance Creation' }
GsSecureSocket class >> newServerIpv6FromGsSocket: aGsSocket [

"Creates a new IPv6 instance and initializes it to act as an SSL server socket.
 Any existing (non-SSL) connection owned by a GsSocket is transferred to the
 new instance and aGsSocket is effectively closed. No SSL connection is made."

^ super newIpv6 initializeAsServerFromGsSocket: aGsSocket

]

{ #category : 'Examples' }
GsSecureSocket class >> preSharedKeyTlsClientExample: logToGciClient usingPort: portNum on: host [

| sslClient data dataBuffer psk |

"Pre-shared key transport layer security (PSK-TLS) requires the client and server
to both know a pre-shared secret key before the connection is initiated. The key
must be at least 8 bytes in size (16 hex digits) and be stored in a ByteArray. 
logToGciClient true: trace output goes to client (topaz console), false: output goes to 
server (linked topaz console or gem log file).
"

"Setup a normal socket connection first."

"Create a new SSL client socket"
sslClient:= self newClient.

[ | failBlock bytesRead dataLength gotEof |

  failBlock := [:sock :shouldLog| | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do:[:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Now connect to the server to get a normal socket connection.
   On error, close the socket and pass to the next exception handler."
  (sslClient connectTo: portNum on: host timeoutMs: 3000)
    ifTrue:[ GsFile gciLog: 'regular socket connect completed OK.'
                    onClient: logToGciClient ] .

  "uncomment next line to put the socket in blocking mode."
  "sslClient makeBlocking."

  "Enable all PSK ciphers.
	-ALL - remove all ciphers from the list but allow adding back later.
	PSK - all PSK ciphers.
	@STRENGTH - sort ciphers by strength."
  (sslClient setCipherListFromString: '-ALL:PSK:@STRENGTH')
    ifFalse:[ GsFile gciLog: 'setCipherListFromString: failed'
                    onClient: logToGciClient
    ].

  "Psk must be a ByteArray containing at least 8 bytes and with size a multiple of 8"
  "PSK used by client/server must exactly match."
  psk := ByteArray fromHexString: 'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa'.
  "Set the PSK for this connection."
  sslClient setPreSharedKey: psk.

  "Attempt to establish a secure connection to the SSL server..."
  ([sslClient secureConnect]
    on: SocketError
    do:[:ex| |sslError|
      GsFile gciLog: 'secureConnect failed' onClient: logToGciClient.
      sslError := sslClient fetchLastIoErrorString .
      sslError ifNil: [
         GsFile gciLog: 'nil result from fetchLastIoErrorString'
                onClient: logToGciClient
          ] ifNotNil: [
          GsFile gciLog: ('SSL error: =>' , sslError) onClient: logToGciClient].
	  ex pass.
    ])
    ifTrue:[ GsFile gciLog: 'Secure connection established'
                    onClient: logToGciClient].

  "We have a secure connection to the server if we get here."

  "Clear PSK from memory"
  sslClient setPreSharedKey: nil.
  
  GsFile gciLog: ('Current cipher in use is: ', sslClient fetchCipherDescription)
         onClient: logToGciClient.

  "Read length from the client"
  GsFile gciLog: 'Starting read' onClient: logToGciClient.

  dataLength := sslClient read: 8.
  (dataLength == nil or: [dataLength size < 8])
       ifTrue: [GsFile gciLog: 'read of 8 bytes of length failed'
  				onClient: logToGciClient .
  	     		failBlock value: sslClient value: logToGciClient]
       ifFalse: [GsFile gciLog: 'read 8 bytes of length, encoded ', dataLength
  				onClient: logToGciClient].
  dataLength := Integer fromHexString: dataLength.
  GsFile gciLog: 'Data length should be ', dataLength asString
  	onClient: logToGciClient.

  dataBuffer := String new.
  gotEof := false .
  [(dataBuffer size < (dataLength + 10)) and:[ gotEof not]] whileTrue: [
     bytesRead := sslClient read: 16272 into: dataBuffer startingAt: dataBuffer size + 1.
     bytesRead ifNil:[
       GsFile gciLog: 'read failed' onClient: logToGciClient .
              failBlock value: sslClient  value: logToGciClient
     ] ifNotNil:[
        bytesRead == 0 ifTrue:[ gotEof := true ]
          ifFalse:[ GsFile gciLog: 'Read ', bytesRead asString, ' bytes from peer.'
                    onClient: logToGciClient ].
     ].
  ].
  gotEof ifFalse:[
    GsFile gciLog: 'did not get EOF' onClient: logToGciClient.
    failBlock value: sslClient value: logToGciClient.
  ].
  dataBuffer size ~= dataLength ifTrue: [
    GsFile gciLog: 'Read failed ' onClient: logToGciClient.
    failBlock value: sslClient value: logToGciClient.
  ].
  GsFile gciLog: 'Finished reading, read ', dataBuffer size asString, ' bytes from peer'
  	onClient: logToGciClient .

] on: SocketError do:[:ex| sslClient close. ex pass ].

sslClient close.

data := (PassiveObject newWithContents: dataBuffer) activate.
GsFile gciLog: 'Result is ', data class asString, ' of size ', data size asString
     onClient: logToGciClient.
^true
]

{ #category : 'Examples' }
GsSecureSocket class >> preSharedKeyTlsServerExample: logToGciClient usingPort: portNum on: host [

"Pre-shared key transport layer security (PSK-TLS) requires the client and server
to both know a pre-shared secret key before the connection is initiated. The key
must be at least 8 bytes in size (16 hex digits) and be stored in a ByteArray. 
logToGciClient true: trace output goes to client (topaz console), false: output goes to 
server (linked topaz console or gem log file).
"

| listener sslServer psk |

[
  | failBlock bigStr result data dataSizeString |

  failBlock := [:sock :shouldLog | | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do: [:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Create a new SSL server socket."
  listener := self newServer.

  "Make it listen for the client."
  (listener makeServerAtPort: portNum) ifNil: [ | err |
  	err := listener lastErrorString.
  	listener close.
  	SocketError signal: 'makeServerAtPort failed, ', err asString.
   ].

  listener port == portNum ifFalse:[ SocketError signal: 'bad port number ', listener port asString ] .

  GsFile gciLog: 'Waiting for GsSecureSocket pskClientExample to connect...'
  	onClient: logToGciClient .
  [listener readWillNotBlockWithin: 5000] whileFalse: [
  	GsFile gciLog: 'waiting...' onClient: logToGciClient
    	].

  GsFile gciLog: 'Client connected.' onClient: logToGciClient .

  "Do the normal accept to establish a standard socket connection."
  sslServer := listener accept .

  GsFile gciLog: 'Normal accept completed OK.' onClient: logToGciClient .

  "uncomment next to put the socket in blocking mode."
  "sslServer makeBlocking."

  "Don't request a certificate from the client."
  sslServer disableCertificateVerification.

  "sslServer has only a normal connection to the peer.  If will have a secure
   connection if secureAccept succeeds."

  "Enable all PSK ciphers.
	-ALL - remove all ciphers from the list but allow adding back later.
	PSK - all PSK ciphers.
	@STRENGTH - sort ciphers by strength."
  (sslServer setCipherListFromString: '-ALL:PSK:@STRENGTH')
    ifFalse:[ GsFile gciLog: 'setCipherListFromString: failed'
                    onClient: logToGciClient
    ].

  "PSK must be a ByteArray containing at least 8 bytes and with size a multiple of 8"
  psk := ByteArray fromHexString: 'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa'.
  "Set the PSK for this connection."
  sslServer setPreSharedKey: psk.

  ([sslServer secureAccept] on: SocketError do:[:ex| | certError |
    GsFile gciLog: 'secureAccept failed.'  onClient: logToGciClient.
    certError := sslServer class fetchLastCertificateVerificationErrorForServer .
    certError ifNil: [
        GsFile gciLog: 'nil result from fetchLastCertificateVerificationErrorForServer'
           onClient: logToGciClient
    ] ifNotNil:[
        GsFile gciLog: ('Cert error: =>' , certError)
           onClient: logToGciClient.].
	ex pass
    ])
      ifTrue:[ GsFile gciLog: 'Secure connection established.'
                      onClient: logToGciClient ].

  "If we get here, we have a secure connection to the client."

  "Clear PSK from memory"
  sslServer setPreSharedKey: nil.
  
  GsFile gciLog: ('Current cipher in use is: ', sslServer fetchCipherDescription)
         onClient: logToGciClient.

  "Build data to send"
  data := Array new.
  4096 timesRepeat: [ data add: (String withAll: '0123456789ABCDEF') ].
  bigStr := String new .
  PassiveObject passivate: data toStream: (WriteStream on: bigStr).
  dataSizeString := bigStr size asHexStringWithLength: 8.
  dataSizeString size > 8 ifTrue:
      [UserDefinedError signal: 'Protocol Error, attempt to send more than 4294967295 bytes'].

  GsFile gciLog: 'Sending data length to client: ', bigStr size asString,
      ' bytes, encoded as ', dataSizeString onClient: logToGciClient.

  result := sslServer write: dataSizeString.
  result == dataSizeString size
    ifTrue: [ GsFile gciLog: 'Finished sending length to peer'
  		onClient: logToGciClient ]
    ifFalse: [ GsFile gciLog: 'write of length bytes failed, result was ', result asString
                  onClient: logToGciClient .
              failBlock value: sslServer value: logToGciClient.
  		].

  GsFile gciLog: 'Sending ', bigStr size asString, ' bytes to client'
         onClient: logToGciClient.
  result := sslServer write: bigStr .
  result == bigStr size
    ifTrue:[ GsFile gciLog: ('Finished sending ', bigStr size asString, ' bytes to peer')
                     onClient: logToGciClient ]
    ifFalse:[ GsFile gciLog: ('write failed, result was ', result asString)
                     onClient: logToGciClient .
              failBlock value: sslServer value: logToGciClient.
              ].
] on: SocketError do:[:ex|
        sslServer ifNotNil:[ sslServer close ].
        listener ifNotNil:[ listener close ].
        ex pass
].

"close method will close the SSL connection and underlying socket."
sslServer close.
listener close .
^ true
]

{ #category : 'Examples' }
GsSecureSocket class >> serverExample [

^ self serverExample: true usingPort: 57785 on: 'localhost'

]

{ #category : 'Examples' }
GsSecureSocket class >> serverExample: logToGciClient usingPort: portNum on: host [

| listener sslServer pw |

[ | failBlock bigStr result data dataSizeString |
  failBlock := [:sock :shouldLog | | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do: [:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Specify the certificate and private key to use for all server connections in
   this gem.  Both must be in PEM format.  Private key and certificate may be
   contained in the same PEM file or in different files as we show here."

   pw := self getPasswordFromFile: '$GEMSTONE/examples/openssl/private/server_1_server_passwd.txt' .

   GsSecureSocket useServerCertificateFile: '$GEMSTONE/examples/openssl/certs/server_1_servercert.pem'
    withPrivateKeyFile: '$GEMSTONE/examples/openssl/private/server_1_serverkey.pem'
    privateKeyPassphrase: pw .

  "Don't request a certificate from the client.  This is typical."
  GsSecureSocket disableCertificateVerificationOnServer .

  "Use all ciphers except NULL ciphers and anonymous Diffie-Hellman
   and sort by strength."
  GsSecureSocket setServerCipherListFromString: 'ALL:!ADH:@STRENGTH' .

  "All class methods have been executed, so now we can create the instance.
   Class methods executed AFTER the instance is created will NOT APPLY to
   the instance!"

  "Create a new SSL server socket."
  listener := self newServer.

  "Make it listen for the client."
  (listener makeServerAtPort: portNum) ifNil: [ | err |
  	err := listener lastErrorString.
  	listener close.
  	SocketError signal: 'makeServerAtPort failed, ', err asString.
          ].

  listener port == portNum ifFalse:[ Error signal: 'bad port number' ] .

  GsFile gciLog: 'Waiting for GsSecureSocket clientExample to connect...'
  	onClient: logToGciClient .
  [listener readWillNotBlockWithin: 5000] whileFalse: [
  	GsFile gciLog: 'waiting...' onClient: logToGciClient
    	].

  GsFile gciLog: 'Client connected.' onClient: logToGciClient .

  "Do the normal accept to establish a standard socket connection."
  "sslServer is our connection to the client."
  sslServer := listener accept .
  sslServer ifNil: [| err |
          err := listener lastErrorString.
          listener close.
  	SocketError signal: 'accept failed, ', err asString.
  	].

  GsFile gciLog: 'Normal accept completed OK.' onClient: logToGciClient .

  "uncomment next to put the socket in blocking mode."
  "sslServer makeBlocking."

  "sslServer has only a normal connection to the peer.  If will have a secure
    connection if secureAccept succeeds."

  ([sslServer secureAccept] on: SocketError do:[:ex| | certError |
    GsFile gciLog: 'secureAccept failed.'  onClient: logToGciClient.
    certError := sslServer class fetchLastCertificateVerificationErrorForServer .
    certError ifNil: [
        GsFile gciLog: 'nil result from fetchLastCertificateVerificationErrorForServer'
           onClient: logToGciClient
    ] ifNotNil:[
        GsFile gciLog: ('Cert error: =>' , certError)
           onClient: logToGciClient.].
	ex pass
    ])
      ifTrue:[ GsFile gciLog: 'Secure connection established.'
                      onClient: logToGciClient ].

  "If we get here, we have a secure connection to the client."

  GsFile gciLog: ('Current cipher in use is: ', sslServer fetchCipherDescription)
         onClient: logToGciClient.

  "Build data to send"
  data := Array new.
  4096 timesRepeat: [ data add: (String withAll: '0123456789ABCDEF') ].
  bigStr := String new .
  PassiveObject passivate: data toStream: (WriteStream on: bigStr).
  dataSizeString := bigStr size asHexStringWithLength: 8.
  dataSizeString size > 8 ifTrue:
      [UserDefinedError signal: 'Protocol Error, attempt to send more than 4294967295 bytes'].

  GsFile gciLog: 'Sending data length to client: ', bigStr size asString,
      ' bytes, encoded as ', dataSizeString onClient: logToGciClient.

  result := sslServer write: dataSizeString.
  result == dataSizeString size
    ifTrue: [ GsFile gciLog: 'Finished sending length to peer'
  		onClient: logToGciClient ]
    ifFalse: [ GsFile gciLog: 'write of length bytes failed, result was ', result asString
                  onClient: logToGciClient .
              failBlock value: sslServer value: logToGciClient.
  		].

  GsFile gciLog: 'Sending ', bigStr size asString, ' bytes to client'
         onClient: logToGciClient.
  result := sslServer write: bigStr .
  result == bigStr size
    ifTrue:[ GsFile gciLog: ('Finished sending ', bigStr size asString, ' bytes to peer')
                     onClient: logToGciClient ]
    ifFalse:[ GsFile gciLog: ('write failed, result was ', result asString)
                     onClient: logToGciClient .
              failBlock value: sslServer value: logToGciClient.
              ].
  "close method will close the SSL connection and underlying socket."
] on: SocketError do:[:ex|
        sslServer ifNotNil:[ sslServer close ].
        listener ifNotNil:[ listener close ].
        ex pass
].

sslServer close.
listener close .
^ true

]

{ #category : 'Examples' }
GsSecureSocket class >> serverExample2 [

^ self serverExample2: true usingPort: 57785 on: 'localhost'

]

{ #category : 'Examples' }
GsSecureSocket class >> serverExample2: logToGciClient usingPort: portNum on: host [

"This version uses GsSecureSocket instance methods for setup rather
 than class methods."

| listener sslServer pw |

[
  | failBlock bigStr result data dataSizeString |

  failBlock := [:sock :shouldLog | | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do: [:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Create a new SSL server socket."
  listener := self newServer.

  "Make it listen for the client."
  (listener makeServerAtPort: portNum) ifNil: [ | err |
  	err := listener lastErrorString.
  	listener close.
  	SocketError signal: 'makeServerAtPort failed, ', err asString.
   ].

  listener port == portNum ifFalse:[ SocketError signal: 'bad port number ', listener port asString ] .

  GsFile gciLog: 'Waiting for GsSecureSocket clientExample to connect...'
  	onClient: logToGciClient .
  [listener readWillNotBlockWithin: 5000] whileFalse: [
  	GsFile gciLog: 'waiting...' onClient: logToGciClient
    	].

  GsFile gciLog: 'Client connected.' onClient: logToGciClient .

  "Do the normal accept to establish a standard socket connection."
  "sslServer is our connection to the client."
  sslServer := listener accept .

  GsFile gciLog: 'Normal accept completed OK.' onClient: logToGciClient .

  "uncomment next to put the socket in blocking mode."
  "sslServer makeBlocking."

  "Don't request a certificate from the client.  This is typical."
  sslServer disableCertificateVerification.

  "sslServer has only a normal connection to the peer.  If will have a secure
   connection if secureAccept succeeds."

  "Example of using an encrypted private key that requires a passphrase."
   pw := self getPasswordFromFile: '$GEMSTONE/examples/openssl/private/server_1_server_passwd.txt' .
   sslServer useCertificateFile: '$GEMSTONE/examples/openssl/certs/server_1_servercert.pem'
    withPrivateKeyFile: '$GEMSTONE/examples/openssl/private/server_1_serverkey.pem'
    privateKeyPassphrase: pw.

  "Use all ciphers except NULL ciphers and anonymous Diffie-Hellman
   and sort by strength."
  sslServer setCipherListFromString: 'ALL:!ADH:@STRENGTH' .

  ([sslServer secureAccept] on: SocketError do:[:ex| | certError |
    GsFile gciLog: 'secureAccept failed.'  onClient: logToGciClient.
    certError := sslServer class fetchLastCertificateVerificationErrorForServer .
    certError ifNil: [
        GsFile gciLog: 'nil result from fetchLastCertificateVerificationErrorForServer'
           onClient: logToGciClient
    ] ifNotNil:[
        GsFile gciLog: ('Cert error: =>' , certError)
           onClient: logToGciClient.].
	ex pass
    ])
      ifTrue:[ GsFile gciLog: 'Secure connection established.'
                      onClient: logToGciClient ].

  "If we get here, we have a secure connection to the client."

  GsFile gciLog: ('Current cipher in use is: ', sslServer fetchCipherDescription)
         onClient: logToGciClient.

  "Build data to send"
  data := Array new.
  4096 timesRepeat: [ data add: (String withAll: '0123456789ABCDEF') ].
  bigStr := String new .
  PassiveObject passivate: data toStream: (WriteStream on: bigStr).
  dataSizeString := bigStr size asHexStringWithLength: 8.
  dataSizeString size > 8 ifTrue:
      [UserDefinedError signal: 'Protocol Error, attempt to send more than 4294967295 bytes'].

  GsFile gciLog: 'Sending data length to client: ', bigStr size asString,
      ' bytes, encoded as ', dataSizeString onClient: logToGciClient.

  result := sslServer write: dataSizeString.
  result == dataSizeString size
    ifTrue: [ GsFile gciLog: 'Finished sending length to peer'
  		onClient: logToGciClient ]
    ifFalse: [ GsFile gciLog: 'write of length bytes failed, result was ', result asString
                  onClient: logToGciClient .
              failBlock value: sslServer value: logToGciClient.
  		].

  GsFile gciLog: 'Sending ', bigStr size asString, ' bytes to client'
         onClient: logToGciClient.
  result := sslServer write: bigStr .
  result == bigStr size
    ifTrue:[ GsFile gciLog: ('Finished sending ', bigStr size asString, ' bytes to peer')
                     onClient: logToGciClient ]
    ifFalse:[ GsFile gciLog: ('write failed, result was ', result asString)
                     onClient: logToGciClient .
              failBlock value: sslServer value: logToGciClient.
              ].
] on: SocketError do:[:ex|
        sslServer ifNotNil:[ sslServer close ].
        listener ifNotNil:[ listener close ].
        ex pass
].

"close method will close the SSL connection and underlying socket."
sslServer close.
listener close .
^ true

]

{ #category : 'Examples' }
GsSecureSocket class >> setCaCertLocation [

"Assume we are running on Linux or Darwin. Try to find the trusted CA cert file
 on this host. If we find it, use it. If we do not, disable cert
 verification so the examples work correctly.

 Ubuntu:         /etc/ssl/certs/ca-certificates.crt
 Centos|Red Hat: /etc/pki/ca-trust/extracted/pem/tls-ca-bundle.pem
 Darwin:         /etc/ssl/cert.pem
 
 Returns the CA cert file used or nil if no file was found and 
 certificate verification has been disabled.
"
 
 | files certFile |
 files := { '/etc/ssl/certs/ca-certificates.crt' .
            '/etc/pki/ca-trust/extracted/pem/tls-ca-bundle.pem' .
	    '/etc/ssl/cert.pem' } .
	    
  certFile := files detect:[:e| GsFile existsOnServer: e] ifNone:[ nil ].
  certFile ifNil:[ GsSecureSocket disableCertificateVerificationOnClient ]
        ifNotNil:[ GsSecureSocket useCACertificateFileForClients: certFile ].
  ^ certFile	
	   
]

{ #category : 'Peer Authentication' }
GsSecureSocket class >> setCertificateVerificationOptionsForServer: anArray [

"Sets the certificate verification options for server sockets using an Array of
 Symbols.  The supported server options are:

 #SSL_VERIFY_FAIL_IF_NO_PEER_CERT
    if the client did not return a certificate, the TLS/SSL handshake is
    immediately terminated with a 'handshake failure' alert.

 #SSL_VERIFY_CLIENT_ONCE
    only request a client certificate on the initial TLS/SSL handshake.
    Do not ask for a client certificate again in case of a renegotiation.

 Refer to the OpenSSL documentation for more information about these options.

 If anArray is empty then any previously set options are cleared.

 Has no effect on instances created before the method was called.

 Raises an error if the array contains any elements besides the above symbols.

 Certificate verification for server sockets must be enabled before executing this
 method, otherwise an error is raised.  Use the #enableCertificateVerificationOnServer
 method to enable certification verification.

 Returns true of success or raises an exception on error."

^ (self _oneArgSslPrim: 6 with: anArray)
    ifTrue:[ true ]
    ifFalse:[ self signalError ]

]

{ #category : 'Ciphers' }
GsSecureSocket class >> setClientCipherListFromString: aString [

"Specify the cipher list to be used by all client connections.  aString must be an
 instance of String in the format described in the man page for ciphers(1). See
   http://www.openssl.org/docs/apps/ciphers.html

 The ciphers loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Returns true if one or more of the specified ciphers are usable or raises
 an exception if an error occurs."

^ (self _oneArgSslPrim: 3 with: aString)
     ifTrue:[ true ]
     ifFalse:[ self signalError ]

]

{ #category : 'Ciphers' }
GsSecureSocket class >> setServerCipherListFromString: aString [

"Specify the cipher list to be used by all server connections.  aString must be an
 instance of String in the format described in the man page for ciphers(1).  See
   http://www.openssl.org/docs/apps/ciphers.html

 The ciphers loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Returns true if one or more of the specified ciphers are usable or raises an error
 if an error occurs."

^ (self _oneArgSslPrim: 4 with: aString)
     ifTrue:[ true ]
     ifFalse:[ self signalError ]

]

{ #category : 'Error Reporting' }
GsSecureSocket class >> signalError [
  ^ SecureSocketError signal: (self lastErrorString ifNil:[ 'no details' ]).

]

{ #category : 'Error Reporting' }
GsSecureSocket class >> signalError: aString [
  ^ SecureSocketError signal:
      (self lastErrorString ifNotNil:[:s | aString , ', ' , s]  ifNil:[ aString ]).

]

{ #category : 'Querying' }
GsSecureSocket class >> sslLibraryVersionString [

"Answer an instance of String which describes the version of OpenSSL library loaded by this session."

^ self _zeroArgSslPrim: 19

]

{ #category : 'Examples' }
GsSecureSocket class >> startTlsClientExample: logToGciClient usingPort: portNum address: serverAddress [

"logToGciClient is a Boolean, true if GsFile-based logging is to be done to
 the RPC Gci client, false if to the server process's log file. Use false if
 executing via topaz 'nbrun'  .

 This client will connect to a server created with serverStartTlsExample.
 Then it will send 4 strings to the server and string 'OK' back to indicate
 the data was received.  Then the client will send the string 'STARTTLS' to
 tell the server to switch to encrypted communication. Once it receives back
 the 'OK' string, it will create a GsSecureSocket from the GsSocket and call
 secureConnect to establish a TLS connection. It will then print the cipher
 being used and close the connection.

 All strings sent and received are expected to be NULL-terminated.

 The server should already be listening for connections when this method is
 invoked. Start the server before starting the client."

| sslClient |

[
  | socket dataString chunk sleepCount outString failBlock |

  socket := GsSocket new.
  (socket connectTo: portNum on: serverAddress) ifFalse:[
    socket close .
    Error signal: 'Unable to connect to port ' , portNum asString ,
      '. Wait 30 seconds or so, then ensure startTlsServerExample is started first.'
    ].
  dataString := String new .
  outString := String new.

  "Exchange some unencrypted data messages"
  1 to: 5 do:[:n| | numWritten|
    sleepCount := 0.
    [ socket writeWillNotBlockWithin: 5000] whileFalse: [
      GsFile gciLog: 'Waiting to write to server...' onClient: logToGciClient .
      sleepCount := sleepCount + 1.
      sleepCount > 200 ifTrue:[
        GsFile gciLog: 'Not ready to write after 20 sec' onClient: logToGciClient.
        Error signal: 'Not ready to read after 20 sec' .
      ].
      System _sleepMs: 100 . "sleep 100 ms then retry"
    ].

    outString size: 0.
    n == 5
      ifTrue:[ outString addAll: 'STARTTLS']
      ifFalse:[ outString addAll: DateTime now asString] .
    outString add: (Character withValue: 0).

    numWritten := socket write: outString.
    GsFile gciLog: 'wrote ', numWritten asString, ' bytes'  onClient: logToGciClient .
    numWritten == outString size ifFalse:[ Error signal: 'error writing to socket'].
    [ socket readWillNotBlockWithin: 10000] whileFalse: [
        GsFile gciLog: 'Waiting for server to write...' onClient: logToGciClient .
    ].

    GsFile gciLog: 'Got something from the server to read.' onClient: logToGciClient .
    dataString size: 0.
    [
      chunk := socket readString: 4000.
      chunk ifNil:[ Error signal: 'Error in reading from socket' ].
      dataString addAll: chunk .
      (chunk at: chunk size) == (Character withValue: 0)  "read until null terminator"
    ] untilTrue .
    n == 5 ifFalse:[ System _sleepMs: 250 ].
  ].

  failBlock := [:sock :shouldLog| | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do:[:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Specify the file which contains a list of trusted CA certificates.
   These will be used to validate the certificate presented by the peer
   during the SSL handshake.  The file must be in PEM format.
   Currently, there is no easy way to specify the CA list from a String.
   This must be done before creating the instance of GsSecureSocket! "
  GsSecureSocket useCACertificateFileForClients:
        '$GEMSTONE/examples/openssl/certs/cacert.pem'.


  "Transfer connection from socket to a new SSL socket."
  sslClient:= GsSecureSocket newClientFromGsSocket: socket .
  sslClient ifNil:[ Error signal: 'Error: newClientFromGsSocket: method failed'].
  "close the old one. This will NOT close the file descriptor."
  socket close.

  "Request a cert from the server"
  sslClient enableCertificateVerification .

  "Set the cipher list for this SSL socket only.
   Use all ciphers except NULL ciphers and anonymous Diffie-Hellman
   and sort by strength."
  (sslClient setCipherListFromString: 'ALL:!ADH:@STRENGTH')
    ifFalse:[ GsFile gciLog: 'setCipherListFromString: failed'
                    onClient: logToGciClient
    ].

  "Attempt to establish a secure connection to the SSL server..."
  ([sslClient secureConnect]
    on: SocketError
    do:[:ex| |certError|
      GsFile gciLog: 'secureConnect failed' onClient: logToGciClient.
      certError := GsSecureSocket fetchLastCertificateVerificationErrorForClient .
      certError ifNil: [
         GsFile gciLog: 'nil result from fetchLastCertificateVerificationErrorForClient'
                onClient: logToGciClient
          ] ifNotNil: [
          GsFile gciLog: ('Cert error: =>' , certError) onClient: logToGciClient].
      ex pass.
    ])
    ifTrue:[ GsFile gciLog: 'Secure connection established'
                    onClient: logToGciClient].

  "We have a secure connection to the server if we get here."
  GsFile gciLog: ('Current cipher in use is: ', sslClient fetchCipherDescription)
         onClient: logToGciClient.
] on: SocketError do:[:ex| sslClient close. ex pass ].
sslClient close.
^ true.

]

{ #category : 'Examples' }
GsSecureSocket class >> startTlsServerExample: logToGciClient usingPort: portNum address: listeningAddress [

"logToGciClient is a Boolean, true if GsFile-based logging is to be done to
 the RPC Gci client, false if to the server process's log file.  Use false if
 executing via topaz 'nbrun'.

 Creates a GsSocket, binds it to port portNum, and waits for a connection.
 When a connection is established, waits for unencrypted data from the client.
 Sends OK back to the client to indicate unencrypted data was received.
 If the unencrypted data contains the string STARTTLS, then a GsSecureSocket
 is created from the GsSocket and secureAccept: is called to negotiate an
 encrypted connection. Once the encrypted connection is established, the
 cipher in use is printed and the connection is closed.

 All strings sent and received are expected to be NULL-terminated.

 You will need two GemStone sessions running from two independent
 interface processes to run both this and the startTlsClientExample.

 Warning: This method will cause your current session to hang until a
 connection is established."

| server client sslServer |
[
  | numWritten errStr sleepTime starttls resultString
    chunk failBlock dataString pw |

  server := GsSocket new.
  (server makeServer: 5 atPort: portNum atAddress: listeningAddress) ifNil: [
    errStr := server lastErrorString.
    server close.
    server := nil.
    Error signal: errStr.
    ].

  server port == portNum ifFalse:[ Error signal: 'bad port number' ] .
  GsFile gciLog: 'Waiting for GsSecureSocket startTlsClientExample to connect...'
  	onClient: logToGciClient .
  [server readWillNotBlockWithin: 10000] whileFalse: [
    GsFile gciLog: 'waiting...' onClient: logToGciClient
  ].

  GsFile gciLog: 'Client connected, starting accept.' onClient: logToGciClient .
  client := server accept .
  GsFile gciLog: 'accept finished. client = ', client asString  onClient: logToGciClient .
  client ifNil: [
    errStr := server lastErrorString.
    server close.
    server := nil .
    GsFile gciLog: 'error from accept, ', errStr onClient: logToGciClient .
    Error signal: errStr.
  ].
  server close.
  server := nil.
  client linger: true length: 10.  "wait after closing until data is processed"
  dataString := String new.
  resultString := String withAll: 'OK'.
  resultString add: (Character withValue: 0).
  starttls := false.

  "Keep receiving normal data and sending OK back until we see STARTTLS"
  [starttls] whileFalse:[
    [ client readWillNotBlockWithin: 10000] whileFalse: [
      GsFile gciLog: 'Waiting for client to write...' onClient: logToGciClient .
    ].
    dataString size: 0.
    GsFile gciLog: 'Got something from the client to read.' onClient: logToGciClient .
    [
      chunk := client readString: 4000 "max bytes".
      chunk ifNil:[ Error signal: 'Error in reading from socket' ].
      dataString addAll: chunk .
      (chunk at: chunk size) == (Character withValue: 0)  "until null terminator"
    ] untilTrue .

    sleepTime := 0 .
    [ client writeWillNotBlock ] whileFalse:[
      sleepTime == 0 ifTrue:[
         GsFile gciLog: 'waiting because write to client would block' onClient: logToGciClient.
      ].
      sleepTime > 5000 ifTrue:[
        GsFile gciLog: 'ERROR, client would block' onClient: logToGciClient .
        Error signal: 'socket write will block'
      ].
      System _sleepMs: 20 .
      sleepTime := sleepTime + 20 .
    ].
    "Got all the data, send back OK"
    numWritten := client write: resultString .
    GsFile gciLog: 'wrote ', numWritten asString, ' bytes'  onClient: logToGciClient .
    numWritten == resultString size ifFalse:[ Error signal: 'error writing to socket'].
    starttls := (dataString findString: 'STARTTLS' startingAt: 1) ~~ 0 .
  ].

  "Client has sent STARTTLS. Create new SSL server socket using existing client
   connection.  Socket file descriptor is transferred to sslServer."
  sslServer := GsSecureSocket newServerFromGsSocket: client .
  sslServer ifNil:[ Error signal: 'Failed to create GsSecureSocket from GsSocket'].
  client close. "will not close the underlyding file descriptor"
  client := nil .
  "do not request a certificate from the client"
  sslServer disableCertificateVerification.

  failBlock := [:sock :shouldLog| | err arr |
    err := sock fetchLastIoErrorString .
    err ifNotNil: [GsFile gciLog: err onClient: shouldLog].
    arr := sock class fetchErrorStringArray .
    1 to: arr size do:[:n| GsFile gciLog: (arr at: n) onClient: shouldLog ].
    sock close.
    SocketError signal: err asString.
  ].

  "Example of using an encrypted private key that requires a passphrase."
   pw := self getPasswordFromFile: '$GEMSTONE/examples/openssl/private/server_1_server_passwd.txt' .
   sslServer useCertificateFile: '$GEMSTONE/examples/openssl/certs/server_1_servercert.pem'
             withPrivateKeyFile: '$GEMSTONE/examples/openssl/private/server_1_serverkey.pem'
             privateKeyPassphrase: pw .

  "Use all ciphers except NULL ciphers and anonymous Diffie-Hellman and sort by strength."
  sslServer setCipherListFromString: 'ALL:!ADH:@STRENGTH' .

  GsFile gciLog: 'Waiting for client to secureConnect: ...' onClient: logToGciClient .

  ([sslServer secureAccept] on: SocketError do:[:ex| | certError |
    GsFile gciLog: 'secureAccept failed.'  onClient: logToGciClient.
    certError := sslServer class fetchLastCertificateVerificationErrorForServer .
    certError ifNil: [
        GsFile gciLog: 'nil result from fetchLastCertificateVerificationErrorForServer'
           onClient: logToGciClient
    ] ifNotNil:[
        GsFile gciLog: ('Cert error: =>' , certError) onClient: logToGciClient.].
  	ex pass
      ])
      ifTrue:[ GsFile gciLog: 'Secure connection established.'
                      onClient: logToGciClient ].

  "If we get here, we have a secure connection to the client."

  GsFile gciLog: ('Current cipher in use is: ', sslServer fetchCipherDescription)
         onClient: logToGciClient.
] on: SocketError do:[:ex|
        sslServer ifNotNil:[ sslServer close ].
        server ifNotNil:[ server close ].
	client ifNotNil:[ client close ].
        ex pass
].

sslServer close .
^ true

]

{ #category : 'TLS Protocol Version' }
GsSecureSocket class >> tlsClientMaxVersion [

"Gets the maximum TLS protocol version for client sockets created
 by the receiver.

 Possible results are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 A return value of #TLS_VERSION_DEFAULT means any supported 
 TLS protocol may be used, which is the default behavior."

^ self _zeroArgSslPrim: 33
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket class >> tlsClientMaxVersion: aStringOrSymbol [

"Sets the maximum TLS protocol version for client sockets created by the receiver.
 aStringOrSymbol must be an instance of String or Symbol.

 Valid options are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 Passing an argument of #TLS_VERSION_DEFAULT means use any supported 
 TLS protocol, which restores the default behavior.

 Returns true on success or raises an exception if the argument is invalid.
 Raises an exception if the argument would set the maximum TLS protocol version
 to be less than a previously set minimum TLS protocol version."

^ self _oneArgSslPrim: 20 with: aStringOrSymbol
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket class >> tlsClientMinVersion [

"Gets the minimum TLS protocol version for client sockets created
 by the receiver.

 Possible results are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 A return value of #TLS_VERSION_DEFAULT means any supported 
 TLS protocol may be used, which is the default behavior."

^ self _zeroArgSslPrim: 32
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket class >> tlsClientMinVersion: aStringOrSymbol [

"Sets the minimum TLS protocol version for client sockets created by the receiver.
 aStringOrSymbol must be an instance of String or Symbol.

 Valid options are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 Passing an argument of #TLS_VERSION_DEFAULT means use any supported 
 TLS protocol, which restores the default behavior.

 Returns true on success or raises an exception if the argument is invalid.
 Raises an exception if the argument would set the minimum TLS protocol version
 to be greater than a previously set maximum TLS protocol version."

^ self _oneArgSslPrim: 19 with: aStringOrSymbol
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket class >> tlsServerMaxVersion [

"Gets the maximum TLS protocol version for server sockets created
 by the receiver.

 Possible results are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 A return value of #TLS_VERSION_DEFAULT means any supported 
 TLS protocol may be used, which is the default behavior."

^ self _zeroArgSslPrim: 31
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket class >> tlsServerMaxVersion: aStringOrSymbol [

"Sets the maximum TLS protocol version for server sockets created by the receiver.
 aStringOrSymbol must be an instance of String or Symbol.

 Valid options are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 Passing an argument of #TLS_VERSION_DEFAULT means use any supported 
 TLS protocol, which restores the default behavior.

 Returns true on success or raises an exception if the argument is invalid.
 Raises an exception if the argument would set the maximum TLS protocol version
 to be less than a previously set minimum TLS protocol version."

^ self _oneArgSslPrim: 18 with: aStringOrSymbol
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket class >> tlsServerMinVersion [

"Gets the minimum TLS protocol version for server sockets created
 by the receiver.

 Possible results are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 A return value of #TLS_VERSION_DEFAULT means any supported 
 TLS protocol may be used, which is the default behavior."

^ self _zeroArgSslPrim: 30
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket class >> tlsServerMinVersion: aStringOrSymbol [

"Sets the minimum TLS protocol version for server sockets created by the receiver.
 aStringOrSymbol must be an instance of String or Symbol.

 Valid options are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 Passing an argument of #TLS_VERSION_DEFAULT means use any supported 
 TLS protocol, which restores the default behavior.

 Returns true on success or raises an exception if the argument is invalid.
 Raises an exception if the argument would set the minimum TLS protocol version
 to be greater than a previously set maximum TLS protocol version."

^ self _oneArgSslPrim: 17 with: aStringOrSymbol
]

{ #category : 'Certificate Authorities' }
GsSecureSocket class >> useCACertificateDirectoryForClients: aDirectoryString [

"Specifies a directory where trusted certificate authority (CA) certificates in
 PEM format are located. The certificates in this directory will be used to
 authenticate certificates provided by servers during the SSL handshake. The
 directory may contain more than one certificate.

 Certificates are loaded into the internal SSL state which is valid for the
 current session only; the SSL state is not committed to the repository.

 Has no effect on instances created before the method was called or server
 instances of GsSecureSocket.

 If successful, this method also enables certificate verification for client
 sockets as if the #enableCertificateVerificationOnClient method had been called
 immediately following this method.

 Returns true on success or false if an error occurs.  Raises an
 exception if the directory does not exist."

^ (self _oneArgSslPrim: 8 with: aDirectoryString)
    ifTrue:[ true ]
    ifFalse:[ self signalError ]

]

{ #category : 'Certificate Authorities' }
GsSecureSocket class >> useCACertificateDirectoryForServers: aDirectoryString [

"Specifies a directory where trusted certificate authority (CA) certificates in
 PEM format are located. The certificates in this directory will be used to
 authenticate certificates provided by client during the SSL handshake. The
 directory may contain more than one certificate.

 Certificates are loaded into the internal SSL state which is valid for the
 current session only; the SSL state is not committed to the repository.

 Has no effect on instances created before the method was called or client
 instances of GsSecureSocket.

 If successful, this method also enables certificate verification for server
 sockets as if the #enableCertificateVerificationOnServer method had been called
 immediately following this method.

 Returns true on success or false if an error occurs.  Raises an
 exception if the directory does not exist."

^ (self _oneArgSslPrim: 9 with: aDirectoryString)
    ifTrue:[ true ]
    ifFalse:[ self signalError ]

]

{ #category : 'Certificate Authorities' }
GsSecureSocket class >> useCACertificateFileForClients: aFileNameString [

"Attempt to load given CA Certificate file which must be in PEM format.
 The file may contain more than one certificate.  If the file is loaded
 successfully, it will be used by SSL clients to validate all certificates
 presented by SSL servers during the SSL handshake.

 The certificates loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Has no effect on instances created before the method was called.

 Also enables certificate verification for client sockets as if
 the #enableCertificateVerificationOnClient method had been called
 immediately following this method.

 Returns true on success raises an exception if an error occurs or the
 file does not exist."

 ^ (self _oneArgSslPrim: 1 with: aFileNameString)
     ifTrue:[ true ]
     ifFalse:[ self signalError: 'useCACertificateFileForClients: failed' ]

]

{ #category : 'Certificate Authorities' }
GsSecureSocket class >> useCACertificateFileForServers: aFileNameString [

"Attempt to load given CA Certificate file which must be in PEM format.
 The file may contain more than one certificate.  The certificates
 are considered to be trusted and will be used to validate certificates
 presented by SSL clients.  Note that authentication of clients is disabled
 by default and must be enabled in order for this method to have any effect.

 The certificates loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Has no effect on instances created before the method was called.

 Also enables certificate verification for server sockets as if
 the #enableCertificateVerificationOnServer method had been called
 immediately following this method.

 Returns true on success or raises an exception if an error occurs or
 the file does not exist."

 ^ (self _oneArgSslPrim: 2 with: aFileNameString)
     ifTrue:[ true ]
     ifFalse:[ self signalError: 'useCACertificateFileForServers: failed' ]

]

{ #category : 'Certificates and Keys' }
GsSecureSocket class >> useClientCertificate: aString withPrivateKey: anotherString privateKeyPassphrase: passPhrase [

"Specifies the certificate string and private key string to be used by all
 client connections.  The private key must match the certificate.

 The certificates loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Both strings must be in PEM format.  If the private key requires a passphrase,
 it is specified in the passPhrase argument as a String.  If no passphrase is
 required, then the passPhrase argument is expected to be nil.

 Both strings must exactly match the contents of the corresponding certificate files
 (including white-space characters) or the strings will not be accepted.

 Returns true on success or raises an exception if an error occurs."

^ (self _threeArgSslPrim: 4 with: aString with: anotherString with: passPhrase)
     ifTrue:[ true ]
     ifFalse:[ self signalError ]

]

{ #category : 'Certificates and Keys' }
GsSecureSocket class >> useClientCertificateFile: certFile withPrivateKeyFile: keyFile privateKeyPassphrase: pass [

"Specifies the certificate file and private key file to be used by all
 client connections.  The private key must match the certificate.

 The certificates loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Both files must be in PEM format.

 The certificate file may contain a certificate chain or a single certificate.

 If the private key requires a passphrase, it is to be specified as String
 passed in using the pass argument.  If the private key does not require a
 passphrase, then the pass argument is expected to be nil.

 Both file arguments must be instances of String.

 Returns true on success or raises an exception if an error occurs or
 the file does not exist."

^ (self _threeArgSslPrim: 7 with: certFile with: keyFile with: pass)
     ifTrue:[ true ]
     ifFalse:[ self signalError ]

]

{ #category : 'Certificates and Keys' }
GsSecureSocket class >> useServerCertificate: aString withPrivateKey: anotherString privateKeyPassphrase: passPhrase [

"Specifies the certificate string and private key string to be used by all
 server connections.  The private key must match the certificate.

 The certificates loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Both strings must be in PEM format.  If the private key requires a passphrase,
 it is specified in the passPhrase argument as a String.  If no passphrase is
 required, then the passPhrase argument is expected to be nil.

 Both strings must exactly match the contents of the corresponding certificate files
 (including white-space characters) or the strings will not be accepted.

 Returns true on success or raises an exception if an error occurs."

^ (self _threeArgSslPrim: 3 with: aString with: anotherString with: passPhrase)
     ifTrue:[ true ]
     ifFalse:[ self signalError ]

]

{ #category : 'Certificates and Keys' }
GsSecureSocket class >> useServerCertificateFile: certFile withPrivateKeyFile: keyFile privateKeyPassphrase: pass [

"Specifies the certificate file and private key file to be used by all
 server connections.  The private key must match the certificate.

 Both files must be in PEM format.

 The certificate file may contain a certificate chain or a single certificate.

 If the private key requires a passphrase, it is to be specified as String
 passed in using the pass argument.  If the private key does not require a
 passphrase, then the pass argument is expected to be nil.

 Both file arguments must be instances of String.

 The certificates loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Returns true on success or raises an exception if an error occurs or
 the file does not exist."

^ (self _threeArgSslPrim: 6 with: certFile with: keyFile with: pass)
     ifTrue:[ true ]
     ifFalse:[ self signalError ]

]

{ #category : 'Private' }
GsSecureSocket >> _installSsl: aCpointer [

"Used in hostagent.  aCpointer must encapsulate a SSL* .
 After this method executes,
  the SSL* will be auto-freed on GC of the receiver."

^ self _oneArgSslPrim: 12 with: aCpointer

]

{ #category : 'Private' }
GsSecureSocket >> _maxReadWaits [
  "The wait loop in GsSocket>>read:into:startingAt can
   get repeated read-ready of the underlying socket
   followed by SSL read apparently peeking
   on the socket and returning EWOULDBLOCK because there is not
   yet enough data to do a decryption cycle."
  ^ SmallInteger maximumValue

]

{ #category : 'Private' }
GsSecureSocket >> _noFreeSslOnGc [
"Used in hostagent code after sslState and fileDescriptor have
 become owned by a pgsvr thread .
 Cancels the effect of _installSsl:   .
 All subsequent ssl read/write operations must be done from the C code
 that now owns the socket.
 Sets C state in the receiver so that GC of the instance
 will neither free the SSL state, nor close the file descriptor.
"

^ self _zeroArgSslPrim: 26

]

{ #category : 'Private' }
GsSecureSocket >> _nonSslWrite: aByteObject startingAt: anOffset ofSize: numBytes [

"GsSocket >> _write:startingAt:ofSize  semantics.
 Returns nil if an error occurred,
   false if non-blocking receiver would block,
   true if EINTR occurred,
   a SmallInteger number of bytes written.
 Generates an error if aByteObject is not a String or ByteArray.
 Clears bits 16r4 from self.readyEvents if no error generated."

<primitive: 882>
self _primitiveFailed: #_write:startingAt:ofSize:
     args: { aByteObject . anOffset . numBytes }

]

{ #category : 'Private' }
GsSecureSocket >> _oneArgSslPrim: opcode with: arg [

"opcode  function
   1        class method: useCACertificateFileForClients:
   2        class method: useCACertificateFileForServers:
   3        class method: setClientCipherListFromString:
   4        class method: setServerCipherListFromString:
   5     instance method: setCipherListFromString:
   6        class method: setCertificateVerificationOptionsForServer:
   7     instance method: setCertificateVerificationOptions:
   8        class method: useCACertificateDirectoryForClients:
   9        class method: useCACertificateDirectoryForServers:
  10     instance method: initializeAsServerFromGsSocket:
  11     instance method: initializeAsClientFromGsSocket:
  12     instance method: _installSsl:
  13     instance method: setServerNameIndication:
  14     instance method: setPreSharedKey:
  15     instance method: tlsMinVersion:
  16     instance method: tlsMaxVersion:
  17:       class method: tlsServerMinVersion:
  18:       class method: tlsServerMaxVersion:
  19:       class method: tlsClientMinVersion:
  20:       class method: tlsClientMaxVersion:
  21     instance method: setExpectedHost:
  22     instance method: addExpectedHost:
  23     instance method: setExpectedHostFlags:
"

<primitive: 910>
^ self _primitiveFailed: #_oneArgSslPrim:with: args: { opcode . arg }

]

{ #category : 'Private' }
GsSecureSocket >> _peek [
  "Returns a SmallInteger in the range 0..255 if there is a byte available
   from the underlying SSL buffer for the socket, otherwise returns nil."

  ^ self _zeroArgSslPrim: 20

]

{ #category : 'Private' }
GsSecureSocket >> _primBlockingReadInto: aByteObject startingAt: anOffset maxBytes: numBytes [

"Returns a SmallInteger indicating how many bytes were read or
 nil if an error occurs.
 Returns 0 if EOF occurs on the socket.
 numBytes must be a SmallInteger > 0 .

 Automatically handles conditions which require waiting for a read-ready
 or write-ready condition."

[ true ] whileTrue:[
  | res |
  "Note: args must be the same each time if -1 or -2 is returned"
  res := self _threeArgSslPrim: 2 with: aByteObject with: anOffset with: numBytes .
  res == -1 ifTrue:[self _waitForReadReady ] "Normal case"
            ifFalse:[ res == -2
                      ifTrue:[ self _waitForWriteReady ] "can happen due to SSL renegotiation"
                      ifFalse:[ ^ res ]. "done or error"
            ].
].

]

{ #category : 'Private' }
GsSecureSocket >> _primNonBlockingReadInto: aByteObject startingAt: anOffset maxBytes: numBytes [

"Returns a SmallInteger indicating how many bytes were read or
 nil if an error occurs.
 Returns 0 if EOF occurs on the socket.
 numBytes must be a SmallInteger > 0 .

 Automatically handles conditions which require waiting for a read-ready
 or write-ready condition."

[ true ] whileTrue:[
  | res |
  "Note: args must be the same each time if -1 or -2 is returned"
  res := self _threeArgSslPrim: 2 with: aByteObject with: anOffset with: numBytes .
  res == -1 ifTrue:[^ false "would block" ]
            ifFalse:[ res == -2
                      ifTrue:[ self _waitForWriteReady ] "can happen due to SSL renegotiation"
                      ifFalse:[ ^ res ]. "done or error"
            ].
].

]

{ #category : 'Private' }
GsSecureSocket >> _rawNbWrite: numBytes from: aByteObject startingAt: anOffset [
 "Does a non-SSL write to the underlying socket .
  Returns false if non-blocking receiver would block,
  or a SmallInteger number of bytes written.
  Retrys if EINTR occurs. "
 | res |
 [ res := self _nonSslWrite: aByteObject startingAt: anOffset ofSize: numBytes  .
   res == true "EINTR ocurred"
 ] whileTrue .
 res ifNil:[ self signalError ].
 ^ res

]

{ #category : 'Private' }
GsSecureSocket >> _rawRead: maxBytes into: byteObj startingAt: index [

"Reads up to the given number of bytes into the given byte object (for
 example, a String).  The first byte read will go into the position
 indicated by index.
 Returns the number of bytes read, or nil if an error,
 or 0 if EOF on the receiver.

 maxBytes must be a SmallInteger > 0 and <= 500000000 .

 An error will be raised if index is greater than the size of byteObj.
 Note that index can be equal to the size of byteObj plus 1 in which
 case any bytes read will be appended to byteObj.

 If no data is available for reading, this returns false .
 This method retrys for EINTR .
 The readWillNotBlock or readWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for reading before calling this method.
 Used in hostagent. "

| status |
[
  "primitive will wait until data available if C socket is blocking."
  status := super _readInto: byteObj startingAt: index maxBytes: maxBytes.
  "status==true from _readInto means got EINTR and must retry"
  (status == false) ifTrue: [
    ^ false
  ] .
  status == true
] whileTrue .
^ status

]

{ #category : 'Private' }
GsSecureSocket >> _rawReadWillNotBlockWithin: msToWait [

^ super readWillNotBlockWithin: msToWait

]

{ #category : 'Private' }
GsSecureSocket >> _readInto: aByteObject startingAt: anOffset maxBytes: numBytes [

"Read up to numBytes of data from the receiver into aByteObject starting at anOffset.
 numBytes must be a SmallInteger > 0 .
 Returns a SmallInteger indicating how many bytes were read,
 or returns 0 if EOF occurs on the socket,
 or signals an error."

| res |
res := self isNonBlocking
   ifTrue:[ self _primNonBlockingReadInto: aByteObject startingAt: anOffset maxBytes: numBytes ]
  ifFalse:[ self _primBlockingReadInto: aByteObject startingAt: anOffset maxBytes: numBytes ].
res ifNil:[
  SocketError signal: self fetchLastIoErrorString
].
^ res

]

{ #category : 'Private' }
GsSecureSocket >> _readLine: maxSize maxWaitMs: waitLimitMs [

"Returns nil or a String .
 Waits up to waitLimitMs milliseconds to receive up to
 maxSize bytes.
 Returns when a linefeed is found at end of data or if max size is reached.
 Used in hostagent.
"
| status waitedMs sz aString |
aString := String new .
waitedMs := 0 .
sz := 0 .
[
  "primitive will wait until data available if C socket is blocking."
  status := self _readInto: aString startingAt: sz + 1 maxBytes: maxSize - sz.
  "status==true from _readInto means got EINTR and must retry"
  status == false ifTrue: [
    "socket is non-blocking and would have blocked"
  ] ifFalse:[
    status == 0 ifTrue:[ ^ nil "EOF"].
    sz := sz + status"num bytes read".
    (aString codePointAt: sz) == 10 ifTrue:[ ^ aString ].
    sz >= maxSize ifTrue:[ ^ aString ].
  ].
  waitedMs < waitLimitMs ifTrue:[
    waitedMs := waitedMs + 1 .
    Delay waitForMilliseconds: 1.
  ] ifFalse:[
    ^ nil "waited too long"
  ].
  true
] whileTrue .

]

{ #category : 'Server Operations' }
GsSecureSocket >> _secureAcceptTimeoutMs: timeoutMs [

"Establishes a secure connection using the receiver as the server  The receiver
 must have already established a regular (non-secure) connection with its
 client.  No certificate management is performed by this method.  A certificate
 and private key must be installed before calling this method.

 This method makes the following OpenSSL calls:
   SSL_new (if not already done on the receiver).
   SSL_set_fd (if not already done on the receiver).
   SSL_accept

 Waits up to the specified time for completion.
 The timeoutMs arugment must be a SmallInteger.

 Returns true to indicate the operation was successful,
 false to indicate timeout, or a String describing an error."

| waitedMs |
waitedMs := 0 .
timeoutMs _isSmallInteger ifFalse:[ timeoutMs _validateClass: SmallInteger ].
[
  | res |
  "Note: args must be the same each time if -1 is returned"
  res := self _zeroArgSslPrim: 2 .
  ((res == nil) or:[res == false]) ifTrue:[ ^ 'secureAccept failed'] .
  res == -1 ifTrue:[ Delay waitForMilliseconds: 2 . waitedMs := waitedMs + 2 ]
           ifFalse:[ res == true ifTrue:[ ^ res "done" ] .
                     ^ 'secureAccept failed with ' , res asString
                   ].
  waitedMs < timeoutMs
] whileTrue .
^ false "timeout"

]

{ #category : 'Server Operations' }
GsSecureSocket >> _secureConnectTimeoutMs: timeoutMs [

"Establishes a secure connection using the receiver as the server  The receiver
 must have already established a regular (non-secure) connection with its
 client.  No certificate management is performed by this method.  A certificate
 and private key must be installed before calling this method.

 This method makes the following OpenSSL calls:
   SSL_new (if not already done on the receiver).
   SSL_set_fd (if not already done on the receiver).
   SSL_connect

 Waits up to the specified time for completion.
 The timeoutMs arugment must be a SmallInteger.

 Returns true to indicate the operation was successful,
 false to indicate timeout, or a String describing an error."

| waitedMs |
waitedMs := 0 .
timeoutMs _isSmallInteger ifFalse:[ timeoutMs _validateClass: SmallInteger ].
[
  | res |
  "Note: args must be the same each time if -1 is returned"
  res := self _zeroArgSslPrim: 1 .
  ((res == nil) or:[res == false]) ifTrue:[ ^ 'secureConnect failed'] .
  res == -1 ifTrue:[ Delay waitForMilliseconds: 1 . waitedMs := waitedMs + 1 ]
           ifFalse:[ res == true ifTrue:[ ^ res "done" ] .
                     ^ 'secureConnect failed with ' , res asString
                   ].
  waitedMs < timeoutMs
] whileTrue .
^ false "timeout"

]

{ #category : 'Private' }
GsSecureSocket >> _threeArgSslPrim: opcode with: arg1 with: arg2 with: arg3 [

"opcode  function
   1     instance method: _write:startingAt:ofSize:
   2     instance method: _readInto:startingAt:maxBytes:
   3        class method: useServerCertificate: withPrivateKey: privateKeyPassphrase:
   4        class method: useClientCertificate: withPrivateKey: privateKeyPassphrase:
   5     instance method: useCertificate: withPrivateKey: privateKeyPassphrase:
   6:       class method: useServerCertificateFile:withPrivateKeyFile:privateKeyPassphrase:
   7:       class method: useClientCertificateFile:withPrivateKeyFile:privateKeyPassphrase:
   8:    instance method: useCertificateFile:withPrivateKeyFile::privateKeyPassphrase:
"
<primitive: 909>
^ self _primitiveFailed: #_threeArgSslPrim:with:with:with:
  args: { opcode . arg1 . arg2 . arg3 }

]

{ #category : 'Private' }
GsSecureSocket >> _write: aByteObject startingAt: anOffset ofSize: numBytes [

"Does not conform exactly to GsSocket >> _write:startingAt:ofSize:
 due to semantics of SSL renegotiation and socket buffering.

 Returns nil if a socket error occurred ,
 or a SmallInteger the number of bytes written which will
 normally be equal to numBytes."

[ true ] whileTrue:[
  | res |
  "Note: args must be the same each time if -1 or -2 is returned"
  res := self _threeArgSslPrim: 1 with: aByteObject with: anOffset with: numBytes .
  res == -2 ifTrue:[
    ^ false
  ] ifFalse:[
    res == -1 ifTrue:[ self _waitForReadReady ] "can happen due to SSL renegotiation"
              ifFalse:[ ^ res ].
  ].
].

]

{ #category : 'Private' }
GsSecureSocket >> _zeroArgSslPrim: opcode [

"opcode  function
   1     instance method: secureConnect
   2     instance method: secureAccept
   3     instance method: hasSecureConnection
   4     instance method: secureClose
   5     instance method: fetchLastIoErrorString
   6        class method: fetchErrorStringArray
   7        class method: clearErrorQueue
   8        class method: disableCertificateVerificationOnClient
   9        class method: enableCertificateVerificationOnClient
  10        class method: disableCertificateVerificationOnServer
  11        class method: enableCertificateVerificationOnServer
  12     instance method: initializeAsClient
  13     instance method: initializeAsServer
  14        class method: fetchLastCertificateVerificationErrorForServer
  15        class method: fetchLastCertificateVerificationErrorForClient
  16     instance method: fetchCipherDescription
  17     instance method: disableCertificateVerification
  18     instance method: enableCertificateVerification
  19        class method: sslLibraryVersionString
  20     instance method: _peek
  21     instance method: certificateVerificationEnabled
  22        class method: certificateVerificationEnabledOnClient
  23        class method: certificateVerificationEnabledOnServer
  24     instance method: fetchCertificateVerificationOptions
  25        class method: fetchCertificateVerificationOptionsForServer
  26     instance method:  _noFreeSslOnGc
  27     instance method: getServerNameIndication
  28     instance method: tlsMinVersion
  29     instance method: tlsMaxVersion
  30        class method: tlsServerMinVersion
  31        class method: tlsServerMaxVersion
  32        class method: tlsClientMinVersion
  33        class method: tlsClientMaxVersion
  34     instance method: tlsActualVersion
  35     instance method: peerCertificate
  36     instance method: peerCertificateChain
  37     instance method: peerName
"
<primitive: 908>
^ self _primitiveFailed: #_zeroArgSslPrim: args: { opcode }

]

{ #category : 'Server Operations' }
GsSecureSocket >> accept [

"Causes the receiver to perform a normal socket accept operation (see the
 accept method in GsSocket for more information).  If the accept is
 successful, then it also initializes the newly created socket as an SSL
 server socket.

 Returns the new socket or raises an exception if an error occurs."

| newServer |
newServer := super accept .
newServer ifNil:[ ^ super signalError: 'accept failed' ]. "Normal accept failed"
newServer initializeAsServer . "SSL init as a server socket"
^ newServer

]

{ #category : 'Server Operations' }
GsSecureSocket >> acceptTimeoutMs: timeoutMs [

  | newServer |
  newServer := super acceptTimeoutMs: timeoutMs .
  newServer ifNil:[ ^ super signalError: 'acceptTimeoutMs: failed' ]. "Normal accept failed"
  newServer initializeAsServer . "SSL init as a server socket"
  ^ newServer

]

{ #category : 'Client Operations' }
GsSecureSocket >> addExpectedHost: aString [

"Adds aString to the list of expected host names the receiver will 
 connect to. 

 This method must be executed before the #secureConnect method and only with
 client sockets. Raises an exception if the receiver is not a client
 socket, #secureConnect has already been executed, or if aString is not
 a non-empty instance of String.

 Returns the receiver."

^ self _oneArgSslPrim: 22 with: aString
]

{ #category : 'Querying' }
GsSecureSocket >> certificateVerificationEnabled [

"Answer true if certificate verification is enabled on the receiver.
 Otherwise answer false."

^ self _zeroArgSslPrim: 21

]

{ #category : 'Socket Operations' }
GsSecureSocket >> close [

"Closes the secure connection and releases memory used by the receiver's
 SSL connection if it exists.  Then closes the underlying socket connection.

 Returns self if the socket was closed successfully or nil if an error occurs."

self hasSecureConnection
  ifTrue: [ self secureClose ].
^ super close

]

{ #category : 'Peer Authentication' }
GsSecureSocket >> disableCertificateVerification [
"Directs the receiver to not request a certificate from its peer.
 This is the default mode for server sockets.

 Returns true on success or raises an exception on error."

^ self _zeroArgSslPrim: 17

]

{ #category : 'Peer Authentication' }
GsSecureSocket >> enableCertificateVerification [
"Directs the receiver to request and verify a certificate from its peer.
 This is the default mode for client sockets.

 Returns true on success or raises an exception on error."

^ self _zeroArgSslPrim: 18

]

{ #category : 'Querying' }
GsSecureSocket >> fetchCertificateVerificationOptions [

"Answers an Array of Symbols which represent the certificate verification
 options used by the receiver.  Certificate verification options are only
 supported by server sockets.  Client sockets always return an empty Array.

 The supported server options are:

 #SSL_VERIFY_FAIL_IF_NO_PEER_CERT
    if the client did not return a certificate, the TLS/SSL handshake is
    immediately terminated with a 'handshake failure' alert.

 #SSL_VERIFY_CLIENT_ONCE
    only request a client certificate on the initial TLS/SSL handshake.
    Do not ask for a client certificate again in case of a renegotiation.

 Refer to the OpenSSL documentation for more information about these options."

^ self _zeroArgSslPrim: 24

]

{ #category : 'Querying' }
GsSecureSocket >> fetchCipherDescription [

"If the receiver has a secure connection with its peer, this method answers a string which
 describes the cipher currently in use.  Otherwise returns nil to indicate
 the receiver does not have a secure connection with its peer."

^ self _zeroArgSslPrim: 16

]

{ #category : 'Error Handling' }
GsSecureSocket >> fetchLastIoErrorString [

"Fetches a string describing the last I/O error encountered by a call to any
 of the following SSL functions:

   SSL_connect
   SSL_accept
   SSL_do_handshake
   SSL_read
   SSL_peek
   SSL_write

 Note that a result containing the substring
   SSL_write:uninitialized
 means that SSL_write was attempted before completing all of the
 handshakes of the sslConnect / sslAccept pair .

 The error is also cleared.  Returns nil if no I/O error has occurred
 on the receiver."

^ self _zeroArgSslPrim: 5

]

{ #category : 'Querying' }
GsSecureSocket >> getServerNameIndication [

"Answer a string indicating the server name indication (SNI) name 
 assigned to the receiver, or nil if no SNI name has been assigned."

^ self _zeroArgSslPrim: 27

]

{ #category : 'Querying' }
GsSecureSocket >> hasSecureConnection [

"Answer true if the receiver has established a secure connection
 with its peer.  Otherwise answer false."

^ self _zeroArgSslPrim: 3

]

{ #category : 'Initialization' }
GsSecureSocket >> initializeAsClient [

"Returns the receiver"
^ self _zeroArgSslPrim: 12

]

{ #category : 'Initialization' }
GsSecureSocket >> initializeAsClientFromGsSocket: aGsSocket [

"Returns the receiver on success"
^ self _oneArgSslPrim: 11 with: aGsSocket

]

{ #category : 'Initialization' }
GsSecureSocket >> initializeAsServer [

"Returns the receiver"
^ self _zeroArgSslPrim: 13

]

{ #category : 'Initialization' }
GsSecureSocket >> initializeAsServerFromGsSocket: aGsSocket [

"Returns the receiver on success"
^ self _oneArgSslPrim: 10 with: aGsSocket

]

{ #category : 'Error Handling' }
GsSecureSocket >> lastErrorString [
  "Provided for compatibility with GsSocket.
   See comments in fetchLastIoErrorString"

  ^ self fetchLastIoErrorString ifNil:[ super lastErrorString]

]

{ #category : 'Client Operations' }
GsSecureSocket >> matchedPeerName [

"Returns a String representing the DNS hostname or subject
 CommonName from the peer certificate that matched one of the
 hosts set by the #setExpectedHost: or #addExpectedHost:  
 methods.

 Returns nil if no host matching was performed.
 Raises an exception if the receiver is not a client socket
 or if the secure connection has not been established."


^ self _zeroArgSslPrim: 37
]

{ #category : 'Peer Certificates' }
GsSecureSocket >> peerCertificate [

"Answer an instance of GsX509Certificate representing the peer's certificate.
 Raises an exception if the receiver has not completed the TLS handshake
 with its peer. Returns nil if no certificate was sent by the peer.
 This will always happen if anonymous TLS is used and can happen when
 certificate verification is disabled."

^ self _zeroArgSslPrim: 35
]

{ #category : 'Peer Certificates' }
GsSecureSocket >> peerCertificateChain [

"Answer an instance of GsX509CertificateChain containing the peer's certificate
 chain. Raises an exception if the receiver has not completed the TLS handshake
 with its peer. Returns nil if no certificate was sent by the peer.
 This will always happen if anonymous TLS is used and may happen when
 certificate verification is disabled."

^ self _zeroArgSslPrim: 36
]

{ #category : 'Testing' }
GsSecureSocket >> readWillNotBlock [

"Returns true if the socket is currently ready to receive input without
 blocking.  Returns false if it is not currently ready.  Returns nil if an error
 occurs.

 The receiver must already be connected for this method to work properly.  If it
 is not connected, then the value that this method returns is indeterminate.
 Use the peerName method to determine if the receiver is connected.

 Call this method to prevent subsequent read or accept operations from hanging.
 If it returns true for a connected socket, then the input operation will not
 hang.  However, a return value of true is no guarantee that the operation
 itself will succeed."

 self _peek ifNotNil:[ ^ true ].
 ^ super readWillNotBlock

]

{ #category : 'Testing' }
GsSecureSocket >> readWillNotBlockWithin: msToWait [

"Returns true if the socket is ready to receive input without blocking within
 msToWait milliseconds from the time that this method is called.  Returns false
 if it is not ready after msToWait milliseconds.  Returns nil if an error
 occurs.

 If msToWait is 0, then this method reports the current readiness of the
 receiver.  If msToWait is -1, then this method never returns false, but waits
 until the receiver is ready to receive input without blocking, and then returns
 true.

 The receiver must already be connected for this method to work properly.  If it
 is not connected, then the value that this method returns is indeterminate.
 Use the peerName method to determine if the receiver is connected.

 Call this method to prevent subsequent read or accept operations from hanging.
 If it returns true for a connected socket, then the input operation will not
 hang.  However, a return value of true is no guarantee that the operation
 itself will succeed."

 self _peek ifNotNil:[ ^ true ].
 ^ super readWillNotBlockWithin: msToWait

]

{ #category : 'Server Operations' }
GsSecureSocket >> secureAccept [

"Establishes a secure connection using the receiver as the server  The receiver
 must have already established a regular (non-secure) connection with its
 client.  No certificate management is performed by this method.  A certificate
 and private key must be installed before calling this method.

 This method makes the following OpenSSL calls:

   SSL_new
   SSL_set_fd
   SSL_accept

 Waits up to 30 seconds for completion.

 Returns true to indicate the operation was successful.  Raises an exception
 if an error or timeout occurred."

^ self secureAcceptTimeoutMs: 30000"milliseconds"

]

{ #category : 'Server Operations' }
GsSecureSocket >> secureAcceptTimeoutMs: timeoutMs [

"Establishes a secure connection using the receiver as the server  The receiver
 must have already established a regular (non-secure) connection with its
 client.  No certificate management is performed by this method.  A certificate
 and private key must be installed before calling this method.

 This method makes the following OpenSSL calls:
   SSL_new (if not already done on the receiver).
   SSL_set_fd (if not already done on the receiver).
   SSL_accept
 Waits up to the specified time for completion.
 The timeoutMs arugment must be a SmallInteger.

 Returns true to indicate the operation was successful.  Raises an exception
 if an error or timeout occurred."

 ^ self secureAcceptTimeoutMs: timeoutMs errorOnTimeout: true

]

{ #category : 'Server Operations' }
GsSecureSocket >> secureAcceptTimeoutMs: timeoutMs errorOnTimeout: aBoolean [

"Establishes a secure connection using the receiver as the server  The receiver
 must have already established a regular (non-secure) connection with its
 client.  No certificate management is performed by this method.  A certificate
 and private key must be installed before calling this method.

 This method makes the following OpenSSL calls:
   SSL_new (if not already done on the receiver).
   SSL_set_fd (if not already done on the receiver).
   SSL_accept
 Waits up to the specified time for completion.
 The timeoutMs arugment must be a SmallInteger.

 Returns true to indicate the operation was successful.
 Raises an exception if an error occurred.
 On timeout, if aBoolean == true, raises an exception otherwise returns false."

 | res |
 res := self _secureAcceptTimeoutMs: timeoutMs  .
 res == true ifTrue:[ ^ true "success" ].
 res == false ifTrue:[
   aBoolean ifFalse:[ ^ false ].
   self signalError: 'secure accept timed out'.
 ].
 self signalError: res asString .

]

{ #category : 'Private' }
GsSecureSocket >> secureClose [

"Closes the secure connection and releases memory used by the receiver's
 SSL connection.  Does not close the underlying socket connection.  Customers
 should use the #close method instead of this one unless the underlying
 (insecure) socket connection is to be retained.

 Returns the receiver."

^ self _zeroArgSslPrim: 4

]

{ #category : 'Client Operations' }
GsSecureSocket >> secureConnect [

"Establishes a secure connection using the receiver as the client  The receiver
 must have already established a regular (non-secure) connection with its
 server.  No certificate management is performed by this method.

 This method makes the following OpenSSL calls:

   SSL_new
   SSL_set_fd
   SSL_connect

 Waits up to 30 seconds for completion.

 Returns true to indicate the operation was successful.  Raises an exception if
 an error or timeout occurred.
"

^ self secureConnectTimeoutMs: 30000"milliseconds"

]

{ #category : 'Client Operations' }
GsSecureSocket >> secureConnectTimeoutMs: timeoutMs [

"Establishes a secure connection using the receiver as the client  The receiver
 must have already established a regular (non-secure) connection with its
 server.  No certificate management is performed by this method.

 This method makes the following OpenSSL calls:

   SSL_new
   SSL_set_fd
   SSL_connect

 Waits up to the specified time for completion.
 The timeoutMs arugment must be a SmallInteger.

 Returns true to indicate the operation was successful.  Raises an exception if
 an error or timeout occurred.  "

| waitedMs |
waitedMs := 0 .
timeoutMs _isSmallInteger
   ifFalse:[ timeoutMs _validateClass: SmallInteger ].
[
  | res |
  "Note: args must be the same each time if -1 is returned"
  res := self _zeroArgSslPrim: 1 .
  ((res == nil) or:[res == false]) ifTrue:[ ^ self signalError: 'secureConnect failed'] .
  res == -1 ifTrue:[ Delay waitForMilliseconds: 20 . waitedMs := waitedMs + 20 ]
           ifFalse:[ res == true ifTrue:[^ res "done" ] .
                     self signalError: 'secureConnect failed with ' , res asString
                  ].
  waitedMs < timeoutMs
] whileTrue .
self signalError: 'secureConnect timed out' .

]

{ #category : 'Peer Authentication' }
GsSecureSocket >> setCertificateVerificationOptions: anArray [

"Sets the certificate verification options for the receiver using an Array of
 Symbols.

 The supported server socket options are:

 #SSL_VERIFY_FAIL_IF_NO_PEER_CERT
    if the client did not return a certificate, the TLS/SSL handshake is
    immediately terminated with a 'handshake failure' alert.

 #SSL_VERIFY_CLIENT_ONCE
    only request a client certificate on the initial TLS/SSL handshake.
    Do not ask for a client certificate again in case of a renegotiation.

 Refer to the OpenSSL documentation for more information about these options.

 Raises an error if the array contains any elements besides the above symbols.

 Currently there are no supported options for client sockets. Using this method
 with a client socket raises an error.

 If anArray is empty then any previously set options are cleared.

 The receiver must enable certificate verification before this method is
 executed, otherwise an error is raised.  Use the #enableCertificateVerification
 method to enable certificate verification.

 This method must be used before the receiver attempts a connection with its peer.
 Using this method with a connected socket raises an error.

 Returns true on success or raises an exception on error."

^ (self _oneArgSslPrim: 7 with: anArray)
    ifTrue:[ true ]
    ifFalse:[ self signalError ]

]

{ #category : 'Ciphers' }
GsSecureSocket >> setCipherListFromString: aString [

"Specify the cipher list to be used by the receiver only.  aString must be an
 instance of String in the format described in the man page for ciphers(1).  See
   http://www.openssl.org/docs/apps/ciphers.html

 Must be executed before the receiver is in a connected state (i.e., before
 the #secureConnect or #secureAccept method has been executed).

 Returns true if one or more of the specified ciphers are usable, raises an
 exception if no specified ciphers are usable.  Raises an exception if the operation would
 have no effect because the receiver is already in a connected state."

^ (self _oneArgSslPrim: 5 with: aString)
    ifTrue:[ true ]
    ifFalse:[ self signalError ]

]

{ #category : 'Client Operations' }
GsSecureSocket >> setExpectedHost: aString [

"Sets the expected host name that the receiver will connect to.
 Any previously set host names are cleared. If aString is nil, all
 previously set host names will be cleared and no host name matching
 will be performed. 

 This method must be executed before the #secureConnect method and only with
 client sockets. Raises an exception if the receiver is not a client
 socket, #secureConnect has already been executed, or if aString is not
 a non-empty instance of String or nil.

 Returns the receiver."

^ self _oneArgSslPrim: 21 with: aString
]

{ #category : 'Client Operations' }
GsSecureSocket >> setExpectedHostFlags: anArray [

"Sets the flags which control host name matching during TLS session 
 negotiation. By default no flags are set.  

 anArray must contain zero or more symbols of the following symbols:

  #X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT
  #X509_CHECK_FLAG_NO_WILDCARDS
  #X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS
  #X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS
  #X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS
  #X509_CHECK_FLAG_NEVER_CHECK_SUBJECT

 An empty array causes all previously set flags to be cleared.

 A detailed description of these flags may be found in the OpenSSL documentation
 at: https://www.openssl.org/docs/man3.0/man3/X509_check_host.html

 Raises an exception if the argument is not an Array or any element is not
 one of the above symbols. Returns the receiver."

^ self _oneArgSslPrim: 23 with: anArray
]

{ #category : 'Peer Authentication' }
GsSecureSocket >> setPreSharedKey: aByteArray [

"Sets the Pre-Shared Key (psk) for the connection to be the bytes contained in aByteArray.
Raises an exception if the receiver is already connected or listening for a connection.
aByteArray must have a size of at least 8 bytes, not more than 64 bytes and be a multiple 
of 8 (16, 24, 32 etc). A key size of at least 16 bytes is recommended.

Invoking this method multiple times overwrites any previously stored PSK.
To clear a the previously set PSK, invoke this method with an argument of nil.

Returns true on success."

^ self _oneArgSslPrim: 14 with: aByteArray
]

{ #category : 'Client Operations' }
GsSecureSocket >> setServerNameIndication: aString [

"Sets the server name indication (SNI) name for the receiver to be
 aString.

 This method is only valid for client sockets which have not yet
 securly connected to a peer.

 Raises an exception if the receiver is not a client socket or if 
 the receiver is already securly connected to its peer.

 Returns true on success, false if an error occurred setting the SNI
 name."

^ self _oneArgSslPrim: 13 with: aString

]

{ #category : 'Error Reporting' }
GsSecureSocket >> signalError [
  ^ SecureSocketError signal: (self lastErrorString ifNil:[ 'no details' ]).

]

{ #category : 'Error Reporting' }
GsSecureSocket >> signalError: aString [
  ^ SecureSocketError signal:
     (self lastErrorString ifNotNil:[:s | aString , ', ' , s]  ifNil:[ aString ]).

]

{ #category : 'TLS Protocol Version' }
GsSecureSocket >> tlsActualVersion [

"Answers a Symbol indicating the actual TLS protocol version used by the 
 receiver. The TLS protocol version for the connection is negotiated by 
 the receiver and its peer during the TLS handshake.

 Possible results are:
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 Raises an exception if the receiver has not completed the TLS handshake
 with its peer. For server sockets, the TLS handshake is complete after the 
 #secureAccept succeeds. For client sockets, the TLS handshake is complete
 after the #secureConnect method succeeds."

^ self _zeroArgSslPrim: 34
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket >> tlsMaxVersion [

"Gets the maximum TLS protocol version for the receiver.

 Possible results are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 A return value of #TLS_VERSION_DEFAULT means any supported 
 TLS protocol may be used, which is the default behavior."

^ self _zeroArgSslPrim: 29
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket >> tlsMaxVersion: aStringOrSymbol [

"Sets the maximum TLS protocol version for the receiver.
 aStringOrSymbol must be an instance of String or Symbol.

 Valid options are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 Passing an argument of #TLS_VERSION_DEFAULT means use any supported 
 TLS protocol, which restores the default behavior.

 Returns true on success or raises an exception if the argument is invalid.
 Raises an exception if the argument would set the maximum TLS protocol version
 to be less than a previously set minimum TLS protocol version."

^ self _oneArgSslPrim: 16 with: aStringOrSymbol
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket >> tlsMinVersion [

"Gets the minimum TLS protocol version for the receiver.

 Possible results are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 A return value of #TLS_VERSION_DEFAULT means any supported 
 TLS protocol may be used, which is the default behavior."

^ self _zeroArgSslPrim: 28
]

{ #category : 'TLS Protocol Version' }
GsSecureSocket >> tlsMinVersion: aStringOrSymbol [

"Sets the minimum TLS protocol version for the receiver.
 aStringOrSymbol must be an instance of String or Symbol.

 Valid options are:
   #TLS_VERSION_DEFAULT = default behavior (see below)
   #TLS1_VERSION        = TLS version 1.0
   #TLS1_1_VERSION      = TLS version 1.1
   #TLS1_2_VERSION      = TLS version 1.2
   #TLS1_3_VERSION      = TLS version 1.3

 Passing an argument of #TLS_VERSION_DEFAULT means use any supported 
 TLS protocol, which restores the default behavior.

 Returns true on success or raises an exception if the argument is invalid.
 Raises an exception if the argument would set the minimum TLS protocol version
 to be greater than a previously set maximum TLS protocol version."

^ self _oneArgSslPrim: 15 with: aStringOrSymbol
]

{ #category : 'Certificates and Keys' }
GsSecureSocket >> useCertificate: aString withPrivateKey: anotherString privateKeyPassphrase: passPhrase [

"Specifies the certificate and private key to be used by
 the receiver only.  The private key must match the certificate.

 The certificates loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 Both strings must be in PEM format.  If the private key requires a passphrase,
 it is specified in the passPhrase argument as a String.  If no passphrase is
 required, then the passPhrase argument is expected to be nil.

 Both strings must exactly match the contents of the corresponding certificate files
 (including white-space characters) or the strings will not be accepted.

 Returns true on success or raises an exception if an error occurs."

^ (self _threeArgSslPrim: 5 with: aString with: anotherString with: passPhrase)
     ifTrue:[ true ]
     ifFalse:[ self signalError ]

]

{ #category : 'Certificates and Keys' }
GsSecureSocket >> useCertificateFile: certFile withPrivateKeyFile: keyFile privateKeyPassphrase: pass [

"Specifies the certificate file and private key file to be used by
 the receiver only.  The private key must match the certificate.

 This method must be run before the #secureAccept (server case)
 or the #secureConnect (client case) methods.
 The certificates loaded into the internal SSL state are valid for the
 current session only; that SSL state is not committed to the repository.

 If the private key requires a passphrase, it is to be specified as String
 passed in using the pass argument.  If the private key does not require a
 passphrase, then the pass argument is expected to be nil.

 Both files must be in PEM format.

 Both arguments must be instances of String.

 Returns true on success or raises an exception if an error occurs or
 the file does not exist."

^ (self _threeArgSslPrim: 8 with: certFile with: keyFile with: pass)
     ifTrue:[ true ]
     ifFalse:[ self signalError ]

]
