"
GsReferencePath holds the result returned from Repository>>#_refPathForObject:limits:.

Instance Variables
   target        <Object>  The object whose path being sought.
   inSearchOops  <Boolean> true if the target was in the searchOops for the last scan.
   status        <Symbol>  Returns #complete if the path starts with an object in the limit set.
                           Returns #cycle if an object occurs more than once in the path.
                           Returns #notConnected if the head of the path has no parents.
                           Returns #noInfo if the RefPathState is not available or a
                             complete path cannot be found.
   moreParents   <Boolean> true if it is known that there are more objects returned than the 
                           numParents value.  This should always be false for objects in the searchOops.
                           If it is true, the other parents can be found by performing another scan
                           with the target in the searchOops.
   numParents    <Integer> The number of parents found for the target.   If the target was NOT
                           in the searchOops for the scan then this value is 0 or 1 (only one
                           parent can be returned for an object not in the searchOops).
   path          <Array>   The objects in the path, starting with an object in the limits,
                           ending with the target object.  An empty array if no path was found.
                           The maximum path returned is 10000.

"
Class {
	#name : 'GsReferencePath',
	#superclass : 'Object',
	#instVars : [
		'target',
		'inSearchOops',
		'status',
		'numParents',
		'moreParents',
		'path'
	],
	#category : 'Repository'
}

{ #category : 'Primitive' }
GsReferencePath class >> _refPathForObject: anObjOrGsBitmap [

"Returns a reference path to a single object.

 This method returns a new GsReferencePath instance.

 If a GsBitmap is passed as an argument, then each object in the bitmap is
 tested and the first object to have a complete path is returned.  If no
 object has a complete path then the status returned is #noInfo.
"

<primitive: 1024>
^ self _primitiveFailed:
    #_refPathForObject:
    args: { anObjOrGsBitmap }

]

{ #category : 'Formatting' }
GsReferencePath class >> pathArrayAsString: path [
"Returns a String describing the path in the form
   <pathIdx> Oop = <objectId> (<className>)
"
| pathLength obj result |

pathLength := path size.
result := String new.

1 to: pathLength do: [:i |  | refPath |
  obj := path at: i.
  result add: ( '  ' , i asString, ' oop = ', (GciInterface _oopForObject: obj) asString,
            ' (', obj class name asString , ')').
  refPath := GsReferencePath _refPathForObject: obj .
  (refPath status == #noInfo)
     ifFalse: [
       (refPath inSearchOops)
          ifTrue: [ result add: (' has ' , refPath numParents asString , ' parents') ]
          ifFalse: [
             (refPath moreParents) ifTrue: [ result add: (' has more than one parent') ]
                                ifFalse: [ result add: (' has only one parent') ]
          ].
     ].
   result add: Character lf.
].
^result

]

{ #category : 'Accessing' }
GsReferencePath >> inSearchOops [
 "Answer whether the target was in the searchOops."

^inSearchOops

]

{ #category : 'Updating' }
GsReferencePath >> inSearchOops: aBoolean [

inSearchOops := aBoolean

]

{ #category : 'Testing' }
GsReferencePath >> isComplete [
  "Answers whether the path returned is complete, i.e., terminates with an object
   in the limit set"

^ status == #complete

]

{ #category : 'Testing' }
GsReferencePath >> isCycle [
  "Answers whether the path contains a cycle, i.e., terminates with an object that appears
   at another position in the path."

^status == #cycle

]

{ #category : 'Testing' }
GsReferencePath >> isNotConnected [

  "Answers true if the head of the path has no parents"

^ status == #notConnected

]

{ #category : 'Accessing' }
GsReferencePath >> moreParents [
 "Answer whether it is known if there are moreParents than specified in numParents."

^moreParents

]

{ #category : 'Updating' }
GsReferencePath >> moreParents: aBoolean [

moreParents := aBoolean

]

{ #category : 'Accessing' }
GsReferencePath >> numParents [
 "Answer the number of parent objects for the target"

^numParents

]

{ #category : 'Updating' }
GsReferencePath >> numParents: anInt [

numParents := anInt

]

{ #category : 'Accessing' }
GsReferencePath >> path [
 "Answer an array of the oops that define a path to the target object"

^path

]

{ #category : 'Updating' }
GsReferencePath >> path: anArray [

path := anArray

]

{ #category : 'Accessing' }
GsReferencePath >> status [
  "Answers a symbol which defines the result for the path.
   Possible values are:
      #complete        if the path starts with an object in the limit set.
      #cycle           if an object occurs more than once in the path.
      #notConnected    if the head of the path has no parents.
  "
^status

]

{ #category : 'Updating' }
GsReferencePath >> status: aSymbol [

status := aSymbol

]

{ #category : 'Accessing' }
GsReferencePath >> target [
 "Answer the target object for the path."

^target

]

{ #category : 'Updating' }
GsReferencePath >> target: anObj [

target := anObj

]
