Extension { #name : 'GsRangeEqualityIndex' }

{ #category : 'Constants' }
GsRangeEqualityIndex class >> _isUnicodeLastElementClass: aClass [
  "returns true if <aClass> is a Unicode class"

  self _validateLastElementClass: aClass.
  Unicode16 usingUnicodeCompares
    ifTrue: [
      (self isBasicClass: aClass)
        ifTrue: [
          (aClass _subclassOf: CharacterCollection)
            ifTrue: [ ^ true ] ] ]
    ifFalse: [
      (aClass _subclassOf: Unicode7)
        ifTrue: [ ^ true ].
      (aClass _subclassOf: Unicode16)
        ifTrue: [ ^ true ].
      (aClass _subclassOf: Unicode32)
        ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Private' }
GsRangeEqualityIndex class >> _validateLastElementClass: aClass [
  aClass isBehavior
    ifFalse: [
      ArgumentTypeError
        signal: 'LastElementClass (' , aClass printString , ') must be a class' ]

]

{ #category : 'Constants' }
GsRangeEqualityIndex class >> isBasicClass: aClass [
  "Returns whether the given class should use BtreePlusBasic*Nodes to store
 B-tree mappings."

  | characterCollectionBasicClasses |
  aClass == nil
    ifTrue: [ ^ false ].
  aClass _isSpecial
    ifTrue: [ "specials do not benefit from basic btree nodes"
      ^ false ].
  self _validateLastElementClass: aClass.
  Unicode16 usingUnicodeCompares
    ifTrue: [ characterCollectionBasicClasses := #().
      (aClass _subclassOf: CharacterCollection)
        ifTrue: [ aClass == CharacterCollection
            ifTrue: [ ^ true ].
          (aClass _subclassOf: String)
            ifTrue: [ ^ true ].
          (aClass _subclassOf: MultiByteString)
            ifTrue: [ ^ true ] ] ]
    ifFalse: [ characterCollectionBasicClasses := {String.
      DoubleByteString.
      QuadByteString.
      Unicode7.
      Unicode16.
      Unicode32} ].
  characterCollectionBasicClasses
    , {Number.	"all non-special subclasses of Number"
      DateTime.
      DateAndTime.
      Date.
      Time} do: [ :basicClass | (aClass _subclassOf: basicClass)
        ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Instance Creation' }
GsRangeEqualityIndex class >> newWithLastElementClass: aClass [
  "Create a new instance and initialize its B-tree root."

  | newOne |
  newOne := super new.
  newOne lastElementClass: aClass.
  ^ newOne

]

{ #category : 'Testing' }
GsRangeEqualityIndex >> _canCompareWith: aKey [
  "Returns whether the receiver can make B-tree comparisons with the given key."

  ^ btreeRoot _canCompare: aKey withClass: lastElementClass

]

{ #category : 'Testing' }
GsRangeEqualityIndex >> _checkSameLastElementClassAsEqualityIndex: equalityIndexObj [
  ^ equalityIndexObj lastElementClass == self lastElementClass

]

{ #category : 'Updating' }
GsRangeEqualityIndex >> _clear [
  "Assigns nil to important instance variables."

  lastElementClass := nil.
  super _clear

]

{ #category : 'Sorting' }
GsRangeEqualityIndex >> _getSecondarySorterOn: indexObjs directions: theBooleans [

"Returns a PathSorter if one is needed to perform secondary sorting;
 otherwise returns nil."

| directions traversers aSize |

(aSize := indexObjs size) > 1 ifTrue: [
    directions := theBooleans copyFrom: 2 to: aSize .
    traversers := indexObjs copyFrom: 2 to: aSize .
    ^ PathSorter on: traversers directions: directions.
].
^ nil  .

]

{ #category : 'Updating' }
GsRangeEqualityIndex >> _setLastPathTermState [
  "Indicate if the last object along the path needs a dependency list."

  | lastPathTerm |
  lastPathTerm := self at: self size.
  lastElementClass allSubclasses , {lastElementClass}
    do: [ :cl | cl instancesInvariant not
        ifTrue: [ "bug 42640 - depList needed unless instances invariant"
          lastPathTerm needsDepList: true.
          ^ self ] ]

]

{ #category : 'Updating' }
GsRangeEqualityIndex >> _setPathTermState [
  "For the last path term, indicate the need to update the B-tree.
   Indicate if the last object along the path needs a dependency list."

  super _setPathTermState.
  self _setLastPathTermState

]

{ #category : 'Sorting' }
GsRangeEqualityIndex >> _sortAscendingOn: indexObjs directions: theBooleans [

""

| array cnt marker sorter prevKey pArr |

" create the Array to hold the objects "
array := Array new: nscRoot size.

" build a sorter to be used for secondary sorts "
sorter := self _getSecondarySorterOn: indexObjs directions: theBooleans.

" start the insertion into the Array at the beginning "
self isIndexOnRootNsc
    ifTrue: [ cnt := 0 ]
    ifFalse: [ cnt := nscRoot _idxOccurrencesOf: nil ].

" gather objects with nil along the path "
pArr := self _getObjectsWithNilOnPath.
1 to: pArr size do: [ :i |
    cnt := cnt + 1.
    array at: cnt put: (pArr at: i)
].
pArr := nil .

" if no secondary sorts "
sorter == nil ifTrue: [
    btreeRoot _putAscendingRootObjectsInto: array startingAt: cnt + 1.
    ^ array
  ].

prevKey := #_incompletePathTraversal.
marker := cnt max: 1.

" Iterate over the key/value/roots in the B-tree leaf nodes "
btreeRoot _leafKeysValuesAndRootsDo: [ :key :value :root |

   cnt := cnt + 1.
   array at: cnt put: root.

    " see if finished a run of duplicate keys "
    key ~= prevKey
      ifTrue: [
        marker < (cnt - 1)
          ifTrue: [ " sort on secondary keys "
            marker to: cnt - 1 do: [ :i | sorter addObject: (array at: i) ].
            sorter sortInto: array startingAt: marker.
            sorter clear
          ].
        marker := cnt
      ].
    prevKey := key.
].
" see if ending with duplicate keys "
( sorter ~~ nil and: [ marker < cnt ] )
    ifTrue: [ " sort on secondary keys "
       marker to: cnt do: [ :i | sorter addObject: (array at: i) ].
       sorter sortInto: array startingAt: marker.
       sorter clear
    ].

^ array

]

{ #category : 'Sorting' }
GsRangeEqualityIndex >> _sortDescendingOn: indexObjs directions: theBooleans [

""

| array cnt prevKey marker sorter pArr |

" create the Array to hold the objects "
array := Array new: nscRoot size.

" build a sorter to be used for secondary sorts "
sorter := self _getSecondarySorterOn: indexObjs directions: theBooleans.

" start the insertion into the Array at the beginning "
self isIndexOnRootNsc
    ifTrue: [ cnt := array size  ]
    ifFalse: [ cnt := array size - (nscRoot _idxOccurrencesOf: nil) ].

" gather objects with nil along the path "
pArr := self _getObjectsWithNilOnPath.
1 to: pArr size do: [ :i |
    array at: cnt put: (pArr at: i).
    cnt := cnt - 1.
].
pArr := nil .

" if no secondary sorts "
sorter == nil ifTrue: [
    btreeRoot _putDescendingRootObjectsInto: array startingAt: cnt.
    ^ array
  ].

prevKey := #_incompletePathTraversal.
marker := cnt.

" iterate over the key/values in the B-tree leaf nodes "
btreeRoot _leafKeysValuesAndRootsDo: [ :key :value :root |

      array at: cnt put: root.
      cnt := cnt - 1.

    " see if finished a run of duplicate keys "
    key ~= prevKey
      ifTrue: [
        marker > (cnt + 1)
          ifTrue: [ " sort on secondary keys "
            cnt + 2 to: marker do: [ :i | sorter addObject: (array at: i) ].
            sorter sortInto: array startingAt: cnt + 2.
            sorter clear
          ].
        marker := cnt + 1
      ].
    prevKey := key.
].

" see if ending with duplicate keys "
( sorter ~~ nil and: [ marker > 1 ] )
  ifTrue: [ " sort on secondary keys "
   1 to: marker do: [ :i | sorter addObject: (array at: i) ].
   sorter sortInto: array startingAt: 1.
     sorter clear
    ].
^ array

]

{ #category : 'Sorting' }
GsRangeEqualityIndex >> _sortOn: indexObjs directions: theBooleans persistentRoot: persistentArrayOrNil [

"Returns an Array of objects from the root NSC that is sorted according to the
 paths traversed by indexObjs (with the direction of the sort specified in
 theBooleans)."

(theBooleans at: 1)
    ifTrue: [ ^ self _sortAscendingOn: indexObjs directions: theBooleans ]
    ifFalse: [ ^ self _sortDescendingOn: indexObjs directions: theBooleans ]

]

{ #category : 'Testing' }
GsRangeEqualityIndex >> _validateCanCompareWith: aKey [
  (self _canCompareWith: aKey)
    ifFalse: [ ^ (ImproperOperation new
        _number:
            (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
        args: { aKey class. self lastPathTerm name. self lastElementClassDescription }) signal ]

]

{ #category : 'Private' }
GsRangeEqualityIndex >> _validateLastElementClass: aClass [
  self class _validateLastElementClass: aClass

]

{ #category : 'Converting' }
GsRangeEqualityIndex >> asIndexSpecification [
  (self class _isUnicodeLastElementClass: self lastElementClass)
    ifTrue: [ ^ self asUnicodeIndexSpecification ].
  ^ (EqualityIndexSpecification
    path: self pathComponentsString
    lastElementClass: self lastElementClass)
    options: self options;
    yourself

]

{ #category : 'Converting' }
GsRangeEqualityIndex >> asQueryEvaluator [

  ^GsEqualityIndexQueryEvaluator on: self

]

{ #category : 'Converting' }
GsRangeEqualityIndex >> asUnicodeIndexSpecification [
  ^ (UnicodeIndexSpecification
    path: self pathComponentsString
    collator: self collator)
    options: self options;
    yourself

]

{ #category : 'Stream Accessing' }
GsRangeEqualityIndex >> btreeRangeComparisonQuerySpec [
  ^ BtreePlusRangeComparisonQuerySpec new
    rangeIndex: self;
    yourself

]

{ #category : 'Comparison Operators' }
GsRangeEqualityIndex >> comparisonForCompare [
  ^ self optimizingComparison
    ifTrue: [ BtreeOptimizedComparison new ]
    ifFalse: [ BtreeComparisonForCompare newForComparison ]

]

{ #category : 'Comparison Operators' }
GsRangeEqualityIndex >> comparisonForSort [
  "called during creation of receiver, so initialize unset instance variables"

  ^ comparisonForSort ifNil: [
      comparisonForSort := self optimizingComparison
        ifTrue: [ BtreeOptimizedComparison new ]
        ifFalse: [ BtreeComparisonForCompare newForSort ] ]

]

{ #category : 'Searching' }
GsRangeEqualityIndex >> findRootObjectMaps: rootObjectMap pathTerm: pathTerm key: aKey value: aValue [
  | evaluator stream |
  evaluator := self asQueryEvaluator.
  stream := evaluator
    _findAllValuesGreaterThan: aKey
    andEquals: true
    andLessThan: aKey
    andEquals: true
    using: evaluator comparisonForSort.
  stream
    _scanForRootObjectMaps: aKey
    value: aValue
    valueOop: aValue asOop
    using: self comparisonForCompare
    do: [:aRoot |
      self
        findRootObjects: rootObjectMap
        rootObject: aRoot
        pathTerm: pathTerm
        key: aKey
        value: aValue ].

]

{ #category : 'Searching' }
GsRangeEqualityIndex >> findRootObjectsForKey: aKey value: aValue [
  | evaluator stream rootObjects |
  evaluator := self asQueryEvaluator.
  stream := evaluator
    _findAllValuesGreaterThan: aKey
    andEquals: true
    andLessThan: aKey
    andEquals: true
    using: evaluator comparisonForSort.
  rootObjects := IdentityBag new.
  [ stream _btreeAtEnd ]
    whileFalse: [
      | rootObject |
      "collect key/value/root tuples for entries in the btree whose intermediate parent object
       is reachable from the root object in the entry."
      ((stream _peekValue == aValue) and: [ stream _peekKey == aKey ])
        ifTrue: [ "select value entries that are identically equal to aValue"
          rootObject := stream _peekRoot.
          rootObjects add: rootObject ].
      stream _btreeNext ].
 ^ rootObjects

]

{ #category : 'Accessing' }
GsRangeEqualityIndex >> indexType [
  ^ #'equality'

]

{ #category : 'Testing' }
GsRangeEqualityIndex >> isRangeEqualityIndex [

"Returns true."

^ true

]

{ #category : 'Accessing' }
GsRangeEqualityIndex >> lastElementClass [

"Returns the value of the instance variable 'lastElementClass'."

^lastElementClass

]

{ #category : 'Updating' }
GsRangeEqualityIndex >> lastElementClass: aClass [
  self _validateLastElementClass: aClass.
  lastElementClass := aClass

]

{ #category : 'Accessing' }
GsRangeEqualityIndex >> lastElementClassDescription [
  "answer a description of the lastElementClass of the receiver,
   suitable for use in an error message"

  | cl qualifier |
  qualifier := ''.
  self optimizingComparison ifTrue: [ qualifier := ' [ with optimized comparison ]'. ].
  cl := self lastElementClass.
  (cl isSubclassOf: CharacterCollection)
    ifTrue: [
      Unicode16 usingUnicodeCompares
        ifTrue: [ ^ 'CharacterCollection', qualifier ]
        ifFalse: [ ^ 'CharacterCollection excluding Unicode String classes', qualifier ] ].
  ^ cl name asString, qualifier

]

{ #category : 'Updating' }
GsRangeEqualityIndex >> nscRoot: newValue [
  super nscRoot: newValue.
  "force lazy initialization of optimizingComparison iv to avoid commit conflicts later..."
  self optimizingComparison

]

{ #category : 'Updating' }
GsRangeEqualityIndex >> objectSecurityPolicy: anObjectSecurityPolicy [
  "Assign the receiver and its B-tree to the given security policy."

  super objectSecurityPolicy: anObjectSecurityPolicy.
  self btreeRoot objectSecurityPolicy: anObjectSecurityPolicy

]

{ #category : 'Testing' }
GsRangeEqualityIndex >> optimizingComparison [
  "if option is set, the ensure that the lastElementClass support optimized comparisons"

  ^optimizingComparison ifNil: [
    optimizingComparison := super optimizingComparison
      ifTrue: [ self lastElementClass _idxCanOptimizeComparison ]
      ifFalse: [ false ] ].

]

{ #category : 'Accessing' }
GsRangeEqualityIndex >> sortNodeClass [
  "Returns the class of SortNode to use."

  ^ self lastElementClass sortNodeClass

]
