"
GsLog provides a logging mechanism which can be used to reopen a FileSystem-based log 
file when a SIGHUP received by a gem results in signalling of a LogRotateNotification .

An instance encapsulates a FileReference and a ZnCharacterWriteStream such that
the underlying file can be reopened by a handler for LogRotateNotification .

GsLog is thread-safe. Writes to the log file are protected by a mutex. 
reopen trys to lock the mutex and signals a Warning it cannot lock.
If a file exists during initial open or during a reopen, the existing file is appended to. 
If the specified file does not exist, a new file is created.

Care should be taken to avoid logging during Ephemeron mourning. Ephemeron mourning, by default, 
occurs on the same thread as active process. If mourning happens during a critical section 
and it writes to the log, the process will deadlock. 
This can be avoided by using processing mourn events in a defined process. 
See GsSignalingSocket class>>_readNotificationExample.
"
Class {
	#name : 'GsLog',
	#superclass : 'Object',
	#instVars : [
		'fileReference',
		'stream',
		'mutex'
	],
	#category : 'gemstoneBaseImage'
}

{ #category : 'instance creation' }
GsLog class >> fileReference: aFileReference [
	"Create a new log writing to the location specified."

	| instance |
	instance := self basicNew.
	instance initializeWith: aFileReference .
	^instance
]

{ #category : 'instance creation' }
GsLog class >> pathString: aPathString [
	"Create the log at the location defined by the provided string."

	^self fileReference: aPathString asFileReference 
]

{ #category : 'closing' }
GsLog >> close [
	"Close the log."

	mutex critical: [stream close. stream := nil]
]

{ #category : 'initializing' }
GsLog >> initializeWith: aFileReference [
	"Initialize the log with the provided location."

	mutex := Semaphore forMutualExclusion.
	fileReference := aFileReference asAbsolute .
	stream := fileReference appendStream.
]

{ #category : 'rotating' }
GsLog >> reopen [
  "Rotate the log file.
  Reopens the log file. Recreates the file if it does not exist.
  Intended to be sent from a handler for  LogRotateNotification."

  mutex tryLock ifTrue:[
		stream close.
		stream := fileReference appendStream .
    mutex signal 
  ] ifFalse:[
    Warning signal:'GsLog >> reopen , failed to get mutex to reopen file'.
  ]
]

{ #category : 'logging' }
GsLog >> write: aMessageString [
	"Write the provided message to the log file.
	This method flushes so that `tail` is useful
	and up-to-date."

	mutex critical:
		[stream
			nextPutAll: aMessageString;
			lf; flush]
]
