"
GsLegacyExternalSession is a variant of GsExternalSession which is used in
environments where the FFI GciLibrary cannot be used. It uses the GCI Interface
which is based on system primitives rather than FFI calls.

At this time, AIX is the only known environment where this class must be used.

"
Class {
	#name : 'GsLegacyExternalSession',
	#superclass : 'GsExternalSession',
	#instVars : [
		'gciInterface',
		'traversalBuffer'
	],
	#category : 'ExternalSessions'
}

{ #category : 'Instance Creation' }
GsLegacyExternalSession class >> gciErrSTypeClass [
  ^ GciLegacyError

]

{ #category : 'Instance Creation' }
GsLegacyExternalSession class >> gciLibrary [
  "No shared library loaded with GsLegacyExternalSession"
  ^ self
]

{ #category : 'Instance Creation' }
GsLegacyExternalSession class >> newDefault [
  ^ self basicNew initialize ; initializeDefaultResources 
]

{ #category : 'Private' }
GsLegacyExternalSession >> _errorIfCallInProgress [

  self isCallInProgress ifTrue: [self error: 'call in progress'].

]

{ #category : 'Private' }
GsLegacyExternalSession >> _freeTraversalBuffer [

	traversalBuffer ifNil: [^self].
	traversalBuffer free.
	traversalBuffer := nil.

]

{ #category : 'Private' }
GsLegacyExternalSession >> _gciInterface [

	^gciInterface

]

{ #category : 'Private' }
GsLegacyExternalSession >> _gciLibrary [
   "Not usable in GsLegacyExternalSession"
   self shouldNotImplement: #_gciLibrary

]

{ #category : 'Private' }
GsLegacyExternalSession >> _getStackForOop: gcierrContextOop [
  | str |
  self isCallInProgress ifTrue:[ ^ 'call in progress, stack not available' ]. 
  str := 'AbstractExternalSession _stackReport: ' , gcierrContextOop asString .
  self forkString: str .
  self waitForResultForSeconds: 120 .
  ^ self nbResult .
]

{ #category : 'Private' }
GsLegacyExternalSession >> _isOnMyStone [
  "value is an approximation, not guaranteed to be accurate"
  | val |
  (val := self dynamicInstVarAt: #_isOnMyStone) ifNil:[
    (GsSession currentSession isSolo) ifTrue:[
      val := false
    ] ifFalse:[
      val := parameters gemStoneName = System stoneName.
    ].
    self dynamicInstVarAt: #_isOnMyStone put: val .
  ].
  ^ val

]

{ #category : 'Private' }
GsLegacyExternalSession >> _logout [

	| descr |
	descr := self _describe.
	self _freeTraversalBuffer.
	gciInterface logout.
	self _waitForLogout.
	self getAndClearLastError.
	stoneSessionId := nil.
	stoneSessionSerial := nil.
	self log: 'GsLegacyExternalSession logout: ' , descr.

]

{ #category : 'Private' }
GsLegacyExternalSession >> _signalIfError [

	| error |
	(error :=  gciInterface getAndClearLastError) ifNil: [^self].
	GciLegacyError
		signal: error
		in: self.

]

{ #category : 'Public' }
GsLegacyExternalSession >> abort [

	self _errorIfCallInProgress.
	gciInterface nbAbort.
	self waitForResult.

]

{ #category : 'Error handling' }
GsLegacyExternalSession >> clearStackFor: anError [

	| contextOop |
	gciInterface sessionId == 0 ifTrue: [^self].
	(contextOop := anError context) == nil asOop ifTrue: [^self].
	self _errorIfCallInProgress.
	gciInterface clearStack: contextOop.
	self _signalIfError.

]

{ #category : 'Public' }
GsLegacyExternalSession >> commit [

	self _errorIfCallInProgress.
	^ gciInterface commit.
]

{ #category : 'Public' }
GsLegacyExternalSession >> commitOrError [
  | res |
  res := self commit .
  res ~~ true ifTrue:[
    TransactionError new signal: 'GciInterface>>commit failed'
  ].
  ^ res .
]

{ #category : 'Error handling' }
GsLegacyExternalSession >> continue: contextOop replacingTopOfStackWithOop: valueOop [
	"Continue execution in the external Gem following an exception,
	 replacing the top of the stack with the specified object.
	 It is an error to specify an oop not visible to the remote Gem."

	self _errorIfCallInProgress.
	^ gciInterface
		continueWith: contextOop
		replaceTopOfStackWith: valueOop
		isSpecial: false
		flags: 0.

]

{ #category : 'Public' }
GsLegacyExternalSession >> executeString: aString [
        "Execute the string expression in the external Gem and answer the result.
         The best values to return are specials.
         Strings and other byte objects are copied to the local Gem.
         All other responses are returned as a Array containing a single OOP."

        ^self
                forkString: aString;
                waitForResult;
                lastResult.

]

{ #category : 'Public' }
GsLegacyExternalSession >> forceLogout [

	self hardBreak.
	self _logout.

]

{ #category : 'Public' }
GsLegacyExternalSession >> forkString: aString [

	self _errorIfCallInProgress.
 gciInterface nbRemoteExecute: aString.

]

{ #category : 'Private' }
GsLegacyExternalSession >> gciErrorClass [
  ^ GciLegacyError
]

{ #category : 'Private' }
GsLegacyExternalSession >> getAndClearLastError [

	^ gciInterface getAndClearLastError.

]

{ #category : 'Public' }
GsLegacyExternalSession >> hardBreak [

	gciInterface hardBreak.

]

{ #category : 'Private' }
GsLegacyExternalSession >> initialize [

	super initialize.
	gciInterface := GciInterface new.

]

{ #category : 'Public' }
GsLegacyExternalSession >> isCallInProgress [
	"The following calls are OK during a nonblocking call:
	GciCallInProgress
	GciErr
	GciGetSessionId
	GciHardBreak
	GciNbEnd
	GciSetSessionId
	GciShutdown
	GciSoftBreak"

	self lastError notNil ifTrue: [self error: self lastError].
	gciInterface sessionId == 0 ifTrue: [^false].
	gciInterface callInProgress == true ifTrue: [^true].
	self getAndClearLastError.
	^false.

]

{ #category : 'Public' }
GsLegacyExternalSession >> isLoggedIn [
  | sid |
  gciInterface ifNil:[ ^ false ].
  sid := gciInterface sessionId  .
  ^ sid ~~ nil and:[ sid > 0 ].
]

{ #category : 'Public' }
GsLegacyExternalSession >> isResultAvailable [

	| status |
	self isCallInProgress ifFalse: [self error: 'no call in progress'].
	status := gciInterface nbEndOop.
	status ifNil: [self _signalIfError].
	^ status == 1.

]

{ #category : 'Private' }
GsLegacyExternalSession >> lastError [

	^ gciInterface lastError.

]

{ #category : 'Public' }
GsLegacyExternalSession >> lastResult [

	| result error1 error2 |
	gciInterface resultIsSpecial ifTrue: [^ gciInterface lastResult].
	error1 := gciInterface getAndClearLastError.
	self traversalBuffer initTraversalBuffer.
	result := gciInterface
		traverseObjects:  { gciInterface lastResult }
		buffer: traversalBuffer.
	(error2 := gciInterface getAndClearLastError) notNil ifTrue: [ | txt |
		txt := error2 message.
		(txt == nil or: [txt trimBlanks isEmpty])
			ifTrue: [(Exception _new: error2 number args:  {} ) signal]
			ifFalse:[self error: txt]
	].
	traversalBuffer currentObjectIsSpecial ifTrue: [^traversalBuffer objId].
	traversalBuffer currentObjectIsByte ifTrue: [^traversalBuffer getByteObject].
	^ { traversalBuffer objId }

]

{ #category : 'Public' }
GsLegacyExternalSession >> login [

	| result onMyStn onMyHost gemHostIdStr |
	stoneSessionId ifNotNil: [
		ImproperOperation signal: 'Stone session ' , stoneSessionId printString ,
			' already associated with this GsExternalSession!'.
	].
	self _errorIfCallInProgress.
	result := gciInterface login: parameters execute: 'System session'.
	self _signalIfError.
	stoneSessionId := result.

  "inline _postLogin"
  result := gciInterface remotePerform: System asOop selector: 'gemProcessId' args: #()  .
  self _signalIfError.
  gemProcessId := result .
  gemHostIdStr := self executeString:'System hostId asString'.
  self _signalIfError.
  onMyHost :=   gemHostIdStr = System hostId asString .

  GsSession isSolo ifTrue:[
    onMyStn := false .
  ] ifFalse:[ | stoneStartupStr |
    stoneStartupStr := self executeString:'System stoneStartupId asString'.
    self _signalIfError.
    onMyStn :=  stoneStartupStr = System stoneStartupId asString .
    stoneSessionSerial := onMyStn ifTrue:[ GsSession serialOfSession: stoneSessionId ]
                 ifFalse:[ self executeString: 'GsSession currentSession serialNumber'].
  ].
  gciSessionId := gciInterface sessionId.

  self dynamicInstVarAt: #_isOnMyStone put: onMyStn .
  self dynamicInstVarAt: #_isOnMyHost put:  onMyHost .

	self log: 'GsLegacyExternalSession login: ', self _describe .
]

{ #category : 'Public' }
GsLegacyExternalSession >> logout [

	self isCallInProgress ifTrue: [
		[
			self waitForResult.
		] on: Error do: [:ex |
			ex return.
		].
	].
	self _logout.

]

{ #category : 'Private' }
GsLegacyExternalSession >> nbLogout [

  ^ self forceLogout

]

{ #category : 'Public' }
GsLegacyExternalSession >> nbResult [
  | status |
  status := gciInterface nbEndOop .
  status == 1 ifFalse:[ Error signal:' result not ready'].
  ^ self lastResult 
]

{ #category : 'Public' }
GsLegacyExternalSession >> send: selector to: remoteOop withArguments: someValues [

  | args |
  (someValues == nil or: [someValues isEmpty]) ifTrue: [
     args := #( )
  ] ifFalse: [
    args := { } .
    someValues do: [:each | args add: each asOop ; add: each isSpecial ].
  ].
  gciInterface remotePerform: remoteOop selector: selector args: args.
  self _signalIfError.
  gciInterface resultIsSpecial ifTrue: [^ gciInterface lastResult].
  ^ { gciInterface lastResult }

]

{ #category : 'Accessors' }
GsLegacyExternalSession >> sessionId [

	^ gciInterface sessionId.

]

{ #category : 'Public' }
GsLegacyExternalSession >> softBreak [

	gciInterface softBreak.

]

{ #category : 'Public' }
GsLegacyExternalSession >> traversalBuffer [

	traversalBuffer ifNil: [
		(traversalBuffer := TraversalBuffer new: 8000)
			level: 1;
			initTraversalBuffer.
	].
	^traversalBuffer.

]

{ #category : 'Public' }
GsLegacyExternalSession >> waitForResultForSeconds: aNumber otherwise: aBlock [
  "Wait as long as the specified seconds for the external Gem
  to complete the current operation. If the operation does not
  complete within that time, answer the result of evaluating aBlock."

  | start |
  start := System timeGmt.
  [ self isResultAvailable ] whileFalse:
    [ Delay waitForMilliseconds: 20.
      aNumber < (System timeGmt - start) ifTrue:[ ^aBlock value]]

]
