Extension { #name : 'GsIndexOptions' }

{ #category : 'Accessing' }
GsIndexOptions class >> btreePlusIndex [
  "Create a btree plus index that uses the new BtreePlusNodes for indexing."

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  "Only one of legacyIndex or btreePlusIndex may be set in an instance"

  ^ self basicNew
    clearLegacyIndex; "record that legacyIndex is false"
    setBtreePlusIndex;
    yourself

]

{ #category : 'Default' }
GsIndexOptions class >> default [
  "default options "

  (SessionTemps current at: #GsDefaultIndexOptions ifAbsent: [])
    ifNotNil: [:sessionDefault | ^ sessionDefault ].
  (Globals at: #GsDefaultIndexOptions ifAbsent: [])
    ifNotNil: [ :default | ^ default ].
  "set state explicitly to avoid infinite loop getting default options when no default defined yet."
  ^ self basicNew
      btreePlusIndex: true;
      legacyIndex: false;
      reducedConflict: false;
      optionalPathTerms: false;
      optimizedComparison: false;
      yourself

]

{ #category : 'Default' }
GsIndexOptions class >> default: aGsIndexOptions [
  "set persistent default options"

  ^ Globals at: #GsDefaultIndexOptions put: aGsIndexOptions _reify immediateInvariant

]

{ #category : 'Accessing' }
GsIndexOptions class >> legacyIndex [
  "Create a legacy index that uses BtreeNodes instead of the new BtreePlusNodes for indexing."

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  ^ self basicNew
    clearBtreePlusIndex; "record that btreePlusIndex is false"
    setLegacyIndex;
    yourself

]

{ #category : 'Accessing' }
GsIndexOptions class >> optimizedComparison [
  "Eliminate per element comparison testing for indexed queries and use native #=
   instead of _idxForCompare* methods when comparisons are needed. In order to be
   able to do these optimizations, the following constraints are imposed on the
   last element values:
    - values must be a kind of the last element class.
    - nil is not allowed as a value.
    - For Float last element class, a NaN is not allowed as a value.
    - For String last element class, Symbols are not allowed as a value.
    - For Symbol last element class, Strings are not allowed as a value.
   "

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  ^ self basicNew
    optimizedComparison: true; "set directly, to allow optimizedComparision option to be set, when lagacyIndex is default (bug 47145)"
    yourself

]

{ #category : 'Accessing' }
GsIndexOptions class >> optionalPathTerms [
  "Do not signal an error if instance variable missing along index path."

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  ^ self basicNew
    setOptionalPathTerms;
    yourself

]

{ #category : 'Accessing' }
GsIndexOptions class >> reducedConflict [
  "Create a reduced conflict index"

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  ^ self basicNew
    setReducedConflict;
    yourself

]

{ #category : 'Default' }
GsIndexOptions class >> sessionDefault: aGsIndexOptions [
  "set default index options for this session only"

  aGsIndexOptions ifNotNil: [ aGsIndexOptions _reify immediateInvariant ].
  ^ SessionTemps current at: #GsDefaultIndexOptions put: aGsIndexOptions

]

{ #category : 'Private' }
GsIndexOptions >> _btreePlusIndex [
  ^ self dynamicInstVarAt: #'btreePlusIndex'

]

{ #category : 'Errors' }
GsIndexOptions >> _errorIncorrectIndexTypeOptionsCombo [

  InternalError signal: 'Invalid GsIndexOptions instance: Only one of either legacyIndex or btreePlusIndex options should be set.'

]

{ #category : 'Errors' }
GsIndexOptions >> _errorOptimizedComparisonAndLegacyIndexSet [

  InternalError signal: 'Invalid GsIndexOptions instance: optimizedComparison should never be set when legacyIndex is set'

]

{ #category : 'Private' }
GsIndexOptions >> _legacyIndex [
  ^ self dynamicInstVarAt: #'legacyIndex'

]

{ #category : 'Private' }
GsIndexOptions >> _optimizedComparison [
  ^ self dynamicInstVarAt: #'optimizedComparison'

]

{ #category : 'Private' }
GsIndexOptions >> _optionalPathTerms [
  ^ self dynamicInstVarAt: #'optionalPathTerms'

]

{ #category : 'Private' }
GsIndexOptions >> _reducedConflict [
  ^ self dynamicInstVarAt: #'reducedConflict'

]

{ #category : 'Private' }
GsIndexOptions >> _reify [
  "Ensure that all slots are set or cleared , no nil slots ... nil slots should be set
   consistent with current default options"

  | default |
  default := self defaultOptions.
  self _reducedConflict ifNil: [ self reducedConflict: default reducedConflict ].
  self _optionalPathTerms ifNil: [ self optionalPathTerms: default optionalPathTerms ].
  self _legacyIndex ifNil: [ self legacyIndex: default legacyIndex ].
  self _btreePlusIndex ifNil: [ self btreePlusIndex: default btreePlusIndex ].
  self _optimizedComparison ifNil: [
    self legacyIndex
      ifTrue: [ self optimizedComparison: false ]
      ifFalse: [ self optimizedComparison: default optimizedComparison ] ].
 self legacyIndex
   ifTrue: [
     self optimizedComparison ifTrue: [ self _errorOptimizedComparisonAndLegacyIndexSet ].
     self btreePlusIndex ifTrue: [ self _errorIncorrectIndexTypeOptionsCombo ]].
 self btreePlusIndex
   ifTrue: [ self legacyIndex ifTrue: [ self _errorIncorrectIndexTypeOptionsCombo ]].


]

{ #category : 'Comparison' }
GsIndexOptions >> = aGsIndexOptions [
  (aGsIndexOptions isKindOf: self class)
    ifFalse: [ ^ false ].
  ^ self reducedConflict = aGsIndexOptions reducedConflict
    and: [ self optionalPathTerms = aGsIndexOptions optionalPathTerms
        and: [ self legacyIndex = aGsIndexOptions legacyIndex
            and: [ self optimizedComparison = aGsIndexOptions optimizedComparison ] ] ]

]

{ #category : 'Accessing' }
GsIndexOptions >> btreePlusIndex [
  ^ self _btreePlusIndex ifNil: [ self defaultOptions btreePlusIndex ]

]

{ #category : 'Private' }
GsIndexOptions >> btreePlusIndex: aBool [
  "Private method ... should never be called by method external to GsIndexOptions.
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'btreePlusIndex' put: aBool.

]

{ #category : 'Operators' }
GsIndexOptions >> clearBtreePlusIndex [
  self btreePlusIndex: false.
  self _legacyIndex == false
    ifTrue: [ self setLegacyIndex ]

]

{ #category : 'Operators' }
GsIndexOptions >> clearLegacyIndex [
  self legacyIndex: false.
  self _btreePlusIndex == false
    ifTrue: [ self setBtreePlusIndex ]

]

{ #category : 'Operators' }
GsIndexOptions >> clearOptimizedComparison [
  self optimizedComparison: false

]

{ #category : 'Operators' }
GsIndexOptions >> clearOptionalPathTerms [
  self optionalPathTerms: false

]

{ #category : 'Operators' }
GsIndexOptions >> clearReducedConflict [
  self reducedConflict: false

]

{ #category : 'Operators' }
GsIndexOptions >> combineWith: aGsIndexOptions [
  "all set/cleared option values in receiver set/cleared in <aGsIndexOptions>"

  self _reducedConflict == true
    ifTrue: [ aGsIndexOptions setReducedConflict ].
  self _reducedConflict == false
    ifTrue: [ aGsIndexOptions clearReducedConflict ].
  self _optionalPathTerms == true
    ifTrue: [ aGsIndexOptions setOptionalPathTerms ].
  self _optionalPathTerms == false
    ifTrue: [ aGsIndexOptions clearOptionalPathTerms ].
  self _legacyIndex == true
    ifTrue: [ aGsIndexOptions setLegacyIndex ].
  self _legacyIndex == false
    ifTrue: [ aGsIndexOptions clearLegacyIndex ].
  self _btreePlusIndex == true
    ifTrue: [ aGsIndexOptions setBtreePlusIndex ].
  self _btreePlusIndex == false
    ifTrue: [ aGsIndexOptions clearBtreePlusIndex ].
  self _optimizedComparison == true
    ifTrue: [ aGsIndexOptions setOptimizedComparison ].
  self _optimizedComparison == false
    ifTrue: [ aGsIndexOptions clearOptimizedComparison ].
  ^ aGsIndexOptions

]

{ #category : 'Accessing' }
GsIndexOptions >> defaultOptions [

  ^ self class default

]

{ #category : 'Comparison' }
GsIndexOptions >> hash [
  "Returns an Integer hash code for the receiver."

  ^ (((((((self btreePlusIndex hash bitShift: -1)
    bitXor: self reducedConflict hash) bitShift: -1)
    bitXor: self optionalPathTerms hash) bitShift: -1)
    bitXor: self legacyIndex hash) bitShift: -1)
    bitXor: self optimizedComparison hash

]

{ #category : 'Initialize-Release' }
GsIndexOptions >> initialize [
  | default |
  super initialize.
  default := self defaultOptions.
  self btreePlusIndex: default btreePlusIndex.
  self legacyIndex: default legacyIndex.
  self reducedConflict: false.
  self optionalPathTerms: false.
  self optimizedComparison: false

]

{ #category : 'Accessing' }
GsIndexOptions >> legacyIndex [
  ^ self _legacyIndex ifNil: [ self defaultOptions legacyIndex ]

]

{ #category : 'Private' }
GsIndexOptions >> legacyIndex: aBool [
  "Private method ... should never be called by method external to GsIndexOptions.
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'legacyIndex' put: aBool.

]

{ #category : 'Operators' }
GsIndexOptions >> negateWith: aGsIndexOptions [
  "all set/cleared option values in receiver cleared/set in <aGsIndexOptions>"

  self _reducedConflict == true
    ifTrue: [ aGsIndexOptions clearReducedConflict ].
  self _reducedConflict == false
    ifTrue: [ aGsIndexOptions setReducedConflict ].
  self _optionalPathTerms == true
    ifTrue: [ aGsIndexOptions clearOptionalPathTerms ].
  self _optionalPathTerms == false
    ifTrue: [ aGsIndexOptions setOptionalPathTerms ].
  self _legacyIndex == true
    ifTrue: [
      aGsIndexOptions clearLegacyIndex.
      aGsIndexOptions setBtreePlusIndex ].
  self _legacyIndex == false
    ifTrue: [
      aGsIndexOptions setLegacyIndex.
      aGsIndexOptions clearBtreePlusIndex ].
  self _btreePlusIndex == true
    ifTrue: [
      aGsIndexOptions clearBtreePlusIndex.
      aGsIndexOptions setLegacyIndex].
  self _btreePlusIndex == false
    ifTrue: [
      aGsIndexOptions setBtreePlusIndex.
      aGsIndexOptions clearLegacyIndex].
  self _optimizedComparison == true
    ifTrue: [ aGsIndexOptions clearOptimizedComparison ].
  self _optimizedComparison == false
    ifTrue: [ aGsIndexOptions setOptimizedComparison ].
  ^ aGsIndexOptions

]

{ #category : 'Accessing' }
GsIndexOptions >> optimizedComparison [
  ^ self _optimizedComparison
    ifNil: [
    self _legacyIndex == true
      ifTrue: [ false ]
      ifFalse: [ self defaultOptions optimizedComparison ] ]

]

{ #category : 'Private' }
GsIndexOptions >> optimizedComparison: aBool [
  "Private method ... should never be called by method external to GsIndexOptions.
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'optimizedComparison' put: aBool

]

{ #category : 'Accessing' }
GsIndexOptions >> optionalPathTerms [
  ^ self _optionalPathTerms ifNil: [ self defaultOptions optionalPathTerms ]

]

{ #category : 'Private' }
GsIndexOptions >> optionalPathTerms: aBool [
  "Private method ... should never be called by method external to GsIndexOptions.
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'optionalPathTerms' put: aBool

]

{ #category : 'Printing' }
GsIndexOptions >> printOn: aStream [
  self legacyIndex
    ifTrue: [
      self btreePlusIndex
        ifTrue: [
          "should never get here"
          self _errorIncorrectIndexTypeOptionsCombo ].
      aStream nextPutAll: 'GsIndexOptions legacyIndex'.
      self optimizedComparison
        ifTrue: [
          "should never get here"
           self _errorOptimizedComparisonAndLegacyIndexSet ].
      self optionalPathTerms
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions optionalPathTerms' ].
      self reducedConflict
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions reducedConflict' ] ]
    ifFalse: [
      self btreePlusIndex
        ifFalse: [
          "should never get here"
          self _errorIncorrectIndexTypeOptionsCombo ].
      aStream nextPutAll: 'GsIndexOptions btreePlusIndex'.
      self optimizedComparison
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions optimizedComparison'].
      self optionalPathTerms
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions optionalPathTerms' ].
      self reducedConflict
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions reducedConflict' ] ]

]

{ #category : 'Accessing' }
GsIndexOptions >> reducedConflict [
  ^ self _reducedConflict ifNil: [ self defaultOptions reducedConflict ]

]

{ #category : 'Private' }
GsIndexOptions >> reducedConflict: aBool [
  "Private method ... should never be called by method external to GsIndexOptions.
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'reducedConflict' put: aBool

]

{ #category : 'Operators' }
GsIndexOptions >> setBtreePlusIndex [
  self btreePlusIndex: true.
  self legacyIndex
    ifTrue: [ self clearLegacyIndex ]

]

{ #category : 'Operators' }
GsIndexOptions >> setLegacyIndex [
  self legacyIndex: true.
  self btreePlusIndex
    ifTrue: [ self clearBtreePlusIndex ].
  self optimizedComparison
    ifTrue: [ self clearOptimizedComparison ]

]

{ #category : 'Operators' }
GsIndexOptions >> setOptimizedComparison [
  self legacyIndex
    ifTrue: [
      "do not set optimizedComparison if legacyIndex already set"
      ^ self ].
  self optimizedComparison: true

]

{ #category : 'Operators' }
GsIndexOptions >> setOptionalPathTerms [
  self optionalPathTerms: true

]

{ #category : 'Operators' }
GsIndexOptions >> setReducedConflict [
  self reducedConflict: true

]
