"
GsExternalSession is a legacy implementation of AbstractExternalSession
that loads $GEMSTONE/lib/librpc*.so  and uses FFI to call Gci functions in
that libgcirpc library.

New code should use GsTsExternalSession.

GsExternalSession is not green-thread-safe, so use of more than one instance 
of GsExternalSession in a session with multiple GsProcesses active can be 
unreliable.

Objects returned by remote session executions can be any type of object. 
 - It is recommended to return specials if possible.
 - Return values that are Unicode7 or String are automatically.
   converted to a string in the local session.
 - Other kinds of byte objects are returned as ByteArrays.
 - All other objects are returned as an Array containing the OOP of the result.
The oops of returned values are added to the remote session's export set, and 
must be manually removed if the remote session has memory pressure. 

GsExternalSession is not supported on AIX; on AIX, use GsLegacyExternalSession.

Example:

| sess |
sess := GsExternalSession
	gemNRS: GsNetworkResourceString defaultGemNRSFromCurrent
	stoneNRS: GsNetworkResourceString defaultStoneNRSFromCurrent
	username: 'DataCurator'
	password: 'swordfish'.
sess login.
sess executeString: 'System stoneName'.
sess executeBlock: [2 + 5].
sess logout.
"
Class {
	#name : 'GsExternalSession',
	#superclass : 'AbstractExternalSession',
	#instVars : [
		'parameters',
		'gciSessionId',
		'stoneSessionId',
		'stoneSessionSerial',
		'gemProcessId',
		'gciErrSType',
		'nbResult',
		'logger'
	],
	#category : 'ExternalSessions'
}

{ #category : 'GciLibrary' }
GsExternalSession class >> gciErrSTypeClass [

	^GciErrSType.

]

{ #category : 'GciLibrary' }
GsExternalSession class >> gciLibrary [

	^SessionTemps current at: #GsExternalSession_gciLibrary
		ifAbsentPut: [
			GciLibrary new
				GciInit;
				yourself.
		].

]

{ #category : 'Instance Creation' }
GsExternalSession class >> gemNRS: gemNRS stoneNRS: stoneNRS username: aUsername password: aPassword [
  | res |
	(res := self new)
		gemNRS: gemNRS;
		stoneNRS: stoneNRS;
		username: aUsername;
		password: aPassword.
  res class gciLibrary .
  ^ res .

]

{ #category : 'Instance Creation' }
GsExternalSession class >> gemNRS: gemNRS stoneNRS: stoneNRS username: gsUsername password: gsPassword hostUsername: hostUsername hostPassword: hostPassword [
  | res |
	(res := self new)
		gemNRS: gemNRS;
		stoneNRS: stoneNRS;
		username: gsUsername;
		password: gsPassword;
		hostUsername: hostUsername;
		hostPassword: hostPassword .
  res class gciLibrary .
  ^ res

]

{ #category : 'Instance Creation' }
GsExternalSession class >> new [
  | res |
  (res := self sessionClass basicNew)
      initialize .
  res class gciLibrary .
  ^ res
]

{ #category : 'Instance Creation' }
GsExternalSession class >> newDefault [
  "This creates an external session that is set to the user, host, and 
  stone of the current gem with the password 'swordfish'. You may update 
  any of these parameters before login for more complex environments."

	 | res |
	 (res := self new)
		  initializeDefaultResources .
   res class gciLibrary .
   ^ res

]

{ #category : 'Private' }
GsExternalSession class >> sessionClass [
	^(System gemVersionAt: 'osName') = 'AIX'
		ifTrue: [GsLegacyExternalSession]
		ifFalse: [ self "change to GsLegacyExternalSession to not use FFI"]

]

{ #category : 'Private' }
GsExternalSession >> _describe [
  | str |
	(str := 'stone session ID ' copy )
     add: stoneSessionId asString ;
     add: ' gem processId '; add: gemProcessId asString .
  self _gemHost ifNotNil:[ :host |
     str add: ' on host ' , host asString.
  ].
  str add: ' stone serialNumber ' ; add: stoneSessionSerial asString.
  ^ str

]

{ #category : 'Private' }
GsExternalSession >> _errorIfCallInProgress [

	self isCallInProgress ifTrue: [self error: 'call in progress'].

]

{ #category : 'Private' }
GsExternalSession >> _errorIfCallInProgress: lib [

	(self _isCallInProgress: lib) ifTrue: [self error: 'call in progress'].

]

{ #category : 'Private' }
GsExternalSession >> _gciLibrary [
  "This entry in SessionTemps initialized by instance creation paths."
	^ SessionTemps current at: #GsExternalSession_gciLibrary

]

{ #category : 'Private' }
GsExternalSession >> _gemHost [
  ^ self dynamicInstVarAt: #gemHost

]

{ #category : 'Private' }
GsExternalSession >> _getBytes: anOop [

	| size bytes fetchedSize classOop |
	classOop := self _gciLibrary GciFetchClass_: anOop.
	self _signalIfError.
	size := self _gciLibrary GciFetchSize__: anOop.
	self _signalIfError.
	bytes := CByteArray gcMalloc: size.
	fetchedSize := self _gciLibrary
		GciFetchBytes__: anOop
		_: 1
		_: bytes
		_: bytes size.
	self _signalIfError.
	fetchedSize ~~ size ifTrue: [self error: 'Unexpected size!'].
	classOop == String asOop ifTrue: [
    size == 0 ifTrue:[ ^ String new ].
		^bytes stringFrom: 0 to: size - 1.
	].
	classOop == Symbol asOop ifTrue: [
    size == 0 ifTrue:[ ^ #'' ].
		^ Symbol withAll: (bytes stringFrom: 0 to: size - 1).
	].
  size == 0 ifTrue:[ ^ ByteArray new ].
	^bytes byteArrayFrom: 0 to: size - 1.

]

{ #category : 'Private' }
GsExternalSession >> _getBytes: anOop lib: lib [
	| size bytes numRet classOop |
	classOop := lib GciFetchClass_: anOop.
  classOop == 20"nil asOop" ifTrue:[
	  self _signalIfError: lib .
  ].
	size := lib GciFetchSize__: anOop.
  size == 0 ifTrue:[
    self _signalIfError: lib
  ].
	bytes := CByteArray gcMalloc: size.
	numRet := lib GciFetchBytes__: anOop _: 1 _: bytes _: bytes size.
  numRet == 0 ifTrue:[
	  self _signalIfError: lib .
  ].
	numRet ~~ size ifTrue: [self error: 'Unexpected size!'].
	classOop == 74753"String asOop" ifTrue: [
    size == 0 ifTrue:[ ^ String new ].
		^ bytes stringFrom: 0 to: size - 1.
	].
  classOop == 154369"Unicode7 asOop" ifTrue:[  "fix 51160"
    size == 0 ifTrue:[ ^ Unicode7 new ]. 
    ^ bytes _copyFrom: 0 to: size - 1 resKind: Unicode7 .
  ].
  classOop == 154113"Utf8 asOop" ifTrue:[ | unicodeMode |
    unicodeMode := Unicode16 usingUnicodeCompares .
    size == 0 ifTrue:[ ^ unicodeMode ifTrue:[ Unicode7 new] ifFalse:[String new]]. "fix 49669"
    ^ bytes decodeUTF8from: 0 to: size - 1 unicode: unicodeMode .
  ].
	classOop == 110849"Symbol asOop" ifTrue: [
    size == 0 ifTrue:[ ^ #'' ].
		^ Symbol withAll: (bytes stringFrom: 0 to: size - 1).
	].
  size == 0 ifTrue:[ ^ ByteArray new ].
	^bytes byteArrayFrom: 0 to: size - 1.

]

{ #category : 'Private' }
GsExternalSession >> _getStackForOop: gcierrContextOop [
  | str start cByteArray |
  nbResult := nil .
  str := 'AbstractExternalSession _stackReport: ' , gcierrContextOop asString .
  self forkString: str .
  start := System timeGmt .
  cByteArray := CByteArray gcMalloc: 8 .
  [ | result lib |
    lib := self _gciLibrary .
    (self _isCallInProgress: lib) ifFalse:[ ^ 'NO STACK, ERROR no call in progress'].
    result := lib GciNbEnd_: cByteArray.
    result >= 2 ifTrue:[
      (self _gciLibrary GciErr_: gciErrSType) == 1 ifTrue:[
        (gciErrSType number  between: 4000 and: 4999) ifTrue: [gciSessionId := 0].
        ^ self gciErrorClass new _error: gciErrSType in: self .
      ].
      nbResult := (cByteArray pointerAt: 0 resultClass: CByteArray) uint64At: 0.
      ^ self _lastResult: lib .
    ].
    Delay waitForMilliseconds: 20 .
    (System timeGmt - start) > 20 ifTrue:[ ^ 'NO STACK, getStack timedout'].
  ] repeat

]

{ #category : 'Private' }
GsExternalSession >> _isCallInProgress: lib [
  | result |
	self _setSessionId: lib .
	result := lib GciCallInProgress. "no Gci error possible"
	^ result == 1

]

{ #category : 'Private' }
GsExternalSession >> _isOnMyHost [
  (self dynamicInstVarAt: #_isOnMyHost) ifNotNil:[ :x | ^ x ].
  ^ false

]

{ #category : 'Private' }
GsExternalSession >> _isOnMyStone [
  | val |
  (val := self dynamicInstVarAt: #_isOnMyStone) ifNil:[
    (GsSession currentSession isSolo) ifTrue:[
      val := false .
      self dynamicInstVarAt: #_isOnMyStone put: val .
    ].
  ].
  ^ val

]

{ #category : 'Private' }
GsExternalSession >> _isResultAvailable: cByteArray [
  "cByteArray is allocated by the caller to avoid a gcMalloc: in each
   invocation of this method."
	| result lib |
	nbResult := nil.
  lib := self _gciLibrary .
	(self _isCallInProgress: lib) ifFalse: [self error: 'no call in progress'].
	result := lib GciNbEnd_: cByteArray.
	result < 2 ifTrue: [ ^ false].
	self _signalIfError: lib .
	nbResult := (cByteArray pointerAt: 0 resultClass: CByteArray) uint64At: 0.
	^ true.

]

{ #category : 'Private' }
GsExternalSession >> _lastResult: lib [
  ^ nbResult ifNotNil:[:r |
     "r is an Integer, the value of a remote OopType"
     self _resolveResult: r lib: lib
  ]

]

{ #category : 'Private' }
GsExternalSession >> _logout [
	| descr |
	descr := self _describe.
	self isLoggedIn ifTrue:[
		self _nbLogout: descr .
		self _waitForLogout.
		gciSessionId := 0.
	].
	stoneSessionId := nil.
	stoneSessionSerial := nil.
	((self dynamicInstVarAt: #quiet) ifNil:[ 0 ]) < 1 ifTrue:[
		self log: 'GsExternalSession logout: ' , descr.
	].

]

{ #category : 'Private' }
GsExternalSession >> _nbLogout [
  | descr |
  descr := self _describe.
	((self dynamicInstVarAt: #quiet) ifNil:[ 0 ]) < 1 ifTrue:[
	  self log: 'GsExternalSession nbLogout: ' , descr.
  ].
  ^ self _nbLogout: descr

]

{ #category : 'Private' }
GsExternalSession >> _nbLogout: descr [
  self isLoggedIn ifTrue:[ | lib |
    lib := self _gciLibrary .
		self _setSessionId: lib .
		lib GciLogout__: 0 .
		[ [ self _signalIfError: lib
			] onException: self gciErrorClass do:[:ex |
				 ex originalNumber == 4100
					 ifTrue:[ "ignore invalid session error from GciError"]
					ifFalse:[ ex pass ].
			].
		] onException: Error do: [:ex | | msg |
			(msg := '---(During GsExternalSession logout:') , descr lf .
			msg := msg , '    ', ex description , ')---'.
			self log: msg lf .
		].
  ].

]

{ #category : 'Private' }
GsExternalSession >> _postLogin: lib [
	| onMyStn onMyHost gemHostIdStr oopSystem fd sock |
  oopSystem := 76033 "System asOop".
	stoneSessionId := Object _objectForOop: (lib GciPerform_: oopSystem _: 'session' _: nil _: 0).
	self _signalIfError: lib .
  gemProcessId := Object _objectForOop: (lib GciPerform_: oopSystem _: 'gemProcessId' _: nil _: 0).
	self _signalIfError: lib .

  gemHostIdStr := self executeString:'System hostId asString' .
  onMyHost :=   gemHostIdStr = System hostId asString .

  GsSession isSolo ifTrue:[
    onMyStn := false .
  ] ifFalse:[ | stoneStartupStr |
    stoneStartupStr := self executeString:'System stoneStartupId asString'.
    onMyStn :=  stoneStartupStr = System stoneStartupId asString .
	  stoneSessionSerial := onMyStn ifTrue:[ GsSession serialOfSession: stoneSessionId ]
                              ifFalse:[ self executeString: 'GsSession currentSession serialNumber'].
  ].
  self dynamicInstVarAt: #_isOnMyStone put: onMyStn .
  self dynamicInstVarAt: #_isOnMyHost put:  onMyHost .
  fd := lib GciNbGetNotifyHandle .
  sock := GsSocket fromFileHandle: fd .
  self dynamicInstVarAt: #_socket put: sock

]

{ #category : 'Private' }
GsExternalSession >> _resolveResult: anOop lib: lib [
	| type |
	self _setSessionId: lib .
	type := lib GciFetchObjImpl_: anOop.
	type == 3 ifTrue: [^Object _objectForOop: anOop]. "result is a special"
	type == 1 ifTrue: [^self _getBytes: anOop lib: lib ].	  "result is a byte object"
  type < 0 ifTrue:[
	  self _signalIfError: lib .
  ].
	^ { anOop } .

]

{ #category : 'Private' }
GsExternalSession >> _setSessionId [

  self isLoggedIn ifFalse:[
     ^ Error signal:'invalid sessionId (session not logged in)'
  ].
  self _gciLibrary GciSetSessionId_: gciSessionId .
  self _signalIfError.

]

{ #category : 'Private' }
GsExternalSession >> _setSessionId: lib [

  self isLoggedIn ifFalse:[
     ^ Error signal:'invalid sessionId (session not logged in)'
  ].
  lib GciSetSessionId_: gciSessionId .
  self _signalIfError: lib .

]

{ #category : 'Private' }
GsExternalSession >> _signalIfError [
	(self _gciLibrary GciErr_: gciErrSType) == 1 ifFalse: [^self].
	(gciErrSType number  between: 4000 and: 4999) ifTrue: [gciSessionId := 0].
	self gciErrorClass 
		signal: gciErrSType
		in: self

]

{ #category : 'Private' }
GsExternalSession >> _signalIfError: lib [
	(lib GciErr_: gciErrSType) == 1 ifFalse: [^self].
	(gciErrSType number  between: 4000 and: 4999) ifTrue: [gciSessionId := 0].
	self gciErrorClass 
		signal: gciErrSType
		in: self

]

{ #category : 'Private' }
GsExternalSession >> _signalIfError: lib arg: detailString [
	(lib GciErr_: gciErrSType) == 1 ifFalse: [^self].
	(gciErrSType number  between: 4000 and: 4999) ifTrue: [gciSessionId := 0].
	self gciErrorClass 
		signal: gciErrSType
		in: self details: detailString

]

{ #category : 'Private' }
GsExternalSession >> _waitForLogout [
  | onMyStn onMyHost |
	stoneSessionId ifNil:[ ^ self].
  onMyStn := self _isOnMyStone .
  onMyHost := self _isOnMyHost .
  (onMyStn or:[ onMyHost]) ifFalse:[  ^ self "no way to wait reliably"].
	1 to: 2000 do:[ :j |
		onMyStn ifTrue: [
			(GsSession serialOfSession: stoneSessionId) = stoneSessionSerial ifFalse:[
         ^self
      ].
		].
    onMyHost ifTrue:[
      (System _hostProcessExists: gemProcessId) ifFalse:[
        ^ self
      ].
    ].
		(Delay forMilliseconds: 10) wait.
	].
  self error: 'session with stone session ID of ' , stoneSessionId printString ,
              ' gemProcessId = ' , gemProcessId printString,
              ' still present 20 seconds after logout'.

]

{ #category : 'Public' }
GsExternalSession >> abort [
  "Abort the current transaction in the external Gem."

  | lib |
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciNbAbort.
	self waitForResult .
	nbResult ~~ 0 ifTrue: [self error: 'Unexpected result!'].

]

{ #category : 'Public' }
GsExternalSession >> begin [
  "Abort current transaction and begin a transaction"
	self _errorIfCallInProgress.
	self _gciLibrary GciNbBegin .
	self waitForResult.
	nbResult ~~ 0 ifTrue: [self error: 'Unexpected result!'].

]

{ #category : 'Error handling' }
GsExternalSession >> clearStackFor: anError [

	| contextOop lib |
	(contextOop := anError context) == 20"nil asOop" ifTrue: [^self].
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciClearStack_: contextOop.
	self _signalIfError: lib .

]

{ #category : 'Public' }
GsExternalSession >> commit [
  "Commit the current transaction in the external Gem."

  | result lib |
  lib := self _gciLibrary .
  self _errorIfCallInProgress: lib .
  result := lib GciCommit.
  result == 1 ifFalse:[
      self _signalIfError: lib .
  ].
  ^ result == 1.

]

{ #category : 'Public' }
GsExternalSession >> commitOrError [
   "Commit the current transaction in the external Gem and return true 
   if success or signal an error."

  | result |
  self _errorIfCallInProgress.
  result := self _gciLibrary GciCommit.
  self _signalIfError.
  result == 1 ifFalse:[ 
    TransactionError new reason: 'commitConflicts' ; signal:'commit conflicts'
    ].
 ^ result == 1

]

{ #category : 'Error handling' }
GsExternalSession >> continue: contextOop replacingTopOfStackWithOop: valueOop [
	"Continue execution in the external Gem following an exception,
	 replacing the top of the stack with the specified object.
	 It is an error to specify an oop not visible to the remote Gem."
  | lib |
	lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciNbContinueWith_: contextOop
		_: valueOop
		_: 16r10"GCI_PERFORM_noClientUseraction"  
		_: nil.
	^self
		waitForResult;
		_lastResult: lib .

]

{ #category : 'Public' }
GsExternalSession >> executeString: aString [
	"Execute the string expression in the external Gem and answer the result.
	 The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

  | lib |
  aString _isOneByteString ifFalse:[ ArgumentError signal:'arg is not a String' ].
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciNbExecuteStr_: aString _: 20"nil asOop". 
  self waitForResult .
  ^ self _lastResult: lib

]

{ #category : 'Public' }
GsExternalSession >> forceLogout [

	stoneSessionId ifNil: [^self].
	[self hardBreak] onException: Error do: [:ex | ].
	self _logout.

]

{ #category : 'Public' }
GsExternalSession >> forkString: aString [
	"Execute the string expression in the external Gem and do not wait for
	 a result.  At some later point, you would check for a result. Otherwise you cannot
	 issue another call, as the current call would remain in progress.
	 Refer to #executeString: for an example of the complete send, wait, response sequence."
  | lib |
  aString _isOneByteString ifFalse:[ ArgumentError signal:'arg is not a String' ].
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciNbExecuteStr_: aString _: 20"nil asOop".  

]

{ #category : 'Private' }
GsExternalSession >> gciErrorClass [
  ^ GciError 
]

{ #category : 'Parameters' }
GsExternalSession >> gemNRS [
   ^ parameters ifNotNil:[:p | p gemService ].

]

{ #category : 'Parameters' }
GsExternalSession >> gemNRS: anNRS [
	"Set the GemService parameters for the logon to the value
	 of anNRS, which may be a String or a GsNetworkResourceString instance."

	parameters gemService: anNRS asString .
  (anNRS isKindOf: GsNetworkResourceString) ifTrue:[
    self dynamicInstVarAt: #gemHost put: anNRS node .
  ].

]

{ #category : 'Accessors' }
GsExternalSession >> gemProcessId [

  ^gemProcessId.

]

{ #category : 'Public' }
GsExternalSession >> hardBreak [
	"Interrupt the external Gem and abort the current transaction."
  | lib |
  lib := self _gciLibrary .
	self _setSessionId: lib .
	lib GciHardBreak.
	self _signalIfError: lib .

]

{ #category : 'Parameters' }
GsExternalSession >> hostPassword: aString [

	parameters hostPassword: aString copy.

]

{ #category : 'Parameters' }
GsExternalSession >> hostUsername: aString [

	parameters hostUsername: aString copy.

]

{ #category : 'Private' }
GsExternalSession >> initialize [

	gciErrSType := self class gciErrSTypeClass new.
	parameters := GemStoneParameters new.
	self loggingToServer.

]

{ #category : 'Private' }
GsExternalSession >> initializeDefaultResources [
  parameters ifNil:[ parameters := GemStoneParameters new].
	self
		gemNRS: GsNetworkResourceString defaultGemNRSFromCurrent;
		stoneNRS: GsNetworkResourceString defaultStoneNRSFromCurrent;
		username: System myUserProfile userId;
		password: 'swordfish' .
]

{ #category : 'Public' }
GsExternalSession >> isCallInProgress [
	"Answer whether there is currently a call in progress to
	 the external Gem.

	 The following calls are OK during a nonblocking call:
		GciCallInProgress
		GciErr
		GciGetSessionId
		GciHardBreak
		GciNbEnd
		GciSetSessionId
		GciShutdown
		GciSoftBreak"

	self isLoggedIn ifFalse:[ ^ false "not logged in"].
  ^ self _isCallInProgress: self _gciLibrary .

]

{ #category : 'Accessors' }
GsExternalSession >> isLoggedIn [
 ^ gciSessionId ~~ nil and:[ gciSessionId > 0].

]

{ #category : 'Public' }
GsExternalSession >> isRemoteServerBigEndian [

	^self _gciLibrary GciServerIsBigEndian ~~ 0

]

{ #category : 'Public' }
GsExternalSession >> isResultAvailable [

	"Check whether the current call in progress has finished
	 and save the result if it has. Most operations cannot be
	 started while another is in progress. You must call this
	 method and receive a true result before starting another
	 remote operation."

  ^ self _isResultAvailable: (CByteArray gcMalloc: 8)

]

{ #category : 'Public' }
GsExternalSession >> lastResult [
  ^ nbResult ifNotNil:[:r |
     "r is an Integer, the value of a remote OopType"
     self resolveResult: r
  ]

]

{ #category : 'Logging' }
GsExternalSession >> log: aString [

	logger value: aString

]

{ #category : 'Logging' }
GsExternalSession >> logger: aOneArgBlock [
	"Use the specified one-argument Block for logging messages.
	 The argument to the Block is the message to log."

	logger := aOneArgBlock

]

{ #category : 'Logging' }
GsExternalSession >> loggingToClient [

	self logger: [:message | GsFile gciLogClient: message]

]

{ #category : 'Logging' }
GsExternalSession >> loggingToServer [

	self logger: [:message | GsFile gciLogServer: message]

]

{ #category : 'Public' }
GsExternalSession >> login [
	| result lib sid |
	stoneSessionId ifNotNil: [
		ImproperOperation signal: 'Stone session ' , stoneSessionId printString ,
			' already associated with this GsExternalSession!'.
	].
  lib := self _gciLibrary .
	result := lib
    GciSetNetEx_: parameters gemStoneName
		_: parameters hostUsername
		_: parameters hostPassword
		_: parameters gemService
		_: parameters passwordIsEncryptedAsIntegerBoolean.	"1 or 0: GCI_LOGIN_PW_ENCRYPTED"
        result == 0 ifTrue:[ self error:'GciSetNetEx_ failed'].
	result := lib
		GciLoginEx_: parameters username
		_: parameters password
		_: parameters loginFlags
		_: -1 . "haltOnErrNum default (use config file)"
  sid := lib GciGetSessionId.
  sid > 0 ifTrue:[ gciSessionId := sid ].
	0 == result ifTrue: [
	  self _signalIfError: lib arg: 'Using ', parameters printString .
	  self error: 'Login failed for unknown reason!'.
	].
  self _postLogin: lib  .
	((self dynamicInstVarAt: #quiet) ifNil:[ 0 ]) < 2 ifTrue:[
	  self log: 'GsExternalSession login: ' , self _describe.
  ].

]

{ #category : 'Public' }
GsExternalSession >> loginSolo [
  "login as a Solo session using  GCI_LOGIN_SOLO flag.
   Requires an appropriate GEM_SOLO_EXTENT value in the config file
   used by the gem process for the new session.
   See GsSession(C)>>isSolo for details of a Solo session."

	stoneSessionId ifNotNil: [
		ImproperOperation signal: 'Stone session ' , stoneSessionId printString ,
			' already associated with this GsExternalSession!'.
	].
  parameters loginFlags: (parameters loginFlags bitOr: parameters soloLoginFlag).
  ^ self login

]

{ #category : 'Public' }
GsExternalSession >> logout [
	stoneSessionId ifNil: [^self].
	self isCallInProgress ifTrue: [
		[
			self waitForResult.
		] onException: Error do: [:ex |
			ex return.
		].
	].
	self _logout.

]

{ #category : 'Private' }
GsExternalSession >> nbLogout [
  "Private.
   Should be followed by a send of _waitForLogout ."

	stoneSessionId ifNil: [^self].
	self isCallInProgress ifTrue: [
		[
			self waitForResultForSeconds: 20 .
		] onException: Error do: [:ex |
			ex return.
		].
	].
	self _nbLogout .

]

{ #category : 'Public' }
GsExternalSession >> nbResult [
  | result lib cByteArray |
  nbResult := nil.
  lib := self _gciLibrary .
  (self _isCallInProgress: lib) ifFalse: [self error: 'no call in progress'].
  cByteArray := CByteArray gcMalloc: 8 .
  result := lib GciNbEnd_: cByteArray .
  result < 2 ifTrue: [ self error:'result not ready'].
  self _signalIfError: lib .
  nbResult := (cByteArray pointerAt: 0 resultClass: CByteArray) uint64At: 0.
  ^ nbResult ifNotNil:[:r |
    "r is an Integer, the value of a remote OopType"
    self resolveResult: r
  ]

]

{ #category : 'Parameters' }
GsExternalSession >> onetimePassword: aString [

   parameters onetimePassword: aString copy
]

{ #category : 'Public' }
GsExternalSession >> parameters [
  "Return the instance of GemStoneParameters"

  ^ parameters

]

{ #category : 'Parameters' }
GsExternalSession >> password: aString [

	parameters password: aString copy.

]

{ #category : 'Public' }
GsExternalSession >> printOn: aStream [

	aStream
		nextPutAll: 'a';
		nextPutAll: self class name;
		nextPutAll: '(';
		nextPutAll: stoneSessionId printString;
		nextPutAll: '/';
		nextPutAll: stoneSessionSerial printString;
		nextPutAll: ')' .

]

{ #category : 'Logging' }
GsExternalSession >> quiet [
  "By default login and logout are logged using GsFile >> gciLogServer: .
   This disables logging of login and logout."
   self dynamicInstVarAt: #quiet put: 2 .

]

{ #category : 'Logging' }
GsExternalSession >> quietLogout [
  "By default login and logout are logged using GsFile >> gciLogServer:.
   This disables logging of logout."
   self dynamicInstVarAt: #quiet put: 1 .

]

{ #category : 'Public' }
GsExternalSession >> resolveResult: anOop [
	"Answer the object, or Array containing the OOP, of the result
	received when the last #isResultAvailable answered true.
	Specials can be fully resolved in the current session, and are the
	preferred return type.
	Results that are byte objects will be copied into a String or ByteArray
	and that will be returned, but the OOP of the byte object will remain
	in the remote Gem's export set.
	For objects of all other types, return a result as an Array containing
	the object's OOP in the remote gem, which is also recorded in the remote
	Gem's export set."

  ^ self _resolveResult: anOop lib: self _gciLibrary .

]

{ #category : 'Public' }
GsExternalSession >> resolveResult: anOop toLevel: anInteger [
	"Similar to resolveResult:, but this recognizes more classes.
	If the class is not recognized, then return a CByteArray
	with the OOP"

	| lib object oop cByteArray |
	lib := self _gciLibrary.
	object := self _resolveResult: anOop lib: lib .
	(object _isArray) ifFalse: [^object].	"a special or a byte object"
	oop := lib GciFetchClass_: anOop.
	(oop == 66817"Array asOop" and: [0 < anInteger]) ifTrue: [
		| size array |
		size := lib GciFetchSize__: anOop.
		array := Array new.
		1 to: size do: [:i |
			oop := lib GciFetchOop_: anOop _: i.
			array add: (self resolveResult: oop toLevel: anInteger - 1).
		].
		^array.
	].
	"Not a recognized object"
	cByteArray := CByteArray gcMalloc: 8.
	cByteArray uint64At: 0 put: anOop.
	^cByteArray.

]

{ #category : 'Public' }
GsExternalSession >> send: selector to: anOop withArguments: someValues [
	"Answer the result of having the specified remote object
	 sent the message with the specified selector and arguments.
	 Argument values are passed by OOP; beware of inconsistent views
	 between the local Gem and the external Gem.
	 The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

	| nArgs args lib |
	lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	(someValues == nil or: [ (nArgs := someValues size) == 0])
		ifFalse: [ | ofs |
        ofs := 0 .
			  args := CByteArray gcMalloc: 8 * nArgs .
			  1 to: nArgs do: [:index | | each |
          each := someValues at: index .
					args uint64At: ofs put: each asOop.
          ofs := ofs + 8 .
        ]
     ].
	lib
		GciNbPerform_: anOop  
		_: selector
		_: args
		_: someValues size.
	^self
		waitForResult;
		_lastResult: lib

]

{ #category : 'Accessors' }
GsExternalSession >> sessionId [
	"0 => not logged in"

	^gciSessionId.

]

{ #category : 'Accessors' }
GsExternalSession >> setGemTrace: anInt [
  "Control tracing of gci commands to the gem process' log file .
   anInt must be >= 0 ,   0 = none, 1 = commands, 2 = commands+args , 3 even more.
   Returns previous value of the tracing control.  
   Additional fields per gemTrace in src/om.hf in debug(slow) VM." 
  | lib |
	lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
  ^ lib GciGemTrace_: anInt
]

{ #category : 'Public' }
GsExternalSession >> softBreak [
	"Interrupt the external Gem, but permit it to be restarted."
  | lib |
  lib := self _gciLibrary .
	self _setSessionId: lib .
	lib GciSoftBreak.
	self _signalIfError: lib .

]

{ #category : 'Parameters' }
GsExternalSession >> stoneNRS: anNRS [
	"Set the Stone parameters for the logon to the value
	 of anNRS, which may be a String or a GsNetworkResourceString instance."

	parameters gemStoneName: anNRS asString

]

{ #category : 'Accessors' }
GsExternalSession >> stoneSessionId [

	^stoneSessionId.

]

{ #category : 'Accessors' }
GsExternalSession >> stoneSessionSerial [

	^stoneSessionSerial.

]

{ #category : 'Logging' }
GsExternalSession >> suppressLogging [

	self logger: [:message | ]

]

{ #category : 'Parameters' }
GsExternalSession >> username [
  ^ parameters username

]

{ #category : 'Parameters' }
GsExternalSession >> username: aString [

	parameters username: aString.

]

{ #category : 'Public' }
GsExternalSession >> waitForReadReady [
  "Use the ProcessorScheduler to wait for this session's socket to
   be ready to read, allowing other GsProcess to run while we are waiting."
  (self dynamicInstVarAt: #_socket) _waitForReadReady

]

{ #category : 'Public' }
GsExternalSession >> waitForResult [
	"Wait as long as it takes for the external Gem to complete
	 the current operation.
   Does not allow other GsProcess to run. "

	self waitForResultForSeconds: 1000000000000

]

{ #category : 'Public' }
GsExternalSession >> waitForResultForSeconds: aNumber [
	"Wait as long as the specified seconds for the external Gem
	 to complete the current operation.
   Does not allow other GsProcess to run. "

	self
		waitForResultForSeconds: aNumber
		otherwise: [self error:
			'Wait time of ' , aNumber printString ,
			' exceeded for session ' , stoneSessionId printString , '/' , stoneSessionSerial printString ,
			' (PID ' , gemProcessId printString , ')']

]

{ #category : 'Public' }
GsExternalSession >> waitForResultForSeconds: aNumber otherwise: aBlock [
	"Wait as long as the specified seconds for the external Gem
	 to complete the current operation. If the operation does not
	 complete within that time, answer the result of evaluating aBlock.
   Does not allow other GsProcess to run. "

	| cByteArray lib res msLeft |
  lib := self _gciLibrary .
  cByteArray := CByteArray gcMalloc: 8 .
  msLeft := aNumber asInteger * 1000 .
  [ msLeft > 0 ] whileTrue:[ | tMs |
    tMs := msLeft min: 2000000000 .
    self _setSessionId: lib .
    res := lib GciNbEndPoll_: cByteArray _: tMs .
    res >= 2 ifTrue:[
      self _signalIfError: lib .
      nbResult := cByteArray uint64At: 0 .
      ^ self
    ].
    msLeft := msLeft - tMs .
  ].
  ^ aBlock value

]
