Extension { #name : 'GsEventLog' }

{ #category : 'Private' }
GsEventLog class >> _logSystemInfoEventFor: subsystem label: aString object: obj [

	GsEventLogSystemEntry info
		subsystem: subsystem;
		label: aString;
		object: obj;
		_addSystemEntryToLog.

]

{ #category : 'Instance Creation' }
GsEventLog class >> _newWithEntries: anRcArray [

	^self new
		_entryArray: anRcArray;
		yourself

]

{ #category : 'Instance Creation' }
GsEventLog class >> current [
  ^ CurrentLog ifNil: [
     		CurrentLog := self new .
     		CurrentLog.
 		].

]

{ #category : 'Immutability' }
GsEventLog class >> entriesModifiable [
	self current entriesModifiable

]

{ #category : 'Immutability' }
GsEventLog class >> entriesUnmodifiable [
	self current entriesUnmodifiable

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logDebug [

	^self current addEntry: GsEventLogEntry debug

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logDebug: aString [

	^self current addEntry:
		(GsEventLogEntry debug
			label: aString;
			yourself)

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logError [

	^self current addEntry: GsEventLogEntry error

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logError: aString [

	^self current addEntry:
		(GsEventLogEntry error
			label: aString;
			yourself)

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logFatal [

	^self current addEntry: GsEventLogEntry fatal

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logFatal: aString [

	^self current addEntry:
		(GsEventLogEntry fatal
			label: aString;
			yourself)

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logInfo [

	^self current addEntry: GsEventLogEntry info

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logInfo: aString [

	^self current addEntry:
		(GsEventLogEntry info
			label: aString;
			yourself)

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logObject: anObject [

	^GsEventLogEntry trace
		object: anObject;
		label: anObject printString;
		addToLog;
		yourself.

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logTrace [

	^self current addEntry: GsEventLogEntry trace

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logTrace: aString [

	^self current addEntry:
		(GsEventLogEntry trace
			label: aString;
			yourself)

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logWarning [

	^self current addEntry: GsEventLogEntry warning

]

{ #category : 'Add Log Entries' }
GsEventLog class >> logWarning: aString [

	^self current addEntry:
		(GsEventLogEntry warning
			label: aString;
			yourself)

]

{ #category : 'Instance Creation' }
GsEventLog class >> new [
	^super new initialize


]

{ #category : 'Actions' }
GsEventLog >> _addSystemEntry: anEntry [

	self _entryArray add: anEntry.
	anEntry makeUnmodifiable.

]

{ #category : 'Private-Deleting system entries' }
GsEventLog >> _basicDeleteAllEntries [
	"Delete all entries in the recevier, including system events and events that are
	configured to not be deleted. "

	self _entryArray size: 0.

]

{ #category : 'Private-Deleting system entries' }
GsEventLog >> _basicDeleteEntry: anEvent [
	"Remove the given event from the receiver"

	self _entryArray remove: anEvent ifAbsent: [].

]

{ #category : 'Private-Deleting system entries' }
GsEventLog >> _deleteEntriesIncludingSystem: anotherEventLog [
	"delete all entries in the argument from the receiver, including system events
	and events marked do not delete."

	self _entryArray == anotherEventLog _entryArray ifTrue: [
		"remove all"
		^self _basicDeleteAllEntries].

	anotherEventLog _entryArray do: [:anEvent | self _basicDeleteEntry: anEvent]

]

{ #category : 'Accessing' }
GsEventLog >> _entryArray [
	^entryArray

]

{ #category : 'Updating' }
GsEventLog >> _entryArray: newValue [
	entryArray := newValue

]

{ #category : 'Accessing' }
GsEventLog >> _newEntriesInvariant [
	^newEntriesInvariant

]

{ #category : 'Updating' }
GsEventLog >> _newEntriesInvariant: newValue [
	newEntriesInvariant := newValue

]

{ #category : 'Deleting' }
GsEventLog >> acquireGsEventLogLock [
	"The lock is not needed for add, only for delete"

	| arr |
	arr := self _entryArray.
	System writeLock: arr
		ifDenied: [ ^false ]
		ifChanged: [
			System addToCommitOrAbortReleaseLocksSet: arr.
			^false ].
	System addToCommitOrAbortReleaseLocksSet: arr.
	^true

]

{ #category : 'Actions' }
GsEventLog >> addEntry: anEntry [

	self _entryArray add: anEntry.
	self _newEntriesInvariant ifTrue: [anEntry makeUnmodifiable].
	^anEntry

]

{ #category : 'Deleting' }
GsEventLog >> deleteAllEntries [
	"Delete all entries in the recevier, except for system events or events that cannot be deleted. "

	self _entryArray removeAllSuchThat: [:entry | entry canBeDeleted]

]

{ #category : 'Deleting' }
GsEventLog >> deleteEntries: anotherEventLog [
	"delete all entries in the argument from the receiver.
	Quiety ignore entries that are not present or are that cannot be deleted.
	This method does not delete system events."

	self _entryArray == anotherEventLog _entryArray ifTrue: [
		"remove all"
		^self deleteAllEntries].

	anotherEventLog _entryArray do: [:anEvent | self deleteEntry: anEvent]

]

{ #category : 'Deleting' }
GsEventLog >> deleteEntry: anEvent [
	"Remove the given event from the receiver, if allowed"

	anEvent canBeDeleted
		ifTrue: [self _basicDeleteEntry: anEvent].

]

{ #category : 'Queries' }
GsEventLog >> entriesDo: aBlock [
	"Execute aBlock for each entry"

	self _entryArray do: aBlock.

]

{ #category : 'Actions' }
GsEventLog >> entriesModifiable [
	self _newEntriesInvariant: false

]

{ #category : 'Queries' }
GsEventLog >> entriesSatisfying: aBlock [
	"return a new instance of the receiver containing the entries from the receiver that satisfy the block"

	| results |
	results := self _entryArray select: aBlock.
	^self class _newWithEntries: results

]

{ #category : 'Actions' }
GsEventLog >> entriesUnmodifiable [
	self _newEntriesInvariant: true

]

{ #category : 'Initialization' }
GsEventLog >> initialize [
	self objectSecurityPolicy: nil.
	self _entryArray:
		(RcArray new
			objectSecurityPolicy: nil;
			yourself).
	self _newEntriesInvariant: false.

]

{ #category : 'Queries' }
GsEventLog >> last [
	^entryArray last

]

{ #category : 'Reporting' }
GsEventLog >> report [
	| str |
	str := AppendStream on: String new.
	self _entryArray do: [:entry | entry printOn: str.  str lf].
	^str contents

]

{ #category : 'Queries' }
GsEventLog >> size [
	^entryArray size

]

{ #category : 'Queries' }
GsEventLog >> systemEvents [
	^self _entryArray entriesSatisfying: [:ev | ev isSystemEvent]

]

{ #category : 'Queries' }
GsEventLog >> userEvents [
	^self entriesSatisfying: [:ev | ev isUserEvent]

]
