Extension { #name : 'GsConstrainedCharacterCollectionIndex' }

{ #category : 'instance creation' }
GsConstrainedCharacterCollectionIndex class >> newWithCollator: anIcuCollatorOrNil [
  "Create a new instance and initialize its B-tree root."

  | newOne |
  newOne := super new.
  newOne collator: anIcuCollatorOrNil.
  ^ newOne

]

{ #category : 'instance creation' }
GsConstrainedCharacterCollectionIndex class >> newWithLastElementClass: aClass [
  self shouldNotImplement: #'newWithLastElementClass:'

]

{ #category : 'Testing' }
GsConstrainedCharacterCollectionIndex >> _canCompareWith: aKey [
  "Returns whether the receiver can make B-tree comparisons with the given key."

  (self constraintType == #string or: [ self constraintType == #unicodeString ])
    ifTrue: [
      | keyClass |
      keyClass := aKey class.
      (aKey _isSymbol not
        and: [ (keyClass _subclassOf: String)
          or: [ (keyClass _subclassOf: DoubleByteString)
              or: [ keyClass _subclassOf: QuadByteString ] ] ])
        ifFalse: [ ^ false ].
      (CharacterCollection isInUnicodeComparisonMode or: [ self constraintType == #unicodeString ])
        ifTrue: [ ^ true ]
        ifFalse: [
          "self constraintType == #string and unicode comparison mode DISABLED"
          ^ ((keyClass == Unicode7)
            or: [ (keyClass == Unicode16)
              or: [ keyClass == Unicode32 ] ]) not ] ]
    ifFalse: [
      "self constraintType == #symbol"
      ^ aKey _isSymbol ]

]

{ #category : 'Updating' }
GsConstrainedCharacterCollectionIndex >> _setLastPathTermState [
  "Last object along the path unconditionally needs a dependency list."

  | lastPathTerm |
  lastPathTerm := self at: self size.
  lastPathTerm needsDepList: true

]

{ #category : 'conversion' }
GsConstrainedCharacterCollectionIndex >> asIndexSpecification [
  ^ (ConstrainedCharacterCollectionIndexSpecification
    path: self pathComponentsString
    collator: self collator)
    options: self options;
    constraintType: self constraintType;
    yourself

]

{ #category : 'Accessing' }
GsConstrainedCharacterCollectionIndex >> collator [
  "Returns IcuCollator to be used when comparing Unicode strings. collator may be nil."

  ^ collator

]

{ #category : 'Accessing' }
GsConstrainedCharacterCollectionIndex >> collator: anIcuCollatorOrNil [
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow
   changes to strength, etc, that might affect the sort ordering. "

  anIcuCollatorOrNil
    ifNil: [ collator := nil ]
    ifNotNil: [ collator := anIcuCollatorOrNil copy immediateInvariant ]

]

{ #category : 'Comparison Operators' }
GsConstrainedCharacterCollectionIndex >> comparisonForCompare [
  collator
    ifNil: [ ^ super comparisonForCompare ].
  ^ BtreeOptimizedUnicodeComparison new
      collator: self collator;
      yourself

]

{ #category : 'Comparison Operators' }
GsConstrainedCharacterCollectionIndex >> comparisonForSort [
  "called during creation of receiver, so initialize unset instance variables"

  ^ comparisonForSort
      ifNil: [
        (self constraintType == #symbol and: [ collator isNil ])
          ifTrue: [
            "use unicode collation for Symbols, so that Symbols with NUL bytes handled
             correctly. See bug 47116."
            self collator: IcuCollator default ].
        collator
          ifNil: [ comparisonForSort := self comparisonForCompare ]
          ifNotNil: [
            comparisonForSort := BtreeOptimizedUnicodeComparison new
            collator: self collator;
            yourself ] ]

]

{ #category : 'Accessing' }
GsConstrainedCharacterCollectionIndex >> constraintType [
  "Returns the constraintType, either #string or #symbol."

  ^ constraintType

]

{ #category : 'Accessing' }
GsConstrainedCharacterCollectionIndex >> constraintType: aSymbol [
  "Set the constraintType, either #string or #symbol."

  (aSymbol ~~ #string and: [ aSymbol ~~ #symbol ])
    ifTrue: [ ArgumentError signal: self class name asString, ' constraintType must #string or #symbol.'].
  constraintType := aSymbol

]

{ #category : 'Accessing' }
GsConstrainedCharacterCollectionIndex >> lastElementClass [
  ^ CharacterCollection

]

{ #category : 'Accessing' }
GsConstrainedCharacterCollectionIndex >> lastElementClassDescription [
  "answer a description of the lastElementClass of the receiver,
   suitable for use in an error message"

  ^ self constraintType == #string
    ifTrue: [ 'String, DoubleByeString, or QuadByteString with optimizedComparison' ]
    ifFalse: [
      "self constraintType == #symbol"
      'Symbol, DoubleByeSymbol, or QuadByteSymbol with optimizedComparison' ].

]

{ #category : 'Testing' }
GsConstrainedCharacterCollectionIndex >> optimizingComparison [
  "optimizedComparison option is uncoditionally true"

  ^ true

]
