Extension { #name : 'GemStoneParameters' }

{ #category : 'Instance creation' }
GemStoneParameters class >> new [

"Create a new initialized instance."

^ self basicNew initialize.

]

{ #category : 'Instance creation' }
GemStoneParameters class >> newDefault [
  ^ self new 
     gemService: GsNetworkResourceString defaultGemNRSFromCurrent asString ;
     gemStoneName: GsNetworkResourceString defaultStoneNRSFromCurrent asString ;
     username: System myUserProfile userId ;
     password: 'swordfish';
     hostUsername: System effectiveUserIdName ; 
     yourself

]

{ #category : 'Constants' }
GemStoneParameters class >> onetimePasswordSize [
"OneTimeLoginToken::TOKEN_BASE64_REAL_SIZE"
^ 64

]

{ #category : 'Private' }
GemStoneParameters >> _encrypt: aString [

"Return a new, encrypted string from the given string. ."

^ self _oneArgPrim: 24 with: aString

]

{ #category : 'Error Handling' }
GemStoneParameters >> _errorBadPassword [

"Raise an error because could encrypt and store a password."

self _error: #rtErrPasswordNotVariant args: #()

]

{ #category : 'Private' }
GemStoneParameters >> _oneArgPrim: opcode with: arg [

"Primitive dispatch. Legal values:
    24 GemStoneParameters | _encrypt:
"

<primitive: 198>

^self _primitiveFailed: #_oneArgPrim: args: { opcode . arg }

]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearEncryptedPasswordFlag [

^ self loginFlags: (self loginFlags bitAnd: self passwordEncryptedLoginFlag bitInvert)

]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearJwtLoginFlag [

^ self loginFlags: (self loginFlags bitAnd: self jwtLoginFlag bitInvert)

]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearLoginDebug [
 
"Disables writing debug info to the gem's log file"
^  self loginFlags: (self loginFlags bitAnd: self debugLoginFlag bitInvert)


]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearOnetimePasswordLoginFlag [

^ self loginFlags: (self loginFlags bitAnd: self onetimePasswordLoginFlag bitInvert)

]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearPasswordIsEnvVarLoginFlag [

^ self loginFlags: (self loginFlags bitAnd: self passwordIsEnvVarLoginFlag bitInvert)

]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearPasswordIsFileNameLoginFlag [

^ self loginFlags: (self loginFlags bitAnd: self passwordIsFileNameLoginFlag bitInvert)

]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearPasswordlessLoginFlag [

^ self loginFlags: (self loginFlags bitAnd: self passwordlessLoginFlag bitInvert)

]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearQuietLogin [

^  self loginFlags: (self loginFlags bitAnd: (self quietLoginFlag bitInvert))

]

{ #category : 'Login Flags - Clearing' }
GemStoneParameters >> clearSoloLoginFlag [

^ self loginFlags: (self loginFlags bitAnd: self soloLoginFlag bitInvert)

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> debugLoginFlag [

^ 16r4000

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> defaultLoginFlags [
^ self passwordEncryptedLoginFlag bitOr: self quietLoginFlag

]

{ #category : 'Accessing' }
GemStoneParameters >> gemService [

   "Return the value of the instance variable 'gemService'."
   ^gemService

]

{ #category : 'Updating' }
GemStoneParameters >> gemService: newValue [

   "Modify the value of the instance variable 'gemService'."
   newValue _isOneByteString ifFalse:[ newValue _validateClass: String ].
   gemService := newValue

]

{ #category : 'Accessing' }
GemStoneParameters >> gemStoneName [

  "Return the value of the instance variable 'gemStoneName'."
  ^ gemStoneName

]

{ #category : 'Updating' }
GemStoneParameters >> gemStoneName: aName [

aName ifNotNil:[
  aName _isOneByteString ifFalse:[ aName _validateClass: String ].
].
gemStoneName := aName.

]

{ #category : 'Accessing' }
GemStoneParameters >> hostPassword [

   "Return the value of the instance variable 'hostPassword'."
   ^hostPassword

]

{ #category : 'Updating' }
GemStoneParameters >> hostPassword: pword [

"Set the host password for the receiver.  The password is encrypted.
The given password's size is set to zero."

pword _isOneByteString ifFalse:[ pword _validateClass: String ].
hostPassword := self _encrypt: pword.
hostPassword immediateInvariant.
pword isInvariant ifFalse:[ pword size: 0 ].
^ pword

]

{ #category : 'Accessing' }
GemStoneParameters >> hostUsername [

   "Return the value of the instance variable 'hostUsername'."
   ^hostUsername

]

{ #category : 'Updating' }
GemStoneParameters >> hostUsername: newValue [

   "Modify the value of the instance variable 'hostUsername'."
   newValue _isOneByteString ifFalse:[ newValue _validateClass: String ].
   hostUsername := newValue

]

{ #category : 'Initialization' }
GemStoneParameters >> initialize [

"Initialize the instance variables of the receiver."
  ^ self loginFlags: self defaultLoginFlags

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> jwtLoginFlag [
^ 16r800 "GCI_JWT_PASSWORD"

]

{ #category : 'Updating' }
GemStoneParameters >> jwtPassword: jwt [

"Set the JSON web token password for the receiver. "
jwt _isOneByteString ifFalse:[ jwt _validateClass: String ].
self setLoginWithJwt .
password := jwt copy.
password immediateInvariant.
^ jwt

]

{ #category : 'Accessing' }
GemStoneParameters >> loginFlags [

   "Return the value of the instance variable 'loginFlags'."

^ (self dynamicInstVarAt: #loginFlags) ifNil:[ self defaultLoginFlags ] .

]

{ #category : 'Private' }
GemStoneParameters >> loginFlags: anInt [
"Do not set loginFlags directly unless you know what you are doing."

self dynamicInstVarAt: #loginFlags put: anInt .
^ self

]

{ #category : 'Updating' }
GemStoneParameters >> onetimePassword: pword [

"Set the onetime password for the receiver. "
pword _isOneByteString ifFalse:[ pword _validateClass: String ].
pword size == self class onetimePasswordSize
	ifFalse:[ pword _error: #rtErrBadSize args: { self class onetimePasswordSize . pword size } ] .
self setLoginWithOnetimePassword .
password := pword copy.
password immediateInvariant.
pword isInvariant ifFalse:[ pword size: 0 ].
^ pword

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> onetimePasswordLoginFlag [
^ 16r400 "GCI_ONETIME_PASSWORD"

]

{ #category : 'Accessing' }
GemStoneParameters >> password [

   "Return the value of the instance variable 'password'."
   ^password

]

{ #category : 'Updating' }
GemStoneParameters >> password: pword [

"Set the password for the receiver.  The password is encrypted.
The given password's size is set to zero."

pword _isOneByteString ifFalse:[ pword _validateClass: String ].

pword size == 0
  ifTrue:[ self setLoginPasswordless .
           password := pword copy ]
  ifFalse:[ self setLoginWithPassword .
            password := self _encrypt: pword ].
password immediateInvariant.
pword isInvariant ifFalse:[ pword size: 0 ].
^ pword

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> passwordEncryptedLoginFlag [

^ 16r1 " GCI_LOGIN_PW_ENCRYPTED "

]

{ #category : 'Updating' }
GemStoneParameters >> passwordEnvVar: envVar [

"External session login using the receiver will use the GemStone password 
 obtained via getenv(envVar) , evaluated in this current session's gem or topaz -l process.
 Applies to JWT and GemStone authentication schemes."

envVar _isOneByteString ifFalse:[ envVar _validateClass: String ].
password := envVar copy.
password immediateInvariant.
self setPasswordIsEnvVar .
^ envVar

]

{ #category : 'Updating' }
GemStoneParameters >> passwordFileName: fileName [

"External session login using the receiver will use the GemStone password 
 contained in the file fileName,  using the current directory of this session's 
 gem or topaz -l process to evaluate fileName.
 Applies to JWT and GemStone authentication schemes."

fileName _isOneByteString ifFalse:[ fileName _validateClass: String ].
password := fileName copy.
password immediateInvariant.
self setPasswordIsFileName .
^ fileName

]

{ #category : 'Accessing' }
GemStoneParameters >> passwordIsEncryptedAsIntegerBoolean [

"Answer a C boolean (1 or 0) indicating whether the password is encrypted.
 The password will always be encrypted unless the login is passwordless."

^ self loginFlags bitAnd: self passwordEncryptedLoginFlag

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> passwordIsEnvVarLoginFlag [
^ 16r2000 "GCI_LOGIN_PW_ENVVAR"

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> passwordIsFileNameLoginFlag [
^ 16r1000 "GCI_LOGIN_PW_FILE"

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> passwordlessLoginFlag [
^ 16r80  "GCI_PASSWORDLESS_LOGIN"

]

{ #category : 'Formatting' }
GemStoneParameters >> printString [
  | res f |
  res := 'stoneNrs:', gemStoneName asString, ' user:', username asString .
  hostUsername ifNotNil:[ res add: ' hostuser:', hostUsername asString ].
  res add: ' gemNrs:', gemService asString .
  (f := self loginFlags) == self defaultLoginFlags ifFalse:[
    res add: ' flags:16r', f asHexString
  ].
  ^ res

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> quietLoginFlag [

^ 16r10

]

{ #category : 'Accessing' }
GemStoneParameters >> remoteRepository [

"Return nil to indicate no remote repository."

^ nil

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setEncryptedPasswordFlag [

^ self loginFlags: (self loginFlags bitOr: self passwordEncryptedLoginFlag)

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setLoginDebug [
 
"Enables writing debug info to the gem's log file"
^  self loginFlags: (self loginFlags bitOr: self debugLoginFlag)

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setLoginPasswordless [

^ self clearEncryptedPasswordFlag ;
	clearJwtLoginFlag ;
	clearOnetimePasswordLoginFlag ;
	loginFlags: (self loginFlags bitOr: self passwordlessLoginFlag)

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setLoginWithJwt [
 
"Password is a JSON Web Token"
^  self 
	clearEncryptedPasswordFlag ; 
	clearOnetimePasswordLoginFlag ;
	clearPasswordlessLoginFlag ;
	clearSoloLoginFlag ;
	loginFlags: (self loginFlags bitOr:  self jwtLoginFlag)

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setLoginWithOnetimePassword [

^ self 
	clearEncryptedPasswordFlag ;
	clearSoloLoginFlag ;
	clearPasswordlessLoginFlag ;
	clearJwtLoginFlag ;
   	loginFlags: (self loginFlags bitOr: self onetimePasswordLoginFlag)

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setLoginWithPassword [

^  self 
	setEncryptedPasswordFlag ;
	clearSoloLoginFlag ;
	clearPasswordlessLoginFlag ;
	clearOnetimePasswordLoginFlag ;
	clearJwtLoginFlag 

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setPasswordIsEnvVar [

^ self 
	clearEncryptedPasswordFlag ;
	clearPasswordIsFileNameLoginFlag ;
	loginFlags: (self loginFlags bitOr: self passwordIsEnvVarLoginFlag)

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setPasswordIsFileName [

^ self 
	clearEncryptedPasswordFlag ;
	clearPasswordIsEnvVarLoginFlag ;
	 loginFlags: (self loginFlags bitOr: self passwordIsFileNameLoginFlag)

]

{ #category : 'Login Flags - Setting' }
GemStoneParameters >> setSoloLogin [
 
^  self 
	clearJwtLoginFlag ;
	clearOnetimePasswordLoginFlag ;
	loginFlags: (self loginFlags bitOr: self soloLoginFlag)

]

{ #category : 'Login Flags - Constants' }
GemStoneParameters >> soloLoginFlag [
^ 16r100 "GCI_LOGIN_SOLO"

]

{ #category : 'Accessing' }
GemStoneParameters >> stoneName [

  ^ gemStoneName
]

{ #category : 'Updating' }
GemStoneParameters >> stoneName: aName [
  ^ self gemStoneName: aName
]

{ #category : 'Accessing' }
GemStoneParameters >> username [

   "Return the value of the instance variable 'username'."
   ^username

]

{ #category : 'Updating' }
GemStoneParameters >> username: newValue [

   "Modify the value of the instance variable 'username'."
   newValue _isOneByteString ifFalse:[ newValue _validateClass: String ].
   username := newValue

]
