Extension { #name : 'EqualityIndexQueryEvaluator' }

{ #category : 'Searching' }
EqualityIndexQueryEvaluator >> _findAllValuesEqualTo: val1 using: aComparison [
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | stream querySpec |
  stream := self
    _findAllValuesGreaterThan: val1
    andEquals: true
    andLessThan: val1
    andEquals: true
    using: aComparison.
  querySpec := self index btreeComparisonQuerySpec
    key: val1 selector: #'=';
    yourself.
  stream streamQuerySpec: querySpec.
  ^ stream

]

{ #category : 'Searching' }
EqualityIndexQueryEvaluator >> _findAllValuesGreaterThan: val1 andEquals: bool1 andLessThan: val2 andEquals: bool2 using: aComparison [
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | array1 array2 stream doNext |
  ((self index _canCompareWith: val1) and: [ self index _canCompareWith: val2 ])
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  stream := self readStreamClass new rangeIndex: self index.	" check to see if the query is even satisfiable "
  (aComparison
    satisfiableQueryGreaterThan: val1
    andEquals: bool1
    andLessThan: val2
    andEquals: bool2)
    ifFalse: [
      stream currentStack: {0}.
      ^ stream ].
  doNext := false.
  array1 := {}.	" see if querying > or >= "
  bool1
    ifTrue: [
      " if >=, perform < and do an '_advance' operation later "
      (self btreeRoot
        _findAllValuesLessThanKey: val1
        andEquals: false
        into: array1
        using: aComparison)
        ifTrue: [ doNext := true ]
        ifFalse: [
          " all were greater than or equal to "
          self btreeRoot
            _putFirstIndexOfFirstChildInto: array1
            ifGreaterThanOrEqualTo: val1
            using: aComparison ] ]
    ifFalse: [
      " if >, ask the B-tree specifically for > "
      (self btreeRoot
        _findAllValuesGreaterThanKey: val1
        into: array1
        using: aComparison)
        ifFalse: [
          " none were greater than "
          stream currentStack: {0}.
          ^ stream ] ].
  array2 := {}.	" ask the B-tree for the second boundary of the query result "
  (self btreeRoot
    _findAllValuesLessThanKey: val2
    andEquals: bool2
    into: array2
    using: aComparison)
    ifFalse: [
      " none were found less than "
      stream currentStack: {0}.
      ^ stream ].
  stream endIndex: (array2 at: 1).
  stream endNode: (array2 at: 2).
  (stream endNode == (array1 at: 2) and: [ stream endIndex < (array1 at: 1) ])
    ifTrue: [
      array1 at: 1 put: 0.
      stream currentStack: array1.
      ^ stream ].
  stream currentStack: array1.
  doNext
    ifTrue: [
      " advance the B-tree positioning "
      stream _advance ].
  ^ stream

]

{ #category : 'Searching' }
EqualityIndexQueryEvaluator >> _findAllValuesGreaterThanKey: aKey andEquals: aBoolean [
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  ^ self
    _findAllValuesGreaterThanKey: aKey
    andEquals: aBoolean
    using: self comparisonForCompare

]

{ #category : 'Searching' }
EqualityIndexQueryEvaluator >> _findAllValuesGreaterThanKey: aKey andEquals: aBoolean using: aComparison [
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | array1 array2 stream doNext querySpec |
  (self index _canCompareWith: aKey)
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  stream := self readStreamClass new rangeIndex: self index.
  doNext := false.
  array1 := {}.	" see if querying > or >= "
  aBoolean
    ifTrue: [
      " if >=, perform < and do a 'next' operation later "
      (self btreeRoot
        _findAllValuesLessThanKey: aKey
        andEquals: false
        into: array1
        using: aComparison)
        ifTrue: [ doNext := true ]
        ifFalse: [
          " all were greater than or equal to "
          self btreeRoot
            _putFirstIndexOfFirstChildInto: array1
            ifGreaterThanOrEqualTo: aKey
            using: aComparison ] ]
    ifFalse: [
      " if >, ask the B-tree specifically for > "
      (self btreeRoot
        _findAllValuesGreaterThanKey: aKey
        into: array1
        using: aComparison)
        ifFalse: [
          " none were greater than "
          stream currentStack: {0}.
          ^ stream ] ].
  array2 := {}.
  self btreeRoot _putLastIndexOfLastChildInto: array2.
  stream currentStack: array1.
  stream endIndex: (array2 at: 1).
  stream endNode: (array2 at: 2).
  doNext
    ifTrue: [ stream _advance ].
  querySpec := self index btreeComparisonQuerySpec
    key: aKey
    selector:
      (aBoolean
        ifTrue: [ #'>=' ]
        ifFalse: [ #'>' ]);
    yourself.
  stream streamQuerySpec: querySpec.
  ^ stream

]

{ #category : 'Searching' }
EqualityIndexQueryEvaluator >> _findAllValuesIdenticalTo: aValue [
  "Returns a set of all values that satisfy the query.  Since this is an identity
 comparison on a range index (which supports equality comparison), this
 method creates a B-tree read stream for equal values, then iterates through the
 values making the identity comparison."

  | array1 array2 stream doNext |
  stream := self readStreamClass on: self index.
  " if can't make the comparison, then none are identical "
  (self index _canCompareWith: aValue)
    ifFalse: [
       stream currentStack: {0}.
       ^ stream ].
  " get boundary entries in the B-tree "
  array1 := {}.
  " >=, perform < and do an 'next' operation later "
  doNext := false.
  (self btreeRoot
    _findAllValuesLessThanKey: aValue
    andEquals: false
    into: array1
    using: self comparisonForCompare)
    ifTrue: [ doNext := true ]
    ifFalse: [ " all were greater than or equal to "
      self btreeRoot
        _putFirstIndexOfFirstChildInto: array1
        ifGreaterThanOrEqualTo: aValue
        using: self comparisonForCompare ].
  array2 := {}.
  (self btreeRoot
    _findAllValuesLessThanKey: aValue
    andEquals: true
    into: array2
    using: self comparisonForCompare)
    ifFalse: [ " none were found less than "
       stream currentStack: {0}.
       ^ stream ].
  " create the read stream "
  stream currentStack: array1.
  stream endIndex: (array2 at: 1).
  stream endNode: (array2 at: 2).
  doNext
    ifTrue: [ " advance the B-tree positioning "
      stream _advance ].
  ^ stream

]

{ #category : 'Query Select' }
EqualityIndexQueryEvaluator >> _findAllValuesIdenticalToAsNsc: aValue [
  "Answer a collecation of all values identical to aValue"

  | stream querySpec |
  stream := self _findAllValuesIdenticalTo: aValue.
  querySpec := self btreeComparisonQuerySpec
    key: aValue selector: #==;
    yourself.
  stream streamQuerySpec: querySpec.
  ^ stream makeNsc

]

{ #category : 'Searching' }
EqualityIndexQueryEvaluator >> _findAllValuesLessThanKey: aKey andEquals: aBoolean using: aComparison [
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | array1 array2 stream querySpec |
  (self index _canCompareWith: aKey)
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  stream := self readStreamClass new rangeIndex: self index.
  array2 := {}.	" ask the B-tree to make the search and put the result path into array2 "
  (self btreeRoot
    _findAllValuesLessThanKey: aKey
    andEquals: aBoolean
    into: array2
    using: aComparison)
    ifTrue: [
      array1 := {}.	" put path to first entry of the B-tree into array1 "
      self btreeRoot _putFirstIndexOfFirstChildInto: array1.
      stream currentStack: array1.
      stream endIndex: (array2 at: 1).
      stream endNode: (array2 at: 2) ]
    ifFalse: [
      " none were found "
      stream currentStack: {0} ].
  querySpec := self index btreeComparisonQuerySpec
    key: aKey
    selector:
      (aBoolean
        ifTrue: [ #'<=' ]
        ifFalse: [ #'<' ]);
    yourself.
  stream streamQuerySpec: querySpec.
  ^ stream

]

{ #category : 'Searching' }
EqualityIndexQueryEvaluator >> _findAllValuesThroughSetsNotIdenticalTo: aKey [
  "Returns a set of all values that satisfy the query.  This method works
 when the path is through a set-valued instance variable."

  | tmpVals tmpHolder startPt pathTerm idx |
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  idx := self index.
  tmpVals := idx btreeRoot _lastValuesNotIdenticalTo: aKey into: Array new.
  startPt := idx lastPathComponentsDictionaryOffset.
  pathTerm := idx at: startPt.
  1 to: tmpVals size do: [ :i |
    idx
      _addAllFor: (tmpVals _at: i)
      into: tmpHolder
      offset: startPt
      pathTerm: pathTerm ].
  ^ tmpHolder completeBag

]

{ #category : 'Searching' }
EqualityIndexQueryEvaluator >> _findFirstKeyNotIdenticalTo: aKey atTerm: pathTerm [
  "Returns the first key in the receiver's B-tree that is not identical to the
 given key."

  | stream key |
  stream := self btreeReadStreamClass on: self btreeRoot.
  [ stream _atEnd ] whileFalse: [ key := stream _peekKey.
      key == aKey
        ifFalse: [ ^ key ].
      stream _btreeNext ].
  ^ #'_incompletePathTraversal'

]

{ #category : 'Accessing' }
EqualityIndexQueryEvaluator >> btreeReadStreamClass [
  "Returns the class of btree read stream to create for query results."

  ^ index btreeReadStreamClass

]

{ #category : 'Accessing' }
EqualityIndexQueryEvaluator >> btreeRoot [

  ^self index btreeRoot

]

{ #category : 'Accessing' }
EqualityIndexQueryEvaluator >> comparisonForCompare [
  ^ self index comparisonForCompare

]

{ #category : 'Accessing' }
EqualityIndexQueryEvaluator >> comparisonForSort [
  ^ BtreeComparisonForCompare newForSort: self collator

]

{ #category : 'Query Select' }
EqualityIndexQueryEvaluator >> findAllValuesEqualTo: aValue [
  "Returns a set of all values that satisfy the query."

  | stream |
  stream := self _findAllValuesEqualTo: aValue using: self comparisonForCompare.
  ^ self doBlock
    ifNotNil: [ [ stream atEnd ] whileFalse: [ self doBlock value: stream next ] ]
    ifNil: [ stream makeNscComparing: #'=' key: aValue ]

]

{ #category : 'Query Select' }
EqualityIndexQueryEvaluator >> findAllValuesGreaterThan: val1 andEquals: bool1 andLessThan: val2 andEquals: bool2 [
  "Returns a set of all values that satisfy the query."

  | selector1 selector2 stream |
  selector1 := bool1
    ifTrue: [ #'>=' ]
    ifFalse: [ #'>' ].
  selector2 := bool2
    ifTrue: [ #'<=' ]
    ifFalse: [ #'<' ].
  stream := self
    _findAllValuesGreaterThan: val1
    andEquals: bool1
    andLessThan: val2
    andEquals: bool2
    using: self comparisonForCompare.
  ^ self doBlock
    ifNotNil: [ stream streamQuerySpec: (stream btreeRangeComparisonQuerySpec
            key: val1 selector: selector1;
            key2: val2 selector2: selector2;
            yourself).
      [ stream atEnd ] whileFalse: [ self doBlock value: stream next ] ]
    ifNil: [ stream
        makeNscComparing: selector1
        key: val1
        and: selector2
        key: val2 ]

]

{ #category : 'Query Select' }
EqualityIndexQueryEvaluator >> findAllValuesGreaterThanKey: aValue andEquals: aBoolean [
  "Returns a set of all values that satisfy the query."

  | selector stream |
  selector := aBoolean
    ifTrue: [ #'>=' ]
    ifFalse: [ #'>' ].
  stream := self
    _findAllValuesGreaterThanKey: aValue
    andEquals: aBoolean
    using: self comparisonForCompare.
  ^ self doBlock
    ifNotNil: [ [ stream atEnd ] whileFalse: [ self doBlock value: stream next ] ]
    ifNil: [ stream makeNscComparing: selector key: aValue ]

]

{ #category : 'Query Select' }
EqualityIndexQueryEvaluator >> findAllValuesIdenticalTo: aValue [
  "Returns a set of all values that satisfy the query.  Since this is an identity
 comparison on a range index (which supports equality comparison), this
 method creates a B-tree read stream for equal values, then iterates through the
 values making the identity comparison."

  | tmpHolder stream |
  " optimize query for objects in the NSC itself "
  self index isIndexOnRootNsc
    ifTrue: [
      doBlock
        ifNil: [
          " optimize query for objects in the NSC itself "
          ^ self _allIdenticalTo: aValue in: self nscRoot ].
      ^ self _allIdenticalTo: aValue in: self nscRoot do: doBlock  ].
  stream := self _findAllValuesIdenticalTo: aValue.
  self doBlock
    ifNotNil: [
      stream atEnd
        ifTrue: [ ^ self nscRoot species new ].
      ^ [ stream atEnd ]
           whileFalse: [
             | nextKey next |
             nextKey := stream _peekKey.
             next := stream next.
             nextKey == aValue
             ifTrue: [ self doBlock value: next ] ] ].
  stream atEnd
    ifTrue: [ ^ self nscRoot species new ].
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  self index size == 1
    ifTrue: [ stream _valuesAddInto: tmpHolder identicalTo: aValue ]
    ifFalse: [ stream _uniqueValuesAddInto: tmpHolder identicalTo: aValue ].
  ^ tmpHolder completeBag

]

{ #category : 'Query Select' }
EqualityIndexQueryEvaluator >> findAllValuesLessThanKey: aValue andEquals: aBoolean [
  "Returns a set of all values that satisfy the query."

  | selector stream |
  selector := aBoolean
    ifTrue: [ #'<=' ]
    ifFalse: [ #'<' ].
  stream := self
    _findAllValuesGreaterThan: nil
    andEquals: true
    andLessThan: aValue
    andEquals: aBoolean
    using: self comparisonForCompare.
  ^ self doBlock
    ifNotNil: [ | querySpec |
      querySpec := self index btreeComparisonQuerySpec
        key: aValue selector: selector;
        yourself.
      stream streamQuerySpec: querySpec.
      [ stream atEnd ] whileFalse: [ self doBlock value: stream next ] ]
    ifNil: [ stream makeNscComparing: selector key: aValue ]

]

{ #category : 'Query Select' }
EqualityIndexQueryEvaluator >> findAllValuesNotEqualTo: aValue [
  "Returns a set of all values that satisfy the query."

  | preStream eqStream postStream querySpec |
  preStream := self
    _findAllValuesGreaterThan: nil
    andEquals: true
    andLessThan: aValue
    andEquals: false
    using: self comparisonForSort.
  eqStream := self
    _findAllValuesGreaterThan: aValue
    andEquals: true
    andLessThan: aValue
    andEquals: true
    using: self comparisonForSort.
  querySpec := self index btreeComparisonQuerySpec
    key: aValue selector: #'~=';
    yourself.
  eqStream streamQuerySpec: querySpec.
  postStream := self
    _findAllValuesGreaterThanKey: aValue
    andEquals: false
    using: self comparisonForSort.
  postStream streamQuerySpec: nil.
  ^ self doBlock
    ifNotNil: [ :aBlock |
      [ preStream atEnd ] whileFalse: [ aBlock value: preStream getNext ].
      [ eqStream atEnd ] whileFalse: [ aBlock value: eqStream getNext ].
      [ postStream atEnd ] whileFalse: [ aBlock value: postStream getNext ] ]
    ifNil: [
      | tmpHolder |
      tmpHolder := NscBuilder
        for: self nscRoot species new
        max: self sizeForNscBuilder.
      self index size == 1
        ifTrue: [
          preStream _valuesAddInto: tmpHolder.
          eqStream _valuesAddInto: tmpHolder notEqualTo: aValue.
          postStream _valuesAddInto: tmpHolder ]
        ifFalse: [
          preStream _uniqueValuesAddInto: tmpHolder.
          eqStream _uniqueValuesAddInto: tmpHolder notEqualTo: aValue.
          postStream _uniqueValuesAddInto: tmpHolder ].
      tmpHolder completeBag ]

]

{ #category : 'Query Detect' }
EqualityIndexQueryEvaluator >> findFirstValueEqualTo: aValue [

"Returns the first value that satisfies the query."

| val |
(index _canCompareWith: aValue)
    ifFalse: [ self _error: #assocErrBadComparison ].

val := self btreeRoot _findFirstValueForKey: aValue using: self comparisonForCompare.

(val == #_incompletePathTraversal)
    ifTrue: [ ^ #_incompletePathTraversal ].

^ self _findFirstValueForKey: val

]

{ #category : 'Query Detect' }
EqualityIndexQueryEvaluator >> findFirstValueGreaterThan: val1
andEquals: bool1
andLessThan: val2
andEquals: bool2 [

"Returns the first value that satisfies the query."

| stream selector1 selector2 |
stream := self
    _findAllValuesGreaterThan: val1
    andEquals: bool1
    andLessThan: val2
    andEquals: bool2
    using: self comparisonForCompare.

stream == nil
  ifTrue: [ ^ #_incompletePathTraversal ].
selector1 := bool1
  ifTrue: [ #_idxForCompareGreaterThanOrEqualTo: ]
  ifFalse: [ #_idxForCompareGreaterThan: ].
selector2 := bool2
  ifTrue: [ #_idxForCompareLessThanOrEqualTo: ]
  ifFalse: [ #_idxForCompareLessThan: ].
^ stream getNextValueComparing: selector1 key: val1 and: selector2 key: val2

]

{ #category : 'Query Detect' }
EqualityIndexQueryEvaluator >> findFirstValueGreaterThanKey: aValue
andEquals: aBoolean [

"Returns the first value that satisfies the query."

| stream selector |
stream := self  _findAllValuesGreaterThanKey: aValue
                    andEquals: aBoolean
                    using: self comparisonForCompare.
stream == nil
  ifTrue: [ ^ #_incompletePathTraversal ].
selector := aBoolean
  ifTrue: [ #_idxForCompareGreaterThanOrEqualTo: ]
  ifFalse: [ #_idxForCompareGreaterThan: ].
^ stream getNextValueComparing: selector key: aValue

]

{ #category : 'Query Detect' }
EqualityIndexQueryEvaluator >> findFirstValueIdenticalTo: aValue [
  "Returns the first value that satisfies the query.  Since this is an identity
 comparison on a range index (which supports equality comparison), create a
 BtreeReadStream for equal values, then iterate through the values making
 the identity comparison."

  | stream |
  stream := self _findAllValuesIdenticalTo: aValue.
  stream atEnd
    ifTrue: [ ^ #'_incompletePathTraversal' ].	" iterate through the stream, checking for an identical value "
  [ stream _atEnd ]
    whileFalse: [
      stream _peekKey == aValue
        ifTrue: [
          ^ self _findFirstValueForKey: stream _peekValue ].
      stream _btreeNext ].
  ^ #'_incompletePathTraversal'

]

{ #category : 'Query Detect' }
EqualityIndexQueryEvaluator >> findFirstValueLessThanKey: aValue
andEquals: aBoolean [

"Returns the first value that satisfies the query."

| stream selector |
stream := self
      _findAllValuesGreaterThan: nil
      andEquals: true
      andLessThan: aValue
      andEquals: aBoolean
      using: self comparisonForCompare.
stream == nil
  ifTrue: [ ^ #_incompletePathTraversal ].
selector := aBoolean
  ifTrue: [ #_idxForCompareLessThanOrEqualTo: ]
  ifFalse: [ #_idxForCompareLessThan: ].
^ stream getNextValueComparing: selector key: aValue

]

{ #category : 'Query Detect' }
EqualityIndexQueryEvaluator >> findFirstValueNotEqualTo: aValue [
  "Returns the first value that satisfies the query."

  | stream totalStream obj bag incomplete |
  " find the ones that are equal "
  stream := self
    _findAllValuesGreaterThan: aValue
    andEquals: true
    andLessThan: aValue
    andEquals: true
    using: self comparisonForCompare.
  stream _atEnd
    ifTrue: [ " if none are equal, pick the first one "
      self hasSetValuedTerm
        ifTrue: [ bag := self nscRoot _asIdentityBag.
          1 to: bag size do: [ :i | nil ~~ (bag _at: i)
              ifTrue: [ ^ bag _at: i ] ].
          ^ #'_incompletePathTraversal' ]
        ifFalse: [ incomplete := #'_incompletePathTraversal'.
          bag := self nscRoot _asIdentityBag.
          1 to: bag size do: [ :i | obj := bag _at: i.
            (self index traverseObject: obj) == incomplete
              ifFalse: [ ^ obj ] ].
          ^ #'_incompletePathTraversal' ] ].
  totalStream := self readStreamClass on: self index.	" get a stream over the entire NSC "
  (totalStream _peekKey _idxForCompareNotEqualTo: stream _peekKey)
    ifTrue: [ " see if very first object is not equal "
      ^ totalStream getNext ].
  (totalStream _hasSameEndAs: stream)
    ifTrue: [ " see if end of original stream is the very last object "
      ^ #'_incompletePathTraversal' ]
    ifFalse: [ " advance the stream to the last object "
      totalStream _positionToEnd.
      ^ totalStream getNext ]

]

{ #category : 'Accessing' }
EqualityIndexQueryEvaluator >> readStreamClass [
  "Returns the class of read stream to create for query results."

  ^ index readStreamClass

]
