Extension { #name : 'DepListBucket' }

{ #category : 'Instance Creation' }
DepListBucket class >> new [

"Returns a DepListBucket with a default capacity of ten entries."

^ self new: 3

]

{ #category : 'Instance Creation' }
DepListBucket class >> new: aSize [

"Returns a DepListBucket with the given capacity of entries."

^ (super new: aSize) initialize

]

{ #category : 'Searching' }
DepListBucket >> _binarySearchFor: aDepList [

"Returns the offset into the receiver where the given dependency list
 can be found.  If not found, returns zero."

| lowPt midPt highPt currDL |

numElements == 0
  ifTrue: [ ^ 0 ].

lowPt := 1.
highPt := numElements.
[ lowPt <= highPt ] whileTrue: [
  midPt := (lowPt + highPt) quo: 2.
  currDL := self _rcAt: midPt.

  "use identity-based comparison of elements of dependency lists"
  (currDL hasIdenticalContents: aDepList)
    ifTrue: [ ^ midPt ].

  aDepList < currDL
    ifTrue: [ highPt := midPt - 1 ]
    ifFalse: [ lowPt := midPt + 1 ]
].

^ 0

]

{ #category : 'Auditing' }
DepListBucket >> _hasEntryOutOfOrder [

"Return whether the receiver has any entry out of order."

| depList prevDepList |
self size > 1
  ifTrue: [
    prevDepList := self _at: 1.
    2 to: self size do: [ :j |
      depList := self _at: j.
      depList < prevDepList
        ifTrue: [ ^ true ]
    ]
  ].
^ false

]

{ #category : 'Searching' }
DepListBucket >> _linearSearchFor: aDepList [

"Private.  Returns the offset into the receiver where the given dependency list
 can be found.  If not found, returns zero."

1 to: numElements do: [ :j |
  aDepList == (self _at: j)
    ifTrue: [ ^ j ]
].
^ 0

]

{ #category : 'Updating' }
DepListBucket >> at: depList [

"Returns the dependency list in the receiver that is equivalent to the given
 dependency list.  If one does not exist, add the given dependency list to the
 receiver and returns the given dependency list.  If one is found, the given
 dependency list is sized to zero. 
 If argument is added to receiver,  sender (DepListTable>>at:logging:) is responsible for
   depList objectSecurityPolicy: self objectSecurityPolicy .
 "
<primitive: 215>
depList _validateInstanceOf: DependencyList .
self _primitiveFailed: #at: args: { depList }

]

{ #category : 'Enumerating' }
DepListBucket >> do: aBlock [

"For each dependency list in the receiver, evaluates the one-argument block
 aBlock with the dependency list as the argument."

| aDL |
1 to: numElements do: [ :j |
  aDL := self _at: j.
  aDL == nil
    ifFalse: [ aBlock value: aDL ]
]

]

{ #category : 'Constants' }
DepListBucket >> growIncrement [

"Returns the number of entries to grow when needed."

^ 10

]

{ #category : 'Initializing' }
DepListBucket >> initialize [

"Initializes the instance variable of the receiver."

numElements := 0.

]

{ #category : 'Removing' }
DepListBucket >> remove: aDepList [

"Remove the given dependency list from the receiver.  Returns whether the
 removal occurred."

| offset |
offset := self _linearSearchFor: aDepList.
offset == 0
  ifTrue: [ ^ false ].

self _deleteNoShrinkFrom: offset to: offset .
numElements := numElements - 1.

^ true

]

{ #category : 'Updating' }
DepListBucket >> reset [

"Reset all entries to nil and the number of elements to zero."

super size: 0.
numElements := 0.
super size: self growIncrement.
System _addEntireObjectToRcReadSet: self

]

{ #category : 'Updating' }
DepListBucket >> resizeIfNecessary [

"Check the size of the receiver to see if it is unnecessarily greater
 than a page in size.  If so, resize it."

( self _basicSize > 2000 and: [ numElements < 2000 ] )
  ifTrue: [ super size: (2000 min: numElements + self growIncrement) ].

]

{ #category : 'Accessing' }
DepListBucket >> size [

"Returns the value of the numElements instance variable."

^ numElements

]
