Extension { #name : 'Delay' }

{ #category : 'GBS Access' }
Delay class >> _highPriorityWaitForMilliseconds: millisecondCount [
  "Set the active process's priority high and then
   suspends the active process for millisecondCount milliseconds.
   Note that this method does not create or use an instance of Delay.
   Returns the receiver."

  | sched oldPrio proc |
  (sched := self _scheduler) _enterCritical .
  proc := sched activeProcess .
  oldPrio := proc _raisePriority .
  sched _waitForMilliseconds: millisecondCount .  "exits critical"
  sched _enterCritical .
  sched activeProcess == proc ifTrue:[
    proc _setPriority: oldPrio .
  ].
  sched _exitCritical .
  1 timesRepeat:[ self class ].  "check for a Break"
  ^ self

]

{ #category : 'Private' }
Delay class >> _scheduler [

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler

]

{ #category : 'Instance Creation' }
Delay class >> forMilliseconds: millisecondCount [
  "Answer a new instance that will suspend the active process
   for millisecondCount milliseconds when sent the message wait."

  ^ self _basicNew interval: millisecondCount

]

{ #category : 'Instance Creation' }
Delay class >> forSeconds: secondCount [
  "Answer a new instance that will suspend the active process
   for secondCount seconds when sent the message wait."

  ^ self forMilliseconds: secondCount * 1000

]

{ #category : 'General Inquiries' }
Delay class >> millisecondClockValue [
  "Returns a SmallInteger, the current value of the millisecond clock.

   Gs64 v2.2, changed to no longer rollover to zero after 524287999 "

<primitive: 651>
^ self _primitiveFailed: #millisecondClockValue

]

{ #category : 'Instance Creation' }
Delay class >> new [

"Disallowed."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
Delay class >> untilMilliseconds: millisecondValue [
  "Answer a new instance that will suspend the active process
   until the millisecond clock reaches the value millisecondValue."

  millisecondValue _isSmallInteger ifFalse:[
    millisecondValue _validateClass: SmallInteger.
  ].
  millisecondValue < 0 ifTrue:[
    ^ millisecondValue _error: #rtErrArgNotPositive
  ].
  ^ self _basicNew _signalTime: millisecondValue

]

{ #category : 'Process Delay' }
Delay class >> waitForMilliseconds: millisecondCount [
  "Suspends the active process for millisecondCount milliseconds.
   Note that this method does not create or use an instance of Delay.
   Returns the receiver."

  | sched |
  millisecondCount < 0 ifTrue:[
    ^ millisecondCount _error: #rtErrArgNotPositive
  ].
  (sched := self _scheduler) _enterCritical .
  sched _waitForMilliseconds: millisecondCount .  "exits critical"
  1 timesRepeat:[ self class ].  "check for a Break"
  ^self

]

{ #category : 'Process Delay' }
Delay class >> waitForSeconds: secondCount [
  "Suspends the active process for secondCount seconds.
   Note that this method does not create or use an instance of Delay.
   Returns the receiver."

  | sched |
  secondCount < 0 ifTrue:[
    ^ secondCount _error: #rtErrArgNotPositive
  ].
  (sched := self _scheduler) _enterCritical .
  sched _waitForMilliseconds: secondCount * 1000 .  "exits critical"
  1 timesRepeat:[ self class ].  "check for a Break"
  ^self

]

{ #category : 'Process Delay' }
Delay class >> waitUntilMilliseconds: millisecondValue [
  "Suspends the active process until the millisecond clock
   reaches the value millisecondValue.
   Note that this method does not create or use an instance of Delay.
   Returns the receiver."

  | delta sched |
  millisecondValue _isSmallInteger ifFalse:[
    millisecondValue _validateClass: SmallInteger.
  ].
  millisecondValue < 0 ifTrue:[
    ^ millisecondValue _error: #rtErrArgNotPositive
  ].
  (sched := self _scheduler) _enterCritical .
  delta := millisecondValue -  sched _now .
  delta < 0 ifTrue:[ delta := 0 ].
  sched _waitForMilliseconds: delta.
  1 timesRepeat:[ self class ].  "check for a Break"
  ^self

]

{ #category : 'Private' }
Delay >> _cancel: aGsProcess [
  "Cancels a delay that is in progress.
   The receiver will not signal any processes that have sent wait to it.
   Only use if you know that the processes that are waiting on the receiver
   will be signalled in some other way."

  | t |
  aGsProcess == (t := target) ifTrue: [
    self _scheduler _delayUnschedule: self.
    target := nil.
  ] ifFalse:[
    t _isArray ifTrue:[
     1 to: t size do:[:j | 
       (t at:j) == aGsProcess ifTrue:[
         t removeAtIndex: j .
         t size == 0 ifTrue:[ 
           target := nil .
           self _scheduler _delayUnschedule: self.
         ].
         ^ self
       ]
     ]
   ]
  ]
]

{ #category : 'Private' }
Delay >> _changePriority: aGsProcess from: oldPriority [
  "Sent by code in GsProcess to change the priority of aGsProcess.waitingOn,
   when aGsProcess.waitingOn is a Delay.  Since a Delay does not contain
   a list of processes sorted by priority, there is nothing to do."

]

{ #category : 'Private' }
Delay >> _finishDelay [
  "Return 1 if receiver ready to run, 0 otherwise.
   For use by the scheduler only."
  | numReady |
  numReady := 0 .
  target ifNotNil:[:t | 
    t _isArray ifTrue:[ 1 to: (numReady := t size) do:[:j | (t at:j) _signalAll ]]
       ifFalse:[ t _signalAll . numReady := 1 ].
    target := nil.
  ].
  ^ numReady
]

{ #category : 'Private' }
Delay >> _scheduler [

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler

]

{ #category : 'Private' }
Delay >> _signalTime [
  ^signalTime

]

{ #category : 'Private' }
Delay >> _signalTime: millisecondClockValue [
  "sets the receiver's signal time w/o regard to the current time
   or delay interval"

  signalTime := millisecondClockValue

]

{ #category : 'Private' }
Delay >> _targetProcess [
  "Returns a GsProcess or an Array of GsProcess that is/are waiting for receiver."
  ^ target
]

{ #category : 'Private' }
Delay >> _unscheduleProcess: aGsProcess [
  "Used by ProcessorScheduler"
  self _cancel: aGsProcess.

]

{ #category : 'Private' }
Delay >> _wait: highPriorityBoolean [

  "Suspend the active process until the millisecond clock
   reaches the appropriate value."

  | interv targ proc oldPrio sched |
  (sched := self _scheduler) _enterCritical .
  (interv := interval) ifNotNil: [
    "If another process is already using this interval delay raise an error"
    target ifNotNil: [
      sched _exitCritical .
      ThreadError new _number: 2365 ; reason: #rtErrDelayInProgress ;
         object: self ; details:'Another GsProcess already waiting on this Delay';
         signal .
      ^ self
    ].
  ].
  highPriorityBoolean ifTrue:[
    proc := sched activeProcess .
    oldPrio := proc _raisePriority .
  ].
  interv ifNotNil: [
    signalTime := interv + (sched _now).
  ] ifNil: [
    signalTime ifNil:[ UncontinuableError signal:'both interval and signalTime of Delay are nil'].
  ].
	targ := sched activeProcess .
	targ _waitingOn: self.
  target ifNil:[
	  target := targ .
    sched _delaySchedule: self 
  ] ifNotNil:[ :t |
    t _isArray ifTrue:[ t add: targ ]
              ifFalse:[ target := { t . targ } . "already scheduled" ].
  ].
  sched _exitCritical .
  targ _wait .
  highPriorityBoolean ifTrue:[
    sched _enterCritical .
    sched activeProcess == proc ifTrue:[
      proc _setPriority: oldPrio .
    ].
    sched _exitCritical .
  ].
  1 timesRepeat:[ self class ].  "check for a Break"
  ^self
]

{ #category : 'Process Delay' }
Delay >> highPriorityWait [
  "Suspend the active process until the millisecond clock
   reaches the appropriate value.  The active process priority
   is raised for the duration of the wait, so that expiration of the delay
   will cause the active process to resume thus interrupting any infinite loop
   or infinite wait for socket activity in other processes."

 ^ self _wait: true

]

{ #category : 'Testing' }
Delay >> inProgress [
  "Returns true if a GsProcess is currently waiting on the receiver."

  ^ target ~~ nil

]

{ #category : 'Private' }
Delay >> interval: millisecondCount [
  "initialize the receiver to delay on an interval"

  millisecondCount _isSmallInteger ifFalse:[
    millisecondCount _validateClass: SmallInteger.
  ].
  millisecondCount < 0 ifTrue:[
    ^ millisecondCount _error: #rtErrArgNotPositive
  ].
  interval := millisecondCount .
  signalTime := nil .
  target ifNotNil:[ Error signal:'Delay already has a target process'].
  target := nil .
]

{ #category : 'Formatting' }
Delay >> printOn: aStream [
  aStream
    nextPutAll: self class name;
    nextPut: $(.
  self asOop printOn: aStream.
  aStream nextPutAll: ',i='.
  interval printOn: aStream.
  aStream nextPutAll: ',t='.
  signalTime printOn: aStream.
  aStream nextPutAll: ',s='.
  target printOn: aStream.
  aStream nextPut: $).

]

{ #category : 'Accessing' }
Delay >> priority [
  | targ |
  (targ := target) class == GsProcess ifTrue:[  ^ targ priority ].
  targ _isArray ifTrue:[ ^ (targ at: 1) priority ].
	^ 15 "inline userSchedulingPriority" .
]

{ #category : 'Accessing' }
Delay >> resumptionTime [
  "Answer the value of the millisecond clock at which
   the delayed process will be resumed.  This will be nil
   if the receiver was created with an interval and no
   wait has been issued."

  interval ifNil: [
    ^signalTime
  ] ifNotNil: [
    target ifNotNil:[ "inline inProgress"
      ^signalTime
    ] ifNil: [
      ^nil
    ].
  ].

]

{ #category : 'Process Delay' }
Delay >> signal [
  "Causes the receiver to signal any processes that have sent wait to it."

  self _scheduler _delayUnschedule: self.
  self _finishDelay .
]

{ #category : 'Process Delay' }
Delay >> wait [
  "Suspend the active process until the millisecond clock
   reaches the appropriate value."

self _wait: false

]
