"
A CanonStringDict is a dictionary that provides
 protocol that is similar to Set in addition to its dictionary protocol.
 It uses sorted collision buckets, and stores only keys. Each slot in
 the dictionary object contains a references to a bucket, or nil  

Constraints:
	numElements: SmallInteger
	tableSize: SmallInteger
	[elements]: CanonStringBucket
"
Class {
	#name : 'CanonStringDict',
	#superclass : 'AbstractDictionary',
	#instVars : [
		'numElements',
		'tableSize'
	],
	#gs_reservedoop : '119297',
	#category : 'Kernel-AllSymbolsSupport'
}

{ #category : 'Instance Creation' }
CanonStringDict class >> _new: aPrime [

"Private"

| newDict |
newDict := super _basicNew: aPrime .
newDict _initializeWithoutClear: aPrime .
^ newDict .

]

{ #category : 'Instance Creation' }
CanonStringDict class >> new [

"Creates an instance of CanonStringDict with a default table size."

^ self new: 20

]

{ #category : 'Instance Creation' }
CanonStringDict class >> new: tableSize [

"Creates an instance of CanonStringDict with the specified table size."

| aPrime |
aPrime := self _tableSizeFor: tableSize.
^ self _new: aPrime

]

{ #category : 'Private' }
CanonStringDict >> _audit [
| cnt |
cnt := 0 .
1 to: tableSize do:[:n |  | bkt |
  bkt := self basicAt: n .
  bkt _audit .
  cnt := cnt + bkt numElements .
].
cnt == numElements ifFalse:[ Error signal: 'bad size']

]

{ #category : 'Hashing' }
CanonStringDict >> _basicHash: aString [

"Private. The hash value is a case-sensitive hash value of the bytes of a
 string without regard to whether the string contains single or
 double byte characters.  The strings are canonicalized so that
 a DoubleByteString with all-ascii characters is converted to
 a String before computing the hash."
<primitive: 519>
aString _validateClass: String .
self _primitiveFailed: #_basicHash: args: { aString }.
self _uncontinuableError

]

{ #category : 'Private' }
CanonStringDict >> _collisionBucketsDo: aBlock [
"Private.  Executes the given one-argument block for each collision bucket
 in the receiver."

| aBucket |
1 to: tableSize do:[:k |
  aBucket := self _at: k .
  aBucket ifNotNil:[ aBlock value: aBucket ].
  ].

]

{ #category : 'Private' }
CanonStringDict >> _initializeWithoutClear: newSize [

"Private. Initializes the instance variables of the receiver to be an empty
 KeyValueDictionary of the specified size. Does not clear the contents
 of the receiver - assumes they are all nil."

tableSize := newSize.
numElements := 0.

^self

]

{ #category : 'Private' }
CanonStringDict >> _nodesObjectSecurityPolicy: anObjectSecurityPolicy [
  "Assigns receiver's components to the given security policy."

self _collisionBucketsDo:[ :aBucket |
   aBucket objectSecurityPolicy: anObjectSecurityPolicy
].

]

{ #category : 'Private' }
CanonStringDict >> _resetParentRef [

"Private. After a become:, the parent refs of the collisionBuckets must
 be reset to point to the correct parent."

self _collisionBucketsDo: [:collisionBkt |
  collisionBkt keyValueDictionary: self].

]

{ #category : 'Comparing' }
CanonStringDict >> = anCanonStringDict [

"Returns true if all of the following conditions are true:

 1.  The receiver and anCanonStringDict are of the same class.
 2.  The two dictionaries are of the same size.
 3.  The corresponding keys and values of the receiver and anCanonStringDict
     are equal."

(self == anCanonStringDict)
  ifTrue: [ ^ true ].

(self class == anCanonStringDict class)
  ifFalse: [ ^ false ].

(self size == anCanonStringDict size)
  ifFalse: [ ^ false ].

self keysDo: [ :aKey |
  (aKey = (anCanonStringDict at: aKey otherwise: nil))
    ifFalse: [ ^ false ]
  ].

^ true.

]

{ #category : 'Enumerating' }
CanonStringDict >> accompaniedBy: anObj keysAndValuesDo: aBlock [

"Evaluates aBlock with each of the receiver's key/value pairs as the
 2nd and 3rd arguments.
 aBlock must be a 3 argument block, with arguments anObj, key value ."

1 to: tableSize do:[ :tableIndex | | collisionBkt |
  collisionBkt := self _at: tableIndex .
  collisionBkt ~~ nil ifTrue:[
      collisionBkt accompaniedBy: anObj keysAndValuesDo: aBlock
  ].
].

]

{ #category : 'Adding' }
CanonStringDict >> add: aString [

"Adds aString if it is not already present in the receiver, and returns
 either aString or the canonical string already present."

<primitive: 478>
aString _validateClass: String .
self _primitiveFailed: #add: args: { aString }.
self _uncontinuableError

]

{ #category : 'Adding' }
CanonStringDict >> addAll: aCollection [

"Adds elements of aCollection to the receiver. If aCollection is
 a Dictionary, adds only the keys to the receiver .
 Returns aCollection."

aCollection == self ifTrue:[ ^ aCollection ].
(aCollection isKindOf: AbstractDictionary)
  ifTrue:[ ^ aCollection keysDo:[:aKey | self add: aKey ] ].
aCollection accompaniedBy: self do:[ :me :aString | me add: aString ].
^ aCollection

]

{ #category : 'Adding' }
CanonStringDict >> addKey: aString [

"Adds aString if it is not already present in the receiver, and returns
 either aString or the canonical string already present."

^ self add: aString

]

{ #category : 'Printing' }
CanonStringDict >> asReportString [

"Returns a String that lists the keys, one on each line."

| result |
result := String new.
self keysDo: [ :key |
  result add: key printString;
    add: Character lf
  ].
^ result

]

{ #category : 'Enumerating' }
CanonStringDict >> associationsDetect: aBlock ifNone: exceptionBlock [

"CanonStringDict's do not hold assocations"

^ self shouldNotImplement: #associationsDetect:ifNone:

]

{ #category : 'Enumerating' }
CanonStringDict >> associationsDo: aBlock [
  ^ self shouldNotImplement: #associationsDo: 
]

{ #category : 'Accessing' }
CanonStringDict >> at: aKey ifAbsent: aBlock [

"Returns the value that corresponds to aKey.  If no such key
 exists, returns the result of evaluating the zero-argument block aBlock."

<primitive: 481>
aBlock _isExecBlock ifFalse:[ aBlock _validateClass: ExecBlock] .
^ aBlock value

]

{ #category : 'Accessing' }
CanonStringDict >> at: aKey ifAbsentPut: aBlock [

"not implemented, key-value pairs not supported "

^ self shouldNotImplement: #at:ifAbsentPut:

]

{ #category : 'Accessing' }
CanonStringDict >> at: aString otherwise: value [

"Returns the key that corresponds to aString.  If no such key
 exists, returns the given alternate value."

<primitive: 481>
aString _validateClass: String .
^ value

]

{ #category : 'Updating' }
CanonStringDict >> at: aKey put: aValue [

"CanonStringDict's do not hold key-value pairs, use add: or addKey: "

^ self shouldNotImplement: #at:put:

]

{ #category : 'Enumerating' }
CanonStringDict >> collectAssociations: aBlock [

"CanonStringDict's do not hold key-value pairs."

^ self shouldNotImplement: #collectAssociations:

]

{ #category : 'Enumerating' }
CanonStringDict >> do: aBlock [
"Iteratively evaluates the one argument block, aBlock, using each key of
 the receiver as the argument to the block. Returns the receiver."

^ self keysDo: aBlock

]

{ #category : 'Hashing' }
CanonStringDict >> hashFunction: aString [

^ ((self _basicHash: aString) \\ self tableSize) + 1


]

{ #category : 'Searching' }
CanonStringDict >> includes: aString [

"Returns true if the receiver contains aString as a key, false otherwise."

| aValue |
aValue := self at: aString otherwise: nil .
^ aValue ~~ nil

]

{ #category : 'Searching' }
CanonStringDict >> includesAssociation: anAssociation [

"CanonStringDict's do not hold assocations"

^ self shouldNotImplement: #includesAssociation:

]

{ #category : 'Searching' }
CanonStringDict >> includesIdentical: aString [

"Returns true if the receiver contains an a String identical to the
 argument, false otherwise."

| elem |
elem := self at: aString otherwise: nil .
elem == nil ifFalse:[ ^ aString == elem ].
^ false .

]

{ #category : 'Searching' }
CanonStringDict >> includesIdenticalAssociation: anAssociation [

"CanonStringDict's do not hold assocations"

^ self shouldNotImplement: #includesIdenticalAssociation:

]

{ #category : 'Searching' }
CanonStringDict >> includesKey: aKey [

"Returns true if the receiver contains an Association or a key-value pair whose
 key is equal to aKey.  Returns false otherwise."

^ self includes: aKey

]

{ #category : 'Initializing' }
CanonStringDict >> initialize: itsSize [

"Initializes the instance variables of the receiver to be an empty
 dictionary of the specified size."

| newSize |

newSize := itsSize .

(newSize <= 0)
  ifTrue: [
    newSize _error: #rtErrArgNotPositive.
    newSize := 3
    ].

self _basicSize: 0.  "Set size to 0 and restore to nil any entries"
self _basicSize: newSize .
self _initializeWithoutClear: newSize.
^self

]

{ #category : 'Accessing' }
CanonStringDict >> keyAtValue: anObject ifAbsent: aBlock [

"not implemented, CanonStringDict's do not hold key-value pairs."

^ self shouldNotImplement: #keyAtValue:ifAbsent:

]

{ #category : 'Accessing' }
CanonStringDict >> keys [

"Returns a Set containing the receiver's keys."

| aSet |

aSet := IdentitySet new.
self keysDo: [ :key | aSet add: key ].
^ aSet

]

{ #category : 'Enumerating' }
CanonStringDict >> keysAndValuesDo: aBlock [

"Evaluates aBlock with each of the receiver's key/value pairs as the
 arguments.  The argument aBlock must be a two-argument block.  The
 first argument is the key and the second argument is the value of
 each key/value pair."

1 to: tableSize do:[ :tableIndex | | collisionBkt |
  collisionBkt := self _at: tableIndex .
  collisionBkt ~~ nil ifTrue:[ collisionBkt keysAndValuesDo: aBlock ].
  ]

]

{ #category : 'Enumerating' }
CanonStringDict >> keysDo: aBlock [

"Iteratively evaluates the one argument block, aBlock, using each key of
 the receiver as the argument to the block. Returns the receiver."

| aBucket |
1 to: tableSize do:[:k |
  aBucket := self _at: k .
  aBucket ~~ nil ifTrue:[ aBucket keysDo: aBlock ].
  ].

]

{ #category : 'Searching' }
CanonStringDict >> occurrencesOf: aValue [

"key value pairs are not supported"

^ self shouldNotImplement: #occurrencesOf:

]

{ #category : 'Searching' }
CanonStringDict >> occurrencesOfIdentical: aValue [

"key value pairs are not supported"

^ self shouldNotImplement: #occurrencesOfIdentical:

]

{ #category : 'Formatting' }
CanonStringDict >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

| count sz myCls |

myCls := self class .
aStream nextPutAll: myCls name describeClassName .
aStream nextPutAll: '( ' .
count := 1 .
sz := self size .
self keysDo:[:aKey |
  aStream isFull ifTrue:[
    "prevent infinite recursion when printing cyclic structures, and
     limit the size of result when printing large collections."
    aStream nextPutAll: '...)' .
    ^ self
    ] .
  aKey printOn: aStream .
  count < sz ifTrue:[ aStream nextPutAll: ', ' ].
  count := count + 1 .
  ].
aStream nextPut: $) .

]

{ #category : 'Hashing' }
CanonStringDict >> rehash [
	"Re-establish any hash invariants of the receiver.
	 Rebuilds the hash table by saving the current state, initializing and
	 changing the size of the table, and adding the key values saved
	 back to the hash dictionary."

	| saveTable |
	saveTable := Array new.
	self keysDo: [:aKey | saveTable add: aKey].
	self initialize: self size.
	self addAll: saveTable.

]

{ #category : 'Removing' }
CanonStringDict >> remove: aString [

"Removes aString if present in the receiver and returns the removed value.  If
 aString is not present, generates an error."

^ self removeKey: aString

]

{ #category : 'Removing' }
CanonStringDict >> remove: aString ifAbsent: aBlock [

"Removes aString if present in the receiver and returns the removed value.  If
 aString is not present, returns the result of evaluating the zero
 argument Block aBlock."

^ self removeKey: aString ifAbsent: aBlock

]

{ #category : 'Removing' }
CanonStringDict >> removeKey: aKey ifAbsent: aBlock [

| elem hash aBucket |
elem := self at: aKey otherwise: nil .
elem ifNotNil:[
  hash := self hashFunction: aKey .
  aBucket := self _at:  hash .
  aBucket removeKey: aKey ifAbsent: [ self _error:'inconsistent bucket'].
  numElements := numElements - 1 .
  ^ elem
] .
^ aBlock value

]

{ #category : 'Accessing' }
CanonStringDict >> size [

"Returns the number of elements contained in the receiver."

^ numElements

]

{ #category : 'Class Membership' }
CanonStringDict >> species [

"Returns a class, an instance of which should be used as the result of
 select: or other projections applied to the receiver."

^ StringKeyValueDictionary

]

{ #category : 'Private' }
CanonStringDict >> tableSize [

"Returns the size of hash table used for storing the entries."

^ tableSize

]

{ #category : 'Enumerating' }
CanonStringDict >> valuesDo: aBlock [
  ^ self shouldNotImplement: #valuesDo: 
]
