Extension { #name : 'BtreeReadStream' }

{ #category : 'Instance Creation' }
BtreeReadStream class >> on: aBtreeNode [

"Create a stream that can access the entire contents of the B-tree whose
 root node is aBtreeNode."

| arr1 arr2 newStream |
arr1 := { } .
arr2 := { } .

aBtreeNode _putFirstIndexOfFirstChildInto: arr1.
aBtreeNode _putLastIndexOfLastChildInto: arr2.

newStream := self new.
newStream endIndex: (arr2 at: 1).
newStream endNode: (arr2 at: 2).
newStream currentStack: arr1.

^ newStream

]

{ #category : 'Testing' }
BtreeReadStream >> _atEnd [
  "Returns true if the receiver is positioned at the end of the stream, and
 false otherwise."

  ^ (currentStack at: 1) == 0

]

{ #category : 'Testing' }
BtreeReadStream >> _btreeAtEnd [

""

^ (currentStack at: 1) == 0

]

{ #category : 'Query Support' }
BtreeReadStream >> _btreeComparisonQuerySpecClass [
  ^ (endNode notNil and: [ endNode collator notNil ])
    ifTrue: [ BtreeUnicodeComparisonQuerySpec ]
    ifFalse: [ BtreeComparisonQuerySpec ]

]

{ #category : 'Accessing' }
BtreeReadStream >> _btreeNext [

"Returns the next value on a stream of B-tree values.  Updates the current
 stack for a subsequent 'next'."

| val leaf index lastValIndex |
" get the index into the leaf node and see if it has reached the end "
index := currentStack at: 1.
index == 0
    ifTrue: [ ^ self _errorEndOfStream ].

" get the leaf and the value within the leaf "
leaf := currentStack at: 2.
val := leaf _basicAt: index.

" if this is the end of the stream, sets the top index to zero and returns "
( leaf == endNode and: [ endIndex == index ] )
    ifTrue: [
        currentStack at: 1 put: 0.
        ^ val
    ].

" see if index refers to last entry in this leaf "
lastValIndex := leaf _lastKeyIndex - 1.
index > lastValIndex
    ifTrue: [ self _error: #rtErrInvalidBtreeReadStream ].

index == lastValIndex
    ifTrue: [
        " must look down the stack for the next leaf node "
        self _nextLeaf
    ]
    ifFalse: [ currentStack at: 1 put: (index + leaf entrySize) ].

^ val

]

{ #category : 'Query Support' }
BtreeReadStream >> _btreeRangeComparisonQuerySpecClass [
  ^ (endNode notNil and: [ endNode collator notNil ])
    ifTrue: [ BtreeUnicodeRangeComparisonQuerySpec ]
    ifFalse: [ BtreeRangeComparisonQuerySpec ]

]

{ #category : 'Error Handling' }
BtreeReadStream >> _errorEndOfStream [

"Raises an error due to an attempt to read beyond the end of the stream."

^ self _error: #rtErrBtreeReadStreamEndOfStream .

]

{ #category : 'Query Support' }
BtreeReadStream >> _hasSameEndAs: otherStream [

"Returns true if the receiver has the same end index and end node as the
 otherStream."

^ endIndex == otherStream endIndex and: [
   endNode == otherStream endNode ]

]

{ #category : 'Accessing' }
BtreeReadStream >> _nextLeaf [

"Looks down the stack for the next leaf node and makes it the current node."

| nodeIndex notPositioned currentNode currentIndex nextIndex node |

nodeIndex := 4.
notPositioned := true.
[ notPositioned ] whileTrue: [
    currentIndex := currentStack at: nodeIndex - 1.
    nil == (currentNode := currentStack at: nodeIndex)
        ifTrue: [ self _error: #rtErrInvalidBtreeReadStream ].


    " if the current index is the last in the node "
    currentIndex == (currentNode _lastKeyIndex - 1)
        ifTrue: [ " must look down the stack another level "
            nodeIndex == currentStack size
                ifTrue: [ ^ self _errorEndOfStream ].
            nodeIndex := nodeIndex + 2
        ]
        ifFalse: [ " current node has a next entry "
            nextIndex := currentIndex + currentNode entrySize.
            " put the next entry (index and node) on the stack "
            currentStack at: nodeIndex - 1 put: nextIndex.
            nil == (node := currentNode _basicAt: nextIndex)
                ifTrue: [ self _error: #rtErrInvalidBtreeReadStream ].
            currentStack at: nodeIndex - 2 put: node.

            " put the first child on any remaining levels in the stack "
            nodeIndex - 3 _downTo: 3 by: 2 do: [ :i |
                currentStack at: i put: 1.
                currentStack at: i -1 put: ((currentStack at: i + 1) _basicAt: 1)
            ].
            currentStack at: 1 put: 1.
            notPositioned := false
        ]
]

]

{ #category : 'Private' }
BtreeReadStream >> _peekKey [

"Returns the key of the current entry in the stream."

| i |
i := currentStack at: 1.
i == 0
    ifTrue: [ ^ self _errorEndOfStream ].
^ (currentStack at: 2) _basicAt: i + 1

]

{ #category : 'Private' }
BtreeReadStream >> _peekValue [
  "Returns the value of the current entry in the stream."

  | i |
  i := currentStack at: 1.
  i == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ (currentStack at: 2) _basicAt: i

]

{ #category : 'Query Support' }
BtreeReadStream >> _positionToEnd [

"Sets the current stack to the end index and end node.  This
 has the effect of advancing the stream to the last object."

currentStack at: 1 put: endIndex.
currentStack at: 2 put: endNode

]

{ #category : 'Query Support' }
BtreeReadStream >> _valuesAddInto: collection [

"Adds each entry value into the given NSC."

| querySpec |
querySpec := BtreeQuerySpec new.
^self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreeReadStream >> _valuesAddInto: collection comparing: aSelector key: aKey [
  "Adds each entry value into the given NSC."

  | querySpec |
  querySpec := self _btreeComparisonQuerySpecClass key: aKey selector: aSelector.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreeReadStream >> _valuesAddInto: collection comparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2 [
  "Adds each entry value into the given NSC."

  | querySpec |
  querySpec := self _btreeRangeComparisonQuerySpecClass
    key: aKey1
    selector: aSelector1
    and: aKey2
    selector: aSelector2.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreeReadStream >> _valuesAddInto: collection identicalTo: aKey [
  "For each entry whose key is identical to the given key, add the value into
 the given collection."

  | querySpec |
  querySpec := self _btreeComparisonQuerySpecClass key: aKey selector: #'=='.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreeReadStream >> _valuesAddInto: collection notEqualTo: aKey [
  "For each entry whose key is not identical to the given key, add the value
 into the given collection."

  | querySpec |
  querySpec := self _btreeComparisonQuerySpecClass key: aKey selector: #'~='.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreeReadStream >> _valuesAddInto: collection notIdenticalTo: aKey [
  "For each entry whose key is not identical to the given key, add the value
 into the given collection."

  | querySpec |
  querySpec := self _btreeComparisonQuerySpecClass key: aKey selector: #'~~'.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreeReadStream >> _valuesAddInto: collection spec: querySpec [

"Adds each entry value into the given NSC."

| leaf index eSize |

" get the index into the leaf node and see if it has reached the end "
index := currentStack at: 1.
index == 0
    ifTrue: [ ^ self ].

leaf := currentStack at: 2.
eSize := endNode entrySize.
" while current node is not the end node "
[ endNode ~~ leaf ] whileTrue: [
    " put the values for the current node in the set "
    querySpec addValuesFrom: leaf
                     start: index
                     end: leaf numElements * eSize
                     into: collection.
    " look down the stack for the next leaf node "
    self _nextLeaf.

    leaf := currentStack at: 2.
    index := 1
].

" put values of end node in the NSC "
querySpec addValuesFrom: leaf
                 start: index
                 end: endIndex
                 into: collection.

]

{ #category : 'Testing' }
BtreeReadStream >> atEnd [
  "Returns true if the receiver is positioned at the end of the stream, and
 false otherwise."

  ^ self _atEnd

]

{ #category : 'Query Support' }
BtreeReadStream >> btreeRangeComparisonQuerySpec [
  ^ self _btreeRangeComparisonQuerySpecClass new

]

{ #category : 'Accessing' }
BtreeReadStream >> currentStack [

"Returns the value of the instance variable 'currentStack'."

^currentStack

]

{ #category : 'Updating' }
BtreeReadStream >> currentStack: newValue [

"Modifies the value of the instance variable 'currentStack'."

currentStack := newValue

]

{ #category : 'Accessing' }
BtreeReadStream >> endIndex [

"Returns the value of the instance variable 'endIndex'."

^endIndex

]

{ #category : 'Updating' }
BtreeReadStream >> endIndex: newValue [

"Modify the value of the instance variable 'endIndex'."

endIndex := newValue

]

{ #category : 'Accessing' }
BtreeReadStream >> endNode [

"Returns the value of the instance variable 'endNode'."

^endNode

]

{ #category : 'Updating' }
BtreeReadStream >> endNode: newValue [

"Modify the value of the instance variable 'endNode'."

endNode := newValue

]

{ #category : 'Accessing' }
BtreeReadStream >> next [

"Returns the next value on a stream of B-tree values.  Update the current
 stack for a subsequent 'next'."

self _btreeAtEnd
    ifTrue: [ ^ self _errorEndOfStream ].

^ self _btreeNext

]

{ #category : 'Adding' }
BtreeReadStream >> nextPut: anObject [

"Disallowed.  You cannot write to a BtreeReadStream."

self shouldNotImplement: #nextPut:

]

{ #category : 'Accessing' }
BtreeReadStream >> size [

"Returns the number of elements contained in the receiver (that is, how many
successful 'next' operations can be performed)."

| array |
self atEnd ifTrue: [
  ^ 0
  ].
array := { 0 . false }.
(currentStack at: currentStack size)
  _totalElementsIn: currentStack
  endNode: endNode
  endIndex: endIndex
  into: array.
^ array at: 1

]

{ #category : 'Accessing' }
BtreeReadStream >> streamQuerySpec [
  ^ streamQuerySpec

]

{ #category : 'Accessing' }
BtreeReadStream >> streamQuerySpec: aBtreeQuerySpec [
  streamQuerySpec := aBtreeQuerySpec

]
