Extension { #name : 'BtreePlusLeafKeyValueNode' }

{ #category : 'Constants' }
BtreePlusLeafKeyValueNode class >> entrySize [
  "Returns the size of an entry with no encryption."

  "in-lined on instance-side for performance"

  ^ 2

]

{ #category : 'Constants' }
BtreePlusLeafKeyValueNode class >> maxNumberOfElements [
  "Returns the number of entries that are allowed in an interior node."

  ^ 1014

]

{ #category : 'Updating' }
BtreePlusLeafKeyValueNode >> _insertKey: aKey value: aValue atIndex: insertionIndex [
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

  | endIndex entrySize |
  entrySize := self entrySize.
  endIndex := numElements * entrySize.

  self _insertAt: insertionIndex
       value: aValue value: aKey
       numToMoveDown: endIndex - insertionIndex + 1 .

  "no prefix encoding"

  numElements := numElements + 1



]

{ #category : 'Updating' }
BtreePlusLeafKeyValueNode >> _insertKey: aKey value: aValue root: ignored atIndex: insertionIndex [

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

self _insertKey: aKey value: aValue atIndex: insertionIndex

]

{ #category : 'Updating' }
BtreePlusLeafKeyValueNode >> _insertKey: aKey value: aValue root: ignored atIndex: insertionIndex selectiveAbortSet: ignoredSet [

"Compat with BtreePlustInteriorNode api ... inserts do not require selective abort logic"

self _insertKey: aKey value: aValue root: ignored atIndex: insertionIndex

]

{ #category : 'Updating' }
BtreePlusLeafKeyValueNode >> at: aKey put: aValue root: ignored [
  "Adds the key/value pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey value: aValue ].
      self _insertKey: aKey value: aValue atIndex: index ]

]

{ #category : 'Updating' }
BtreePlusLeafKeyValueNode >> at: aKey put: aValue root: ignored selectiveAbortSet: selectiveAbortSetOrNil [
  "Adds the key/value pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue selectiveAbortSet: selectiveAbortSetOrNil ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey value: aValue ].
      self _insertKey: aKey value: aValue atIndex: index ]

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyValueNode >> compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex root: rootObject [
  "Performs a = comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, use the OOP of the value as the next basis for
 comparison."

  ^ self compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyValueNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: ignoredRootObject [
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, then use the OOP of the value for the comparison."

  ^self compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyValueNode >> compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: ignored [
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is
   at the given index. "

  ^ self compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyValueNode >> compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: ignored [
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  ^ self compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex

]

{ #category : 'Constants' }
BtreePlusLeafKeyValueNode >> entrySize [
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  ^ 2

]

{ #category : 'Constants' }
BtreePlusLeafKeyValueNode >> parentNodeClass [
  "Returns the class of node to be created as the parent when a split occurs."

  ^ BtreePlusInteriorKeyValueRootObjectNode

]

{ #category : 'Accessing' }
BtreePlusLeafKeyValueNode >> rootIndexOffset [
  "Answer the offset from entry index for the root object. Typical order is value/key/root.
   The value and root object is the same for the receiver."

  ^ 0

]
