Extension { #name : 'BtreePlusGsReversedIndexReadStream' }

{ #category : 'Instance Creation' }
BtreePlusGsReversedIndexReadStream class >> on: aGsRangeIndex [
  "Create a stream that can access the entire contents of the B-tree whose
 root node is aBtreeNode."

  | arr1 arr2 newStream btreeNode |
  arr1 := {}.
  arr2 := {}.
  btreeNode := aGsRangeIndex btreeRoot.
  btreeNode _putFirstIndexOfFirstChildInto: arr2.
  btreeNode _putLastIndexOfLastChildInto: arr1.
  newStream := self new.
  newStream endIndex: (arr2 at: 1).
  newStream endNode: (arr2 at: 2).
  newStream currentStack: arr1.
  newStream rangeIndex: aGsRangeIndex.
  ^ newStream

]

{ #category : 'Accessing' }
BtreePlusGsReversedIndexReadStream >> _btreeNextRootObject [
  "Returns the previous root object on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'previous'."

  ^self _btreePreviousRootObject

]

{ #category : 'Accessing' }
BtreePlusGsReversedIndexReadStream >> _btreeNextValue [

"Returns the previous value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'previous'."

^ self _btreePreviousValue

]

{ #category : 'Accessing' }
BtreePlusGsReversedIndexReadStream >> _btreePreviousNoValue [
  "Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

  " get the index into the leaf node and see if it has reached the end "
  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].	" get the leaf and the value within the leaf "
  (currentNode == endNode and: [ endIndex == currentIndex ])
    ifTrue: [ 
      currentIndex := 0.
      ^ self ].	" see if index refers to first entry in this leaf "
  currentIndex == 1
    ifTrue: [ 
      " must look down the stack for the next leaf node "
      self _previousLeaf ]
    ifFalse: [ currentIndex := currentIndex - currentEntrySize ].

]

{ #category : 'Accessing' }
BtreePlusGsReversedIndexReadStream >> _btreePreviousRootObject [
  "Returns the next root object on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

  | root |
  " get the root value within the leaf "
  root := currentNode rootObjectAt: currentIndex.
  " if this is the end of the stream, sets the top currentIndex to zero and returns "
  (currentNode == endNode and: [ endIndex == currentIndex ])
    ifTrue: [ currentIndex := 0.
      ^ root ].
  " see if currentIndex refers to last entry in this leaf "
  currentIndex == 1
    ifTrue: [ self _previousLeaf ]
    ifFalse: [ currentIndex := currentIndex - currentEntrySize ].
  ^ root

]

{ #category : 'Accessing' }
BtreePlusGsReversedIndexReadStream >> _btreePreviousValue [

"Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

| val    |
" get the index into the leaf node and see if it has reached the end "
currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].

" get the leaf and the value within the leaf "
val := currentNode at: currentIndex.

" if this is the end of the stream, sets the top index to zero and returns "
( currentNode == endNode and: [ endIndex == currentIndex ] )
    ifTrue: [
        currentIndex := 0.
        ^ val
    ].

" see if index refers to first entry in this leaf "
currentIndex == 1
    ifTrue: [
        " must look down the stack for the next leaf node "
        self _previousLeaf
    ]
    ifFalse: [ currentIndex := (currentIndex - currentEntrySize) ].

^ val

]

{ #category : 'Accessing' }
BtreePlusGsReversedIndexReadStream >> _previousLeaf [
  "Get the previous leaf node and make it the current node."

  self currentNode: currentNode previousLeaf.
  currentNode ifNil: [ ^ self _errorEndOfStream ].
  currentIndex := currentLastValIndex

]

{ #category : 'Query Support' }
BtreePlusGsReversedIndexReadStream >> _valuesAddInto: collection spec: querySpec [
  "Adds each entry value into the given NSC."

  | eSize |
  " get the index into the leaf node and see if it has reached the end "
  currentIndex == 0
    ifTrue: [ ^ self ].
  eSize := endNode entrySize.
  " while current node is not the end node "
  [ endNode ~~ currentNode ]
    whileTrue: [ " put root objects for the current node in the collection "
      querySpec
        addRootObjectsFrom: currentNode
        start: 1
        end: currentIndex
        into: collection.
      " get the previous leaf node "
      self _previousLeaf ].
  " put root objects of end node in the collection "
  querySpec
    addRootObjectsFrom: currentNode
    start: endIndex
    end: currentIndex
    into: collection

]

{ #category : 'Private' }
BtreePlusGsReversedIndexReadStream >> reversedRangeIndexClass [
  ^ BtreePlusGsIndexReadStream

]

{ #category : 'Accessing' }
BtreePlusGsReversedIndexReadStream >> size [

"Returns the number of elements contained in the receiver (that is, how many
successful 'previous' operations can be performed)."

| total theNode theIndex lastNode |
self atEnd ifTrue: [
  ^ 0
  ].
total := 0.
theNode := currentNode.
theIndex := currentIndex.
lastNode := false.
[ lastNode ] whileFalse: [
  lastNode := theNode == endNode.
  total := total + (theNode
    _totalElementsReversedFrom: theIndex
    toEndNode: endNode
    endIndex: endIndex).
  lastNode
    ifFalse: [
      theNode := theNode previousLeaf.
      theIndex := theNode _lastIndex ] ].
^ total

]

{ #category : 'Accessing' }
BtreePlusGsReversedIndexReadStream >> skip: amount [
  "advance the receiver's position by amount elements"

  | count |
  count := 0.
  [ self atEnd not and: [ count < amount ] ]
    whileTrue: [ 
      self _btreePreviousNoValue.
      count := count + 1 ]

]
