Extension { #name : 'BtreePlusGsIndexReadStream' }

{ #category : 'Instance Creation' }
BtreePlusGsIndexReadStream class >> on: aGsRangeIndex [
  "Create a stream that can access the entire contents of the given RangeEqualityIndex."

  | newStream |
  newStream := super on: aGsRangeIndex btreeRoot.
  newStream rangeIndex: aGsRangeIndex.
  ^ newStream

]

{ #category : 'Private' }
BtreePlusGsIndexReadStream >> _advance [
   "Advances the B-tree portion of the stream without advancing any of the set
     iteration indexes."

   ^ self next

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> _scanForRootObjectMaps: aKey value: aValue valueOop: valueOop using: comparisonForCompare do: aBlock [
  "Adds each entry value into the given NSC."

  | eSize |
  " get the index into the leaf node and see if it has reached the end "
  currentIndex == 0
    ifTrue: [ ^ self ].
  eSize := endNode entrySize.
  " while current node is not the end node "
  [ endNode ~~ currentNode ]
    whileTrue: [
      " scan key and values of currentNode "
      currentNode
        _scanForRootObjectMapsStartingAt: currentIndex
        end: currentLastValIndex
        key: aKey
        value: aValue
        valueOop: valueOop
        using: comparisonForCompare
        do: aBlock.
      " get the next leaf node "
      self _nextLeaf ].
  " scan key and values of endNode "
  currentNode
    _scanForRootObjectMapsStartingAt: currentIndex
    end: endIndex
    key: aKey
    value: aValue
    valueOop: valueOop
    using: comparisonForCompare
    do: aBlock.

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> _valuesAddInto: collection [

"Adds each entry value into the given NSC."

| querySpec |
querySpec := BtreeQuerySpec new.
^self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> _valuesAddInto: collection comparing: aSelector key: aKey [
  "Adds each entry value into the given NSC."

  | querySpec |
  querySpec := self btreeComparisonQuerySpec
    key: aKey selector: aSelector;
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> _valuesAddInto: collection comparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2 [
  "Adds each entry value into the given NSC."

  | querySpec |
  querySpec := self btreeRangeComparisonQuerySpec
    key: aKey1 selector: aSelector1;
    key2: aKey2 selector2: aSelector2;
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> _valuesAddInto: collection identicalTo: aKey [
  "For each entry whose key is identical to the given key, add the value into
 the given collection."

  | querySpec |
  querySpec := self btreeComparisonQuerySpec
    key: aKey selector: #'==';
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> _valuesAddInto: collection notEqualTo: aKey [
  "For each entry whose key is not identical to the given key, add the value
 into the given collection."

  | querySpec |
  querySpec := self btreeComparisonQuerySpec
    key: aKey selector: #'~=';
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> _valuesAddInto: collection notIdenticalTo: aKey [
  "For each entry whose key is not identical to the given key, add the value
 into the given collection."

  | querySpec |
  querySpec := self btreeComparisonQuerySpec
    key: aKey selector: #'~~';
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> _valuesAddInto: collection spec: querySpec [
  "Adds each entry value into the given NSC."

  | eSize |
  " get the index into the leaf node and see if it has reached the end "
  currentIndex == 0
    ifTrue: [ ^ self ].
  eSize := endNode entrySize.
  " while current node is not the end node "
  [ endNode ~~ currentNode ]
    whileTrue: [ " put the values for the current node in the collection "
      querySpec
        addRootObjectsFrom: currentNode
        start: currentIndex
        end: currentNode numElements * eSize
        into: collection.
      " get the next leaf node "
      self _nextLeaf ].
  " put values of end node in the collection "
  querySpec
    addRootObjectsFrom: currentNode
    start: currentIndex
    end: endIndex
    into: collection

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> approxNumElements [

"Returns the number of leaf node entries represented by the receiver.  This is
 approximately the number of elements because objects may share sub-objects
 which converge to the same leaf node entry."

^ self size

]

{ #category : 'Testing' }
BtreePlusGsIndexReadStream >> atEnd [
  "Returns true if there are no more elements to return through the logical
 iteration of the stream."

  mustCompare
    ifTrue: [ "bug 44079: need to skip key/value pairs that do not satisfy the querySpec"
      [ | aKey |
      currentIndex == 0
        ifTrue: [ ^ true ].
      aKey := currentNode keyAt: currentIndex.
      streamQuerySpec compareKey: aKey ] whileFalse: [
          self _btreeNextRootObject.
          currentIndex == 0
            ifTrue: [ ^ true ] ] ].
  ^ currentIndex == 0

]

{ #category : 'Accessing' }
BtreePlusGsIndexReadStream >> btreeComparisonQuerySpec [
  ^ self gsIndex btreeComparisonQuerySpec

]

{ #category : 'Accessing' }
BtreePlusGsIndexReadStream >> btreeRangeComparisonQuerySpec [
  ^ self gsIndex btreeRangeComparisonQuerySpec

]

{ #category : 'Private' }
BtreePlusGsIndexReadStream >> getNext [
  "Returns the next value on a stream of range index values."

  "QueryExecutor compat"

  ^ self next

]

{ #category : 'Updating' }
BtreePlusGsIndexReadStream >> gsIndex [
  "Answer the value of the instance variable 'gsIndex'."

  ^gsIndex

]

{ #category : 'Updating' }
BtreePlusGsIndexReadStream >> identityIndex: aGsIndex [
  "Modify the value of the instance variable 'gsIndex'."

  gsIndex := aGsIndex

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> makeNsc [

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: gsIndex nscRoot species new max: gsIndex nscRoot size.

streamQuerySpec
  ifNil: [
    self _valuesAddInto: tmpHolder ]
  ifNotNil: [
    self _valuesAddInto: tmpHolder spec: streamQuerySpec ].

" make the stream be at the end "
currentIndex := 0.

^ tmpHolder completeBag

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> makeNscComparing: aSelector key: aKey [

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: gsIndex nscRoot species new max: gsIndex nscRoot size.

self _valuesAddInto: tmpHolder comparing: aSelector key: aKey.

" make the stream be at the end "
currentIndex := 0.

^ tmpHolder completeBag

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> makeNscComparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2 [

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: gsIndex nscRoot species new max: gsIndex nscRoot size.

self _valuesAddInto: tmpHolder comparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2.

" make the stream be at the end "
currentIndex := 0.
^ tmpHolder completeBag

]

{ #category : 'Accessing' }
BtreePlusGsIndexReadStream >> next [
  "Returns the next root object on a stream of B-tree values and root objects.  Update the current
 stack for a subsequent 'next'."

  optimizedComparison
    ifTrue: [
      currentIndex == 0
        ifTrue: [ ^ self _errorEndOfStream ].
     ^ self _btreeNextRootObject ].
  "bug 44079: use #atEnd to skip key/value pairs that do not satisfy the querySpec ... "
  self atEnd
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ self _btreeNextRootObject

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> nextRootComparing: aSelector key: aKey [
  " iterate through the stream, return first value that satisfies query "

  [ self _atEnd not ]
    whileTrue: [
      (self _peekKey perform: aSelector with: aKey)
        ifTrue: [ ^ self next ].
      self _btreeNext ].
  ^ #'_incompletePathTraversal'

]

{ #category : 'Query Support' }
BtreePlusGsIndexReadStream >> nextRootComparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2 [
  " iterate through the stream, return first value that satisfies query "

  [ self _atEnd not ]
    whileTrue: [
      | key |
      key := self _peekKey.
      ((key perform: aSelector1 with: aKey1)
        and: [ key perform: aSelector2 with: aKey2 ])
        ifTrue: [ ^ self next ].
      self _btreeNext ].
  ^ #'_incompletePathTraversal'

]

{ #category : 'Updating' }
BtreePlusGsIndexReadStream >> rangeIndex: aGsIndex [
  "Modify the value of the instance variable 'gsIndex'."

  gsIndex := aGsIndex.
  optimizedComparison := gsIndex optimizingComparison.
  mustCompare := false

]

{ #category : 'Conversion' }
BtreePlusGsIndexReadStream >> reversed [

  | reversed |
  reversed := (self reversedRangeIndexClass on: self gsIndex).

  self atEnd
    ifTrue: [ reversed currentIndex: 0 ]
    ifFalse: [
      reversed
        currentIndex: endIndex;
        currentNode: endNode;
        endIndex: currentIndex;
        endNode: currentNode;
        streamQuerySpec: self streamQuerySpec;
        yourself ].
  ^ reversed

]

{ #category : 'Private' }
BtreePlusGsIndexReadStream >> reversedRangeIndexClass [
  ^ BtreePlusGsReversedIndexReadStream

]

{ #category : 'Accessing' }
BtreePlusGsIndexReadStream >> skip: amount [
  "advance the receiver's position by amount elements"

  | count |
  count := 0.
  [ self atEnd not and: [ count < amount ] ]
    whileTrue: [ 
      self _btreeNextNoValue.
      count := count + 1 ]

]

{ #category : 'Accessing' }
BtreePlusGsIndexReadStream >> streamQuerySpec [
  ^ streamQuerySpec

]

{ #category : 'Accessing' }
BtreePlusGsIndexReadStream >> streamQuerySpec: aBtreeQuerySpec [
  streamQuerySpec := aBtreeQuerySpec.
  (optimizedComparison
    and: [ aBtreeQuerySpec == nil or: [ aBtreeQuerySpec canUseOptimizedComparison ] ])
    ifTrue: [ mustCompare := false ]
    ifFalse: [
      "bug 44079: need to skip key/value pairs that do not satisfy the streamQuerySpec.
       If optimizedComparison is true, then not exposed to bug 44079. "
      mustCompare := aBtreeQuerySpec notNil ]

]
