Extension { #name : 'BtreeNode' }

{ #category : 'Constants' }
BtreeNode class >> entrySize [

"Returns the size of an entry with no encryption."

^ 2

]

{ #category : 'Constants' }
BtreeNode class >> maxNumberOfElements [

"Returns the number of entries that are allowed in an interior node."

^ 1000

]

{ #category : 'Instance Creation' }
BtreeNode class >> new [

"Returns a new initialized instance with the correct size."

^ (super new: self entrySize * self maxNumberOfElements) numElements: 0.

]

{ #category : 'Audit' }
BtreeNode >> _auditKey: key value: value bag: bag nsc: nsc for: pathTerm offset: offset enumBag: enumBag on: aString [
  "Private. "

  | num ar index startPt |
  index := pathTerm at: offset.
  index indexDictionary ~~ nil
    ifTrue: [
      ar := {}.
      index isIndexOnNscElements
        ifTrue: [ startPt := index size ]
        ifFalse: [ startPt := index lastPathComponentsDictionaryOffset ].
      startPt := startPt max: 1.
      index
        _addAllFor: value
        into: ar
        offset: 0
        pathTerm: (index at: startPt).
      num := ar size ]
    ifFalse: [ num := nsc occurrencesOf: value ].
  pathTerm isEnumeratedTerm
    ifTrue: [
      pathTerm == (index at: index size)
        ifTrue: [ num := (enumBag occurrencesOf: value) ]
        ifFalse: [ num := (enumBag occurrencesOf: value) * num ] ].
  bag size == num
    ifFalse: [
      pathTerm
        auditInfo:
          {key.
          nil.
          nil}
        on: aString.
      aString
        add: 'Btree [';
        add: self asOop asString;
        add: '] has incorrect number of entries (';
        add: bag size asString;
        add: ') for [';
        add: value asOop asString;
        add: '] of class ';
        add: value class asString;
        add: ' (should be ';
        add: num asString;
        add: ')';
        add: Character lf . self checkDebug: aString ].
  bag removeAll: bag.
  enumBag removeAll: enumBag

]

{ #category : 'Audit' }
BtreeNode >> _auditLeafKeysValuesAndEncryptedFor: pathTerm on: aString do: aBlock [
  self subclassResponsibility: #'_auditLeafKeysValuesAndEncryptedFor:on:do:'

]

{ #category : 'Testing' }
BtreeNode >> _basicCanCompare: aKey withClass: aClass [
  "Returns true if the receiver can make comparisons with the given key, and
false otherwise.  Used for basic nodes."

  ^ aKey _idxBasicCanCompareWithClass: aClass

]

{ #category : 'Searching' }
BtreeNode >> _binarySearchCoveringKey: aKey totalOrder: aBoolean [
  "Returns the index for the first entry in which aKey is found utilizing a
binary search.  This is the first entry whose key >= aKey."

  | lowPt midPt highPt entrySize index entryGreaterThan comparisonForSort |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  [ lowPt <= highPt ] whileTrue: [ midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 2.
      entryGreaterThan := false.
      (self _compare: comparisonForSort key: aKey lessThanEntryAt: index)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ (self _compare: comparisonForSort key: aKey greaterThanEntryAt: index)
            ifTrue: [ lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ highPt := midPt - 1 ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - 1 ].
  ^ index - 1

]

{ #category : 'Searching' }
BtreeNode >> _binarySearchCoveringKey: aKey value: aValue [
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys
 are inserted by the value's OOP ordering).  If the receiver is empty, returns
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  [ lowPt <= highPt ] whileTrue: [ entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 2.
      (self
        _compare: comparisonForSort
        key: aKey
        value: aValue
        lessThanEntryAt: index)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ (self
            _compare: comparisonForSort
            key: aKey
            value: aValue
            greaterThanEntryAt: index)
            ifTrue: [ lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ (self _compare: comparisonForSort key: aKey equalToEntryAt: index)
                ifTrue: [ (self _compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: index)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - 1 ].
  (entryEqualTo
    and: [ self _compareValueOop: aValue greaterThanEntryValueOopAt: index ])
    ifTrue: [ ^ index + entrySize - 1 ].
  ^ index - 1

]

{ #category : 'Searching' }
BtreeNode >> _binarySearchForKey: aKey value: aValue [
  "Returns the 'value' index whose key equality matches aKey.  If no match is
found, returns zero.  Uses a binary search."

  | lowPt midPt highPt entrySize index comparisonForSort |
  numElements == 0
    ifTrue: [ ^ 0 ].
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  comparisonForSort := self comparisonForSort.
  [ lowPt <= highPt ] whileTrue: [ midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 2.
      (self
        _compare: comparisonForSort
        key: aKey
        value: aValue
        equalToEntryAt: index)
        ifTrue: [ ^ index - 1 ].
      (self
        _compare: comparisonForSort
        key: aKey
        value: aValue
        lessThanOrEqualToEntryAt: index)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ lowPt := midPt + 1 ] ].
  ^ 0

]

{ #category : 'Testing' }
BtreeNode >> _canCompare: aKey withClass: aClass [

"Returns whether the receiver can make comparisons with the given key.
Returns true since the comparison will actually be attempted."

^ true

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compare: aComparison key: aKey equalToEntryAt: index [

"Performs a = comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey equalTo: (self _at: index)

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compare: aComparison key: aKey greaterThanEntryAt: index [

"Perform a > comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey greaterThan: (self _at: index)

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compare: aComparison key: aKey greaterThanOrEqualToEntryAt: index [

"Perform a >= comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey greaterThanOrEqualTo: (self _at: index)

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compare: aComparison key: aKey lessThanEntryAt: index [

"Perform a < comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey lessThan: (self _at: index)

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compare: aComparison key: aKey lessThanOrEqualToEntryAt: index [

"Perform a <= comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey lessThanOrEqualTo: (self _at: index)

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compare: comparisonForSort key: aKey value: aValue greaterThanEntryAt: index [
  "Performs a > comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison.  The default implementation uses no encryption."

  " first compare the keys "

  (comparisonForSort compareKey: aKey greaterThan: (super at: index))
    ifTrue: [ ^ true ].
  ^ false

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compare: comparisonForSort key: aKey value: aValue lessThanEntryAt: index [
  "Performs a < comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison.  The default implementation uses no encryption."

  " first compare the keys "

  (comparisonForSort compareKey: aKey lessThan: (super at: index))
    ifTrue: [ ^ true ].
  ^ false

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compareKey: aKey equalToEntryAt: index [
  "Performs a = comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

  ^ self _compare: self comparisonForSort key: aKey equalToEntryAt: index

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compareKey: aKey greaterThanEntryAt: index [
  "Perform a > comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

  ^ self _compare: self comparisonForSort key: aKey greaterThanEntryAt: index

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compareKey: aKey greaterThanOrEqualToEntryAt: index [
  "Perform a >= comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

  ^ self
    _compare: self comparisonForSort
    key: aKey
    greaterThanOrEqualToEntryAt: index

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compareKey: aKey lessThanEntryAt: index [
  "Perform a < comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

  ^ self _compare: self comparisonForSort key: aKey lessThanEntryAt: index

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compareKey: aKey lessThanOrEqualToEntryAt: index [
  "Performs a <= comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

  ^ self
    _compare: self comparisonForSort
    key: aKey
    lessThanOrEqualToEntryAt: index

]

{ #category : 'Comparison Operators' }
BtreeNode >> _compareLastKeyLessThanOrEqualToKey: aKey value: aValue [
  "Returns whether the receivers last key/value pair is greater than or equal to
the given key/value pair."

  numElements == 0
    ifTrue: [ ^ false ].	" return aKey <= last key "
  ^ self
    _compare: self comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: self _lastKeyIndex

]

{ #category : 'Updating' }
BtreeNode >> _createSibling [
  "Creates a new sibling node and moves half of the receiver's entries into the
sibling.  Returns the new sibling."

  | newSibling sz splitIndex halfNumElements |
  newSibling := self class new.
  newSibling
    objectSecurityPolicy: self objectSecurityPolicy;
    collator: self collator.
  sz := numElements * self entrySize.
  halfNumElements := numElements quo: 2.
  splitIndex := halfNumElements * self entrySize + 1.	" splitIndex is where to split the receiver "
  newSibling
    _insertAt: 1
    from: self
    fromStart: splitIndex
    fromEnd: sz
    numToMoveDown: 0.	" copy the last half of entries into the new sibling "
  self _deleteNoShrinkFrom: splitIndex to: sz.	" nil out last half of receiver "
  newSibling numElements: numElements - halfNumElements.	" update the element counts in the receiver..."
  numElements := halfNumElements.	"... and the sibling "
  ^ newSibling

]

{ #category : 'Constants' }
BtreeNode >> _encryptionSize [

"There is no default encryption."

^ 0

]

{ #category : 'Searching' }
BtreeNode >> _findAllValuesForIdenticalKey: aKey into: aCollection [
  "Finds all the values for the given identical key, placing them in the
 collection.  Return whether the last key was equal or not (so the caller
 knows it must check the next leaf node)."

  ^ self
    _findAllValuesForIdenticalKey: aKey
    into: aCollection
    using: self comparisonForSort

]

{ #category : 'Searching' }
BtreeNode >> _findAllValuesForKey: aKey into: aCollection [
  "Finds all the values for the given key, placing them in the collection."

  ^ self
    _findAllValuesForKey: aKey
    into: aCollection
    using: self comparisonForSort

]

{ #category : 'Searching Support' }
BtreeNode >> _findAllValuesGreaterThanKey: aKey into: anArray [
  "Puts into anArray the index and the leaf node of the entry that is greater
 than the given key.  Returns true if a value was found, false otherwise."

  ^ self
    _findAllValuesGreaterThanKey: aKey
    into: anArray
    using: (BtreeComparisonForCompare newForComparison: self collator)

]

{ #category : 'Searching Support' }
BtreeNode >> _findAllValuesLessThanKey: aKey andEquals: aBoolean into: anArray [
  "Puts into anArray information about the location of the match.  Returns true
 if a value was found, false otherwise."

  ^ self
    _findAllValuesLessThanKey: aKey
    andEquals: aBoolean
    into: anArray
    using: (BtreeComparisonForCompare newForComparison: self collator)

]

{ #category : 'Searching Support' }
BtreeNode >> _findAllValuesLessThanKey: aKey into: anArray [
  "Puts into anArray information about the location of the match.  Returns true
 if a value was found, false otherwise."

  ^ self
    _findAllValuesLessThanKey: aKey
    into: anArray
    using: (BtreeComparisonForCompare newForComparison: self collator)

]

{ #category : 'Searching' }
BtreeNode >> _findCoveringIndexForKey: aKey totalOrder: bool1 [

"Returns the index for the first entry in which aKey is found.  This is the
first entry whose key >= aKey.  If the receiver's last key is greater than
aKey, or if the receiver is empty, returns 0.  Uses a binary search."

| index |
numElements == 0 ifTrue: [ ^ 0 ].

index := self _binarySearchCoveringKey: aKey totalOrder: bool1.
" check if not found "
^ index > self _lastKeyIndex
ifTrue: [ 0 ]
ifFalse: [ index ]

]

{ #category : 'Searching' }
BtreeNode >> _findFirstValueForKey: aKey [
  "Returns the first value associated with the given key."

  ^ self
    _findFirstValueForKey: aKey
    using: (BtreeComparisonForCompare newForComparison: self collator)

]

{ #category : 'Updating' }
BtreeNode >> _incrementNumElements [

"Increment the value of the instance variable 'numElements' by 1."

numElements := numElements + 1

]

{ #category : 'Updating' }
BtreeNode >> _insertKey: aKey value: aValue atIndex: insertionIndex [

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

| endIndex entrySize |

" see if any existing entries have to be moved "
entrySize := self entrySize .
endIndex := (numElements * entrySize) .

"move entries down to make room for a new entry, or add room at the end"
self _insertAt: insertionIndex
     from: nil "from abstract Array of nils"
     fromStart: 1
     fromEnd: entrySize
     numToMoveDown: (endIndex - insertionIndex + 1) .


" add the new entry "
self _at: insertionIndex put: aValue.
self _at: (insertionIndex + 1) put: aKey.

self _insertEncryptionFor: aKey value: aValue startingAt: (insertionIndex + 2).

numElements := numElements + 1

]

{ #category : 'Accessing' }
BtreeNode >> _lastIndex [

"Returns the index of the last used slot in the node."

^ numElements * self entrySize

]

{ #category : 'Accessing' }
BtreeNode >> _lastKeyIndex [

"Returns the index of the key of the last entry, or zero if the node is empty."

^ numElements == 0
ifTrue: [ 0 ]
ifFalse: [ numElements - 1 * self entrySize + 2 ]

]

{ #category : 'Searching Support' }
BtreeNode >> _putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey [
  "Puts first entry that is greater than or equal to aKey into the Array."

  ^ self
    _putFirstIndexOfFirstChildInto: anArray
    ifGreaterThanOrEqualTo: aKey
    using: (BtreeComparisonForCompare newForComparison: self collator)

]

{ #category : 'Searching' }
BtreeNode >> _scanForAllValuesForKey: aKey into: aCollection [
  "Places all values associated with the given key into aCollection."

  ^ self
    _scanForAllValuesForKey: aKey
    into: aCollection
    using: self comparisonForSort

]

{ #category : 'Audit' }
BtreeNode >> auditNsc: nsc for: pathTerm offset: offset on: aString [
  "Verifies that the btree structure consistent relative to the given pathTerm."

  | indicatesIndexOnNscElements count key value bag ivOffset bagOfKeys setOfKeys comparisonForSort 
     previousKey previousValue sentinel enumBag blk |
  indicatesIndexOnNscElements := pathTerm indicatesIndexOnNscElements.
  count := 0.
  bag := IdentityBag new.
  enumBag := IdentityBag new.
  bagOfKeys := IdentityBag new.
  setOfKeys := IdentitySet new.
  comparisonForSort := self comparisonForSort.
  sentinel := Object new.
  previousKey := sentinel.
  blk := [:prevArg :kArg | | prev aKey | 
    "avoid using Fix 50599, 51126 on legacy indexes, relops.c still using coercion to float ..."
     prev := prevArg.  aKey := kArg.
     (prev _isNumber and:[ aKey _isNumber ]) ifTrue:[
       (prev _generality) ~~ (aKey _generality) ifTrue:[
	 (prev _isFloat) ~~ (aKey _isFloat) ifTrue:[
	   prev := prev asFloat .  aKey := aKey asFloat
         ]. 
       ].
     ].
     (comparisonForSort compareKey: prev lessThanOrEqualTo: aKey) ifTrue:[
         (comparisonForSort compareKey: prev equalTo: aKey) ifTrue:[ 0 ] ifFalse:[ -1 ].
      ] ifFalse:[ 1 ].
  ].
  self
    _auditLeafKeysValuesAndEncryptedFor: pathTerm
    on: aString
    do: [ :k :v :encryptedEntry :encryptedKey | 
      previousKey == sentinel ifFalse:[ | cmp |
         cmp := blk value: previousKey value: k .
         cmp == 0 ifTrue: [ 
           previousValue asOop <= v asOop ifFalse: [ 
                pathTerm auditInfo: {k. nil. nil} on: aString.
                aString add: 'Btree ('; add: self asOop asString;
                    add: ') has out of oop order key/value pairs where the key ';
                    add: k printString printString; add: ' ('; add: k asOop asString;
                    add: '( is equal to previous key ';
                    add: previousKey printString printString;
                    add: ' ('; add: previousKey asOop asString; add: '( but the oop of the value ';
                    add: v printString printString;
                    add: ' ('; add: v asOop asString; add: '( is less than the oop of the previous value ';
                    add: previousValue printString printString;
                    add: ' ('; add: previousValue asOop asString; add: '(' ; lf .
                    self checkDebug: aString 
            ]
         ] ifFalse:[
           cmp == -1 ifFalse:[
              pathTerm auditInfo: {k . nil. nil} on: aString.
              aString
                add: 'Btree ('; add: self asOop asString; add: ') has key ';
                add: k printString printString; add: ' out of order ('; add: k asOop asString;
                add: ') that sorts greater than previous key'; add: previousKey printString printString;
                add: ' ('; add: previousKey asOop asString; add: ')'; lf . self checkDebug: aString 
          ].
        ]
      ].
      previousKey := k.
      previousValue := v.
      encryptedEntry ~~ nil
        ifTrue: [ encryptedEntry = encryptedKey
            ifFalse: [ pathTerm auditInfo: {k.
                  nil.
                  nil} on: aString.
              aString
                add: 'Btree [';
                add: self asOop asString;
                add: '] has encryptedKey for [';
                add: v asOop asString;
                add: '] of class ';
                add: v class asString;
                add: ' associated with key [';
                add: k asOop asString;
                add: '] that is [';
                add: (encryptedEntry at: 1) asOop asString;
                add: ', ';
                add: (encryptedEntry at: 2) asOop asString;
                add: '] but should be [';
                add: (encryptedKey at: 1) asOop asString;
                add: ', ';
                add: (encryptedKey at: 2) asOop asString;
                add: ']';
                add: Character lf . self checkDebug: aString ] ].
      count := count + 1.
      indicatesIndexOnNscElements
        ifTrue: [ v == k
            ifFalse: [ pathTerm auditInfo: {k.
                  nil.
                  nil} on: aString.
              aString
                add: 'Btree [';
                add: self asOop asString;
                add: '] has entry for [';
                add: v asOop asString;
                add: '] of class ';
                add: v class asString;
                add: ' associated with incorrect key [';
                add: k asOop asString;
                add: '] should be ( ';
                add: v asOop asString;
                add: ') ';
                add: Character lf . self checkDebug: aString ] ]
        ifFalse: [ | hasEntryFor expectedEntryValue |
          hasEntryFor := false.
          pathTerm isEnumeratedTerm
            ifTrue: [ | ivOffsets |
              ivOffsets := pathTerm _ivOffsetsFor: v.
              ivOffsets do: [ :off | | expected |
                  off ifNotNil: [ (expected := v instVarAt: off) == k
                        ifTrue: [ hasEntryFor := true.
                          expectedEntryValue := expected ] ] ] ]
            ifFalse: [ pathTerm isSelectorTerm
                ifTrue: [ expectedEntryValue := v perform: pathTerm termSelector.
                  hasEntryFor := expectedEntryValue _idxForSortEqualTo: k ]
                ifFalse: [ ivOffset := pathTerm _ivOffsetFor: v.
                  hasEntryFor := (expectedEntryValue := v instVarAt: ivOffset)
                    == k ] ].
          hasEntryFor
            ifFalse: [ pathTerm auditInfo: {k.
                  nil.
                  nil} on: aString.
              aString
                add: 'Btree [';
                add: self asOop asString;
                add: '] has entry for [';
                add: v asOop asString;
                add: '] of class ';
                add: v class asString;
                add: ' associated with incorrect key [';
                add: k asOop asString;
                add: '] should be ( ';
                add: expectedEntryValue asOop asString;
                add: ') ';
                add: Character lf . self checkDebug: aString ] ].
      count > 1
        ifTrue: [ value == v
            ifFalse: [ self
                _auditKey: key
                value: value
                bag: bag
                nsc: nsc
                for: pathTerm
                offset: offset
                enumBag: enumBag
                on: aString ].
          (comparisonForSort compareKey: key equalTo: k)
            ifTrue: [ bagOfKeys add: k.
              setOfKeys add: k ]
            ifFalse: [ 1 to: setOfKeys size do: [ :n | | aKey |
                aKey := setOfKeys _at: n.
                (DependencyList for: aKey) ~~ nil
                  ifTrue: [ pathTerm
                      auditDepListForLastElementObject: aKey
                      occurrences: (bagOfKeys occurrencesOf: aKey)
                      index: (pathTerm at: offset)
                      on: aString ] ].
              bagOfKeys removeAll: bagOfKeys.
              setOfKeys removeAll: setOfKeys.
              bagOfKeys add: k.
              setOfKeys add: k ] ]
        ifFalse: [ bagOfKeys add: k.
          setOfKeys add: k ].
      value := v.
      key := k.
      bag add: v.
      pathTerm isEnumeratedTerm
        ifTrue: [ enumBag add: v ] ].
  count > 0
    ifTrue: [ self
        _auditKey: key
        value: value
        bag: bag
        nsc: nsc
        for: pathTerm
        offset: offset
        enumBag: enumBag
        on: aString ].
  1 to: setOfKeys size do: [ :n | | aKey |
    aKey := setOfKeys _at: n.
    (DependencyList for: aKey) ~~ nil
      ifTrue: [ pathTerm
          auditDepListForLastElementObject: aKey
          occurrences: (bagOfKeys occurrencesOf: aKey)
          index: (pathTerm at: offset)
          on: aString ] ].
  ^ count
]

{ #category : 'Audit' }
BtreeNode >> checkDebug: aString [
  BtreePlusNodeAuditor checkDebug: aString 

]

{ #category : 'Accessing' }
BtreeNode >> collator [
  ^ self dynamicInstVarAt: #'collator'

]

{ #category : 'Accessing' }
BtreeNode >> collator: anIcuCollator [
  ^ self dynamicInstVarAt: #'collator' put: anIcuCollator

]

{ #category : 'Comparison Operators' }
BtreeNode >> comparisonForSort [
  ^ BtreeComparisonForCompare newForSort: self collator

]

{ #category : 'Constants' }
BtreeNode >> entrySize [

"Returns the size of an entry with no encryption."

^ 2

]

{ #category : 'Testing' }
BtreeNode >> isBtreeNode [

"Returns true: the receiver is a B-tree node."

^ true

]

{ #category : 'Testing' }
BtreeNode >> isLeaf [

"Returns whether the B-tree node is a leaf node.  The default return is false."

^ false

]

{ #category : 'Indexing Support' }
BtreeNode >> maxNumWhenMerging [

"Returns the maximum number of entries to add when building interior nodes
during a merge sort on index creation."

" ^ (self mergeThreshold + self class maxNumberOfElements) // 2 "
^ self class maxNumberOfElements - 5

]

{ #category : 'Accessing' }
BtreeNode >> numElements [

"Returns the value of the instance variable 'numElements'."

^ numElements

]

{ #category : 'Updating' }
BtreeNode >> numElements: newValue [

"Modify the value of the instance variable 'numElements'."

numElements := newValue

]

{ #category : 'Constants' }
BtreeNode >> parentNodeClass [

"Returns the class of node to be created as the parent when a split occurs."

^ BtreeInteriorNode

]

{ #category : 'Printing' }
BtreeNode >> prettyPrint [

""

| aStream |
aStream := PrintStream printingOn: String new.
self printOn: aStream withIndentationLevel: 0 includingValues: false.
^ aStream contents

]

{ #category : 'Updating' }
BtreeNode >> primitiveAt: anIndex put: aValue [

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, if the receiver is not indexable, or if the
 receiver is not of the right class to store the given value.

 The primitive is equivalent to GciStoreIdxOop or GciStoreByte,
 depending on implementation of the receiver."

<primitive: 268>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
(self class isIndexable) "not an indexable object"
  ifFalse: [ ^ self _errorNotIndexable].
(anIndex < 1 or:[ anIndex > (self size + 1)]) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(self class isBytes)
ifTrue: [ ((aValue class ~~ SmallInteger) or: [ (aValue < 0) | (aValue > 255) ])
    ifTrue: [^ aValue _error: #rtErrExpectedByteValue].
  ].

self _primitiveFailed: #at:put: args: { anIndex . aValue }.
self _uncontinuableError

]

{ #category : 'Formatting' }
BtreeNode >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

"Default printing for non-protected access."


aStream nextPutAll: self asString

]
