"
AzureDataKey is a class that holds a local encryption key. In order
to use the key, it must be decrypted by accessing a key in the key vault
stored in the Azure cloud.

Instance variables:

keyVaultUrl (String) the URL of the key vault that contains the key.
keyName (String) - the name of the key to use for encryption/decryption.
encryptedDataKey -(String)  the local data key in encrypted form, stored
                   as base 64 text.

AzureDataKeys may be in a locked or unlocked state. Locked means the
key has not yet been decrypted by the current session. Unlocked means the
data key has been decrypted and is stored in session memory. AzureDataKeys
may only be used to encrypt or decrypt data in the unlocked state.

If an AzureDataKey is committed to GemStone, it is always stored on disk 
in its locked state. Once an AzureDataKey is unlocked, it remains unlocked
until it is sent the #lock message or the session logs out.

Azure support is only available on Linux.
"
Class {
	#name : 'AzureDataKey',
	#superclass : 'AbstractCloudKey',
	#instVars : [
		'keyVaultUrl',
		'keyName',
		'encryptedDataKey'
	],
	#gs_reservedoop : '251137',
	#category : 'Credentials'
}

{ #category : 'Private' }
AzureDataKey class >> _threeArgClassPrim: opCode with: obj1 with: obj2 with: obj3 [

"
opCode		class method
0		AwsDataKey createKeyUsingAwsCredentials:cmsKeyId:keySizeBytes:
1		AzureDataKey createKeyUsingAzureCredentials:keyVaultUrl:keyName: 

opCode		instance method
2		AzureDataKey _changeKeyNameTo:inKeyVault:usingCredentials:
"

<primitive: 1128>
^ self _primitiveFailed: #_threeArgClassPrim:with:with: args: { opCode . obj1 . obj2 . obj3 }

]

{ #category : 'Instance Creation' }
AzureDataKey class >> createKeyUsingAzureCredentials: azureCreds keyVaultUrl: keyVaultUrl keyName: keyName [

"Creates a new instance of the receiver using azureCreds, keyVaultUrl and keyName.
 The new instance is unlocked.  azureCreds must be valid for keyVaultUrl and keyName. 
 The keyVault and keyName must already exist in Azure and the permissions must be set
 to allow azureCreds to encrypt and decrypt keys.

 Raises an exception on error."

^ self _threeArgClassPrim: 1 with: azureCreds with: keyVaultUrl with: keyName

]

{ #category : 'Exceptions' }
AzureDataKey class >> errorClass [
  ^ AzureError
]

{ #category : 'Private' }
AzureDataKey >> _changeKeyNameTo: newKeyName inKeyVault: newKeyVaultUrl usingCredentials: newCreds [

"Updates the receiver to use newKeyName in newKeyVaultUrl which is accessed with newCreds.
The receiver must first be unlocked using the #unlockWithAzureCredentials: method before this method
can be successfully invoked.

Private. Do not call this method directly. Use the public method #changeKeyIdTo:inKeyVault:usingCredentials:
to change the data key."

^self _threeArgInstPrim: 2 with: newKeyName with: newKeyVaultUrl with: newCreds

]

{ #category : 'Private' }
AzureDataKey >> _copyKeyTo: newAzureDataKey [

"Private. Do not call this method directly unless you know what you are doing.
If the receiver is unlocked, copy the decrypted data key to newObject.
Copies nothing if the receiver is locked. It is an error if newAzureDataKey
already has data key and and is unlocked. 

Returns newAzureDataKey."

^ self _oneArgInstPrim: 3 with: newAzureDataKey

]

{ #category : 'Private' }
AzureDataKey >> _oneArgInstPrim: opCode with: obj [

"
OpCode		Method
-------------------------
0			AwsDataKey unlockWithAwsCredentials:
1			AwsDataKey _copyKeyTo:
2			AzureDataKey unlockWithAzureCredentials:
3			AzureDataKey _copyKeyTo:
"

<primitive: 1126>
^ self _primitiveFailed: #_oneArgInstPrim: args: { opCode . obj }

]

{ #category : 'Private' }
AzureDataKey >> _threeArgInstPrim: opCode with: obj1 with: obj2 with: obj3 [

"
opCode		class method
0		AwsDataKey createKeyUsingAwsCredentials:cmsKeyId:keySizeBytes:
1		AzureDataKey createKeyUsingAzureCredentials:keyVaultUrl:keyName: 

opCode		instance method
2		AzureDataKey _changeKeyNameTo:inKeyVault:usingCredentials:
"

<primitive: 1128>
^ self _primitiveFailed: #_threeArgInstPrim:with:with: args: { opCode . obj1 . obj2 . obj3 }

]

{ #category : 'Private' }
AzureDataKey >> _twoArgInstPrim: opCode with: obj1 with: obj2 [

"
OpCode		Method
-------------------------
0			AwsDataKey _changeCmsKeyIdTo:usingCredentials:
1			AwsDataKey encrypt:into:
2			AwsDataKey decrypt:into:
3			AzureDataKey encrypt:into:
4			AzureDataKey decrypt:into:

"

<primitive: 1127>
^ self _primitiveFailed: #_twoArgInstPrim: args: { opCode . obj1 . obj2 }

]

{ #category : 'Private' }
AzureDataKey >> _zeroArgInstPrim: opCode [

"
OpCode		Method
-------------------------
0		AwsDataKey	isLocked
1		AwsDataKey	lock
2		AzureDataKey	isLocked
3		AzureDataKey	lock
"

<primitive: 1125>
^ self _primitiveFailed: #_zeroArgInstPrim: args: { opCode }

]

{ #category : 'Comparing' }
AzureDataKey >> = anotherKey [

(self == anotherKey) ifTrue:[ ^ true ].
(self class == anotherKey class) ifFalse:[ ^ false ].
(self keyVaultUrl = anotherKey keyVaultUrl) ifFalse: [^ false].
(self keyName = anotherKey keyName) ifFalse: [^ false].
(self encryptedDataKey = anotherKey encryptedDataKey) ifFalse: [^false].
^ true

]

{ #category : 'Key Rotation' }
AzureDataKey >> changeKeyNameTo: newName inKeyVault: newVault usingCredentials: newCreds [

"Atomically updates the receiver to use newName in newVault  which is accessed with newCreds in a single operation.
The receiver must first be unlocked using the #unlockWithAzureCredentials: method before this method
can be successfully invoked.
Fails with an exception if the session has uncommitted changes or if the receiver is locked.
Fails with an exception if the session cannot obtain a write lock on the receiver.
Returns true on success and leaves the receiver in an unlocked state."

| sys |
self assertNotSolo; assertUnlocked ; assertNoUncommittedChanges .
sys := System .
sys beginTransaction.
^ [ sys writeLock: self ;
	addToCommitOrAbortReleaseLocksSet: self .
self _changeKeyNameTo: newName inKeyVault: newVault usingCredentials: newCreds .
sys commit
] on: Exception do:[:ex| sys abortTransaction. ex pass ]

 
]

{ #category : 'Copying' }
AzureDataKey >> copy [

"Answer a new object which is a deep copy of the receiver. The lock state of the receiver is preserved
when copied, i.e.: if the receiver is unlocked the resulting copy will also
be unlocked."

| result |
result := self class basicNew.
result keyVaultUrl: self keyVaultUrl copy ;
       keyName: self keyName copy ;
       encryptedDataKey: self encryptedDataKey copy .

^self _copyKeyTo: result

]

{ #category : 'Encrypting' }
AzureDataKey >> decrypt: srcByteObj into: destByteObj [

"Uses the receiver to decrypt srcByteObj and stores the resulting decrypted bytes into destByteObj.
The receiver must be unlocked. srcByteObj must be a non-empty byte object containg Base64 text
obtained by calling one of the encrypt methods in this class.
destByteObj must be a mutable byte object. The contents of destByteObj, if any, are
overwritten by this method.

Encryption is performed using the AES-OCB authenticated encryption, which ensures data that has been 
successfully decrypted has not been modified in any way.

Returns the destByteObj on success or raises an exception on error.
"

^self _twoArgInstPrim: 4 with: srcByteObj with: destByteObj

]

{ #category : 'Encrypting' }
AzureDataKey >> encrypt: srcByteObj into: destByteObj [

"Uses the receiver to encrypt srcByteArg and stores the resulting encrypted bytes into destByteObj 
as a Base64 string. The receiver must be unlocked. srcByteArg must be a non-empty byte object.
destByteArg must be a mutable byte object. Any data present in destByteArg will be overwritten.

Encryption is performed using the AES-OCB authenticated encryption, which ensures data that has been 
successfully decrypted has not been modified in any way.

Returns destByteObj on success and raises an exception on error.
"

| result |
result := self _twoArgInstPrim: 3 with: srcByteObj with: destByteObj .
^ result

]

{ #category : 'Encrypting' }
AzureDataKey >> encryptAndErase: srcByteArg into: destByteObj [

"Uses the receiver to encrypt and erase srcByteArg and store the resulting encrypted bytes into 
destByteObj as a Base64 string. The receiver must be unlocked. srcByteArg must be a non-empty byte 
object. destByteArg must be a mutable byte object. Any data present in destByteArg will be overwritten.

Encryption is performed using the AES-OCB authenticated encryption, which ensures data that has been 
successfully decrypted has not been modified in any way.

Returns destByteObj and sets srcByteArg to empty (zero size) upon success.
Raises an exception on error.
"

| result |
result := self _twoArgInstPrim: 3 with: srcByteArg with: destByteObj .
srcByteArg size: 0 .
^ result

]

{ #category : 'Accessing' }
AzureDataKey >> encryptedDataKey [
	^encryptedDataKey

]

{ #category : 'Updating' }
AzureDataKey >> encryptedDataKey: newValue [
	encryptedDataKey := newValue

]

{ #category : 'Hashing' }
AzureDataKey >> hash [

"Returns an Integer hash code for the receiver."

^ self keyVaultUrl hash bitXor: self encryptedDataKey hash

]

{ #category : 'Testing' }
AzureDataKey >> isLocked [

"Returns a boolean indicating if the receiver is locked.
Unlocked keys may be used to encrypt and decrypt data.
Locked keys must be unlocked using the #unlockWithAzureCredentials: method
before usage."

^ self _zeroArgInstPrim: 2

]

{ #category : 'Testing' }
AzureDataKey >> isUnlocked [

"Returns a boolean indicating if the receiver is unlocked."

^ self isLocked not

]

{ #category : 'Accessing' }
AzureDataKey >> keyName [
	^keyName

]

{ #category : 'Updating' }
AzureDataKey >> keyName: newValue [
	keyName := newValue

]

{ #category : 'Accessing' }
AzureDataKey >> keyVaultUrl [
	^keyVaultUrl

]

{ #category : 'Updating' }
AzureDataKey >> keyVaultUrl: newValue [
	keyVaultUrl := newValue

]

{ #category : 'Locking' }
AzureDataKey >> lock [

"Locks the receiver and securly removes the encryption key from memory.
Returns the receiver."
^ self _zeroArgInstPrim: 3

]

{ #category : 'Locking' }
AzureDataKey >> unlockWithAzureCredentials: awsCreds [

"Attempts to unlock the receiver using awsCreds.
Returns the receiver on success or if the receiver is already unlocked.
Raises an exception on error."

^self _oneArgInstPrim: 2 with: awsCreds

]
