"
A GciTsLibrary represents a loaded CLibrary for FFI access to the
thread safe GCI functions in $GEMSTONE/include/gcits.hf. 
The CLibrary encapsulated by a GciTsLibrary is a loaded
  $GEMSTONE/lib/libgcits*.so  .

You can load a libgcits library from a different version of GemStone
to create a GsTsExternalSession that logs into to a stone that is 
runing a different version of GemStone. 
Use   GciTsLibrary class >> newForVersion:product: 
to load a different version of libgcits.  This method is relatively expensive 
(a second of CPU time to parse the other version $GEMSTONE/include/gcits.hf).
so you may want to cache the instance of GciTsLibrary in SessionTemps,
similar to the caching in   GciTsLibrary class >> newDefault .

Note that loading a libgcits library for a different version of GemStone 
requires the cpp; you will need gcc/g++ to be installed.
"
Class {
	#name : 'GciTsLibrary',
	#superclass : 'Object',
	#instVars : [
		'version',
		'callouts',
		'calloutSigs',
		'cLibrary'
	],
	#classVars : [
		'Gemstone_GciTsLibrary_default'
	],
	#category : 'GCI-Support'
}

{ #category : 'Private' }
GciTsLibrary class >> _initializeDefault [
 "executed by image upgrade and slowfilein"
 | val f |
 (f := GsFile openReadOnServer: '$upgradeDir/GciTsLibrary.dat') ifNotNil:[
    val := (PassiveObject newOnStream: f) activate.
    GsFile gciLogServer:'loaded GciTsLibrary.dat' .
 ] ifNil:[ "this path executes in slowfilein only"
   val := self newForVersion: System _gemVersion  product: '$GEMSTONE'. "run FFI parsing..."
   f := GsFile openWriteOnServer:  'GciTsLibrary.dat' . "to current dir of filein"
   PassiveObject passivate: val toStream: f .
   GsFile gciLogServer:'generated  GciTsLibrary.dat' .
 ].
 f close .
 self _addClassVar: #Gemstone_GciTsLibrary_default value: val .  

]

{ #category : 'Instance Creation' }
GciTsLibrary class >> newDefault [
  ^ (self classVarAt: #Gemstone_GciTsLibrary_default otherwise: nil) ifNil:[
     Error signal: 'Gemstone_GciTsLibrary_default  not initialized'.
  ].

]

{ #category : 'Instance Creation' }
GciTsLibrary class >> newForVersion: verString product: pathArg [
  "Requires cpp; should have gcc/g++ installed"

  | res hdr path hdrPath libPath suffix |
  (res := self new )
    version: verString . "checks verString"
  path := pathArg.
  path last == $/ ifFalse:[ (path := path copy) add: $/ ].
  hdrPath := path, 'include/gcits.hf'  .
  suffix := ((System gemVersionReport at: 'gsBuildArchitecture') at:1 equals:'Darwin')
             ifTrue:[ '-64.dylib' ] ifFalse:[ '-64.so' ].
  libPath := path, 'lib/libgcits-', (res version) , suffix .
  hdr := CHeader path: hdrPath .
  res
    libPath: libPath ;
    _initCallouts: hdr .
  ^ res

]

{ #category : 'Private' }
GciTsLibrary >> _initCallouts: aCHeader [
 | fcts list |
 callouts := { } .
 "calloutSigs not used, send   signatureString  to a callout to  see the signature"
 fcts := aCHeader functions .
 "Callouts used by GsTsExternalSession must be at fixed offsets in array of callouts."
 list := #(
    'GciTsLogin'  "order of list must match offsets used in GsTsExternalSession"
    'GciTsLogout'
    'GciTsAbort'
    'GciTsCommit'
    'GciTsExecute'
    'GciTsPerform'
    'GciTsContinueWith' "7"
    'GciTsClearStack'
    'GciTsFetchObjInfo'
    'GciTsNbLogout'  "10"
  ) copy .
  version >= '3.5.3' ifTrue:[
   list addAll: #(
    'GciTsNbPerform' "11"
    'GciTsNbResult'
    'GciTsSocket'
    'GciTsNbExecute' ).
  ].
  version >= '3.6.0' ifTrue:[
   list addAll: #(
    'GciTsNbLogin'   "15" 
    'GciTsNbLoginFinished'
    'GciTsBreak'
    'GciTsBegin'
    'GciTsX509Login'
    'GciTsNewUtf8String' "20" 
    'GciTsReleaseObjs').
  ].
  version >= '3.7.0' ifTrue:[
   list addAll: #(
    'GciTsDebugConnectToGem'   "22"
    'GciTsDebugStartDebugService'
    'GciTsGemTrace' ).
  ].
  "todo, add more functions"
  list  do:[ :fn | | f c |
    f := fcts at: fn .
    c := (f cCalloutForLibrary: cLibrary) copy .
    c coerceNameToSymbol . "for efficient consistency checks in GsTsExternalSession"
    callouts add: c .
    fcts removeKey: fn .
  ].
  "now add remaining callouts to the arrays ."
  list := SortedCollection withAll:( fcts keys ).
  list do:[:fn | | f c |
    (fn at: 1 equals: 'Gci') ifTrue:[
      f := fcts at: fn .
      c := (f cCalloutForLibrary: cLibrary) copy .
      c coerceNameToSymbol . "for efficient consistency checks in GsTsExternalSession"
      callouts add: c .
    ].
  ].
]

{ #category : 'Accessing' }
GciTsLibrary >> callouts [
  ^ callouts
]

{ #category : 'Updating' }
GciTsLibrary >> libPath: aPath [
  cLibrary := CLibrary named: aPath .

]

{ #category : 'Accessing' }
GciTsLibrary >> version [
 "Returns a String of the form  3.6.0 "
  ^ version
]

{ #category : 'Updating' }
GciTsLibrary >> version: verString [
  "can't yet load v3.4.x libs into a 3.5 VM , see bug 48749"
  | ok ofs vs |
  ofs := (vs := verString) indexOf: $, startingAt: 1 .
  ofs > 0 ifTrue:[ 
    vs := vs copyFrom: 1 to: ofs - 1 .
  ].
  ok := false .
  (vs at: 1 equals: '3.5') ifTrue:[ | x |
     x := vs copyFrom: 1 to: 5 .
     ok := x >= '3.5.3' . 
  ] ifFalse:[ | v |
    ofs := vs indexOf: $. startingAt: 1 .
    ofs > 0 ifTrue:[
      ofs := vs indexOf: $. startingAt: ofs + 1  . "second . "
      ofs > 0 ifTrue:[
        v := Float fromString:(vs copyFrom: 1 to: ofs - 1 ) .
        ok :=   v >= 3.6 .
      ].
    ]. 
  ].
  ok ifTrue:[ version := vs ]
     ifFalse:[ Error signal:'Cannot load library that has version ', verString].
]
