"
GsTraitImpl is an internal class that implements Traits, which allow sets of 
methods to be shared between classes. This class is not meant to be used 
directly. See the Trait class for more information.
"
Class {
	#name : 'GsTraitImpl',
	#superclass : 'Object',
	#instVars : [
		'name',
		'instanceSourceStrings',
		'classSourceStrings',
		'instanceCategories',
		'classCategories',
		'instanceDependents',
		'classDependents',
		'classForCompiles',
		'instanceTrait',
		'classTrait',
		'traitCategory',
		'extraDict'
	],
	#category : 'Kernel-Methods'
}

{ #category : 'instance creation' }
GsTraitImpl class >> name: aString instVars: ivNamesArray classVars: cvNamesArray classInstVars: civNamesArray [
  | tr sym |
  tr := self new .
  tr name: (sym := aString asSymbol)  instVars: ivNamesArray  classVars: cvNamesArray classInstVars: civNamesArray.
  ^ tr
]

{ #category : 'private' }
GsTraitImpl >> _basicCompile: selector source: sourceString category: categoryString in: classOrMetaclass origin: aClassOrInstanceTrait [
	"Compile trait method in classOrMetaclass, 
  if the method is not present in the class or the receiver IS the origin of the method"
	| existingMethod compiledMethod |
	existingMethod := classOrMetaclass compiledMethodAt: selector otherwise: nil.
	(existingMethod isNil or: [ existingMethod origin = aClassOrInstanceTrait ])
		ifTrue: [ 
			"compile method unconditionally as a non-Rowan method ... since we are a trait method override"
			GsObjectSecurityPolicy
				setCurrent: self objectSecurityPolicy
				while: [ 
					compiledMethod := classOrMetaclass
						compileMethod: sourceString
						dictionaries: GsCurrentSession currentSession symbolList
						category: categoryString asSymbol
						intoMethodDict: nil
						intoCategories: nil
						environmentId: 0 ].
			compiledMethod _origin: aClassOrInstanceTrait ]
]

{ #category : 'private' }
GsTraitImpl >> _basicRemove: selector from: classOrMetaclass [
	"remove trait method from aCls, if the method is not present in the class or the receiver IS the origin of the method"

	| existingMethod |
	existingMethod := classOrMetaclass compiledMethodAt: selector otherwise: nil.
	(existingMethod notNil and: [ existingMethod origin == self ])
		ifTrue: [ 
			classOrMetaclass removeSelector: selector.
			existingMethod _origin: nil ]
]

{ #category : 'private' }
GsTraitImpl >> _basicRemove: selector from: classOrMetaclass origin: aClassOrInstanceTrait [
	"remove trait method from aCls, if the method is not present in the class or the receiver IS the origin of the method"

	| existingMethod |
	existingMethod := classOrMetaclass compiledMethodAt: selector otherwise: nil.
	(existingMethod notNil and: [ existingMethod origin = aClassOrInstanceTrait ])
		ifTrue: [ 
			classOrMetaclass removeSelector: selector.
			existingMethod _origin: nil ]
]

{ #category : 'private' }
GsTraitImpl >> _category [
	"Returns the receiver's traitCategoyr or nil."

	^ traitCategory
]

{ #category : 'private' }
GsTraitImpl >> _category: newCategory [
"Sets the traitCategory variable of the receiver.
 The argument should be a kind of CharacterCollection or nil."
"category/packageName used in Trait tonel file"

newCategory ifNil:[
	traitCategory := nil.
	^ self ].

(newCategory _validateClass: CharacterCollection ) ifFalse:[ ^ nil ].

traitCategory := newCategory asString
]

{ #category : 'browser methods' }
GsTraitImpl >> _categoryOfSelectorForClassTrait: aString [
	"Returns the category of the given selector, or 'unknown' if it isn't found."

	^ (self categoryOfSelectorForClassTrait: aString) ifNil: [ 'unknown' ]
]

{ #category : 'browser methods' }
GsTraitImpl >> _categoryOfSelectorForInstanceTrait: aString [
	"Returns the category of the given selector, or 'unknown' if it isn't found."

	^ (self categoryOfSelectorForInstanceTrait: aString) ifNil: [ 'unknown' ]
]

{ #category : 'browser methods' }
GsTraitImpl >> _classCategoriesDo: aBlock [
	classCategories keysAndValuesDo: aBlock
]

{ #category : 'private' }
GsTraitImpl >> _classDependents [
	^ classDependents
]

{ #category : 'browser methods' }
GsTraitImpl >> _classTopazMethodAt: aString [
	"Returns a GsNMethod, or signals an Error."

	| sel sourceString |
	sel := aString asSymbol.
	sourceString := (self classSourceCodeAt: sel) ifNil: [ ^ nil ].
	^ self classForCompiles class
		_compileMethod: sourceString
		symbolList: System myUserProfile symbolList
]

{ #category : 'private' }
GsTraitImpl >> _classVars [
	^ classForCompiles _classVars
]

{ #category : 'private' }
GsTraitImpl >> _comment [
	"answer nil if no comment defined"

	^ self _extraDictAt: #'comment'
]

{ #category : 'browser methods' }
GsTraitImpl >> _definitionInContext: aUserProfile [
	"Returns a description of the receiver using object names taken from the given
 UserProfile."

	| result lfsp civs iVs |
	result := String new.
	result addAll: 'Trait'.
	(lfsp := Character lf asString) addAll: '  '.

	result
		addAll: ' name: ''';
		addAll: name;
		addLast: $'.	
	iVs := self instVarNames.
	result
		addAll: lfsp;
		addAll: 'instVars: #(';
		addAll: (self _instVarNamesWithSeparator: lfsp , '                 ');
		add: $).	" instVars: #( <list of strings> ) "
	result
		addAll: lfsp;
		addLast: 'classVars: #('.
	self _sortedClassVarNames
		accompaniedBy: result
		do: [ :res :aKey | 
			res addLast: $ .
			(aKey includesValue: $')
				ifTrue: [ res addAll: aKey _asSource ]
				ifFalse: [ res addAll: aKey ] ].
	result addLast: $).	" classVars: #( <list of strings> ) "
	result
		addAll: lfsp;
		addLast: 'classInstVars: #('.
	civs := self classForCompiles class allInstVarNames.
	civs removeFrom: 1 to: self classForCompiles class superClass instSize.
	civs
		accompaniedBy: result
		do: [ :res :civName | 
			res addLast: $ .
			(civName includesValue: $')
				ifTrue: [ res addAll: civName _asSource ]
				ifFalse: [ res addAll: civName ] ].
	result addLast: $).	" classInstVars: #( <list of strings> ) "
	result
		addAll: lfsp;
		addAll: 'inDictionary: ';
		addAll: (self _dictionaryNameForFileout: aUserProfile). " inDictionary: <name of containing dictionary> "
	result
		add: lfsp;
		add: Character lf.
	^ result
]

{ #category : 'browser methods' }
GsTraitImpl >> _dictionaryNameForFileout: aUserProfile [
	| anArray |
	anArray := aUserProfile dictionariesAndSymbolsOf: self instanceTrait.
	anArray size ~~ 0
		ifTrue: [ 
			anArray := aUserProfile dictionariesAndSymbolsOf: ((anArray at: 1) at: 1).
			anArray size == 0
				ifTrue: [ ^ '(dictionary not in your dictionaries)' ]
				ifFalse: [ 
					| dName |
					(dName := (anArray at: 1) at: 2) isValidIdentifier
						ifTrue: [ ^ dName ] ] ].
	^ 'UserGlobals'
]

{ #category : 'private' }
GsTraitImpl >> _extraDictAt: key [
	"Return value for key in extraDict.
   Return nil if the key is not present. "

	^ extraDict at: key otherwise: nil
]

{ #category : 'private' }
GsTraitImpl >> _extraDictAt: key put: value [
	"Add value for key to extraDict. "

	^ extraDict at: key put: value
]

{ #category : 'private' }
GsTraitImpl >> _extraDictRemoveKey: key [
	" Remove key/value from extraDict. "

	extraDict removeKey: key ifAbsent: [  ]
]

{ #category : 'fileout' }
GsTraitImpl >> _fileoutHeaderOn: stream [
  | prevEnv |
  stream isEmpty ifTrue:[
    (stream isKindOf: GsFile) ifTrue:[
      stream nextPutAll: 'fileformat utf8' ; lf .
    ].
    "Gs64 v3.3, no SET SOURCESTRINGCLASS directives in fileouts."
  ].
  prevEnv := stream dynamicInstVarAt: #environmentId .
  prevEnv ~~ 0 ifTrue:[ 
    stream nextPutAll:'set compile_env: ' ; nextPutAll: '0' ; lf .
    stream dynamicInstVarAt: #environmentId put: 0 .
  ]
]

{ #category : 'browser methods' }
GsTraitImpl >> _includesSelectorForClassTrait: aString [
	| sel |
	sel := aString asSymbol.
	^ self classCategories includesKey: sel
]

{ #category : 'browser methods' }
GsTraitImpl >> _includesSelectorForInstanceTrait: aString [
	| sel |
	sel := aString asSymbol.
	^ self instanceCategories includesKey: sel
]

{ #category : 'browser methods' }
GsTraitImpl >> _instanceCategoriesDo: aBlock [
	instanceCategories keysAndValuesDo: aBlock
]

{ #category : 'private' }
GsTraitImpl >> _instanceDependents [
	^ instanceDependents
]

{ #category : 'browser methods' }
GsTraitImpl >> _instanceTopazMethodAt: aString [
	"Returns a GsNMethod, or signals an Error.  Returns nil if method not found."

	| sel sourceString |
	sel := aString asSymbol.
	sourceString := (self instanceSourceCodeAt: sel) ifNil: [ ^ nil ].
	^ self classForCompiles
		_compileMethod: sourceString
		symbolList: System myUserProfile symbolList
]

{ #category : 'browser methods' }
GsTraitImpl >> _instVarNamesWithSeparator: sep [
	"Returns a string showing my instance variables, with the given
 separator string inserted after every three names."

	^ self classForCompiles _instVarNamesWithSeparator: sep
]

{ #category : 'browser methods' }
GsTraitImpl >> _selectorsReportForClassSide: forClassSide selectors: list matching: aString primitivesOnly: primsBoolean includeDeprecated: inclDeprecBool [
	"Used by topaz (and GBS?).
  Result is a sorted SequenceableCollection  of Symbols, plus an optional string 'Omitted .. deprecated methods'.
  aString if not nil restricts the result to only include those selectors containing
  the case-insensitive substring aString .
  primsBoolean if true further restricts the result to only include only
  selectors of methods which are primitives."

	| res deprecSet numDeprecated selectors symList |
	numDeprecated := 0.
	symList := System myUserProfile symbolList.
	inclDeprecBool
		ifFalse: [ deprecSet := Object _selectorsInBaseCategory: #'Deprecated Notification' ].
	res := SortedCollection new.
	selectors := forClassSide
		ifTrue: [ self classSelectors ]
		ifFalse: [ self instanceSelectors ].
	selectors
		do: [ :sym | 
			| sel |
			(aString == nil or: [ sym includesString: aString ])
				ifTrue: [ 
					sel := sym.
					(primsBoolean or: [ deprecSet ~~ nil ])
						ifTrue: [ 
							| sourceString |
							(forClassSide
								ifTrue: [ 
									sourceString := self classSourceCodeAt: sel.
									self classForCompiles class
										_compileMethod: sourceString
										symbolList: symList ]
								ifFalse: [ 
									sourceString := self instanceSourceCodeAt: sel.
									self classForCompiles _compileMethod: sourceString symbolList: symList ])
								ifNotNil: [ :meth | 
									primsBoolean
										ifTrue: [ 
											meth _isPrimitive
												ifFalse: [ sel := nil ] ].
									sel
										ifNotNil: [ 
											deprecSet
												ifNotNil: [ 
													(meth _selectorPool * deprecSet) size ~~ 0
														ifTrue: [ 
															sel := nil.
															numDeprecated := numDeprecated + 1 ] ] ] ] ].
					sel ifNotNil: [ res add: sel ] ] ].
	numDeprecated > 0
		ifTrue: [ 
			res := Array withAll: res.
			res add: '(Omitted ' , numDeprecated asString , ' deprecated methods)' ].
	^ res
]

{ #category : 'browser methods' }
GsTraitImpl >> _sortedClassVarNames [
	^ self classForCompiles _sortedClassVarNames
]

{ #category : 'browser methods' }
GsTraitImpl >> _topazFileoutTrait: headerStr asUtf8: utf8Bool [
	| strm |
	strm := AppendStream on: String new.
	strm dynamicInstVarAt: #'utf8Bool' put: utf8Bool.
	strm nextPutAll: headerStr.
	self fileOutTraitOn: strm.
	^ utf8Bool
		ifTrue: [ strm contents encodeAsUTF8 ]
		ifFalse: [ strm contents ]
]

{ #category : 'private' }
GsTraitImpl >> _traitCategory [
	^ traitCategory
]

{ #category : 'updating' }
GsTraitImpl >> addClassDependent: aClass [
	classDependents ifNil: [ classDependents := IdentitySet new ].
	(classDependents includes: aClass)
		ifTrue: [ ^ self ].
	classDependents add: aClass.
	classSourceStrings
		keysAndValuesDo: [ :selector :sourceString | 
			self
				_basicCompile: selector
				source: sourceString
				category: (classCategories at: selector)
				in: aClass class
				origin: self classTrait ]
]

{ #category : 'updating' }
GsTraitImpl >> addInstanceDependent: aClass [
	instanceDependents ifNil: [ instanceDependents := IdentitySet new ].
	(instanceDependents includes: aClass)
		ifTrue: [ ^ self ].
	instanceDependents add: aClass.
	instanceSourceStrings
		keysAndValuesDo: [ :selector :sourceString | 
			self
				_basicCompile: selector
				source: sourceString
				category: (instanceCategories at: selector)
				in: aClass
				origin: self instanceTrait ]
]

{ #category : 'formatting' }
GsTraitImpl >> asString [

^ self name asString
]

{ #category : 'browser methods' }
GsTraitImpl >> categoryOfSelectorForClassTrait: aString [
	"Returns the category of the given selector, or nil if it isn't found."

	| sel |
	sel := aString asSymbol.
	^ self classCategories at: sel ifAbsent: [  ]
]

{ #category : 'browser methods' }
GsTraitImpl >> categoryOfSelectorForInstanceTrait: aString [
	"Returns the category of the given selector, or nil if it isn't found."

	| sel |
	sel := aString asSymbol.
	^ self instanceCategories at: sel ifAbsent: [  ]
]

{ #category : 'accessing' }
GsTraitImpl >> classCategories [
	^classCategories
]

{ #category : 'accessing' }
GsTraitImpl >> classCategories: object [
	classCategories := object
]

{ #category : 'accessing' }
GsTraitImpl >> classForCompiles [
  ^ classForCompiles
]

{ #category : 'accessing' }
GsTraitImpl >> classInstVarNames [
	^ classForCompiles class instVarNames
]

{ #category : 'accessing' }
GsTraitImpl >> classSelectors [
	^ self classSourceStrings keys
]

{ #category : 'accessing' }
GsTraitImpl >> classSourceCodeAt: selectorSymbol [
  ^ classSourceStrings at: selectorSymbol otherwise: nil
]

{ #category : 'accessing' }
GsTraitImpl >> classSourceStrings [
  ^ classSourceStrings
]

{ #category : 'accessing' }
GsTraitImpl >> classTrait [
	^classTrait
]

{ #category : 'accessing' }
GsTraitImpl >> classTrait: object [
	classTrait := object
]

{ #category : 'accessing' }
GsTraitImpl >> classVarNames [
	^ classForCompiles classVarNames
]

{ #category : 'accessing' }
GsTraitImpl >> comment [
	| str |
	self _comment ifNotNil: [ :cmt | ^ cmt ].
	str := 'No Trait-specific documentation for ' , self name.
	^ str
]

{ #category : 'accessing' }
GsTraitImpl >> comment: aString [

  (aString isKindOf: CharacterCollection) ifFalse: [
    ArgumentTypeError signal: 'Comment must be a String' ].
  self _extraDictAt: #comment put: aString
]

{ #category : 'fileout' }
GsTraitImpl >> commentForFileout [
	"Returns a non-empty class comment or nil."

	| str |
	str := self _extraDictAt: #'comment'.
	str size = 0
		ifTrue: [ ^ nil ].
	^ str
]

{ #category : 'compiling' }
GsTraitImpl >> compileClassMethod: sourceString [
	^ self compileClassMethod: sourceString category: #'(as yet unclassified)'
]

{ #category : 'compiling' }
GsTraitImpl >> compileClassMethod: sourceString category: aCategoryString [
	| sel cat |
	self classForCompiles class
		_compileMethod: sourceString
		symbolList: System myUserProfile symbolList.
	sel := self class extractSelector: sourceString.
	cat := aCategoryString
		ifNil: [ #'(as yet unclassified)' ]
		ifNotNil: [ :str | str asSymbol ].
	self classSourceStrings at: sel put: sourceString.
	self classCategories at: sel put: cat.
	classDependents
		do: [ :aCls | 
			self
				_basicCompile: sel
				source: sourceString
				category: cat
				in: aCls class
				origin: self classTrait ]
]

{ #category : 'compiling' }
GsTraitImpl >> compileMethod: sourceString [
	^ self compileMethod: sourceString category: #'(as yet unclassified)'
]

{ #category : 'compiling' }
GsTraitImpl >> compileMethod: sourceString category: aCategoryString [
	| sel cat |
	self classForCompiles
		_compileMethod: sourceString
		symbolList: System myUserProfile symbolList.
	sel := self class extractSelector: sourceString.
	cat := aCategoryString
		ifNil: [ #'(as yet unclassified)' ]
		ifNotNil: [ :str | str asSymbol ].
	self instanceSourceStrings at: sel put: sourceString.
	self instanceCategories at: sel put: cat.
	instanceDependents
		do: [ :aCls | 
			self
				_basicCompile: sel
				source: sourceString
				category: cat
				in: aCls
				origin: self instanceTrait ]
]

{ #category : 'browser methods' }
GsTraitImpl >> definition [
	"Returns a String containing a GemStone Smalltalk definition for the receiver
 (that is, a trait creation message).  This method uses the UserProfile
 of the owner of the current session as the correct context."
	"For use with the Topaz run command."

	^self _definitionInContext: System myUserProfile
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutClassMethod: selector on: stream [
	"Writes the given method's source to the given stream in Topaz Filein format."

	self fileOutMethod: selector forClassTrait: true on: stream
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutCommentOn: stream [
	"Writes code to create trait comment onto the given stream in
 filein format."

	self _fileoutHeaderOn: stream.
	self commentForFileout
		ifNotNil: [ :cmt | 
			stream
				nextPutAll: 'expectvalue /Trait';
				lf;
				nextPutAll: 'doit';
				lf;
				_fileOutAll: self name;
				nextPutAll: ' comment: ';
				lf;
				_fileOutAll: cmt printString;
				lf;
				nextPut: $%;
				lf ]
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutInstanceMethod: selector on: stream [
	"Writes the given method's source to the given stream in Topaz Filein format."

	self fileOutMethod: selector forClassTrait: false on: stream
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutMethod: selector forClassTrait: forClassTrait on: stream [
	"Writes the given method's source to the given stream in Topaz Filein format."

	| lf cat src |
	self _fileoutHeaderOn: stream.
	src := forClassTrait
		ifTrue: [ self classSourceCodeAt: selector ]
		ifFalse: [ self instanceSourceCodeAt: selector ].
	src class ~~ Unicode7
		ifTrue: [ 
			(stream dynamicInstVarAt: #'utf8Bool') == false
				ifTrue: [ 
					(Unicode7 _withAll: src)
						ifNotNil: [ :s | 
							"source ok"
							 ]
						ifNil: [ 
							Error
								signal:
									'source for ' , selector printString
										, ' contains codePoint > 127  when non-Utf8 fileout requested' ] ] ].
	lf := Character lf.
	cat := forClassTrait
		ifTrue: [ self _categoryOfSelectorForClassTrait: selector ]
		ifFalse: [ self _categoryOfSelectorForInstanceTrait: selector ].
	cat ifNil: [ ^ self _error: #'rtErrKeyNotFound' args: {selector} ].
	stream
		nextPutAll: 'category: ''';
		_fileOutAll: cat;
		nextPut: $';
		nextPut: lf.
	forClassTrait
		ifTrue: [ stream nextPutAll: 'trclassmethod: ' ]
		ifFalse: [ stream nextPutAll: 'trmethod: ' ].
	stream
		_fileOutAll: self name;
		nextPut: lf.
	stream _fileOutAll: src.
	src last == lf
		ifFalse: [ stream nextPut: lf ].
	stream
		nextPut: $%;
		nextPut: lf
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutMethodRemovalOn: stream name: traitname [
	"Writes code to remove all the receiver's methods onto the given stream
  in filein format."

	self _fileoutHeaderOn: stream.
	stream
		nextPutAll: '! ------------------- Remove existing behavior from ';
		_fileOutAll: traitname;
		lf;
		nextPutAll: 'trremoveallmethods ';
		_fileOutAll: traitname;
		lf;
		nextPutAll: 'trremoveallclassmethods ';
		_fileOutAll: traitname;
		lf
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutMethodsOn: stream [
	"File out this trait's methods, but sort the selectors alphabetically."

	| sels nm |
	nm := self name.
	self fileOutMethodRemovalOn: stream name: nm.

	stream
		nextPutAll: '! ------------------- Class methods for ';
		_fileOutAll: nm;
		lf.
	sels := SortedCollection withAll: self classSelectors.
	1 to: sels size do: [ :i | self fileOutClassMethod: (sels at: i) on: stream ].
	stream
		nextPutAll: '! ------------------- Instance methods for ';
		_fileOutAll: nm;
		lf.
	sels := SortedCollection withAll: self instanceSelectors.
	1 to: sels size do: [ :i | self fileOutInstanceMethod: (sels at: i) on: stream ].
	^ stream
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutTraitCategoryOn: stream [
	"Writes out trait category, if there is one for this trait."

	self _fileoutHeaderOn: stream.
	self _traitCategory
		ifNotNil: [ :cat | 
			stream
				nextPutAll: 'expectvalue /Trait';
				lf;
				nextPutAll: 'doit';
				lf;
				_fileOutAll: self name asString;
				nextPutAll: ' category: ';
				_fileOutAll: cat quoted;
				lf;
				nextPut: $%;
				lf ]
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutTraitDefinitionOn: stream [
	"Writes the receiver's preclass, trait definition, and comment onto
 the given stream in filein format."

	self _fileoutHeaderOn: stream.
	stream
		nextPutAll: '! ------------------- Trait definition for ';
		_fileOutAll: self name;
		lf.
	stream
		nextPutAll: 'expectvalue /Trait';
		lf;
		nextPutAll: 'doit';
		lf;
		_fileOutAll: self definition;
		lf;
		nextPut: $%;
		lf.
	self fileOutCommentOn: stream.
	self fileOutTraitCategoryOn: stream
]

{ #category : 'fileout' }
GsTraitImpl >> fileOutTraitOn: stream [
"Writes the receiver's definition and methods onto the given stream in
 filein format."

self _fileoutHeaderOn: stream  .
self fileOutTraitDefinitionOn: stream .
self fileOutMethodsOn: stream .
]

{ #category : 'accessing' }
GsTraitImpl >> instanceCategories [
	^instanceCategories
]

{ #category : 'accessing' }
GsTraitImpl >> instanceCategories: object [
	instanceCategories := object
]

{ #category : 'accessing' }
GsTraitImpl >> instanceSelectors [
	^ self instanceSourceStrings keys
]

{ #category : 'accessing' }
GsTraitImpl >> instanceSourceCodeAt: selectorSymbol [
  ^ instanceSourceStrings at: selectorSymbol otherwise: nil
]

{ #category : 'accessing' }
GsTraitImpl >> instanceSourceStrings [
  ^ instanceSourceStrings
]

{ #category : 'accessing' }
GsTraitImpl >> instanceTrait [
	^instanceTrait
]

{ #category : 'accessing' }
GsTraitImpl >> instanceTrait: object [
	instanceTrait := object
]

{ #category : 'accessing' }
GsTraitImpl >> instVarNames [
	^ classForCompiles instVarNames
]

{ #category : 'updating' }
GsTraitImpl >> instVars: ivNamesArray classVars: cvNamesArray classInstVars: civNamesArray [
	classForCompiles := Object
		subclass: name , '_traitClass'
		instVarNames: ivNamesArray
		classVars: cvNamesArray
		classInstVars: civNamesArray
		poolDictionaries: #()
		inDictionary: nil
		options: #(#'subclassesDisallowed' #'instancesNonPersistent')
]

{ #category : 'accessing' }
GsTraitImpl >> name [
	^ name
]

{ #category : 'initialization' }
GsTraitImpl >> name: aSymbol instVars: ivNamesArray classVars: cvNamesArray classInstVars: civNamesArray [
	name := aSymbol.
	instanceSourceStrings := SymbolDictionary new.
	classSourceStrings := SymbolDictionary new.
	instanceCategories := SymbolDictionary new.
	classCategories := SymbolDictionary new.
	instanceDependents := IdentitySet new.
	classDependents := IdentitySet new.
	extraDict := SymbolDictionary new.
	extraDict objectSecurityPolicy: self objectSecurityPolicy.
	self
		instVars: ivNamesArray
		classVars: cvNamesArray
		classInstVars: civNamesArray
]

{ #category : 'updating' }
GsTraitImpl >> objectSecurityPolicy: anObjectSecurityPolicy [
	"Assigns the receiver and subcomponents to the given security policy."

	super objectSecurityPolicy: anObjectSecurityPolicy.
	instanceSourceStrings objectSecurityPolicy: anObjectSecurityPolicy.
	classSourceStrings objectSecurityPolicy: anObjectSecurityPolicy.
	instanceCategories objectSecurityPolicy: anObjectSecurityPolicy.
	classCategories objectSecurityPolicy: anObjectSecurityPolicy.
	instanceDependents objectSecurityPolicy: anObjectSecurityPolicy.
	classDependents objectSecurityPolicy: anObjectSecurityPolicy.
	classForCompiles objectSecurityPolicy: anObjectSecurityPolicy.
	extraDict objectSecurityPolicy: anObjectSecurityPolicy
]

{ #category : 'updating' }
GsTraitImpl >> removeAllClassSelectors [
	self classSourceStrings keys do: [ :sel | self removeClassSelector: sel ]
]

{ #category : 'updating' }
GsTraitImpl >> removeAllInstanceSelectors [
	self instanceSourceStrings keys do: [ :sel | self removeSelector: sel ]
]

{ #category : 'updating' }
GsTraitImpl >> removeClassDependent: aClass [
	"remove dependent AND remove the methods from aClass"

	classDependents ifNil: [ ^ self ].
	self removeClassDependentOnly: aClass.
	classSourceStrings
		keysAndValuesDo: [ :selector :sourceString | self _basicRemove: selector from: aClass class origin: self classTrait ]
]

{ #category : 'updating' }
GsTraitImpl >> removeClassDependentOnly: aClass [
	"remove the dependent, but leave the trait methods in aClass"

	classDependents ifNil: [ ^ self ].
	classDependents remove: aClass
]

{ #category : 'updating' }
GsTraitImpl >> removeClassSelector: aString [
	| sel |
	classDependents do: [ :aCls | aCls class removeSelector: aString ].
	sel := aString asSymbol.
	self classSourceStrings removeKey: sel.
	self classCategories removeKey: sel
]

{ #category : 'updating' }
GsTraitImpl >> removeFromSystem [
	"When a trait is removed from the system it should remove instanceTrait and classTrait 
		from dependents -- trait methods will be removed from dependents as well."

	| trait |
	trait := self classTrait.
	classDependents copy
		do: [ :dependentClass | dependentClass removeClassTrait: trait ].
	trait := self instanceTrait.
	instanceDependents copy
		do: [ :dependentClass | dependentClass removeTrait: trait ]
]

{ #category : 'updating' }
GsTraitImpl >> removeInstanceDependent: aClass [
	"remove the dependent, but leave the trait methods in aClass"

	instanceDependents ifNil: [ ^ self ].
	self removeInstanceDependentOnly: aClass.
	instanceSourceStrings
		keysAndValuesDo: [ :selector :sourceString | self _basicRemove: selector from: aClass origin: self instanceTrait ]
]

{ #category : 'updating' }
GsTraitImpl >> removeInstanceDependentOnly: aClass [
	"remove the dependent, but leave the trait methods in aClass"

	instanceDependents ifNil: [ ^ self ].
	instanceDependents remove: aClass.
]

{ #category : 'updating' }
GsTraitImpl >> removeSelector: aString [
	| sel |
	sel := aString asSymbol.
	self instanceSourceStrings removeKey: sel.
	self instanceCategories removeKey: sel.
	instanceDependents do: [ :aCls | aCls removeSelector: aString ].
]
