"
UpgradeStreams is a class used during image upgrade.  
Methods should only be executed via the image upgrade scripts.
"
Class {
	#name : 'UpgradeStreams',
	#superclass : 'PrivateObject',
	#category : 'Kernel-Build'
}

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _checkLegacyStreams [
GemStone_Legacy_Streams valuesDo:[:v |
 v isBehavior ifTrue:[ | key lgAssoc assoc |
   key := v name .
   lgAssoc := GemStone_Legacy_Streams associationAt: key otherwise: nil .
   assoc := Globals associationAt: key otherwise: nil .
   lgAssoc == assoc ifTrue:[
     Error signal: key , ' has identical association in Globals and GemStone_Legacy_Streams'.
   ].
   lgAssoc ifNil:[ Error signal: key , ' not found in GemStone_Legacy_Streams'].
   "may not be in Globals at this point during slowfilein"
 ].
].

]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _copyMethodsTo_legacyStreams [
"copy methods to classes in GemStone_Legacy_Streams"
| suffixSize report |
report := String new .
suffixSize := 'Legacy' size .
{ WriteStreamLegacy . ReadStreamLegacy . PositionableStreamLegacy . 
  ReadByteStreamLegacy } do:[:cls| | nam destNam dest |
   nam := cls name  .
   destNam := (nam copyFrom: 1 to: (nam size - suffixSize)) asSymbol  .
   dest := GemStone_Legacy_Streams at: destNam .
   report addAll: (cls _copyMethodsTo: dest ) ; lf .
].
GsFile gciLogServer: report .

]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _copyMethodsTo_Portable_Streams [
"copy methods to classes in GemStone_Portable_Streams"
| suffixSize report |
report := String new .
suffixSize := 'Portable' size .
{ PositionableStreamPortable . ReadStreamPortable . WriteStreamPortable .
  ReadWriteStreamPortable . FileStreamPortable . ReadByteStreamPortable } do:[:cls| | nam destNam dest |
   nam := cls name  .
   destNam := (nam copyFrom: 1 to: (nam size - suffixSize)) asSymbol  .
   dest := GemStone_Portable_Streams at: destNam .
   report addAll: (cls _copyMethodsTo: dest ) ; lf .
].
GsFile gciLogServer: report .
]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _initializeForInstall [
  self _initStreamLiterals .

  Globals at: #PositionableStream_position  put: #ANSI .
  self _copyMethodsTo_Portable_Streams ;
   _copyMethodsTo_legacyStreams ;
   _installPortableStreams ;
   _initTranscriptStreamPortable ;
   _testStreamLiterals ;
   _initRandoms .
  ^ true
]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _initRandoms [
  (Globals at: #HostRandom) initialize .
  #( Random HostRandom SeededRandom Lag1MwcRandom Lag25000CmwcRandom )
  do:[:sym | (Globals at: sym) immediateInvariant ]

]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _initStreamLiterals [
  Stream _initializeWriteStreamClassVars: (GemStone_Portable_Streams at: #WriteStream) .
  Stream _initializeWriteStreamClassVars:(GemStone_Legacy_Streams at: #WriteStream) .
  Stream _initializeStreamClassVars .

  Stream _initializeWriteStreamClassVars: (Globals at: #WriteStreamLegacy).
  Stream _initializeWriteStreamClassVars: (Globals at: #WriteStreamPortable).

]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _initTranscriptStreamPortable [
  (Globals at: #TranscriptStreamPortable) initialize.

]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _installLegacyStreams: positionableStream_position [
	"Install legacy classes without 'Legacy' in name into Globals

	positionableStream_position dictates which position methods implementation to use, based on original value"

	self _installStreamImplementationFrom: GemStone_Legacy_Streams.
	Globals at: #'PositionableStream_position' put: positionableStream_position.
	PositionableStream compilePositionMethods
]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _installPortableStreams [
"Install portable classes without 'Portable' in name into Globals"

self _installStreamImplementationFrom: GemStone_Portable_Streams
]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _installStreamImplementationFrom: aSymbolDictionary [
  | streamClass installBlock |
  GsFile gciLogServer:'_installStreamImplementation from ', aSymbolDictionary name printString.

  installBlock := [:className |
    streamClass := aSymbolDictionary at: className.
    Globals at: className  put: streamClass.
    GsFile gciLogServer:'   -- Globals at: ', className printString, ' put: ', streamClass asOop asString].

  installBlock value: #PositionableStream.
  installBlock value: #ReadStream.
  installBlock value: #WriteStream.
  installBlock value: #ReadByteStream.

"ReadWriteStream and FileStream defined in Portable implementation, but not in
 Legacy implementation"
(aSymbolDictionary at: #ReadWriteStream otherwise: nil) ifNil: [ 
   (Globals removeKey: #ReadWriteStream otherwise:  nil )
     ifNotNil:[:removedClass | GsFile gciLogServer: '  -- Globals removeKey: #ReadWriteStream (', removedClass asOop asString, ')' ] 
   ] ifNotNil: [:readWriteStreamClass | 
     installBlock value: #ReadWriteStream 
   ].
(aSymbolDictionary at: #FileStream otherwise: nil ) ifNil: [ 
   (Globals removeKey: #FileStream otherwise: nil)
      ifNotNil:[:removedClass | GsFile gciLogServer: '  -- Globals removeKey: #FileStream (', removedClass asOop asString, ')' ] 
   ] ifNotNil: [:fileStreamClass | 
      installBlock value: #FileStream 
   ].
]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> _testStreamLiterals [
  | strm ary rpt exp |
  rpt := String new .
  strm := PrintStream on: String new .
  strm cr; lf ; space ; tab . "note cr same as lf for PrintStream"
  ary := { } .
  strm contents do:[:c | ary add: c  codePoint ] .
  ary = #( 10 10 32 9 ) ifFalse:[
    rpt add: 'PrintStream; ', ary printString; lf
  ].
  strm := AppendStream on: String new .
  strm cr ; crlf; crtab ; crtab: 2 ; lf ; space ; tab ; space: 2 ; tab: 2 .
  ary := { } .
  strm contents do:[:c | ary add: c  codePoint ] .
  ary = (exp := #( 13 13 10 13 9 13 9 9 10 32 9 32 32 9 9 ))
     ifFalse:[ rpt add: 'AppendStream; ', ary printString ; lf;
                   add: 'expected ', exp printString; lf ].

  "Assume portable streams are installed in virgin dbf"
  "Note WriteStreamLegacy inherits Stream>>cr which appends codepoint 10"
  { WriteStreamLegacy . (GemStone_Legacy_Streams at: #WriteStream) } do:[:cls |
    strm := cls on: String new .
    strm cr ; crlf; crtab ; crtab: 2 ; lf ; space ; tab ; space: 2; tab: 2 .
    ary := { } .
    strm contents do:[:c | ary add: c  codePoint ] .
    ary = (exp := #( 10 13 10 13 9 13 9 9 10 32 9 32 32 9 9 ))
     ifFalse:[ rpt add: cls name ,' oop ', cls asOop asString, ' ', ary printString ; lf;
                   add: 'expected ', exp printString; lf ].
  ].
  { WriteStreamPortable . (Globals at: #WriteStream) .
    (GemStone_Portable_Streams at: #WriteStream)  } do:[:cls |
    strm := cls on: String new .
    strm cr ; crlf ; crtab ; crtab: 2 ; lf ; space ; tab ; space: 2 ; tab: 2 .
    ary := { } .
    strm contents do:[:c | ary add: c  codePoint ] .
    ary = (exp := #( 13 13 10 13 9 13 9 9 10 32 9 32 32 9 9 ))
     ifFalse:[ rpt add: cls name ,' oop ', cls asOop asString, ' ', ary printString; lf;
                   add: 'expected ', exp printString; lf ].
  ].
  ary := { Character backspace .
    Character lf .
    Character cr .
    Character esc .
    Character newPage .
    Character tab } collect:[:c | c codePoint ] .
  ary = (exp := #( 8 10 13 27 12 9 )) ifFalse:[ rpt add: 'Character; ', ary printString; lf;
                   add: 'expected ', exp printString; lf ].
  rpt size == 0 ifFalse:[
     GsFile gciLogServer:rpt .
     Error signal:'Bad results(s)'.
  ].
  ^ true

]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> initializeForInstall [

  (Globals at: #PositionableStream_position otherwise: nil) ifNotNil:[:v |
    v == #ANSI ifFalse:[ Error signal:'about to overwrite Legacy Streams with ANSI'].
  ] ifNil:[
    self _checkLegacyStreams . "should still be in Globals"
  ].
  ^ self _initializeForInstall
]

{ #category : 'Image Upgrade' }
UpgradeStreams class >> initializeForUpgrade [
	"called during upgradeImage only ...records which flavor of Stream is currently installed (Legacy or Portable); 
    upgrades Stream classes using initialize; and then ensures that original flavof of Stream is installed"

	| installedStreamImageStatus positionableStream_position |
	installedStreamImageStatus := {false.	"was GemStone_Legacy_Streams present at start of install?"
	false	"If GemStone_Legacy_Streams present (i.e. upgrade from 3.x), was the 
					Portable Stream implementation installed at start of install?
					If not present (build or upgrade from .x), is this an upgrade?"}.
	positionableStream_position := Globals
		at: #'PositionableStream_position'
		ifAbsent: [ 
			(Globals includesKey: #'DbfHistory')
				ifTrue: [ 
					"this is a database upgrade so keep old behavior"
					#'Legacy' ]
				ifFalse: [ 
					"this is a new database; switch to #'ANSI' in 3.0"
					#'ANSI' ] ].
	(installedStreamImageStatus
		at: 1
		put: (Globals includesKey: #'GemStone_Legacy_Streams'))
		ifTrue: [ 
			"True means Portable installed, false means Legacy installed (Bug45450)"
			(installedStreamImageStatus
				at: 2
				put:
					((Globals at: #'PositionableStream') instVarNames includes: #'collection'))
				ifTrue: [ self _installLegacyStreams: positionableStream_position ] ]
		ifFalse: [ 
			"True means upgrade, false means build --- only called during upgrade"
			installedStreamImageStatus at: 2 put: true ].
	self _initializeForInstall.	"install streams - Portable streams will be installed when completed"
	PositionableStream isPortableStreamImplementation
		ifFalse: [ 
			"we always install portable stream during upgrade - at this point portable streams should be installed"
			Error signal: 'Expected PortableStreams to be installed' ].
	(installedStreamImageStatus at: 1)
		ifTrue: [ 
			"install stream implementation based on original status of the image"
			(installedStreamImageStatus at: 2)
				ifTrue: [ 
					self _installPortableStreams.
					GsFile gciLogServer: 'installPortableStreamImplementation ' ]
				ifFalse: [ 
					self _installLegacyStreams: positionableStream_position.
					GsFile
						gciLogServer:
							'installLegacyStreamImplementation position: '
								, positionableStream_position printString ] ]
		ifFalse: [ 
			"Conversion or upgrade from 2.x install legacy stream implementation"
			(installedStreamImageStatus at: 2)
				ifTrue: [ 
					"true means upgrade from 2.x"
					self _installLegacyStreams: positionableStream_position.
					GsFile
						gciLogServer:
							'installLegacyStreamImplementation position: '
								, positionableStream_position printString ]
				ifFalse: [ 
					"false means build ... should never get here, but if we do, Portable streams should be already installed"
					((Globals at: #'PositionableStream') instVarNames includes: #'collection')
						ifFalse: [ Error signal: 'Legacy Stream implementation is unexpectedly installed' ] ] ].
	^ true
]
