Extension { #name : 'TestResult' }

{ #category : 'Exceptions' }
TestResult class >> error [
	^self exError

]

{ #category : 'Exceptions' }
TestResult class >> exError [
	^SUnitNameResolver errorObject

]

{ #category : 'Exceptions' }
TestResult class >> failure [
	^TestFailure

]

{ #category : 'Init / Release' }
TestResult class >> new [
	^super new initialize

]

{ #category : 'Exceptions' }
TestResult class >> resumableFailure [
	^ResumableTestFailure

]

{ #category : 'Exceptions' }
TestResult class >> signalErrorWith: aString [
	self error sunitSignalWith: aString

]

{ #category : 'Exceptions' }
TestResult class >> signalFailureWith: aString [
	self failure sunitSignalWith: aString

]

{ #category : 'Adding' }
TestResult >> addError: aTestCase [
	"We cannot use self errors as that incorporates test expectations and so does not return the stored collection."

	^self errors add: aTestCase

]

{ #category : 'Adding' }
TestResult >> addFailure: aTestCase [

	^self failures add: aTestCase
]

{ #category : 'Adding' }
TestResult >> addPass: aTestCase [
	"We cannot use self passed as that incorporates test expectations and so does not return the stored collection."

	^self passed add: aTestCase

]

{ #category : 'Deprecated' }
TestResult >> correctCount [
	"deprecated - use #passedCount"

	self deprecated: 'TestResult>>correctCount deprecated v3.2, use passedCount instead'.
	^self passedCount

]

{ #category : 'Accessing' }
TestResult >> defects [
	^OrderedCollection new
		addAll: self errors;
		addAll: self failures; yourself

]

{ #category : 'Accessing' }
TestResult >> errorCount [

	^self errors size

]

{ #category : 'Accessing' }
TestResult >> errors [

	errors == nil
		ifTrue: [errors := OrderedCollection new].
	^errors

]

{ #category : 'Accessing' }
TestResult >> failureCount [

	^self failures size

]

{ #category : 'Accessing' }
TestResult >> failures [
	"We use a Set, not an OrderedCollection as #errors and #passed do, because a resumable test failure in a loop can raise many failures against the same test. In current SUnit UIs, this could result in bizarre test count reporting (-27 tests run, and suchlike). This will be reviewed."

	failures ifNil: [failures := Set new].
	^failures

]

{ #category : 'Testing' }
TestResult >> hasErrors [

	^self errors size > 0

]

{ #category : 'Testing' }
TestResult >> hasFailures [

	^self failures size > 0

]

{ #category : 'Testing' }
TestResult >> hasPassed [

	^self hasErrors not and: [self hasFailures not]

]

{ #category : 'Init / Release' }
TestResult >> initialize [

]

{ #category : 'Testing' }
TestResult >> isError: aTestCase [

	^self errors includes: aTestCase

]

{ #category : 'Testing' }
TestResult >> isFailure: aTestCase [
	^self failures includes: aTestCase

]

{ #category : 'Testing' }
TestResult >> isPassed: aTestCase [

	^self passed includes: aTestCase

]

{ #category : 'Accessing' }
TestResult >> passed [

	passed == nil
		ifTrue: [passed := OrderedCollection new].

	^passed

]

{ #category : 'Accessing' }
TestResult >> passedCount [

	^self passed size

]

{ #category : 'Printing' }
TestResult >> printOn: aStream [

	aStream
		nextPutAll: self runCount printString;
		nextPutAll: ' run, ';
		nextPutAll: self passedCount printString;
		nextPutAll: ' passed, ';
		nextPutAll: self failureCount printString;
		nextPutAll: ' failed, ';
		nextPutAll: self errorCount printString;
		nextPutAll: ' error'.

	self errorCount ~= 1
		ifTrue: [aStream nextPut: $s]

]

{ #category : 'Running' }
TestResult >> runCase: aTestCase [
	[aTestCase runCase.
	self addPass: aTestCase]
		on: self class failure , self class error
		do: [:ex | ex sunitAnnounce: aTestCase toResult: self]

]

{ #category : 'Accessing' }
TestResult >> runCount [

	^self passedCount + self failureCount + self errorCount

]

{ #category : 'Accessing' }
TestResult >> tests [
	^(OrderedCollection new: self runCount)
		addAll: passed;
		addAll: failures;
		addAll: errors;
		yourself

]
