Extension { #name : 'ReadWriteStreamPortable' }

{ #category : 'Passivation - Support' }
ReadWriteStreamPortable >> _fastNext [
  "Returns the next item on the input stream without end of stream checks.
 This may result in a bad access error instead of an end of stream error."

  ^ collection at: (position := position + 1)

]

{ #category : 'Passivation - Support' }
ReadWriteStreamPortable >> _fastPosition [
  "Returns the receiver's current position reference for accessing the sequence
 of objects.  The position is actually the next element of the collection to be
 read or written; the position is incremented by each read or write.  In
 general, to start reading or writing at the nth element of a collection, the
 position must be set to n."

  "See bug 42596: Compatible with Legacy Stream positioning"

  ^ position + 1

]

{ #category : 'Passivation - Support' }
ReadWriteStreamPortable >> _fastPosition: pos [
  "Sets the position of the receiver without checking the validity of the
 argument 'pos'.  This may result in bad index errors instead of end
 of stream errors in subsequent access of the stream."

  "See bug 42596: Compatible with Legacy Stream positioning"

  position := pos - 1

]

{ #category : 'Accessing' }
ReadWriteStreamPortable >> contents [
"Answer with a copy of my collection from 1 to readLimit."

readLimit := readLimit max: position.
^collection copyFrom: 1 to: readLimit

]

{ #category : 'Accessing' }
ReadWriteStreamPortable >> next [
"Returns the next object that the receiver can access for reading.
 Returns nil if an attempt is made to read beyond the end of the stream."

position >= readLimit
	ifTrue: [^nil]
	ifFalse: [^collection at: (position := position + 1)]

]

{ #category : 'Accessing' }
ReadWriteStreamPortable >> next: amount [
"Answer the next amount elements of my collection.  overriden for efficiency"

| ans endPosition |
amount < 0 ifTrue: [ ^self error: 'amount may not be less than 0' ].
readLimit := readLimit max: position.

endPosition := position + amount  min:  readLimit.
ans := collection copyFrom: position + 1 to: endPosition.
position := endPosition.
^ans

]

{ #category : 'Passivation - Support' }
ReadWriteStreamPortable >> nextBytes: count addTo: anObject [
  | pos |
  count <= 0 ifTrue:[ ^ anObject ].
  (self atEnd)
   ifTrue: [ ^ EndOfStream signal ].
  anObject addAllBytes: (collection copyFrom: (pos := position) + 1 to: pos + count ).
  position := pos + count .
  ^ anObject .

]

{ #category : 'Passivation - Support' }
ReadWriteStreamPortable >> nextElements: count into: aCollection [
  "Stores the next count elements that the receiver can access for reading
 into aCollection.  The receiver's collection and aCollection must be compatible
 SequenceableCollections.  Returns the count of elements read.

 Generates an error if an attempt is made to read beyond the end of the stream."

  "Used by PassiveObject."

  ^ self next: count into: aCollection startingAt: 1

]

{ #category : 'Testing' }
ReadWriteStreamPortable >> nextMatchFor: anObject [

	"The first object is removed from the receiver's future sequence value and appended to the end of
	the receiver's past sequence values. The value that would result from sending #= to the object with
	anObject as the argument is returned.
	The results are undefined if there are no future sequence values in the receiver.

	ANSI 5.9.2.6"

	^self next = anObject.

]
