Extension { #name : 'ReadByteStreamPortable' }

{ #category : 'Instance Creation' }
ReadByteStreamPortable class >> on: aCollection [
  (aCollection _stringCharSize == 0) ifTrue:[
     ArgumentError signal:'expected a String, MultiByteString, or ByteArray'.
  ].
  ^ super on: aCollection
]

{ #category : 'Instance Creation' }
ReadByteStreamPortable class >> on: aCollection from: firstIndex to: lastIndex [
  "disallowed"
  self shouldNotImplement: #on:from:to:
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> atEnd [
  ^ (collection atOrNil: position + 1) == nil 
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> beforeEnd [
  ^ (collection atOrNil: position + 1) ~~ nil 
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> contents [
"Answer with a copy of my collection from 1 to readLimit."

^collection copyFrom: 1 to: collection size

]

{ #category : 'Testing' }
ReadByteStreamPortable >> isEmpty [
"Answer whether the receiver's contents has no elements."

"Returns true if both the set of past and future sequence values of
the receiver are empty. Otherwise returns false"

 ^ position ifNotNil:[:p | position == 0 ] ifNil: [ ^ true ]
]

{ #category : 'Comparing' }
ReadByteStreamPortable >> match: subCollection [
"Set the access position of the receiver to be past the next occurrence of the subCollection. Answer whether subCollection is found.  No wildcards, and case does matter."
| pattern startMatch |
pattern := ReadStreamPortable on: subCollection.
startMatch := nil.
[ pattern atEnd ] whileFalse:
  [
    self atEnd ifTrue:[ ^ false ].
    self next = pattern next
      ifTrue: [ pattern position = 1 ifTrue: [ startMatch := self position ] ]
      ifFalse:
        [ pattern position: 0.
        startMatch ifNotNil:
          [ self position: startMatch.
          startMatch := nil ] ] ].
^ true

]

{ #category : 'Accessing' }
ReadByteStreamPortable >> next [
 "Returns the next object that the receiver can access for reading.
  Returns nil if an attempt is made to read beyond the end of the stream."
  | pos |
  (collection atOrNil: (pos := position + 1)) ifNotNil:[:res | 
     position := pos  .
     ^ res
  ].
  ^ nil
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> next: amount [
 "Answer the next amount elements of my collection."

 | ans endPosition pos |
 amount < 0 ifTrue: [ ^self error: 'amount may not be less than 0' ].
 
 endPosition := (pos := position) + amount  min:  collection size .
 ans := collection copyFrom: pos+1 to: endPosition.
 position := endPosition.
 ^ans
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> next: n into: aCollection startingAt: startIndex [
 "Read n objects into the given collection.
  Return aCollection or a partial copy if less than
  n elements have been read."
  | max pos |
  max := (collection size  - (pos := position)) min: n.
  aCollection
    replaceFrom: startIndex
    to: startIndex+max-1
    with: collection
    startingAt: pos +1.
  position := pos + max.
  max = n
    ifTrue:[^aCollection]
    ifFalse:[^aCollection copyFrom: 1 to: startIndex+max-1]
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> nextLine [
"Answer next line (may be empty) without line end delimiters, or nil if at end.
Leave the stream positioned after the line delimiter(s).
Handle a zoo of line delimiters CR, LF, or CR-LF pair"

| cr lf result ch |
self atEnd ifTrue: [ ^nil ].
cr := Character cr.
lf := Character lf.
result := self collectionSpecies new.
[ ch := self next .
  (ch == cr or:[ ch == lf ]) ifTrue:[
    ch == cr ifTrue:[ self peekFor: lf ].
    ^ result
  ].
  result add: ch .
  self atEnd
] whileFalse .
^ result

]

{ #category : 'Accessing' }
ReadByteStreamPortable >> nextOrNil [
 "Returns the next object that the receiver can access for reading.
  Returns nil if an attempt is made to read beyond the end of the stream."
  | pos |
  (collection atOrNil: (pos := position + 1)) ifNotNil:[:res | 
     position := pos  .
     ^ res
  ].
  ^ nil
]

{ #category : 'Positioning' }
ReadByteStreamPortable >> on: aCollection [
  collection := aCollection.
  readLimit := nil .
  position := 0.
]

{ #category : 'Private' }
ReadByteStreamPortable >> on: aCollection from: firstIndex to: lastIndex [
  "disallowed"
  self shouldNotImplement: #on:from:to:
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> peek [
  "Answer what would be returned if the message next were sent to the
  receiver. If the receiver is at the end, answer nil. 
  Reimplemented as an optimization"
  ^ collection atOrNil: position + 1 .
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> peek2 [
 "Peeks at the second incoming object.   
  Returns nil if there are not 2 more elements in the receiver."

  ^ collection atOrNil: position + 2 
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> peekFor: anObject [
 "Answer false and do not move over the next element if it is not equal to
 the argument, anObject, or if the receiver is at the end. Answer true
 and increment the position for accessing elements, if the next element is
 equal to anObject."

  | nextObject pos |
  nextObject := collection atOrNil: (pos:= position + 1) .
  nextObject ifNil:[ ^ false ].
  anObject = nextObject ifTrue:[ 
    position := pos .
    ^true
  ].
  ^ false
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> peekN: anInteger [
  "Peek returning the next N characters of collection."
  | end start lim |
  position >= (lim := collection size) ifTrue:[ ^ collection class new ].
  start := position + 1 .
  end := start + anInteger - 1.
  end >= lim ifTrue:[ end := lim - 1 ].
  ^ collection copyFrom: start to: end
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> peekTwice [
  "Answer an array containing the two elements that would would be returned
  if the message #next were sent to the receiver twice. If the receiver is at
  or reaches the end, the array will include one or two nils."

	| array |
  self atEnd
		ifTrue: [^ {  nil . nil } ].
	array := { self next .  self peek } .
	position := position - 1.
	^array

]

{ #category : 'Positioning' }
ReadByteStreamPortable >> position: anInteger [
   "Set the current position for accessing the objects to be anInteger, as long
   as anInteger is within the bounds of the receiver's contents. If it is not,
   create an error notification."
   
   (anInteger >= 0 and: [anInteger <= collection size ])
     ifTrue: [position := anInteger]
     ifFalse: [self positionError: anInteger]
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> size [
  ^ collection size
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> skip: anInteger [
 "Set the receiver's position to be the current position+anInteger. Do not
  throw error if skipAmount would exceed collection bounds - ANSI compliance. "

 self position: ((position + anInteger max: 0) min: collection size )
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> upTo: anObject [
 "fast version using indexOf:"
  | start end |
  start := position+1.
  end := collection indexOf: anObject startingAt: start .
  "not present--return rest of the collection"
  end == 0 ifTrue: [ ^self upToEnd ].

  "skip to the end and return the data passed over"
  position := end.
  ^ collection copyFrom: start to: (end-1)
]

{ #category : 'Accessing' }
ReadByteStreamPortable >> upToEnd [
  | start pos |
  start := 1 + position .
  position := (pos := collection size ) .
  ^ collection copyFrom: start to: pos
]
