Extension { #name : 'PositionableStreamPortable' }

{ #category : 'Portable Methods' }
PositionableStreamPortable class >> isLegacyStreamImplementation [

^false

]

{ #category : 'Portable Methods' }
PositionableStreamPortable class >> isPortableStreamImplementation [

^true

]

{ #category : 'Instance Creation' }
PositionableStreamPortable class >> on: aCollection [
"Answer an instance of me, streaming over the elements of aCollection."

^self basicNew on: aCollection.

]

{ #category : 'Instance Creation' }
PositionableStreamPortable class >> on: aCollection from: firstIndex to: lastIndex [
"Answer an instance of me, streaming over the elements of aCollection
starting with the element at firstIndex and ending with the one at
lastIndex."

	^self basicNew on: (aCollection copyFrom: firstIndex to: lastIndex)

]

{ #category : 'Accessing' }
PositionableStreamPortable >> _collection [

"Returns the collection of the receiver."

^collection

]

{ #category : 'Converting' }
PositionableStreamPortable >> asPetitStream [
	"Some of my subclasses do not use the instance-variables collection, position and readLimit but instead have a completely different internal representation. In these cases just use the super implementation that is inefficient but should work in all cases."

"
	Disabled until we agree on some way how to optimize this

	^ (collection isNil or: [ position isNil or: [ readLimit isNil ] ])
		ifFalse: [ PPStream on: collection from: ( position + 1 ) to: readLimit ]
      ifTrue: [ super asPetitStream ]
"
	^ super asPetitStream

]

{ #category : 'Testing' }
PositionableStreamPortable >> atBeginning [
"Answer true if the stream is positioned at the beginning"

^position == 0

]

{ #category : 'Testing' }
PositionableStreamPortable >> atEnd [
"Returns true if the receiver cannot access any more objects, false if it can."

^position >= readLimit
]

{ #category : 'Testing' }
PositionableStreamPortable >> beforeEnd [
"Returns true if the receiver can access more objects, false if not .
 GemStone extension. "

^position < readLimit
]

{ #category : 'Accessing' }
PositionableStreamPortable >> collection [
	^ collection

]

{ #category : 'Accessing' }
PositionableStreamPortable >> collectionSpecies [

  ^ collection species

]

{ #category : 'Accessing' }
PositionableStreamPortable >> contents [
"Answer with a copy of my collection from 1 to readLimit."

^collection copyFrom: 1 to: (readLimit ifNil:[ collection size])

]

{ #category : 'Testing' }
PositionableStreamPortable >> isEmpty [
"Answer whether the receiver's contents has no elements."

"Returns true if both the set of past and future sequence values of
the receiver are empty. Otherwise returns false"

 ^ (position ifNotNil:[:p | p >= readLimit] ifNil:[ self atEnd]) and:[ position == 0 ]
]

{ #category : 'Comparing' }
PositionableStreamPortable >> match: subCollection [
"Set the access position of the receiver to be past the next occurrence of the subCollection. Answer whether subCollection is found.  No wildcards, and case does matter."
| pattern startMatch |
pattern := ReadStreamPortable on: subCollection.
startMatch := nil.
[ pattern atEnd ] whileFalse:
	[ 
   (position ifNotNil:[:p | p >= readLimit] ifNil:[ self atEnd]) ifTrue:[ ^ false ].
	  self next = pattern next
		  ifTrue: [ pattern position = 1 ifTrue: [ startMatch := self position ] ]
		  ifFalse:
			  [ pattern position: 0.
			  startMatch ifNotNil:
				  [ self position: startMatch.
				  startMatch := nil ] ] ].
^ true

]

{ #category : 'Accessing' }
PositionableStreamPortable >> next: amount [
"Answer the next amount elements of my collection. Must override
because default uses self contents species, which might involve a large
collection."

| newArray |
amount < 0 ifTrue: [ ^self error: 'amount may not be less than 0' ].
newArray := collection species new: amount.
1 to: amount do: [:index | newArray at: index put: self next].
^newArray

]

{ #category : 'Accessing' }
PositionableStreamPortable >> next: n into: aCollection [
"Read n objects into the given collection.
Return aCollection or a partial copy if less than
n elements have been read."
^self next: n into: aCollection startingAt: 1

]

{ #category : 'Accessing' }
PositionableStreamPortable >> next: n into: aCollection startingAt: startIndex [
"Read n objects into the given collection.
Return aCollection or a partial copy if less than
n elements have been read."
| obj |
0 to: n-1 do:[:i|
	(obj := self next) == nil ifTrue:[^aCollection copyFrom: 1 to: startIndex+i-1].
	aCollection at: startIndex+i put: obj].
^aCollection

]

{ #category : 'Adding' }
PositionableStreamPortable >> next: anInteger putAll: aCollection startingAt: startIndex [
"Store the next anInteger elements from the given collection."
(startIndex == 1 and:[anInteger == aCollection size])
	ifTrue:[^self nextPutAll: aCollection].
^self nextPutAll: (aCollection copyFrom: startIndex to: startIndex+anInteger-1)

]

{ #category : 'Accessing' }
PositionableStreamPortable >> nextLine [
"Answer next line (may be empty) without line end delimiters, or nil if at end.
Leave the stream positioned after the line delimiter(s).
Handle a zoo of line delimiters CR, LF, or CR-LF pair"

| cr lf result ch |
(position ifNotNil:[:p | p >= readLimit] ifNil:[ self atEnd]) ifTrue:[ ^ nil ].
cr := Character cr.
lf := Character lf.
result := self collectionSpecies new.
[ ch := self next .
  (ch == cr or:[ ch == lf ]) ifTrue:[
    ch == cr ifTrue:[ self peekFor: lf ].
    ^ result
  ].
  result add: ch .
  self atEnd 
] whileFalse .
^ result

]

{ #category : 'Positioning' }
PositionableStreamPortable >> on: aCollection [

collection := aCollection.
readLimit := aCollection size.
position := 0.
self reset

]

{ #category : 'Accessing' }
PositionableStreamPortable >> peek [
"Answer what would be returned if the message next were sent to the
receiver. If the receiver is at the end, answer nil."

| nextObject |
(position ifNotNil:[:p | p >= readLimit] ifNil:[ self atEnd]) ifTrue:[ ^ nil ].
nextObject := self next.
position := position - 1.
^nextObject

]

{ #category : 'Accessing' }
PositionableStreamPortable >> peek2 [
"Peeks at the second incoming object."

| nextObject |
position + 1 >= readLimit ifTrue: [^nil].
nextObject := self next; next.
position := position - 2.
^nextObject

]

{ #category : 'Accessing' }
PositionableStreamPortable >> peekFor: anObject [
"Answer false and do not move over the next element if it is not equal to
the argument, anObject, or if the receiver is at the end. Answer true
and increment the position for accessing elements, if the next element is
equal to anObject."

| nextObject |
(position ifNotNil:[:p | p >= readLimit] ifNil:[ self atEnd]) ifTrue:[ ^ false ].
nextObject := self next.
"peek for matching element"
anObject = nextObject ifTrue: [^true].
"gobble it if found"
position := position - 1.
^false

]

{ #category : 'Accessing' }
PositionableStreamPortable >> peekTwice [
  "Answer an array containing the two elements that would would be returned
  if the message #next were sent to the receiver twice. If the receiver is at
  or reaches the end, the array will include one or two nils."

	| array |
  (position ifNotNil:[:p | p >= readLimit] ifNil:[ self atEnd]) 
		ifTrue: [^ {  nil . nil } ].
	array := { self next .  self peek } .
	position := position - 1.
	^array

]

{ #category : 'Positioning' }
PositionableStreamPortable >> position [
"Answer the current position of accessing the sequence of objects."

^position

]

{ #category : 'Positioning' }
PositionableStreamPortable >> position: anInteger [
"Set the current position for accessing the objects to be anInteger, as long
as anInteger is within the bounds of the receiver's contents. If it is not,
create an error notification."

(anInteger >= 0 and: [anInteger <= readLimit])
	ifTrue: [position := anInteger]
	ifFalse: [self positionError: anInteger]

]

{ #category : 'Positioning' }
PositionableStreamPortable >> positionError: anInteger [
"Returns an error message that anInteger is out of bounds of the Collection."

^ self _error: #rtErrBadStreamPosition args: { anInteger }

]

{ #category : 'Positioning' }
PositionableStreamPortable >> reset [
"Set the receiver's position to the beginning of the sequence of objects."

position := 0

]

{ #category : 'Positioning' }
PositionableStreamPortable >> setToEnd [
"Set the position of the receiver to the end of the sequence of objects."

position := readLimit

]

{ #category : 'Accessing' }
PositionableStreamPortable >> skip: anInteger [
"Set the receiver's position to be the current position+anInteger. A
subclass might choose to be more helpful and select the minimum of the
receiver's size and position+anInteger, or the maximum of 0 and
position+anInteger for the repositioning."

self position: position + anInteger

]

{ #category : 'Accessing' }
PositionableStreamPortable >> skipSeparators [
[self atEnd]
	whileFalse:
	[self next isSeparator ifFalse: [^ self position: self position-1]]

]

{ #category : 'Accessing' }
PositionableStreamPortable >> skipTo: anObject [
"Set the access position of the receiver to be past the next occurrence of
anObject. Answer whether anObject is found."

[self atEnd]
	whileFalse: [self next = anObject ifTrue: [^true]].
^false

]

{ #category : 'Accessing' }
PositionableStreamPortable >> upTo: anObject [
"Answer a subcollection from the current access position to the
occurrence (if any, but not inclusive) of anObject in the receiver. If
anObject is not in the collection, answer the entire rest of the receiver."
| newStream element |
newStream := AppendStream on: collection species new.
[self atEnd or: [(element := self next) = anObject]]
	whileFalse: [newStream nextPut: element].
^newStream contents

]

{ #category : 'Accessing' }
PositionableStreamPortable >> upToAll: aCollection [
"Answer a subcollection from the current access position to the occurrence (if any, but not inclusive) of aCollection. If aCollection is not in the stream, answer the entire rest of the stream."

| startPos endMatch result |
startPos := self position.
(self match: aCollection)
	ifTrue: [endMatch := self position.
		self position: startPos.
		result := self next: endMatch - startPos - aCollection size.
		self position: endMatch.
		^ result]
	ifFalse: [self position: startPos.
		^ self upToEnd]

]

{ #category : 'Accessing' }
PositionableStreamPortable >> upToAny: aCollection [
"Answer a subcollection from the current access position to the
occurrence (if any, but not inclusive) of any objects in the given collection in the receiver. If
any of these is not in the collection, answer the entire rest of the receiver."

| newStream element |
newStream := AppendStream on: collection species new.
[self atEnd or: [aCollection includes: (element := self next)]]
	whileFalse: [newStream nextPut: element].
^newStream contents

]

{ #category : 'Accessing' }
PositionableStreamPortable >> upToAnyOf: subcollection do: aBlock [
"Answer a subcollection from the current access position to the occurrence (if any, but not inclusive) of any object in the collection.
Evaluate aBlock with this occurence as argument.
If no matching object is found, don't evaluate aBlock and answer the entire rest of the receiver."

| stream ch |
stream := AppendStream on: collection species new.
[ self atEnd or: [ (subcollection includes: (ch := self next)) and: [aBlock value: ch. true] ] ]
	whileFalse: [ stream nextPut: ch ].
^ stream contents

]

{ #category : 'Accessing' }
PositionableStreamPortable >> upToEnd [
"Answer a subcollection from the current access position through the last element of the receiver."

| newStream |
newStream := AppendStream on: collection species new.
[self atEnd] whileFalse: [ newStream nextPut: self next ].
^ newStream contents

]
