"
AbstractExternalSession is the superclass of a number of implementations
of external sessions.
An external session provides a means of running Smalltalk expressions in a different Gem,
possibly on another server or against a different Stone. The expression to execute can
be provided as a String (via #executeString: and #forkString:) or as a Block (via #executeBlock:
and #forkBlock:).

The #executeBlock: protocol allows the use of all the usual Smalltalk tools for working
with Smalltalk code, but does not support arguments to the Block and also requires that
the code compiles in the local image.

The #executeString: protocol supports the use of any String, but being a String, it is
opaque to the operation of most Smalltalk tools.

The methods #fork* and #nbSend* are asynchronous and require checking for results via
#nbResult and  #wait*  methods .

The Gem processes created by external session logins are not managed by the system, 
so ensure sessions are logged out to prevent lingering sessions consuming system 
resources unnecessarily.
"
Class {
	#name : 'AbstractExternalSession',
	#superclass : 'Object',
	#category : 'ExternalSessions'
}

{ #category : 'Private' }
AbstractExternalSession class >> _stackReport: contextOop [
 "contextOop is from   aGciErrSType context"
 | aGsProcess |
 aGsProcess := (Object _objectForOop: contextOop) ifNil:[ ^ ' < NO PROCESS FOUND > '].
 ^ [ aGsProcess stackReportToLevel: 300 withArgsAndTemps: true andMethods: false
   ] on: Error do:[:ex | 'ERROR during stack report ', ex asString ].
]

{ #category : 'Instance Creation' }
AbstractExternalSession class >> gemNRS: gemNRS stoneNRS: stoneNRS username: aUsername password: aPassword [
  ^ self subclassResponsibility: #gemNRS:stoneNRS:username:password:
]

{ #category : 'Instance Creation' }
AbstractExternalSession class >> gemNRS: gemNRS stoneNRS: stoneNRS username: gsUsername password: gsPassword hostUsername: hostUsername hostPassword: hostPassword [

  ^ self subclassResponsibility: #gemNRS:stoneNRS:username:password:hostUsername:hostPassword:
]

{ #category : 'Instance Creation' }
AbstractExternalSession class >> new [
  "disallowed"
  self shouldNotImplement: #new
]

{ #category : 'Instance Creation' }
AbstractExternalSession class >> newDefault [
  ^ self subclassResponsibility: #newDefault 
]

{ #category : 'Private' }
AbstractExternalSession >> _stringForBlock: aBlock [
  | string |
  string := aBlock method _sourceStringForBlock .
  (string at: 1) == $[  ifFalse:[ Error signal:'malformed source'].
  "replace [ ]  with spaces"
  string at: 1 put: $  ; at: string size put: $  .
  ^ string
]

{ #category : 'Private' }
AbstractExternalSession >> _stringForBlock: aBlock withArguments: someValues [
  | str nArgs |
  aBlock numArgs == (nArgs := someValues size) ifFalse: [self error: 'Wrong number of arguments'].
  str := aBlock method _sourceStringForBlock .
  nArgs == 0 ifTrue:[
    str addAll: ' value '.
  ] ifFalse:[ | sep stream |
    stream := AppendStream on: String new .
    sep := '' .
    stream nextPutAll: str ; nextPutAll: ' valueWithArguments: {'.
    1 to: nArgs do: [:index | | each |
      each := someValues at: index .
      stream nextPutAll: sep .
      each printOn: stream.
      sep := ' . '.
    ].
    stream nextPut: $}  .
    str := stream contents.
  ].
  (str at: 1) == $[  ifFalse:[ Error signal:'malformed source'].
  ^ str
]

{ #category : 'Public' }
AbstractExternalSession >> abort [
    ^ self subclassResponsibility: #abort
]

{ #category : 'Public' }
AbstractExternalSession >> begin [
    ^ self subclassResponsibility: #begin
]

{ #category : 'Error handling' }
AbstractExternalSession >> clearStackFor: aGciError [
    ^ self subclassResponsibility: #clearStackFor:
]

{ #category : 'Public' }
AbstractExternalSession >> commit [
    ^ self subclassResponsibility: #commit
]

{ #category : 'Public' }
AbstractExternalSession >> commitOrError [
    ^ self subclassResponsibility: #commitOrError
]

{ #category : 'Error handling' }
AbstractExternalSession >> continue: contextOop [
  ^ self continue: contextOop replacingTopOfStackWithOop: 1"OOP_ILLEGAL"
]

{ #category : 'Error handling' }
AbstractExternalSession >> continue: contextOop replacingTopOfStackWithOop: anOop [
    ^ self subclassResponsibility: #continue:replacingTopOfStackWithOop:
]

{ #category : 'Error handling' }
AbstractExternalSession >> continue: contextOop with: anObject [
  ^ self continue: contextOop replacingTopOfStackWithOop: anObject asOop
]

{ #category : 'Calls' }
AbstractExternalSession >> executeBlock: aBlock [
  "Execute the code in the Block argument in the external Gem
   and answer the result. The best values to return are specials.
   Strings and other byte objects are copied to the local Gem.
   All other responses are returned as a Array of one or more OOPs, 
   see resolveResult:."

   ^self executeString: (self _stringForBlock: aBlock)
]

{ #category : 'Calls' }
AbstractExternalSession >> executeBlock: aBlock with: aValue [
  "Execute the code in the Block argument in the external Gem, passing
   in the specified value, and answer the result. The value passed to the
   Block must be one whose printString allows the correct object state to
   be recreated (such as numbers and strings, for example).
   The best values to return are specials.
   Strings and other byte objects are copied to the local Gem.
   All other responses are returned as a Array of one or more OOPs, 
   see resolveResult:."

^ self executeBlock: aBlock withArguments: {aValue}.
]

{ #category : 'Calls' }
AbstractExternalSession >> executeBlock: aBlock with: vOne with: vTwo [
  ^ self executeBlock: aBlock withArguments:{ vOne . vTwo }

]

{ #category : 'Calls' }
AbstractExternalSession >> executeBlock: aBlock withArguments: someValues [
  "Execute the code in the Block argument in the external Gem, passing
   in the specified values, and answer the result. The values passed to the
   Block must be ones whose printString allows the correct object state to
   be recreated (such as numbers and strings, for example).
   The best values to return are specials.
   Strings and other byte objects are copied to the local Gem.
   All other responses are returned as a Array of one or more OOPs, 
   see resolveResult:."

  ^self executeString: (self _stringForBlock: aBlock withArguments: someValues)
]

{ #category : 'Public' }
AbstractExternalSession >> executeString: aString [
    ^ self subclassResponsibility: #executeString:
]

{ #category : 'Public' }
AbstractExternalSession >> forceLogout [
    ^ self subclassResponsibility: #forceLogout
]

{ #category : 'Calls' }
AbstractExternalSession >> forkBlock: aBlock [

^ self forkString:(self _stringForBlock: aBlock )

]

{ #category : 'Calls' }
AbstractExternalSession >> forkBlock: aBlock with: aValue [

^   self forkBlock: aBlock withArguments: {aValue }

]

{ #category : 'Calls' }
AbstractExternalSession >> forkBlock: aBlock with: aValue with: anotherValue [

^   self forkBlock: aBlock withArguments: {aValue. anotherValue}.

]

{ #category : 'Calls' }
AbstractExternalSession >> forkBlock: aBlock withArguments: someValues [
  "Execute the code in the Block argument in the external Gem, passing
   in the specified values, and do not wait for a result. The values passed
   to the Block must be ones whose printString allows the correct object
   state to be recreated (such as numbers and strings, for example).
   At some later point, you would check for a result. Otherwise you cannot
   issue another call, as the current call would remain in progress.
   Refer to #executeString: for an example of the complete send, wait, response sequence."

  self forkString: (self _stringForBlock: aBlock withArguments: someValues)
]

{ #category : 'Public' }
AbstractExternalSession >> forkString: aString [
    ^ self subclassResponsibility: #forkString:
]

{ #category : 'Public' }
AbstractExternalSession >> gemProcessId [
    ^ self subclassResponsibility: #gemProcessId
]

{ #category : 'Public' }
AbstractExternalSession >> hardBreak [
    ^ self subclassResponsibility: #hardBreak
]

{ #category : 'Public' }
AbstractExternalSession >> isCallInProgress [
    ^ self subclassResponsibility: #isCallInProgress
]

{ #category : 'Public' }
AbstractExternalSession >> isLoggedIn [
    ^ self subclassResponsibility: #isLoggedIn
]

{ #category : 'Public' }
AbstractExternalSession >> isRemoteServerBigEndian [
    ^ self subclassResponsibility: #isRemoteServerBigEndian
]

{ #category : 'Public' }
AbstractExternalSession >> isResultAvailable [
    ^ self subclassResponsibility: #isResultAvailable
]

{ #category : 'Public' }
AbstractExternalSession >> lastResult [
    ^ self subclassResponsibility: #lastResult
]

{ #category : 'Public' }
AbstractExternalSession >> login [
    ^ self subclassResponsibility: #login
]

{ #category : 'Public' }
AbstractExternalSession >> loginSolo [
    ^ self subclassResponsibility: #loginSolo
]

{ #category : 'Public' }
AbstractExternalSession >> logout [
    ^ self subclassResponsibility: #logout
]

{ #category : 'Public' }
AbstractExternalSession >> nbLogout [
    ^ self subclassResponsibility: #nbLogout
]

{ #category : 'Public' }
AbstractExternalSession >> nbResult [
    ^ self subclassResponsibility: #nbResult
]

{ #category : 'Public' }
AbstractExternalSession >> parameters [
    ^ self subclassResponsibility: #parameters
]

{ #category : 'Public' }
AbstractExternalSession >> resolveResult: anOop [
    ^ self subclassResponsibility: #resolveResult:
]

{ #category : 'Public' }
AbstractExternalSession >> resolveResult: anOop toLevel: anInteger [
    ^ self subclassResponsibility: #resolveResult:toLevel:
]

{ #category : 'Public' }
AbstractExternalSession >> send: selector to: anOop [
  "Answer the result of having the specified remote object
   sent the message with the specified selector.
   The best values to return are specials.
   Strings and other byte objects are copied to the local Gem.
   All other responses are returned as a Array of one or more OOPs, 
   see resolveResult:."

  ^self send: selector to: anOop withArguments: nil
]

{ #category : 'Public' }
AbstractExternalSession >> send: aSelector to: anObject withArguments: anArray [
    ^ self subclassResponsibility: #send:to:withArguments:
]

{ #category : 'Public' }
AbstractExternalSession >> softBreak [
    ^ self subclassResponsibility: #softBreak
]

{ #category : 'Public' }
AbstractExternalSession >> stoneSessionId [
    ^ self subclassResponsibility: #stoneSessionId
]

{ #category : 'Public' }
AbstractExternalSession >> waitForReadReady [
    ^ self subclassResponsibility: #waitForReadReady
]

{ #category : 'Public' }
AbstractExternalSession >> waitForResult [
    ^ self subclassResponsibility: #waitForResult
]
