"
An RcPipe is similar to an GsPipe except that it has been extended to allow
concurrent additions to the pipe without generating a concurrency conflict.

The RcPipe is a very efficient for situations in which concurrent additions 
occur, but are not likely for every commit.  If it is known that almost every
addition is concurrent with some other additions, the RcQueue is likely 
a more efficient implementation.

Multiple concurrent producers (sessions adding to the RcPipe) will NOT experience
concurrency conflicts because as part of every add operation, they record the add
in a redo log in the session state.  If a transation containing a RcPipe object 
detects a conflict in its commit, it breaks serialization and then replays the 
addition before competing the commit.  The replays are serialized so that if 
there are multiple sessions performing additions, they will all complete 
sucessfully in the order in which they are serialized.  Because the RcPipe must 
perform the logging and replay operations, when there are lots of concurrent updaters, 
it may be more effient to use an RcQueue.  If the probability of concurrent updates
is relatively small, it can be much more efficient because it does not use as
much space and does not require cleanup like the RcQueue does.

Values in the RcPipe are always removed in the order that they were committed.

"
Class {
	#name : 'RcPipe',
	#superclass : 'GsPipe',
	#category : 'Collections-IOOrdered'
}

{ #category : 'Instance Creation' }
RcPipe class >> new [

"Returns a new RcPipe."

^super new

]

{ #category : 'Reduced Conflict Support' }
RcPipe >> _abortAndReplay: conflictObjects [

"Abort the receiver and replay operations on the receiver from the redo log."

| redoLog logEntries |
redoLog := System _redoLog.

" if no log entries to replay, then we're done "
redoLog == nil ifTrue: [ ^ false ].
logEntries := redoLog getLogEntriesFor: self .
logEntries == nil ifTrue:[ ^ false ].

" cannot perform selective abort if receiver has a dependency tag "
self _hasDependencyList ifTrue: [ ^ false ].

" refresh the state of the receiver and the tail.
  these should only be done once!"
self _selectiveAbort.
self tail _selectiveAbort.


" tell the redo log to replay any operations on the receiver "
^ redoLog _redoOperationsForEntries: logEntries

]

{ #category : 'Private' }
RcPipe >> _redoAdd: anObject [

"Performs the replay of adding anObject to the RcPipe and returns true."
self add: anObject.
^ true

]

{ #category : 'Adding' }
RcPipe >> add: aValue [

"Adds aValue to the RcPipe and returns aValue."

| redo logEntry systm |
redo := (systm := System) redoLog.
logEntry := LogEntry new.
logEntry receiver: self selector: #_redoAdd: argArray: { aValue }.
redo addLogEntry: logEntry.
redo addConflictObject: tail for: self.
systm _addToRcReadSet: self includingAllNodes: false.
systm _addToRcReadSet: head includingAllNodes: false.
systm _addToRcReadSet: tail includingAllNodes: false.
systm _addToRcReadSet: tail next  includingAllNodes: false.
^ super add: aValue

]
