"
An RcIdentitySet is a special kind of IdentitySet that provides for handling
 modifications to an individual instance by multiple concurrent sessions.  
 When concurrent sessions update an RcIdentitySet and an rc conflict is 
 detected during commit the conflict is resolved by replaying the operation.

 There is no commit conflict with multiple sessions adding and a single session 
 removing objects from the RcIdentitySet.

 In an application with high rates of concurrency, the conflict resolution must
 be serialized, which may create delays.  In this case an RcQueue may be more 
 efficient.

"
Class {
	#name : 'RcIdentitySet',
	#superclass : 'IdentitySet',
	#category : 'Collections-Unordered'
}

{ #category : 'Instance Creation' }
RcIdentitySet class >> new [

"Returns a new RcIdentitySet."

^super new

]

{ #category : 'Reduced Conflict Support' }
RcIdentitySet >> _abortAndReplay: conflictObjects [

"Abort the receiver and replay operations on the receiver from the redo log."

| redoLog logEntries |
_indexedPaths
  ifNotNil: [
    _indexedPaths _anyIndexIsLegacy
      ifTrue: [
        "abort and replay not supported when legacy index is involved"
        ^false ] ].

redoLog := System _redoLog.

" if no log entries to replay, then we're done "
redoLog == nil ifTrue: [ ^ false ].
logEntries := redoLog getLogEntriesFor: self .
logEntries == nil ifTrue:[ ^ false ].

" cannot perform selective abort if receiver has a dependency tag "
self _hasDependencyList ifTrue: [ ^ false ].

" Refresh the state of the receiver."

self _selectiveAbort.

" tell the redo log to replay any operations on the receiver "
^ redoLog _redoOperationsForEntries: logEntries

]

{ #category : 'Private' }
RcIdentitySet >> _redoAdd: anObject [

"Performs the replay of adding anObject to the RcIdentitySet and returns true."

self _addForReplay: anObject.
^ true

]

{ #category : 'Private' }
RcIdentitySet >> _redoAddAll: aCollection [
  "Performs the replay of adding aCollection to the receiver and returns true."
 | coll |
 System _addToRcReadSet: self includingAllNodes: true.
 coll := aCollection _isRcIdentityBag ifTrue:[ aCollection _asIdentityBag ] ifFalse:[ aCollection ].
 self _addAll: coll _asIdentityBag forReplay: true .
 ^ true

]

{ #category : 'Private' }
RcIdentitySet >> _redoRemove: anObject [

"Performs the replay of removing anObject from the receiver and returns true."

^ self _removeForReplay: anObject.

]

{ #category : 'Private' }
RcIdentitySet >> _redoRemoveAll: aCollection [

 "Performs the replay of removing aCollection elements from the receiver and returns true."
 | coll |
 System _addToRcReadSet: self includingAllNodes: true.
 coll := aCollection _isRcIdentityBag ifTrue:[ aCollection _asIdentityBag ] ifFalse:[ aCollection ].
 ^ self _removeAll: coll _asIdentityBag errIfAbsent: true forReplay: true 
      rc: true"handle leaf nodes created by another session"

]

{ #category : 'Private' }
RcIdentitySet >> _redoRemoveAllPresent: aCollection [

 "Performs the replay of removing the elements in aCollection from the receiver and returns true."
 | coll |
 System _addToRcReadSet: self includingAllNodes: true.
 coll := aCollection _isRcIdentityBag ifTrue:[ aCollection _asIdentityBag ] ifFalse:[ aCollection ].
 ^ self _removeAll: aCollection _asIdentityBag errIfAbsent: false forReplay: true 
      rc: true"handle leaf nodes created by another session"
]

{ #category : 'Reduced Conflict Support' }
RcIdentitySet >> _validateLegacyRcIndexSupport [
  "legacy indexes are not supported on RcIndentitySet or RcLowMaintenanceIdentityBag (bug47179)"

  self error: 'Creating a legacy index on an RcIdentitySet or RcLowMaintenanceBag is not supported.'

]

{ #category : 'Adding' }
RcIdentitySet >> add: anObject [

"Adds anObject to the RcIdentitySet and returns anObject.
 Nils are ignored, i.e., not added to the set."

anObject == nil ifTrue:[ ^ anObject "ignore nils" ].
(self includes: anObject) ifFalse:[
  self addRedoLogEntryFor: #_redoAdd:  withArgs: { anObject } .
  self _rcAdd: anObject withOccurrences: 1 .
].
^ anObject

]

{ #category : 'Adding' }
RcIdentitySet >> addAll: aCollection [

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

self addRedoLogEntryFor: #_redoAddAll:  withArgs: { aCollection copy } .
System _addToRcReadSet: self includingAllNodes: true.

^ super addAll: aCollection

]

{ #category : 'Private' }
RcIdentitySet >> addRedoLogEntryFor: aSelector withArgs: args [

"Creates a redo log entry for the selector with the specified argument array,
 adds it to the redolog for this session.
 Sender responsible for adding to rcReadSet .  "

| redo logEntry |

redo := System redoLog.
logEntry := LogEntry new.
logEntry receiver: self selector: aSelector argArray: args.
redo addLogEntry: logEntry.
redo addConflictObject: self for: self.

]

{ #category : 'Converting' }
RcIdentitySet >> asIdentitySet [

^ IdentitySet withAll: self

]

{ #category : 'Removing' }
RcIdentitySet >> remove: anObject [

"Removes anObject if present from the receiver.
 Generates an error if anObject is not in the receiver.  Returns anObject.
"

anObject ifNotNil:[
  (self includes: anObject) ifTrue:[
    self addRedoLogEntryFor: #_redoRemove:  withArgs: { anObject } .
    ^ self _rcRemove: anObject .
  ] ifFalse:[
    ^ self _errorNotFound: anObject
  ]
].
^ anObject

]

{ #category : 'Removing' }
RcIdentitySet >> remove: anObject ifAbsent: aBlock [

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  If anObject is not present in the receiver,
 evaluates aBlock and returns the result of the evaluation."

(self includes: anObject) ifTrue:[
  self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
  ^ self _rcRemove: anObject
]
ifFalse: [ ^ aBlock value ]

]

{ #category : 'Removing' }
RcIdentitySet >> remove: anObject otherwise: notFoundValue [

"Removes from the receiver an object that is identical to anObject and returns
 anObject.  If anObject is not present in the receiver, returns notFoundValue."

(self includes: anObject) ifTrue:[
  self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
  ^ self _rcRemove: anObject
]
ifFalse: [ ^ notFoundValue ]

]

{ #category : 'Removing' }
RcIdentitySet >> removeAll: aCollection [
  "Removes all of the elements of aCollection from the receiver and returns aCollection.
 Generates an error if any object in the collection is not in the receiver.
"

  self addRedoLogEntryFor: #_redoRemoveAll: withArgs: {aCollection copy}.
  System _addToRcReadSet: self includingAllNodes: true.
  ^ super removeAll: aCollection

]

{ #category : 'Removing' }
RcIdentitySet >> removeAllPresent: aCollection [

"Removes from the receiver each element of aCollection that is also an
 element of the receiver.  Differs from removeAll: in that, if some
 elements of aCollection are not present in the receiver, no error is generated.
 Returns aCollection."

self addRedoLogEntryFor: #_redoRemoveAllPresent: withArgs: { aCollection } .
System _addToRcReadSet: self includingAllNodes: true.

^super removeAllPresent: aCollection

]

{ #category : 'Removing' }
RcIdentitySet >> removeIdentical: anObject [

"Same as remove:."

^self remove: anObject

]

{ #category : 'Removing' }
RcIdentitySet >> removeIdentical: anObject ifAbsent: aBlock [

"Same as remove:ifAbsent:."

^self remove: anObject ifAbsent: aBlock

]

{ #category : 'Removing' }
RcIdentitySet >> removeIdentical: anObject otherwise: notFoundValue [

"Same as remove:otherwise:."

^ self remove: anObject otherwise: notFoundValue

]

{ #category : 'Removing' }
RcIdentitySet >> removeIfPresent: anObject [

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  Returns nil if anObject is not present in the receiver."

(self includes: anObject) ifTrue:[
  self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
  ^ self _rcRemove: anObject
]
ifFalse: [ ^ nil ]

]

{ #category : 'Class Membership' }
RcIdentitySet >> species [

"Returns the class to use to select and reject queries."

^ IdentitySet

]
