"
PrintStream is designed for use in place of WriteStream to handle printing
of mixed string types. The default size of the stream is limited
to 100000 bytes.

Constraints:
	itsCollection: SequenceableCollection
	maxSize: Object
	approxPosition: Object
"
Class {
	#name : 'PrintStream',
	#superclass : 'Stream',
	#instVars : [
		'itsCollection',
		'maxSize',
		'approxPosition'
	],
	#gs_reservedoop : '108033',
	#category : 'Collections-Streams'
}

{ #category : 'Instance Creation' }
PrintStream class >> on: aCollection [

"Returns an instance of the receiver that can stream over the elements of
 aCollection.  The maximum size of the receiver is set to 536870911."

| newStream |

newStream := self _basicNew.
newStream _initStreamWith: aCollection maxSize: 536870911 .
^ newStream

]

{ #category : 'Instance Creation' }
PrintStream class >> printingOn: aCollection [

"Returns an instance of the receiver that can stream over the elements of
 aCollection. The maximum size of the receiver is set to 100000."

^ self printingOn: aCollection maxSize: 100000

]

{ #category : 'Instance Creation' }
PrintStream class >> printingOn: aCollection maxSize: n [


| newStream |

newStream := self _basicNew.
newStream _initStreamWith: aCollection maxSize: n .
^ newStream

]

{ #category : 'Accessing' }
PrintStream >> _collection [

"Returns the collection of the receiver."

^itsCollection

]

{ #category : 'Converting' }
PrintStream >> _convertToISOLatin [

"Converts the receiver's collection from String to ISOLatin.
 Has no effect if the receiver's collection is already an ISOLatin, or
 if the receiver's collection is something other than a String, such as
 a GsFile or a JISString."

| oldStr |

itsCollection class == String ifTrue:[
  oldStr := itsCollection .
  itsCollection := ISOLatin withAll: oldStr  .
  ]

]

{ #category : 'Positioning' }
PrintStream >> _initStreamWith: aCollection maxSize: aMaxSize [

"Initializes the receiver's 'itsCollection' instance variable to be
 aCollection."

itsCollection := aCollection.
maxSize := aMaxSize .
approxPosition := 0 .

]

{ #category : 'Adding' }
PrintStream >> _nextPut: anObject [

"Add anObject to the stream, regardless of maxSize."

 itsCollection addLast: anObject .
 ^ anObject 

]

{ #category : 'Debug' }
PrintStream >> _overflow [
  ^ self
]

{ #category : 'Testing' }
PrintStream >> atEnd [
  ^ self shouldNotImplement: #atEnd 
]

{ #category : 'Accessing' }
PrintStream >> contents [

"Returns the Collection associated with the receiver (that is,
 the sequence of objects that the receiver may access)."

^ itsCollection

]

{ #category : 'Testing' }
PrintStream >> isEmpty [

"Returns true if the collection that the receiver accesses contains
 no elements; otherwise returns false."

^ itsCollection size == 0

]

{ #category : 'Testing' }
PrintStream >> isFull [

"Returns true if the size of the collection that the receiver accesses
 is equal or greater than maxSize."

^ approxPosition >= maxSize .

]

{ #category : 'Accessing' }
PrintStream >> maxSize [

"Returns the maxSize of the receiver."

^ maxSize

]

{ #category : 'Accessing' }
PrintStream >> next [
  ^ self shouldNotImplement: #next
]

{ #category : 'Adding' }
PrintStream >> nextPut: anObject [

"If there's room, add anObject to the stream.
 If adding anObject would exceed maxSize, trigger overflow behavior.

 Returns anObject.
"
| ap maxSz |
ap := approxPosition .
ap < (maxSz := maxSize) ifTrue:[
  itsCollection addLast: anObject .
  ap := ap + 1 .
  approxPosition := ap .
  (ap >= maxSz) ifTrue: [ self overflow ]
].
^ anObject

]

{ #category : 'Adding' }
PrintStream >> nextPutAll: aCollection [

"If there's room, add the elements of aCollection to the stream.
 If adding aCollection would exceed maxSize, add part of aCollection
 to stream up to maxSize and then trigger overflow behavior.
 Use #_basicSize to estimate size of aCollection, since #size can be
 expensive for certain classes.

 Returns aCollection.  "
| ap argSiz maxSz numLeft |
ap := approxPosition.
maxSz := maxSize .
numLeft := maxSz - ap .
numLeft > 0 ifTrue:[
  argSiz := aCollection size .
  (argSiz <= numLeft or:[ argSiz <= 2]) ifTrue:[
    itsCollection addAll: aCollection  .
    ap := ap + argSiz .
    approxPosition := ap .
  ] ifFalse:[ | coll sz |
    coll := itsCollection .   sz := coll size .
    coll replaceFrom: sz + 1 to: sz + numLeft with: aCollection startingAt: 1 .
    ap := ap + numLeft .
    approxPosition := ap . 
  ].
  ap >= maxSz ifTrue:[ self overflow  ].
].
^ aCollection

]

{ #category : 'Adding' }
PrintStream >> nextPutAllBytes: aCharacterCollection [
  ^ self shouldNotImplement: #nextPutAllBytes: 
]

{ #category : 'Positioning' }
PrintStream >> overflow [

"Do whatever is appropriate when we try to exceed maxSize.
 For a CharacterCollection, add ellipses.
 For anything else, trigger #rtErrBadStreamPosition.
"
| coll |
coll := itsCollection .
self _overflow .
(coll isKindOf: CharacterCollection)
  ifTrue: [ coll addAll: '...' ]
  ifFalse: [self _error: #rtErrBadStreamPosition args: { approxPosition }]

]

{ #category : 'Positioning' }
PrintStream >> position [

"Returns the receiver's approximate position reference for accessing the
 sequence of objects. "

^ itsCollection _basicSize

]

{ #category : 'Adding' }
PrintStream >> print: anObject [

	anObject printOn: self.

]
