Extension { #name : 'LargeInteger' }

{ #category : 'Private' }
LargeInteger class >> _initializeMaximumValues [
"Returns the maximum allowable LargeInteger value."
| nd v f df |
nd := MaximumDigits"classVar installed by bom.c" .
v := 1 bitShift: (nd * 32) - 1 .
v := (v - 1) bitOr: v .
v _digitLength = nd  ifFalse:[ self error:'incorrect size'].
(v bitXor: v) = 0 ifFalse:[ self error:'incorrect value'].
f := Float fmax asInteger. "largest Float < PlusInfinity"
df := 9.9999999999999999999f15000 asInteger. "largest DecimalFloat < Infinity"
LargeInteger _addInvariantClassVar: #MaximumValue value: v immediateInvariant .
LargeInteger _addInvariantClassVar: #MaximumFloat value: f asInteger immediateInvariant .
LargeInteger _addInvariantClassVar: #MaximumDecimalFloat value: df asInteger immediateInvariant .

Float _addInvariantClassVar: #MaximumFloat value: f immediateInvariant .
DecimalFloat _addInvariantClassVar: #MaximumFloat value: df immediateInvariant .
^ v

]

{ #category : 'Instance Creation' }
LargeInteger class >> basicNew [
  "Returns a instance of LargeInteger with specified size and value = 0"
  ^ self basicNew: 0

]

{ #category : 'Instance Creation' }
LargeInteger class >> basicNew: aSize [
  "Returns a instance of LargeInteger with specified size and value = 0"

^ self _new: aSize

]

{ #category : 'Storing and Loading' }
LargeInteger class >> loadFrom: passiveObject [

"Reads from passiveObject and convert to active. This method creates
 subnormal (LargeIntegers in the SmallInteger range) if necessary."

| value instance |
value := passiveObject readInt.
instance := value _isSmallInteger
   ifTrue: [self _newWithValue: value]
   ifFalse: [value].
passiveObject hasRead: instance.
^instance

]

{ #category : 'Queries' }
LargeInteger class >> maximumDecimalFloat [
 ^ MaximumDecimalFloat .

]

{ #category : 'Queries' }
LargeInteger class >> maximumFloat [
 ^ MaximumFloat .

]

{ #category : 'Queries' }
LargeInteger class >> maximumValue [
"Returns the maximum allowable LargeInteger value."

^ MaximumValue "computed during filein"

]

{ #category : 'Private' }
LargeInteger >> __recursiveSize: arr [
  ^ self . "do not iterate over the digits"
  
]

{ #category : 'Updating' }
LargeInteger >> _digitAt: anIndex put: aValue [

"Store aValue into the receiver's base 4294967296 digit specified by anIndex .
 indicated by index.  Fail if the value is negative or is larger than 4294967295.
 Fail if the index is not an Integer or is out of bounds.  Returns the value
 that was stored."

<primitive: 257>

anIndex _validateClass: SmallInteger.
aValue _validateClass: SmallInteger.
self _primitiveFailed: #_digitAt:Put: args: { anIndex . aValue } .
self _uncontinuableError

]

{ #category : 'Arithmetic' }
LargeInteger >> abs [

"Returns a LargeInteger that is the absolute value of the receiver."

| sign |
sign := self _digitAt:0 .
sign == 0 ifTrue:[ ^ self ].
^ 0 - self

]

{ #category : 'Converting' }
LargeInteger >> asCanonicalForm [
	"Answer self, or, if I am a LargeInteger with an equivalent
	SmallInteger, answer that SmallInteger."

	| res |
	res := self + 0.
	res class == SmallInteger
		ifTrue: [ ^ res ].
	^ self
]

{ #category : 'Converting' }
LargeInteger >> asInteger [

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #asInteger .
self _uncontinuableError

]

{ #category : 'Accessing' }
LargeInteger >> at: anIndex [

"Disallowed."

self shouldNotImplement: #at:

]

{ #category : 'Accessing' }
LargeInteger >> at: anIndex put: aNumber [

"Disallowed.  You may not change the value of a LargeInteger."

self shouldNotImplement: #at:put:

]

{ #category : 'Truncation and Rounding' }
LargeInteger >> ceiling [

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #ceiling .
self _uncontinuableError

]

{ #category : 'Truncation and Rounding' }
LargeInteger >> floor [

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #floor .
self _uncontinuableError

]

{ #category : 'Bit Manipulation' }
LargeInteger >> highBit [

"Returns the index of the high-order bit that is set
 in the binary representation of the receiver.  (If the receiver is a negative
 integer, takes its absolute value first.)
 As of v3.7, if the receiver is zero, this returns zero; in previous releases 
 this returned nil.
 As of Gs64 v3.2  the least significant bit has index 1 ,
 in previous versions the least significant bit had index 0 .

 Example:   4 highBit == 3  "

| len |
len := self _digitLength .
^ (self _digitAt: len) highBit + (32 * (len - 1))

]

{ #category : 'Testing' }
LargeInteger >> negative [

^ (self _digitAt:0) ~~ 0

]

{ #category : 'Testing' }
LargeInteger >> positive [

^ (self _digitAt:0) == 0

]

{ #category : 'Copying' }
LargeInteger >> postCopy [
  ^ self immediateInvariant

]

{ #category : 'Truncation and Rounding' }
LargeInteger >> rounded [

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #rounded .
self _uncontinuableError

]

{ #category : 'Accessing' }
LargeInteger >> sign [
  self = 0 ifTrue:[ ^ 0 ].
  ^ (self _digitAt: 0) ~~ 0 ifTrue:[ -1 ] ifFalse:[ 1]

]

{ #category : 'Truncation and Rounding' }
LargeInteger >> truncated [

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #truncated .
self _uncontinuableError

]
