"
 InstVarMappingArray is used in instance migration to control migration of named and dynamic 
 instance variables.  

 It identifies the instance variables by using the integer offset of the name of the instance 
 variable in the old and new classes; 
 the offset in the new class is the index into the InstVarMappingArray, and the offset in the 
 old class is the value at that index.
 A value of 0 means no corresponding value from the old instance will be migrated to the new 
 class; the migrated instance will have nil at that instance variable offset. 

 For example, the mapping of OrigClass (instvars aa bb cc) to NewClass (instcars aa cc dd), using 
   InstVarMappingArray mappingFrom: OldClass to: NewClass.
 would be 1, 3, 0.  

 In most cases for migration you do not need to be concerned about the InstVarMappingArray.  
 By implementing migrateFrom:instVarMap: on your new instance's class (and invoking super 
 migrateFrom:instVarMap:) you can write code to perform the appropriate migrate changes.  

 This class is used directly for multi-threaded migration using Repository>>mtMigrate:.  

 instance variables
   oldCl              - the old Class
   newCl              - the new Class  
   dynamicIvsToNamed  - behavior not yet implemented
   namedIvsToDynamic  - behavior not yet implemented
   preserveDynamic    aBoolean

 varying instanceVariables 
   offset N: a SmallInteger, offset of instVar in old object that will correspond to instVarAt: N in new object
"
Class {
	#name : 'InstVarMappingArray',
	#superclass : 'Array',
	#instVars : [
		'oldCl',
		'newCl',
		'dynamicIvsToNamed',
		'namedIvsToDynamic',
		'preserveDynamic'
	],
	#category : 'Kernel-Classes'
}

{ #category : 'Instance Creation' }
InstVarMappingArray class >> mappingFrom: oldClass to: newClass [
 "Create a default instance. This default preserves dynamic instVars on instances of oldClass.
  Named instVars in oldClass that do not exist in newClass are not preserved.

  Instances of InstVarMappingArray are not explicitly created for ordinary migration. 
  Multi-threaded migrate (Repository mtMigrate: does require creating and optionally 
  customizing an instance of InstVarMappingArray, using preserveDynamic:, mapInstVarNamed:to:, and 
  mapInstVarToNil:."

^ self new initializeFrom: oldClass to: newClass

]

{ #category : 'Other' }
InstVarMappingArray >> convertForMigrateMt [
" Generates an array containing the information in the InstVarMappingArray for passing to migrateMt:"

  | arr |
  (oldCl isSubclassOf: IdentitySet) ifTrue: [
       ArgumentError signal: 'multithreaded migration not supported for IdentitySets'
    ].
  (oldCl isSubclassOf: Collection) ifTrue: [ 
       ArgumentError signal: 'multithreaded migration not supported for Collections'
    ].
  arr := Array with: oldCl with: newCl with: preserveDynamic with: (oldCl instSize) with: (newCl instSize).
  arr addAll: self.
  ^arr
]

{ #category : 'Initialization' }
InstVarMappingArray >> initializeFrom: oldClass to: newClass [

^ self initializeFrom: oldClass to: newClass preserveDynamic: true

]

{ #category : 'Initialization' }
InstVarMappingArray >> initializeFrom: oldClass to: newClass preserveDynamic: aBoolean [
  "Initializes the receiver. 
   If aBoolean==true, dynamic instance variables are preserved in the
   new instance, if false, dynamic instance variables are not preserved. "

  oldCl := oldClass .
  newCl := newClass .
  preserveDynamic := aBoolean .
  self addAll: (newClass instVarMappingTo: oldClass) .

]

{ #category : 'Updating' }
InstVarMappingArray >> mapInstVarNamed: oldName to: newName [
  "The value at the instance variable named <oldName> in the old class should be migrated
  to the instance variable named <newName> in the new class."

  | o n |
  o := oldCl allInstVarNames indexOfIdentical: oldName.
  n := newCl allInstVarNames indexOfIdentical: newName.
  self at: n put: o.
]

{ #category : 'Updating' }
InstVarMappingArray >> mapInstVarToNil: newName [
  "The migration should not retain the value of any instance variable <newName> present 
  in the instance of the old class. After migration; the object will have a nil value 
  at this instance variable."

  | i |
  i := newCl allInstVarNames indexOfIdentical: newName.
  self at: i put: 0.
]

{ #category : 'Accessing' }
InstVarMappingArray >> newClass [
  ^ newCl

]

{ #category : 'Accessing' }
InstVarMappingArray >> oldClass [
  ^ oldCl

]

{ #category : 'Accessing' }
InstVarMappingArray >> preserveDynamic [
  ^ preserveDynamic

]

{ #category : 'Updating' }
InstVarMappingArray >> preserveDynamic: aBoolean [
  preserveDynamic := aBoolean

]
