"
GsHostProcess represents a forked child process whose stdin, stdout, and
stderr may be accessable via instances of GsSocket.

Lookup in the PATH environment variable is not performed. The argument
to fork:  must specify a complete path to an executable or script.
Instances may not be committed.
The argument to execute: or fork:  does not undergo any shell 
expansion of characters such as '*' or '?' .  

The GsSocket instance variables in, out, and err either hold GsSockets
representing  the parent process's ends of the pipes, or those
instance variables specify paths of files to be opened.
If a child process should not access one of those files,  it should
be given a path of '/dev/null' .  For example a child process that should
not read anything from stdin should be sent   stdinPath: '/dev/null' 
before forking or executing it.  If child stdout should be ignored , then
stdoutPath: '/dev/zero'  may be appropriate.  
See detailed comments in GsHostProcess >> fork.

The instance variable errSocket represents parent end of a pipe
receiving stderr data in the event a stderr file or stdout file cannot be opened.

For pipes, the forked child process has the other end of the pipe.  
The child accepts input on the stdin and puts output on  stdout and stderr.  
For the parent end, stdin is write-only and  stdout and stderr are read only.

When an instance is garbage collected, and the child has not yet been
reaped,  the child process represented by that instance is killed with
kill -TERM, and waitpid is called to reap the child.

For stdout or stderr pipes of a GsHostProcess, if the child writes data while
it is running then the corresponding GsSocket can read that data. Use
GsSocket>>readWillNotBlock or GsSocket>>readWillNotBlockWithin:
to determine whether data is ready to read.

GsSocket>>read:into:startingAt: will yield to other GsProcess(s) as for
reading from a GsSocket represending an AF_INET or AF_INET6 socket .
If the child has exited and there is no more data to read, this read
method will wait forever.

When GsHostProcess>>status returns non-nil, then any data not yet read
from stdout/stderr pipe is still available to read. The parent end of the
pipe is not closed until the corresponding GsSocket is garbage collected
or explicitly sent GsSocket>>close

If the child tries to read from a stdin pipe it will block until data is
written by the parent to the stdin of the GsHostProcess. If the child 
tries to write a lot of data to stdout or stderr pipe, the child may block
until the parent reads enough data from the respective GsSocket to 
unblock the pipe. 

For example,
  GsHostProcess fork: '$GEMSTONE/bin/topaz -r'
will fork a topaz process which will do blocking reads on stdin and
execute each command that is written to the stdin of the GsHostProcess 
instance.

Dynamic instVars
  args   -  Array of argument strings, if non-nil they are appended to
            any space separated args within cmd .

"
Class {
	#name : 'GsHostProcess',
	#superclass : 'Object',
	#instVars : [
		'cmd',
		'in',
		'out',
		'err',
		'childPid',
		'childStatus',
		'appendToFiles',
		'errSocket'
	],
	#gs_options : [
		'instancesNonPersistent'
	],
	#gs_reservedoop : '160769',
	#category : 'OSAccess'
}

{ #category : 'Private' }
GsHostProcess class >> _basicNew [

"creates an instance registered with VM for finalization of cData"

<primitive: 674>
^ self _primitiveFailed: #_basicNew

]

{ #category : 'Instance creation' }
GsHostProcess class >> _execute: commandLineString input: stdinString [
  "Returns an array of the form { stdout . stderr }
  containing stdout and stderr from the child.

  Otherwise see GsHostProcess class >> execute:input:"

  | inst |
  (inst := self new) commandLine: commandLineString .
  ^ inst _executeWithInput: stdinString .
]

{ #category : 'Instance creation' }
GsHostProcess class >> basicNew [

"disallowed"

self shouldNotImplement: #basicNew

]

{ #category : 'Instance creation' }
GsHostProcess class >> execute: commandLineString [
  "Execute the command in a child process. For details, see comments in the 
  method execute:input:args:"

 ^ self execute: commandLineString input: nil args: nil 
]

{ #category : 'Instance creation' }
GsHostProcess class >> execute: commandLineString args: anArray [
  "Execute the command in a child process, passing in the arguments in anArray. 
  For details, see comments in the method execute:input:args:"

 ^ self execute: commandLineString input: nil args: anArray 
]

{ #category : 'Instance creation' }
GsHostProcess class >> execute: commandLineString input: stdinString [
  "Execute the command in a child process, writing stdinString to the child's
  stdin. For details, see comments in the method execute:input:args:"

 ^ self execute: commandLineString input: stdinString args: nil 
]

{ #category : 'Instance creation' }
GsHostProcess class >> execute: commandLineString input: stdinString args: anArray [
 "Executes commandLineString in a child process,
  writing stdinString to the stdin of the child.
  stdinString may be nil, in which case nothing is written to stdin of child
  If child returns non-zero exit status, signals an Error using
  contents of stderr from the child.  Otherwise returns a String
  containing stdout from the child.
  If child produces exit status
  zero and non-empty stderr the result is prefixed with
  'WARNING: <contents of stderr>' .
  Lookup in the PATH environment variable is not performed,
  the commandLineString must specify a complete path to an executable
  or script.

  commandLineString is parsed for space separated arguments.
  anArray may be nil or an Array. 
  commandLineString and elements of anArray must each be an instance of String , Unicode7 or Utf8.
  Elements of anArray are appended to the arguments contained in commandLineString, to
  form the total argv array of the child, any quoting or whitespace within an element
  of anArray is passed directly to the child in the corresponding element of child's argv.
  If any argument has complicated quoting or whitespace, 
  commandLineString should contain only the full path to the executable, 
  and arguments should be passed as elements of anArray.

  Signals an error if the privilege NoPerformOnServer is true and
  commandLineString is not present in the allowlist of allowed commands
  for the session's UserProfile. See the methods in UserProfile under
  category PerformOnServer for more information."

  | inst |
  (inst := self new) commandLine: commandLineString ;
                     args: anArray .
  ^ inst executeWithInput: stdinString .
]

{ #category : 'Instance creation' }
GsHostProcess class >> fork: commandLineString [
  "Create a new instance, fork it, and exex the command in the child procss.
  For details, see comments in the method fork:args:"

  ^ self fork: commandLineString args: nil .
]

{ #category : 'Instance creation' }
GsHostProcess class >> fork: commandLineString args: anArray [
" Create a new instance of the receiver, fork a child process,
 and exec the commandLineString in the child.
 Utf16 command lines are not yet supported by the fork primitive.
 Lookup in the PATH environment variable is not performed,
 the commandLineString must specify a complete path to an executable
 or script to be exec'ed .

 WARNING, if you are not using a variant of GsHostProcess>>execute ,
 it is recommended to use one or more of GsHostProcess>>stderrPath: ,
  GsHostProcess>>stdinPath: , GsHostProcess>>stdoutPath:  to
 specify where the child should go for its standard files.  Otherwise the
 child may hang when trying to read or write to a pipe connected to
 the parent process , if the parent is not executing the data read and writes
 implemented in  GsHostProcess>>_executeWithInput:  .

 commandLineString is parsed for space separated arguments.
 anArray may be nil or an Array. 
 commandLineString and elements of anArray must each be an instance of String , Unicode7 or Utf8.
 Elements of anArray are appended to the arguments contained in commandLineString, to
 form the total argv array of the child, any quoting or whitespace within an element
 of anArray is passed directly to the child in the corresponding element of child's argv.
 If any argument has complicated quoting or whitespace, 
 commandLineString should contain only the full path to the executable, 
 and arguments should be passed as elements of anArray.

 If running in a topaz -l process and (GsFile stdin isTerminal == true)
 The child process will ignore SIGINT.

 Signals an error if the privilege NoPerformOnServer is true and
 commandLineString is not present in the allowlist of allowed commands
 for the session's UserProfile. See the methods in UserProfile under
 category PerformOnServer for more information."

  | inst |
  (inst := self _basicNew) commandLine: commandLineString ;
     args: anArray;
     fork .
  ^ inst

]

{ #category : 'Instance creation' }
GsHostProcess class >> new [
  ^ self _basicNew
]

{ #category : 'Private' }
GsHostProcess >> _asPath: sockOrPath [
  sockOrPath ifNil:[ ^ nil ].
  (sockOrPath _isOneByteString or:[ sockOrPath isKindOfClass: Utf8 ])
     ifTrue:[ ^ sockOrPath ].
  ^ nil
]

{ #category : 'Private' }
GsHostProcess >> _asSocket: sockOrPath [
  sockOrPath ifNil:[ ^ nil ].
  (sockOrPath _isOneByteString or:[ sockOrPath isKindOfClass: Utf8 ])
     ifTrue:[ ^ nil ].
  ^ sockOrPath
]

{ #category : 'Private' }
GsHostProcess >> _closeSockets [
  (err isKindOf: GsSocket) ifTrue:[ err close ].
  (out isKindOf: GsSocket) ifTrue:[ out close ].
  (in isKindOf: GsSocket) ifTrue:[ in close ].
]

{ #category : 'Execution' }
GsHostProcess >> _executeWithInput: stdinString [
 "Executes the command line specified by instVar  cmd   in a child process .
  If stdinString ~~ nil, writes stdinString to the child's stdin.
  If child returns non-zero exit status, signals a ChildError using
  status and contents of stderr from the child.  
  Returns an Array of the form { stdout . stderr } .
  containing stdout and stderr from the child.

  If either of stdout, stderr are specified to be files via
  GsHostProcess>>stdoutrPath:, GsHostProcess>>stderrPath: prior to  invoking
  this method, those respective output strings will be empty.

  Lookup in the PATH environment variable is not performed,
  the commandLineString must specify a complete path to an executable
  or script.

  Signals an error if the privilege NoPerformOnServer is true and
  commandLineString is not present in the allowlist of allowed commands
  for the session's UserProfile. See the methods in UserProfile under
  category PerformOnServer for more information."

  | outStr errStr status readDataBlk writeOfs nToWrite 
    stdoutIsSocket stderrSock |
  writeOfs := 1 .
  stdinString ifNil:[
    nToWrite := 0 .
    in ifNil:[ in := '/dev/null' ].
  ] ifNotNil:[
    nToWrite := stdinString size .
    (nToWrite > 0 and:[ in ~~ nil ]) ifTrue:[
       Error signal: 'stdinString should be nil or empty after GsHostProcess>>stdinPath: sent'.
    ]
  ].
  stdoutIsSocket := out == nil .
  self fork.  "creates the GsSockets for pipes"
  stderrSock := self _stderr .
  outStr := String new .
  errStr := String new .
  readDataBlk := [ :timeOutMs | 
    stdoutIsSocket ifTrue:[
      self _readFromSocket: out into: outStr timeout: timeOutMs .
    ].
    self _readFromSocket: stderrSock into: errStr timeout: timeOutMs .
  ].
  [ status := self childStatus .
    status == nil
  ] whileTrue:[
    (nToWrite > 0 and:[ in writeWillNotBlock]) ifTrue:[ | nWrote |
      nWrote := in write: nToWrite from: stdinString startingAt: writeOfs .
      nWrote ifNil:[ Error signal:'lost stdin' , in lastErrorString ].
      writeOfs := writeOfs + nWrote .
      nToWrite := nToWrite - nWrote .
      nToWrite == 0 ifTrue:[ in close ].
    ].
    readDataBlk value: nil .  "read while waiting for child exit, so pipes don't block"
    Delay waitForMilliseconds: 10 .
  ].
  readDataBlk value: 5000 .  "final read allows 5 secs for data to arrive"
  self _closeSockets .  "close file handles sooner than finalization would"
  status ~~ 0 ifTrue:[ | ex |
    (ex := ChildError new) status: status ; stderr: errStr ; stdout: outStr .
    ex signal.
  ].
  ^ { outStr . errStr } .

]

{ #category : 'Execution' }
GsHostProcess >> _fork: argsArray [

"Forks the child process specified by the instVars. 
 The instVar cmd must be a valid String or Utf8 containing full path of command
 to execute.
 The instVars in , out , err of the receiver , if nil , specify that each is to
 be opened as a pipe , with a GsSocket connected to the pipe.
 The instVars in , out , err of the receiver , if not nil , must each be a String 
 or a Utf8 specifying a valid path to redirect the corresponding standard file . 
 A path may be '/dev/null' for no-access semantics or '/dev/zero'  
 for output ignored semantics.

 For  out and err , file will be created using the specified path .  
 If both are to be redirected to the same file,  values of out and err should be equal;
 they may be identical.

 For in, if not nil, the file must exist for opening as stdin .

 If instVar appendToFiles == true,  then out and/or err specifing a file will be opened
 for append , otherwise out and/or err to a file will create a new file.

 commandLineString is parsed for space separated arguments.
 anArray may be nil or an Array. 
 commandLineString and elements of anArray must each be an instance of String , Unicode7 or Utf8.
 Elements of anArray are appended to the arguments contained in commandLineString, to
 form the total argv array of the child, any quoting or whitespace within an element
 of anArray is passed directly to the child in the corresponding element of child's argv.
 If any argument has complicated quoting or whitespace, 
 commandLineString should contain only the full path to the executable, 
 and arguments should be passed as elements of anArray.


 WARNING, if standard files have not been redirected to the filesystem 
 by use of one or more of GsHostProcess>>stderrPath: ,
  GsHostProcess>>stdinPath: , GsHostProcess>>stdoutPath: ,
 the child may hang when trying to read or write to a pipe connected to
 the parent process , if the parent is not executing the data read and writes
 equivalent to those implemented in  GsHostProcess>>_executeWithInput:  .

 If running in a topaz -l process and (GsFile stdin isTerminal == true)
 The child process will ignore SIGINT.

 Signals an error if the privilege NoPerformOnServer is true and
 commandLineString is not present in the allowlist of allowed commands
 for the session's UserProfile. See the methods in UserProfile under
 category PerformOnServer for more information."

 <primitive: 956>
 (cmd _isOneByteString or:[ cmd isKindOfClass: Utf8 ]) ifFalse:[
   ArgumentTypeError signal:'cmd must be a String or Utf8'.
 ].
 argsArray ifNotNil:[
   argsArray _validateInstanceOf: Array .
   1 to: argsArray size do:[:n | | elem |
     elem := argsArray at: n .
     (elem _isOneByteString or:[ elem isKindOfClass: Utf8 ]) ifFalse:[
       ArgumentTypeError signal:'argument must be a String or Utf8'.
     ].
   ].
 ].
 self _primitiveFailed: #_fork: args: { argsArray }
]

{ #category : 'Status' }
GsHostProcess >> _killChild: timeoutSeconds [
  "Attempts to kill the child. Waits specified time for child to go way.
   Returns child status."
  | s endMs |
  endMs := DateTime now asMillisecondsGmt + (1000 * timeoutSeconds).
  (s := childStatus) ifNil:[
     self _waitChild: true . "kill with SIGTERM"
     [
       Delay waitForMilliseconds:10.
       (s := self childStatus) ifNotNil:[ ^ s ].
       DateTime now asMillisecondsGmt < endMs
     ] whileTrue .
  ].
  ^ s

]

{ #category : 'Private' }
GsHostProcess >> _readFromSocket: aSocket into: aString timeout: timeoutMs [
  "If timeoutMs not nil, waits for up to that time for data.  
   If socket not ready and timeoutMs==nil , returns number of bytes read,
   otherwise waits for timeoutMs .
   Returns total number of bytes read in this call."
  | total |
  total := 0 .
  [ true ] whileTrue:[ | nRead |
    nRead := aSocket _readInto: aString startingAt: aString size + 1  maxBytes: 16272 .
    nRead _isSmallInteger ifTrue:[
      nRead == 0 ifTrue:[ ^ total ] .
      total := total + nRead . 
    ] ifFalse:[
      nRead == true ifTrue:[ 
        "got EINTR, retry"
      ] ifFalse:[
        nRead == false "EWOULDBLOCK" ifTrue:[  | status |
          timeoutMs ifNil:[ ^ total "caller will try again"].
          status := aSocket readWillNotBlockWithin: timeoutMs .
          status ifNil:[ ^ aSocket signalError ].
          status ifFalse:[ ^ Error signal:'timeout reading from pipe to child'].
        ] ifFalse:[
          "aSocket is a GsSignalingSocket, _readInto: prim should signal socket errors directly" 
          Error signal:'Unexpected nRead ', nRead asString .
        ].
      ].
    ]
  ]
]

{ #category : 'Private' }
GsHostProcess >> _stderr [
  "After fork: , returns a non-blocking read-only GsSocket
  representing the parent's end of a pipe (see   man 2 pipe )
  connected to stderr of the child.  If the GsSocket is not read
  from sufficiently to consume data, the child process may block
  waiting to write more data to its stderr .

  The instVar errSocket is parent's end of a pipe used to read 
  data for details of failure to open a stdin,stdout or stderr file
  during the fork.  Once the child has successfully opened all files,
  no more data will be written to this socket.  "

  ^ (self _asSocket: err) ifNil:[ errSocket ].
]

{ #category : 'Private' }
GsHostProcess >> _waitChild: killBoolean [

"calls waitpid() .
 If child is still running and killBoolean == true, attempts to kill
 the child with a SIGTERM.
 The return status is computed before attempting any kill of the child.

 Returns nil if child still running, otherwise returns a SmallInteger
 representing child status from waitpid() .

  result == 0 means child exited with exit code of 0 .

  ((result bitAnd: 16r100) ~~ 0)  means child exited due to a signal
     and  (result bitAnd: 16rFF) is the number of the signal .

  ((result bitAnd: 16r200) ~~ 0) means child was stopped by a signal
     and  (result bitAnd: 16rFF) is the number of the signal .

  otherwize non-zero result means child exited with that error status.
"
<primitive: 957>
killBoolean _validateClass: Boolean .
self _primitiveFailed: #_waitChild: args: { killBoolean }

]

{ #category : 'Accessing' }
GsHostProcess >> appendToFiles: aBoolean [
  "If aBoolean == true and out or err are paths for files to be opened, 
   causes already existing files to be opened for append. "
  aBoolean _validateClass: Boolean .
  appendToFiles := aBoolean .
]

{ #category : 'Accessing' }
GsHostProcess >> args: anArray [
  self dynamicInstVarAt: #args put: anArray
]

{ #category : 'Status' }
GsHostProcess >> childHasExited [
| s |
 s := self childStatus .
 s ifNil:[ ^ false "still running"].
 ^ (s  bitAnd: 16r200) == 0   "if not stopped with a signal it has exited"

]

{ #category : 'Status' }
GsHostProcess >> childStatus [
  "Returns nil if the child has not been reaped,
   or returns a SmallInteger as follows

  result == 0 means child exited with exit code of 0 .

  ((result bitAnd: 16r100) ~~ 0)  means child exited due to a signal
     and  (result bitAnd: 16rFF) is the number of the signal .

  ((result bitAnd: 16r200) ~~ 0) means child was stopped by a signal
     and  (result bitAnd: 16rFF) is the number of the signal .

  otherwize non-zero result means child exited with that error status."

  | s |
  (s := childStatus) ifNil:[
    s := self _waitChild: false  .
    s ifNotNil:[
      (s bitAnd: 16r200) == 0 ifTrue:[  "child exited"
        childStatus := s .
      ]
    ].
  ].
  ^ s

]

{ #category : 'Accessing' }
GsHostProcess >> commandLine [
  "After fork: , returns the command line that was used to
   exec the child process."
  ^ cmd

]

{ #category : 'Accessing' }
GsHostProcess >> commandLine: aString [
  cmd := aString .
]

{ #category : 'Execution' }
GsHostProcess >> execute [
  ^ self executeWithInput: nil .  
]

{ #category : 'Execution' }
GsHostProcess >> executeWithInput: stdinString [
  | arr outStr errStr result |
  arr := self _executeWithInput: stdinString .
  outStr := arr at: 1 .
  errStr := arr at: 2 .
  errStr size > 0 ifTrue:[
    (result := 'WARNING: <' copy) addAll: errStr asString; add: '> ' ; lf;
      addAll: outStr .
  ] ifFalse:[
    result := outStr
  ].
  ^ result
]

{ #category : 'Execution' }
GsHostProcess >> fork [
  "Fork a child process. See comments in the method _fork:"
  ^ self _fork: (self dynamicInstVarAt: #args)
]

{ #category : 'Status' }
GsHostProcess >> killChild [
  "Attempts to kill the child. Waits one second for child to go way.
   Returns child status. Reading from childs stdout or sterr after
   a kill may block due to no data available. (use GsSocket>>readWillNotBlock)"

^ self  _killChild: 1

]

{ #category : 'Status' }
GsHostProcess >> killChild: timeoutSeconds [
  "Attempts to kill the child. Waits specified time for child to go way.
   Returns child status. Reading from childs stdout or sterr after
   a kill may block due to no data available. (use GsSocket>>readWillNotBlock)"

  ^ self  _killChild: timeoutSeconds

]

{ #category : 'Accessing' }
GsHostProcess >> processId [
  "After fork: , returns a SmallInteger , the operating system
   process id of the child."
  ^ childPid

]

{ #category : 'Accessing' }
GsHostProcess >> readOutErr [
  "assumming both out and err are sockets, attempt to read from both
   and return combined result."
  | outStr |
  outStr := String new .
  self _readFromSocket: out into: outStr timeout: 5000 .
  self _readFromSocket: self _stderr into: outStr timeout: 5000 .
  ^ outStr
]

{ #category : 'Accessing' }
GsHostProcess >> redirectStderrToStdout [
  "This should be used rather than stderrPath:  if you want both stdout and stderr
   redirected to the same file."
  out ifNil:[ Error signal:'redirectStderrToStdout must be preceeded by stdoutPath:'].
  err := out 
]

{ #category : 'Accessing' }
GsHostProcess >> stderr [
  "After fork: , returns a non-blocking read-only GsSocket
  representing the parent's end of a pipe (see   man 2 pipe )
  connected to stderr of the child.  If the GsSocket is not read
  from sufficiently to consume data, the child process may block
  waiting to write more data to its stderr .

  If child's stderr was opened as a file, returns nil"

  ^ self _asSocket: err
]

{ #category : 'Accessing' }
GsHostProcess >> stderrPath [
  ^ self _asPath: err
]

{ #category : 'Accessing' }
GsHostProcess >> stderrPath: aStringOrUtf8 [
  err ifNotNil:[ Error signal:'path for stderr has already been specified.' ].
  (self _asPath: aStringOrUtf8) ifNotNil:[
     err := aStringOrUtf8 .
     ^ self .
  ].
  Error signal:'expected a String or Utf8'.
]

{ #category : 'Accessing' }
GsHostProcess >> stdin [
  "After fork: , returns a non-blocking write-only GsSocket
   representing the parent's end of a pipe (see   man 2 pipe )
   connected to stdin of the child.  The child's end of the pipe
   is blocking and a read by the child on stdin will block until
   Smalltalk code writes sufficient data to this GsSocket
   to satisfy the read.  
   If stdin was opened as a file, returns nil"
  ^ self _asSocket: in 
]

{ #category : 'Accessing' }
GsHostProcess >> stdinPath [
  ^ self _asPath: in
]

{ #category : 'Accessing' }
GsHostProcess >> stdinPath: aStringOrUtf8 [
  "If a child process is not expected to read from stdin,  
   the recommended usage is      stdinPath: '/dev/null'  
   so that the child gets an EOF error if it does try to read from stdin."
  (self _asPath: aStringOrUtf8) ifNotNil:[
     in := aStringOrUtf8 .
     ^ self .
  ].
  Error signal:'expected a String or Utf8'.
]

{ #category : 'Accessing' }
GsHostProcess >> stdout [
  "After fork: , returns a non-blocking read-only GsSocket
  representing the parent's end of a pipe (see   man 2 pipe )
  connected to stdout of the child.  If the GsSocket is not read
  from sufficiently to consume data, the child process may block
  waiting to write more data to its stdout . 

  If stdout was opened as a file, returns nil"

  ^ self _asSocket: out 
]

{ #category : 'Accessing' }
GsHostProcess >> stdoutPath [
  ^ self _asPath: out
]

{ #category : 'Accessing' }
GsHostProcess >> stdoutPath: aStringOrUtf8 [
  out ifNotNil:[ Error signal:'path for stdout has already been specified'].
  (self _asPath: aStringOrUtf8) ifNotNil:[
     out := aStringOrUtf8 .
     ^ self .
  ].
  Error signal:'expected a String or Utf8'.
]
